<?php
/**
 * WooCommerce Cart Customizations
 *
 * Handles member information in cart
 *
 * @package    Karate_Club_Manager
 * @subpackage Karate_Club_Manager/includes/woocommerce
 * @since      1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class MACM_Cart
 *
 * Manages cart customizations for dojo products
 *
 * @since 1.0.0
 */
class MACM_Cart {

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// Add member data to cart item.
		add_filter( 'woocommerce_add_cart_item_data', array( $this, 'add_member_to_cart_item' ), 10, 2 );

		// Display member info in cart.
		add_filter( 'woocommerce_get_item_data', array( $this, 'display_member_in_cart' ), 10, 2 );

		// Load cart item data from session.
		add_filter( 'woocommerce_get_cart_item_from_session', array( $this, 'get_cart_item_from_session' ), 10, 2 );
	}

	/**
	 * Add member ID to cart item data
	 *
	 * This creates unique cart items for the same product with different members
	 *
	 * @since 1.0.0
	 * @param array $cart_item_data Cart item data.
	 * @param int   $product_id     Product ID.
	 * @return array Modified cart item data.
	 */
	public function add_member_to_cart_item( $cart_item_data, $product_id ) {
		// Check if this is a dojo product.
		if ( ! MACM_Products::is_dojo_product( $product_id ) ) {
			return $cart_item_data;
		}

		// Check if member ID is being submitted.
		$member_id_input = isset( $_POST['macm_member_id'] ) ? absint( wp_unslash( $_POST['macm_member_id'] ) ) : 0;

		// Only process if member ID is provided.
		if ( $member_id_input > 0 ) {
			// Verify our member selection nonce - required when member_id is submitted.
			$macm_nonce = isset( $_POST['macm_member_nonce'] )
				? sanitize_text_field( wp_unslash( $_POST['macm_member_nonce'] ) )
				: '';

			if ( ! wp_verify_nonce( $macm_nonce, 'macm_member_selection' ) ) {
				// Invalid nonce - don't add member to cart.
				return $cart_item_data;
			}
			// Get member details.
			$member = MACM_Member::get( $member_id_input );

			if ( $member ) {
				$cart_item_data['macm_member_id']   = $member_id_input;
				$cart_item_data['macm_member_name'] = $member->full_name;

				// This makes each member a unique cart item.
				$cart_item_data['unique_key'] = md5( $product_id . '_' . $member_id_input );
			}
		}

		return $cart_item_data;
	}

	/**
	 * Display member information in cart
	 *
	 * @since 1.0.0
	 * @param array $item_data Item data.
	 * @param array $cart_item Cart item.
	 * @return array Modified item data.
	 */
	public function display_member_in_cart( $item_data, $cart_item ) {
		if ( isset( $cart_item['macm_member_id'] ) && isset( $cart_item['macm_member_name'] ) ) {
			$item_data[] = array(
				'key'   => __( 'For Member', 'martial-arts-club-manager' ),
				'value' => esc_html( $cart_item['macm_member_name'] ),
			);
		}

		return $item_data;
	}

	/**
	 * Load cart item data from session
	 *
	 * @since 1.0.0
	 * @param array $cart_item Cart item.
	 * @param array $values    Values from session.
	 * @return array Modified cart item.
	 */
	public function get_cart_item_from_session( $cart_item, $values ) {
		if ( isset( $values['macm_member_id'] ) ) {
			$cart_item['macm_member_id'] = $values['macm_member_id'];
		}

		if ( isset( $values['macm_member_name'] ) ) {
			$cart_item['macm_member_name'] = $values['macm_member_name'];
		}

		return $cart_item;
	}
}

// Initialize.
new MACM_Cart();
