<?php
/**
 * Training Videos Management
 *
 * Handles the training videos page with capability-based access control
 *
 * @package    Karate_Club_Manager
 * @subpackage Karate_Club_Manager/includes/public
 * @since      1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class MACM_Training_Videos
 *
 * Manages training videos page and access control
 *
 * @since 1.0.0
 */
class MACM_Training_Videos {

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_shortcode( 'macm_training_videos', array( $this, 'render_shortcode' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_assets' ) );
	}

	/**
	 * Enqueue frontend assets
	 *
	 * @since 1.0.27
	 */
	public function enqueue_assets() {
		// Only enqueue on training videos page.
		if ( ! is_page() ) {
			return;
		}

		global $post;
		if ( ! $post || ! has_shortcode( $post->post_content, 'macm_training_videos' ) ) {
			return;
		}

		// Enqueue CSS.
		wp_enqueue_style(
			'kcm-training-videos',
			MACM_PLUGIN_URL . 'assets/css/training-videos.css',
			array(),
			MACM_VERSION
		);

		// Enqueue JS.
		wp_enqueue_script(
			'kcm-training-videos',
			MACM_PLUGIN_URL . 'assets/js/training-videos.js',
			array( 'jquery' ),
			MACM_VERSION,
			true
		);

		// Localize script.
		wp_localize_script(
			'kcm-training-videos',
			'macmVideosPublic',
			array(
				'ajaxurl' => admin_url( 'admin-ajax.php' ),
			)
		);
	}

	/**
	 * Render training videos shortcode
	 *
	 * @since 1.0.0
	 * @return string HTML output.
	 */
	public function render_shortcode() {
		ob_start();

		// Check if user is logged in.
		if ( ! is_user_logged_in() ) {
			$this->render_login_required();
			return ob_get_clean();
		}

		// Check if user has access based on page access rules.
		// This will check if the user has at least one member with the required membership type.
		$page_id = get_the_ID();
		if ( $page_id && ! MACM_Page_Access::check_user_access( $page_id, get_current_user_id() ) ) {
			$this->render_access_denied();
			return ob_get_clean();
		}

		// User has access - render content.
		$this->render_content();

		return ob_get_clean();
	}

	/**
	 * Render login required message
	 *
	 * @since 1.0.0
	 */
	private function render_login_required() {
		$template_path = MACM_PLUGIN_DIR . 'templates/public/training-videos-login-required.php';

		if ( file_exists( $template_path ) ) {
			include $template_path;
		} else {
			// Fallback if template doesn't exist.
			echo '<div class="kcm-message kcm-message-info">';
			echo '<h2>' . esc_html__( 'Login Required', 'martial-arts-club-manager' ) . '</h2>';
			echo '<p>' . esc_html__( 'You must be logged in to view training videos.', 'martial-arts-club-manager' ) . '</p>';
			echo '<p><a href="' . esc_url( wp_login_url( get_permalink() ) ) . '" class="button button-primary">' . esc_html__( 'Login', 'martial-arts-club-manager' ) . '</a></p>';
			echo '</div>';
		}
	}

	/**
	 * Render access denied message
	 *
	 * @since 1.0.0
	 */
	private function render_access_denied() {
		$template_path = MACM_PLUGIN_DIR . 'templates/public/training-videos-access-denied.php';

		if ( file_exists( $template_path ) ) {
			include $template_path;
		} else {
			// Fallback if template doesn't exist.
			echo '<div class="kcm-message kcm-message-warning">';
			echo '<h2>' . esc_html__( 'Premium Membership Required', 'martial-arts-club-manager' ) . '</h2>';
			echo '<p>' . esc_html__( 'Access to training videos is only available to premium members.', 'martial-arts-club-manager' ) . '</p>';
			echo '<p>' . esc_html__( 'Please contact the club administrator to upgrade your membership.', 'martial-arts-club-manager' ) . '</p>';
			echo '</div>';
		}
	}

	/**
	 * Render training videos content
	 *
	 * @since 1.0.0
	 */
	private function render_content() {
		$template_path = MACM_PLUGIN_DIR . 'templates/public/training-videos-content.php';

		if ( file_exists( $template_path ) ) {
			include $template_path;
		} else {
			// Fallback if template doesn't exist.
			echo '<div class="kcm-training-videos">';
			echo '<h2>' . esc_html__( 'Training Videos', 'martial-arts-club-manager' ) . '</h2>';
			echo '<p>' . esc_html__( 'Welcome to the premium training videos section!', 'martial-arts-club-manager' ) . '</p>';
			echo '<p class="description">' . esc_html__( 'This is a placeholder. Add your video content here.', 'martial-arts-club-manager' ) . '</p>';
			echo '</div>';
		}
	}
}

// Initialize - PREMIUM feature.
// This block will be removed from the free version by Freemius.
if ( function_exists( 'macm_fs' ) && macm_fs()->is__premium_only() ) {
	if ( macm_fs()->can_use_premium_code() ) {
		new MACM_Training_Videos();
	}
}
