<?php
/**
 * Public-facing functionality
 *
 * @package KarateClubManager
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Public class
 */
class MACM_Public {

	/**
	 * Sanitize a $_FILES array entry for secure processing.
	 *
	 * Validates structure, sanitizes filename, and verifies upload.
	 *
	 * @param array $file The $_FILES array entry to sanitize.
	 * @return array|WP_Error Sanitized file array or WP_Error on failure.
	 */
	private function sanitize_file_upload( $file ) {
		// Validate required keys exist.
		$required_keys = array( 'name', 'type', 'tmp_name', 'error', 'size' );
		foreach ( $required_keys as $key ) {
			if ( ! isset( $file[ $key ] ) ) {
				return new WP_Error( 'invalid_file', __( 'Invalid file upload structure.', 'martial-arts-club-manager' ) );
			}
		}

		// Check for upload errors.
		if ( UPLOAD_ERR_OK !== $file['error'] ) {
			return new WP_Error( 'upload_error', __( 'File upload error.', 'martial-arts-club-manager' ) );
		}

		// Verify this is an actual uploaded file (security check).
		if ( ! is_uploaded_file( $file['tmp_name'] ) ) {
			return new WP_Error( 'not_uploaded', __( 'File is not a valid upload.', 'martial-arts-club-manager' ) );
		}

		// Sanitize the filename.
		$sanitized_name = sanitize_file_name( $file['name'] );
		if ( empty( $sanitized_name ) ) {
			return new WP_Error( 'invalid_filename', __( 'Invalid filename.', 'martial-arts-club-manager' ) );
		}

		// Return sanitized file array.
		return array(
			'name'     => $sanitized_name,
			'type'     => sanitize_mime_type( $file['type'] ),
			'tmp_name' => $file['tmp_name'], // Cannot sanitize - system path.
			'error'    => absint( $file['error'] ),
			'size'     => absint( $file['size'] ),
		);
	}
	/**
	 * Constructor
	 */
	public function __construct() {
		// Register shortcodes.
		add_shortcode( 'macm_member_area', array( $this, 'member_area_shortcode' ) );

		// AJAX handlers for frontend member management.
		add_action( 'wp_ajax_macm_add_member', array( $this, 'ajax_add_member' ) );
		add_action( 'wp_ajax_macm_edit_member', array( $this, 'ajax_edit_member' ) );
		add_action( 'wp_ajax_macm_delete_member', array( $this, 'ajax_delete_member' ) );
		add_action( 'wp_ajax_macm_upload_photo', array( $this, 'ajax_upload_photo' ) );
		add_action( 'wp_ajax_macm_delete_photo', array( $this, 'ajax_delete_photo' ) );
		add_action( 'wp_ajax_macm_get_member', array( $this, 'ajax_get_member' ) );
		add_action( 'wp_ajax_macm_get_member_classes', array( $this, 'ajax_get_member_classes' ) );

		// PREMIUM: Events AJAX handler - only register if premium features available.
		if ( function_exists( 'macm_fs' ) && macm_fs()->can_use_premium_code() ) {
			add_action( 'wp_ajax_macm_get_member_events', array( $this, 'ajax_get_member_events' ) );
			add_action( 'wp_ajax_macm_get_member_belt_history', array( $this, 'ajax_get_member_belt_history' ) );
		}
	}

	/**
	 * Enqueue public styles
	 */
	public function enqueue_styles() {
		// Enqueue Dashicons for frontend (needed for member area icons).
		wp_enqueue_style( 'dashicons' );

		wp_enqueue_style(
			'kcm-public-styles',
			MACM_PLUGIN_URL . 'assets/css/public.css',
			array( 'dashicons' ),
			MACM_VERSION,
			'all'
		);

		// PREMIUM: Enqueue class schedule styles (used by shortcode) - only if premium features available.
		if ( function_exists( 'macm_fs' ) && macm_fs()->can_use_premium_code() ) {
			wp_enqueue_style(
				'kcm-class-schedule',
				MACM_PLUGIN_URL . 'assets/css/class-schedule.css',
				array( 'dashicons' ),
				MACM_VERSION,
				'all'
			);
		}

		// Add dynamic button gradient CSS.
		$button_gradient_start_raw = get_option( 'macm_button_gradient_start', '#B11226' );
		$button_gradient_end_raw   = get_option( 'macm_button_gradient_end', '#8F0E1E' );

		// Sanitize hex colors - fall back to defaults if invalid.
		$button_gradient_start = sanitize_hex_color( $button_gradient_start_raw );
		$button_gradient_end   = sanitize_hex_color( $button_gradient_end_raw );

		if ( ! $button_gradient_start ) {
			$button_gradient_start = '#B11226';
		}
		if ( ! $button_gradient_end ) {
			$button_gradient_end = '#8F0E1E';
		}

		$custom_css = sprintf(
			'.kcm-submit-button,
			.kcm-btn-primary {
				background: linear-gradient(135deg, %s 0%%, %s 100%%) !important;
			}',
			esc_attr( $button_gradient_start ),
			esc_attr( $button_gradient_end )
		);

		wp_add_inline_style( 'kcm-public-styles', $custom_css );
	}

	/**
	 * Enqueue public scripts
	 */
	public function enqueue_scripts() {
		wp_enqueue_script(
			'kcm-public-scripts',
			MACM_PLUGIN_URL . 'assets/js/public.js',
			array( 'jquery' ),
			MACM_VERSION,
			true
		);

		// PREMIUM: Enqueue class schedule scripts (used by shortcode) - only if premium features available.
		if ( function_exists( 'macm_fs' ) && macm_fs()->can_use_premium_code() ) {
			wp_enqueue_script(
				'kcm-class-schedule',
				MACM_PLUGIN_URL . 'assets/js/class-schedule.js',
				array( 'jquery' ),
				MACM_VERSION,
				true
			);
		}

		// Localize script for AJAX.
		wp_localize_script(
			'kcm-public-scripts',
			'macmPublic',
			array(
				'ajaxurl' => admin_url( 'admin-ajax.php' ),
				'nonce'   => wp_create_nonce( 'kcm-public-nonce' ),
				'strings' => array(
					'confirmDelete'       => __( 'Are you sure you want to delete this member? This action cannot be undone.', 'martial-arts-club-manager' ),
					'error'               => __( 'An error occurred. Please try again.', 'martial-arts-club-manager' ),
					'success'             => __( 'Operation completed successfully.', 'martial-arts-club-manager' ),
					'myClassesTitle'      => __( 'My Enrolled Classes', 'martial-arts-club-manager' ),
					'myEventsTitle'       => __( 'My Registered Events', 'martial-arts-club-manager' ),
					'loadingClasses'      => __( 'Loading classes...', 'martial-arts-club-manager' ),
					'loadingEvents'       => __( 'Loading events...', 'martial-arts-club-manager' ),
					'noClassesEnrolled'   => __( 'This member is not enrolled in any classes yet.', 'martial-arts-club-manager' ),
					'noEventsRegistered'  => __( 'This member has not registered for any events yet.', 'martial-arts-club-manager' ),
					'schedule'            => __( 'Schedule', 'martial-arts-club-manager' ),
					'location'            => __( 'Location', 'martial-arts-club-manager' ),
					'instructor'          => __( 'Instructor', 'martial-arts-club-manager' ),
					'enrolledDate'        => __( 'Enrolled Date', 'martial-arts-club-manager' ),
					'eventDates'          => __( 'Event Dates', 'martial-arts-club-manager' ),
					'description'         => __( 'Description', 'martial-arts-club-manager' ),
					'registeredDate'      => __( 'Registered Date', 'martial-arts-club-manager' ),
					'currentFutureEvents' => __( 'Current & Future', 'martial-arts-club-manager' ),
					'pastEvents'          => __( 'Past Events', 'martial-arts-club-manager' ),
					'noCurrentEvents'     => __( 'This member has no current or upcoming events.', 'martial-arts-club-manager' ),
					'noPastEvents'        => __( 'This member has no past events.', 'martial-arts-club-manager' ),
					// Belt History (Premium).
					'beltHistoryTitle'    => __( 'Belt Grading History', 'martial-arts-club-manager' ),
					'loadingBeltHistory'  => __( 'Loading belt history...', 'martial-arts-club-manager' ),
					'noBeltHistory'       => __( 'No grading history found for this member.', 'martial-arts-club-manager' ),
					'gradingDate'         => __( 'Date', 'martial-arts-club-manager' ),
					'beltColor'           => __( 'Belt Color', 'martial-arts-club-manager' ),
					'examiner'            => __( 'Examiner', 'martial-arts-club-manager' ),
					'score'               => __( 'Score', 'martial-arts-club-manager' ),
					'notes'               => __( 'Notes', 'martial-arts-club-manager' ),
					'exportCsv'           => __( 'Export CSV', 'martial-arts-club-manager' ),
				),
			)
		);
	}

	/**
	 * Member area shortcode
	 *
	 * Usage: [macm_member_area]
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string HTML output.
	 */
	public function member_area_shortcode( $atts ) {
		// Check if user is logged in.
		if ( ! is_user_logged_in() ) {
			return '<div class="kcm-notice kcm-notice-error">' .
					'<p>' . __( 'You must be logged in to view this page.', 'martial-arts-club-manager' ) . '</p>' .
					'<p><a href="' . esc_url( wp_login_url( get_permalink() ) ) . '">' . __( 'Log in', 'martial-arts-club-manager' ) . '</a></p>' .
					'</div>';
		}

		// Get current user's members.
		$user_id = get_current_user_id();
		$members = MACM_Member::get_by_user( $user_id );

		// Start output buffering.
		ob_start();

		// Include template.
		include MACM_PLUGIN_DIR . 'templates/member-area.php';

		return ob_get_clean();
	}

	/**
	 * AJAX handler: Add member
	 */
	public function ajax_add_member() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'kcm-public-nonce', 'nonce', false ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'martial-arts-club-manager' ) ) );
		}

		// Check if user is logged in.
		if ( ! is_user_logged_in() ) {
			wp_send_json_error( array( 'message' => __( 'You must be logged in.', 'martial-arts-club-manager' ) ) );
		}

		// Get form data.
		$data = array(
			'full_name'          => isset( $_POST['full_name'] ) ? sanitize_text_field( wp_unslash( $_POST['full_name'] ) ) : '',
			'date_of_birth'      => isset( $_POST['date_of_birth'] ) ? sanitize_text_field( wp_unslash( $_POST['date_of_birth'] ) ) : '',
			'belt_color'         => isset( $_POST['belt_color'] ) ? sanitize_text_field( wp_unslash( $_POST['belt_color'] ) ) : '',
			'weight'             => isset( $_POST['weight'] ) && '' !== $_POST['weight'] ? MACM_Member::weight_to_kg( floatval( wp_unslash( $_POST['weight'] ) ) ) : '',
			'height'             => isset( $_POST['height'] ) && '' !== $_POST['height'] ? MACM_Member::height_to_cm( floatval( wp_unslash( $_POST['height'] ) ) ) : '',
			'license_number'     => isset( $_POST['license_number'] ) ? sanitize_text_field( wp_unslash( $_POST['license_number'] ) ) : '',
			'license_expiration' => isset( $_POST['license_expiration'] ) ? sanitize_text_field( wp_unslash( $_POST['license_expiration'] ) ) : '',
		);

		// Validate: license_expiration requires license_number.
		if ( ! empty( $data['license_expiration'] ) && empty( $data['license_number'] ) ) {
			wp_send_json_error(
				array(
					'message' => __( 'License expiration date requires a license number. Please enter a license number first.', 'martial-arts-club-manager' ),
				)
			);
		}

		// Add club_id.
		if ( isset( $_POST['club_id'] ) && ! empty( $_POST['club_id'] ) ) {
			$data['club_id'] = absint( wp_unslash( $_POST['club_id'] ) );
		}

		// Add membership type (only admins can set this).
		if ( isset( $_POST['membership_type_id'] ) && current_user_can( 'manage_options' ) ) {
			$membership_type_id = absint( wp_unslash( $_POST['membership_type_id'] ) );
			if ( $membership_type_id > 0 ) {
				$data['membership_type_id'] = $membership_type_id;
			}
		}

		// Create member.
		$user_id   = get_current_user_id();
		$member_id = MACM_Member::create( $user_id, $data );

		if ( is_wp_error( $member_id ) ) {
			wp_send_json_error( array( 'message' => $member_id->get_error_message() ) );
		}

		// Get the created member.
		$member = MACM_Member::get( $member_id );

		wp_send_json_success(
			array(
				'message' => __( 'Member added successfully.', 'martial-arts-club-manager' ),
				'member'  => $member,
			)
		);
	}

	/**
	 * AJAX handler: Edit member
	 */
	public function ajax_edit_member() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'kcm-public-nonce', 'nonce', false ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'martial-arts-club-manager' ) ) );
		}

		// Check if user is logged in.
		if ( ! is_user_logged_in() ) {
			wp_send_json_error( array( 'message' => __( 'You must be logged in.', 'martial-arts-club-manager' ) ) );
		}

		// Get member ID.
		$member_id = isset( $_POST['member_id'] ) ? absint( wp_unslash( $_POST['member_id'] ) ) : 0;
		if ( ! $member_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid member ID.', 'martial-arts-club-manager' ) ) );
		}

		// Get form data.
		$data = array(
			'full_name'          => isset( $_POST['full_name'] ) ? sanitize_text_field( wp_unslash( $_POST['full_name'] ) ) : '',
			'date_of_birth'      => isset( $_POST['date_of_birth'] ) ? sanitize_text_field( wp_unslash( $_POST['date_of_birth'] ) ) : '',
			'belt_color'         => isset( $_POST['belt_color'] ) ? sanitize_text_field( wp_unslash( $_POST['belt_color'] ) ) : '',
			'weight'             => isset( $_POST['weight'] ) && '' !== $_POST['weight'] ? MACM_Member::weight_to_kg( floatval( wp_unslash( $_POST['weight'] ) ) ) : '',
			'height'             => isset( $_POST['height'] ) && '' !== $_POST['height'] ? MACM_Member::height_to_cm( floatval( wp_unslash( $_POST['height'] ) ) ) : '',
			'license_number'     => isset( $_POST['license_number'] ) ? sanitize_text_field( wp_unslash( $_POST['license_number'] ) ) : '',
			'license_expiration' => isset( $_POST['license_expiration'] ) ? sanitize_text_field( wp_unslash( $_POST['license_expiration'] ) ) : '',
		);

		// Validate: license_expiration requires license_number.
		if ( ! empty( $data['license_expiration'] ) && empty( $data['license_number'] ) ) {
			wp_send_json_error(
				array(
					'message' => __( 'License expiration date requires a license number. Please enter a license number first.', 'martial-arts-club-manager' ),
				)
			);
		}

		// Add club_id.
		if ( isset( $_POST['club_id'] ) && ! empty( $_POST['club_id'] ) ) {
			$data['club_id'] = absint( wp_unslash( $_POST['club_id'] ) );
		} else {
			$data['club_id'] = null;
		}

		// Add membership type (only admins can set this).
		if ( isset( $_POST['membership_type_id'] ) && current_user_can( 'manage_options' ) ) {
			$membership_type_id = absint( wp_unslash( $_POST['membership_type_id'] ) );
			if ( $membership_type_id > 0 ) {
				$data['membership_type_id'] = $membership_type_id;
			}
		}

		// Update member.
		$result = MACM_Member::update( $member_id, $data );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		// Get the updated member.
		$member = MACM_Member::get( $member_id );

		wp_send_json_success(
			array(
				'message' => __( 'Member updated successfully.', 'martial-arts-club-manager' ),
				'member'  => $member,
			)
		);
	}

	/**
	 * AJAX handler: Delete member
	 */
	public function ajax_delete_member() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'kcm-public-nonce', 'nonce', false ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'martial-arts-club-manager' ) ) );
		}

		// Check if user is logged in.
		if ( ! is_user_logged_in() ) {
			wp_send_json_error( array( 'message' => __( 'You must be logged in.', 'martial-arts-club-manager' ) ) );
		}

		// Get member ID.
		$member_id = isset( $_POST['member_id'] ) ? absint( wp_unslash( $_POST['member_id'] ) ) : 0;
		if ( ! $member_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid member ID.', 'martial-arts-club-manager' ) ) );
		}

		// Delete member.
		$result = MACM_Member::delete( $member_id );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		wp_send_json_success( array( 'message' => __( 'Member deleted successfully.', 'martial-arts-club-manager' ) ) );
	}

	/**
	 * AJAX handler: Upload photo
	 */
	public function ajax_upload_photo() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'kcm-public-nonce', 'nonce', false ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'martial-arts-club-manager' ) ) );
		}

		// Check if user is logged in.
		if ( ! is_user_logged_in() ) {
			wp_send_json_error( array( 'message' => __( 'You must be logged in.', 'martial-arts-club-manager' ) ) );
		}

		// Get member ID.
		$member_id = isset( $_POST['member_id'] ) ? absint( wp_unslash( $_POST['member_id'] ) ) : 0;
		if ( ! $member_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid member ID.', 'martial-arts-club-manager' ) ) );
		}

		// Check if file was uploaded.
		if ( ! isset( $_FILES['photo'] ) || ! isset( $_FILES['photo']['tmp_name'] ) ) {
			wp_send_json_error( array( 'message' => __( 'No file uploaded.', 'martial-arts-club-manager' ) ) );
		}

		// Build sanitized file array from individual validated parts.
		$photo_file = array(
			'name'     => isset( $_FILES['photo']['name'] ) ? sanitize_file_name( wp_unslash( $_FILES['photo']['name'] ) ) : '',
			'type'     => isset( $_FILES['photo']['type'] ) ? sanitize_mime_type( wp_unslash( $_FILES['photo']['type'] ) ) : '',
			'tmp_name' => isset( $_FILES['photo']['tmp_name'] ) ? sanitize_text_field( wp_unslash( $_FILES['photo']['tmp_name'] ) ) : '',
			'error'    => isset( $_FILES['photo']['error'] ) ? absint( $_FILES['photo']['error'] ) : UPLOAD_ERR_NO_FILE,
			'size'     => isset( $_FILES['photo']['size'] ) ? absint( $_FILES['photo']['size'] ) : 0,
		);

		// Validate the file.
		$sanitized_file = $this->sanitize_file_upload( $photo_file );
		if ( is_wp_error( $sanitized_file ) ) {
			wp_send_json_error( array( 'message' => $sanitized_file->get_error_message() ) );
		}

		// Upload photo with sanitized file data.
		$result = MACM_Member::upload_photo( $member_id, $sanitized_file );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		// Get photo URL.
		$photo_url = wp_get_attachment_image_url( $result, 'thumbnail' );

		wp_send_json_success(
			array(
				'message'   => __( 'Photo uploaded successfully.', 'martial-arts-club-manager' ),
				'photo_id'  => $result,
				'photo_url' => $photo_url,
			)
		);
	}

	/**
	 * AJAX handler: Delete photo
	 */
	public function ajax_delete_photo() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'kcm-public-nonce', 'nonce', false ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'martial-arts-club-manager' ) ) );
		}

		// Check if user is logged in.
		if ( ! is_user_logged_in() ) {
			wp_send_json_error( array( 'message' => __( 'You must be logged in.', 'martial-arts-club-manager' ) ) );
		}

		// Get member ID.
		$member_id = isset( $_POST['member_id'] ) ? absint( wp_unslash( $_POST['member_id'] ) ) : 0;
		if ( ! $member_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid member ID.', 'martial-arts-club-manager' ) ) );
		}

		// Delete photo.
		$result = MACM_Member::delete_photo( $member_id );

		if ( ! $result ) {
			wp_send_json_error( array( 'message' => __( 'Failed to delete photo.', 'martial-arts-club-manager' ) ) );
		}

		wp_send_json_success( array( 'message' => __( 'Photo deleted successfully.', 'martial-arts-club-manager' ) ) );
	}

	/**
	 * AJAX handler: Get member data
	 */
	public function ajax_get_member() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'kcm-public-nonce', 'nonce', false ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'martial-arts-club-manager' ) ) );
		}

		// Check if user is logged in.
		if ( ! is_user_logged_in() ) {
			wp_send_json_error( array( 'message' => __( 'You must be logged in.', 'martial-arts-club-manager' ) ) );
		}

		// Get member ID.
		$member_id = isset( $_GET['member_id'] ) ? absint( wp_unslash( $_GET['member_id'] ) ) : 0;
		if ( ! $member_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid member ID.', 'martial-arts-club-manager' ) ) );
		}

		// Get member.
		$member = MACM_Member::get( $member_id );

		if ( ! $member ) {
			wp_send_json_error( array( 'message' => __( 'Member not found.', 'martial-arts-club-manager' ) ) );
		}

		// Verify user owns this member.
		if ( ! MACM_Member::can_manage_member( $member_id ) ) {
			wp_send_json_error( array( 'message' => __( 'You do not have permission to view this member.', 'martial-arts-club-manager' ) ) );
		}

		// Add membership type name if set.
		if ( $member->membership_type_id ) {
			$membership_type              = MACM_Membership_Type::get( $member->membership_type_id );
			$member->membership_type_name = $membership_type ? $membership_type->type_name : null;
		} else {
			$member->membership_type_name = null;
		}

		// Convert weight/height to display units for form pre-fill.
		if ( $member->weight ) {
			$member->weight_display = MACM_Member::kg_to_display( $member->weight );
		}
		if ( $member->height ) {
			$member->height_display = MACM_Member::cm_to_display( $member->height );
		}

		// Add formatted weight/height for display.
		$member->weight_formatted = MACM_Member::format_weight( $member->weight );
		$member->height_formatted = MACM_Member::format_height( $member->height );

		wp_send_json_success( array( 'member' => $member ) );
	}

	/**
	 * AJAX handler to get member's enrolled classes
	 *
	 * @since 1.0.78
	 */
	public function ajax_get_member_classes() {
		// Verify nonce.
		check_ajax_referer( 'kcm-public-nonce', 'nonce' );

		// Check if user is logged in.
		if ( ! is_user_logged_in() ) {
			wp_send_json_error( array( 'message' => __( 'You must be logged in.', 'martial-arts-club-manager' ) ) );
		}

		// Get member ID.
		$member_id = isset( $_POST['member_id'] ) ? absint( wp_unslash( $_POST['member_id'] ) ) : 0;

		if ( ! $member_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid member ID.', 'martial-arts-club-manager' ) ) );
		}

		// Get member.
		$member = MACM_Member::get( $member_id );
		if ( ! $member ) {
			wp_send_json_error( array( 'message' => __( 'Member not found.', 'martial-arts-club-manager' ) ) );
		}

		// Verify user owns this member.
		if ( ! MACM_Member::can_manage_member( $member_id ) ) {
			wp_send_json_error( array( 'message' => __( 'You do not have permission to view this member.', 'martial-arts-club-manager' ) ) );
		}

		// Get enrolled classes with caching.
		global $wpdb;
		$cache_key   = 'macm_member_enrollments_' . $member_id;
		$enrollments = wp_cache_get( $cache_key, 'macm' );

		if ( false === $enrollments ) {
			$enrollments = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT e.*, c.class_name, c.day_of_week, c.start_time, c.end_time,
					        l.location_name, i.full_name as instructor_name
					 FROM {$wpdb->prefix}macm_class_enrollments e
					 INNER JOIN {$wpdb->prefix}macm_classes c ON e.class_id = c.id
					 LEFT JOIN {$wpdb->prefix}macm_locations l ON c.location_id = l.id
					 LEFT JOIN {$wpdb->prefix}macm_class_instructors ci ON c.id = ci.class_id
					 LEFT JOIN {$wpdb->prefix}macm_instructors i ON ci.instructor_id = i.id
					 WHERE e.member_id = %d
					 AND e.removed_at IS NULL
					 AND c.is_archived = 0
					 ORDER BY c.day_of_week ASC, c.start_time ASC",
					$member_id
				)
			);
			wp_cache_set( $cache_key, $enrollments, 'macm', 300 );
		}

		// Format the data.
		$classes      = array();
		$days_of_week = array(
			0 => __( 'Sunday', 'martial-arts-club-manager' ),
			1 => __( 'Monday', 'martial-arts-club-manager' ),
			2 => __( 'Tuesday', 'martial-arts-club-manager' ),
			3 => __( 'Wednesday', 'martial-arts-club-manager' ),
			4 => __( 'Thursday', 'martial-arts-club-manager' ),
			5 => __( 'Friday', 'martial-arts-club-manager' ),
			6 => __( 'Saturday', 'martial-arts-club-manager' ),
		);

		foreach ( $enrollments as $enrollment ) {
			$classes[] = array(
				'class_name'    => $enrollment->class_name,
				'day'           => $days_of_week[ $enrollment->day_of_week ] ?? '',
				'time'          => wp_date( 'g:i A', strtotime( $enrollment->start_time ) ) . ' - ' . wp_date( 'g:i A', strtotime( $enrollment->end_time ) ),
				'location'      => $enrollment->location_name ?? __( 'Not set', 'martial-arts-club-manager' ),
				'instructor'    => $enrollment->instructor_name ?? __( 'Not assigned', 'martial-arts-club-manager' ),
				'enrolled_date' => date_i18n( get_option( 'date_format' ), strtotime( $enrollment->enrolled_at ) ),
			);
		}

		wp_send_json_success(
			array(
				'classes'     => $classes,
				'member_name' => $member->full_name,
			)
		);
	}

	/**
	 * AJAX handler to get member's registered events
	 *
	 * @since 1.0.78
	 */
	public function ajax_get_member_events() {
		// Verify nonce.
		check_ajax_referer( 'kcm-public-nonce', 'nonce' );

		// Check if user is logged in.
		if ( ! is_user_logged_in() ) {
			wp_send_json_error( array( 'message' => __( 'You must be logged in.', 'martial-arts-club-manager' ) ) );
		}

		// Get member ID.
		$member_id = isset( $_POST['member_id'] ) ? absint( wp_unslash( $_POST['member_id'] ) ) : 0;

		if ( ! $member_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid member ID.', 'martial-arts-club-manager' ) ) );
		}

		// Get member.
		$member = MACM_Member::get( $member_id );
		if ( ! $member ) {
			wp_send_json_error( array( 'message' => __( 'Member not found.', 'martial-arts-club-manager' ) ) );
		}

		// Verify user owns this member.
		if ( ! MACM_Member::can_manage_member( $member_id ) ) {
			wp_send_json_error( array( 'message' => __( 'You do not have permission to view this member.', 'martial-arts-club-manager' ) ) );
		}

		// Get registered events with caching.
		global $wpdb;
		$cache_key     = 'macm_member_events_' . $member_id;
		$registrations = wp_cache_get( $cache_key, 'macm' );

		if ( false === $registrations ) {
			$registrations = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT r.*, e.title, e.start_date, e.end_date, e.location,
					        e.description, e.closing_date
					 FROM {$wpdb->prefix}macm_event_registrations r
					 INNER JOIN {$wpdb->prefix}macm_events e ON r.event_id = e.id
					 WHERE r.member_id = %d
					 AND e.status = 'active'
					 ORDER BY e.start_date ASC",
					$member_id
				)
			);
			wp_cache_set( $cache_key, $registrations, 'macm', 300 );
		}

		// Format the data.
		$events = array();
		$today  = current_time( 'Y-m-d' );

		foreach ( $registrations as $registration ) {
			$is_past     = $registration->end_date < $today;
			$is_upcoming = $registration->start_date > $today;

			$status_label = '';
			if ( $is_past ) {
				$status_label = __( 'Past', 'martial-arts-club-manager' );
			} elseif ( $is_upcoming ) {
				$status_label = __( 'Upcoming', 'martial-arts-club-manager' );
			} else {
				$status_label = __( 'Ongoing', 'martial-arts-club-manager' );
			}

			$events[] = array(
				'title'           => $registration->title,
				'start_date'      => date_i18n( get_option( 'date_format' ), strtotime( $registration->start_date ) ),
				'end_date'        => date_i18n( get_option( 'date_format' ), strtotime( $registration->end_date ) ),
				'location'        => $registration->location ?? __( 'Not set', 'martial-arts-club-manager' ),
				'description'     => $registration->description ?? '',
				'registered_date' => date_i18n( get_option( 'date_format' ), strtotime( $registration->registration_date ) ),
				'status'          => $status_label,
				'is_past'         => $is_past,
				'is_upcoming'     => $is_upcoming,
			);
		}

		wp_send_json_success(
			array(
				'events'      => $events,
				'member_name' => $member->full_name,
			)
		);
	}

	/**
	 * AJAX handler to get member's belt grading history (PREMIUM)
	 *
	 * @since 1.0.114
	 */
	public function ajax_get_member_belt_history() {
		// Verify nonce.
		check_ajax_referer( 'kcm-public-nonce', 'nonce' );

		// Check if user is logged in.
		if ( ! is_user_logged_in() ) {
			wp_send_json_error( array( 'message' => __( 'You must be logged in.', 'martial-arts-club-manager' ) ) );
		}

		// Get member ID.
		$member_id = isset( $_POST['member_id'] ) ? absint( wp_unslash( $_POST['member_id'] ) ) : 0;

		if ( ! $member_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid member ID.', 'martial-arts-club-manager' ) ) );
		}

		// Get member.
		$member = MACM_Member::get( $member_id );
		if ( ! $member ) {
			wp_send_json_error( array( 'message' => __( 'Member not found.', 'martial-arts-club-manager' ) ) );
		}

		// Verify user owns this member.
		if ( ! MACM_Member::can_manage_member( $member_id ) ) {
			wp_send_json_error( array( 'message' => __( 'You do not have permission to view this member.', 'martial-arts-club-manager' ) ) );
		}

		// Check if grading history class exists (premium feature).
		if ( ! class_exists( 'MACM_Grading_History' ) ) {
			wp_send_json_error( array( 'message' => __( 'This feature requires a premium license.', 'martial-arts-club-manager' ) ) );
		}

		// Get belt history.
		$history = MACM_Grading_History::get_member_history( $member_id, array( 'order' => 'DESC' ) );

		// Format the data.
		$records = array();
		foreach ( $history as $record ) {
			$examiner = '';
			if ( ! empty( $record->instructor_name ) ) {
				$examiner = $record->instructor_name;
			} elseif ( ! empty( $record->examiner_name ) ) {
				$examiner = $record->examiner_name;
			}

			$records[] = array(
				'belt_color_key' => $record->belt_color_key,
				'color_name'     => $record->color_name ?? $record->belt_color_key,
				'grading_date'   => date_i18n( get_option( 'date_format' ), strtotime( $record->grading_date ) ),
				'examiner'       => $examiner,
				'score'          => $record->score ?? '',
				'notes'          => $record->notes ?? '',
			);
		}

		// Generate export URL.
		$export_url = add_query_arg(
			array(
				'action'    => 'macm_export_grading_csv',
				'member_id' => $member_id,
				'_wpnonce'  => wp_create_nonce( 'macm_export_grading_csv' ),
			),
			admin_url( 'admin.php' )
		);

		wp_send_json_success(
			array(
				'records'     => $records,
				'member_name' => $member->full_name,
				'export_url'  => $export_url,
			)
		);
	}
}
