<?php
/**
 * Access Control Class
 *
 * Handles page-level access control for the plugin.
 * Restricts access to selected pages based on login status.
 *
 * @package    Karate_Club_Manager
 * @subpackage Karate_Club_Manager/includes/public
 * @since      1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * KCM Access Control Class
 *
 * Manages access control for WordPress pages.
 *
 * @since 1.0.0
 */
class MACM_Access_Control {

	/**
	 * Initialize the class and set up hooks.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'template_redirect', array( $this, 'check_page_access' ), 10 );
	}

	/**
	 * Check if current page requires login and handle access control.
	 *
	 * Hooks into template_redirect to check page access before rendering.
	 * If page is protected and user does not have access:
	 * - Not logged in: redirect to login
	 * - Logged in but no access: let the page load (shortcode will show access denied message)
	 *
	 * @since 1.0.0
	 */
	public function check_page_access() {
		// Only check for pages (not posts or other content types).
		if ( ! is_page() ) {
			return;
		}

		// Get the current page ID.
		$current_page_id = get_the_ID();

		// Check if page is protected.
		if ( ! MACM_Page_Access::is_page_protected( $current_page_id ) ) {
			return;
		}

		// Check if current user has access.
		$current_user_id = get_current_user_id();
		if ( MACM_Page_Access::check_user_access( $current_page_id, $current_user_id ) ) {
			return;
		}

		// User does not have access.
		// If not logged in, redirect to login.
		// If logged in, let the page load so the shortcode can show a proper access denied message.
		if ( 0 === $current_user_id ) {
			$this->redirect_to_login( $current_page_id );
		}
		// Otherwise, do nothing and let the shortcode handle the access denied message.
	}

	/**
	 * Redirect user to login page with return URL.
	 *
	 * @since 1.0.0
	 * @param int $page_id The ID of the protected page.
	 */
	private function redirect_to_login( $page_id ) {
		// Get the permalink of the current page for return URL.
		$redirect_url = get_permalink( $page_id );

		// Build login URL with redirect parameter.
		$login_url = wp_login_url( $redirect_url );

		/**
		 * Filter the login redirect URL.
		 *
		 * Allows modification of the login URL and redirect parameter.
		 *
		 * @since 1.0.0
		 * @param string $login_url    The complete login URL with redirect parameter.
		 * @param string $redirect_url The URL to redirect to after login.
		 * @param int    $page_id      The ID of the protected page.
		 */
		$login_url = apply_filters( 'macm_access_control_login_url', $login_url, $redirect_url, $page_id );

		// Perform redirect.
		wp_safe_redirect( $login_url );
		exit;
	}

	/**
	 * Check if a specific page ID is protected.
	 *
	 * Utility method to check if a page requires login or has access rules.
	 *
	 * @since 1.0.0
	 * @param int $page_id The page ID to check.
	 * @return bool True if page is protected, false otherwise.
	 */
	public static function is_page_protected( $page_id ) {
		return MACM_Page_Access::is_page_protected( $page_id );
	}

	/**
	 * Get all protected page IDs.
	 *
	 * @since 1.0.0
	 * @return array Array of protected page IDs.
	 */
	public static function get_protected_pages() {
		$protected_pages = MACM_Page_Access::get_all_protected_pages();
		$page_ids        = array();

		foreach ( $protected_pages as $page_access ) {
			$page_ids[] = $page_access->page_id;
		}

		return $page_ids;
	}

	/**
	 * Check if user has access to a specific page.
	 *
	 * @since 1.0.0
	 * @param int $page_id The page ID to check.
	 * @param int $user_id The user ID to check (0 for current user).
	 * @return bool True if user has access, false otherwise.
	 */
	public static function check_user_access( $page_id, $user_id = 0 ) {
		return MACM_Page_Access::check_user_access( $page_id, $user_id );
	}
}
