<?php
/**
 * Email Notifications for Member Registration
 *
 * Handles sending email notifications when new members are registered
 *
 * @package KarateClubManager
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Email Notifications class
 */
class MACM_Email_Notifications {

	/**
	 * Constructor
	 */
	public function __construct() {
		// Hook into member creation.
		add_action( 'macm_member_created', array( $this, 'send_member_registration_emails' ), 10, 3 );

		// Hook into member deletion.
		add_action( 'macm_member_deleted', array( $this, 'send_member_deletion_emails' ), 10, 2 );
	}

	/**
	 * Send emails when a new member is created
	 *
	 * @param int   $member_id Member ID.
	 * @param int   $user_id   User ID.
	 * @param array $data      Member data.
	 */
	public function send_member_registration_emails( $member_id, $user_id, $data ) {
		// Get member and user data.
		$member = MACM_Member::get( $member_id );
		$user   = get_user_by( 'id', $user_id );

		if ( ! $member || ! $user ) {
			return;
		}

		// Send confirmation email to user.
		$this->send_user_confirmation_email( $member, $user );

		// Send notification email to admin.
		$this->send_admin_notification_email( $member, $user );
	}

	/**
	 * Send confirmation email to user
	 *
	 * @param object $member Member object.
	 * @param object $user   User object.
	 */
	private function send_user_confirmation_email( $member, $user ) {
		$to = $user->user_email;
		/* translators: %s: member full name */
		$subject = sprintf( __( 'Member Registration Confirmation - %s', 'martial-arts-club-manager' ), $member->full_name );

		// Build email content.
		$message = $this->get_user_email_template( $member, $user );

		// Set HTML content type.
		add_filter( 'wp_mail_content_type', array( $this, 'set_html_content_type' ) );

		// Send email.
		$sent = wp_mail( $to, $subject, $message );

		// Reset content type.
		remove_filter( 'wp_mail_content_type', array( $this, 'set_html_content_type' ) );
	}

	/**
	 * Send notification email to admin
	 *
	 * @param object $member Member object.
	 * @param object $user   User object.
	 */
	private function send_admin_notification_email( $member, $user ) {
		$recipients = macm_parse_notification_emails( get_option( 'macm_member_notification_email', '' ) );
		/* translators: %s: member full name */
		$subject = sprintf( __( 'New Member Registration: %s', 'martial-arts-club-manager' ), $member->full_name );

		// Build email content.
		$message = $this->get_admin_email_template( $member, $user );

		// Set HTML content type.
		add_filter( 'wp_mail_content_type', array( $this, 'set_html_content_type' ) );

		// Send email.
		$sent = wp_mail( $recipients, $subject, $message );

		// Reset content type.
		remove_filter( 'wp_mail_content_type', array( $this, 'set_html_content_type' ) );
	}

	/**
	 * Get user confirmation email template
	 *
	 * @param object $member Member object.
	 * @param object $user   User object.
	 * @return string Email HTML content.
	 */
	private function get_user_email_template( $member, $user ) {
		$site_name = get_bloginfo( 'name' );

		// Get related data.
		$belt_color      = $this->get_belt_color_name( $member->belt_color );
		$club            = $member->club_id ? MACM_Club::get( $member->club_id ) : null;
		$group           = $member->group_id ? MACM_Group::get( $member->group_id ) : null;
		$membership_type = $member->membership_type_id ? MACM_Membership_Type::get( $member->membership_type_id ) : null;

		ob_start();
		?>
		<!DOCTYPE html>
		<html>
		<head>
			<meta charset="UTF-8">
			<style>
				body {
					font-family: Arial, sans-serif;
					line-height: 1.6;
					color: #111827;
					max-width: 600px;
					margin: 0 auto;
				}
				.header {
					background: linear-gradient(135deg, #B11226 0%, #8F0E1E 100%);
					color: white;
					padding: 30px 20px;
					text-align: center;
				}
				.content {
					padding: 30px 20px;
					background: #F4F4F2;
				}
				.info-section {
					background: white;
					padding: 20px;
					margin: 20px 0;
					border-radius: 5px;
					box-shadow: 0 2px 4px rgba(0,0,0,0.1);
				}
				.info-row {
					padding: 8px 0;
					border-bottom: 1px solid #eee;
				}
				.info-row:last-child {
					border-bottom: none;
				}
				.label {
					font-weight: bold;
					color: #B11226;
					display: inline-block;
					width: 180px;
				}
				.value {
					color: #111827;
				}
				.footer {
					padding: 20px;
					text-align: center;
					font-size: 12px;
					color: #6B7280;
				}
			</style>
		</head>
		<body>
			<div class="header">
				<h1><?php echo esc_html( $site_name ); ?></h1>
				<p>Member Registration Confirmation</p>
			</div>

			<div class="content">
				<p>Dear <?php echo esc_html( $user->display_name ); ?>,</p>

				<p>Thank you for registering a new member with <?php echo esc_html( $site_name ); ?>. Below is a summary of the information provided:</p>

				<div class="info-section">
					<h3 style="margin-top: 0; color: #B11226;">Member Information</h3>

					<div class="info-row">
						<span class="label">Full Name:</span>
						<span class="value"><?php echo esc_html( $member->full_name ); ?></span>
					</div>

					<div class="info-row">
						<span class="label">Date of Birth:</span>
						<span class="value"><?php echo esc_html( $member->date_of_birth ? wp_date( 'F j, Y', strtotime( $member->date_of_birth ) ) : 'Not provided' ); ?></span>
					</div>

					<div class="info-row">
						<span class="label">Belt Color:</span>
						<span class="value"><?php echo esc_html( $belt_color ); ?></span>
					</div>

					<?php if ( $membership_type ) : ?>
					<div class="info-row">
						<span class="label">Membership Type:</span>
						<span class="value"><?php echo esc_html( $membership_type->type_name ); ?></span>
					</div>
					<?php endif; ?>

					<?php if ( $club ) : ?>
					<div class="info-row">
						<span class="label">Club:</span>
						<span class="value"><?php echo esc_html( $club->club_name ); ?></span>
					</div>
					<?php endif; ?>

					<?php if ( $group ) : ?>
					<div class="info-row">
						<span class="label">Group:</span>
						<span class="value"><?php echo esc_html( $group->group_name ); ?></span>
					</div>
					<?php endif; ?>

					<?php if ( $member->weight ) : ?>
					<div class="info-row">
						<span class="label">Weight:</span>
						<span class="value">
							<?php
							/* translators: %s: weight value in kilograms */
							printf( esc_html__( '%s kg', 'martial-arts-club-manager' ), esc_html( $member->weight ) );
							?>
						</span>
					</div>
					<?php endif; ?>

					<?php if ( $member->height ) : ?>
					<div class="info-row">
						<span class="label">Height:</span>
						<span class="value">
							<?php
							/* translators: %s: height value in centimeters */
							printf( esc_html__( '%s cm', 'martial-arts-club-manager' ), esc_html( $member->height ) );
							?>
						</span>
					</div>
					<?php endif; ?>

					<?php if ( $member->license_number ) : ?>
					<div class="info-row">
						<span class="label">License Number:</span>
						<span class="value"><?php echo esc_html( $member->license_number ); ?></span>
					</div>
					<?php endif; ?>

					<?php if ( $member->license_expiration ) : ?>
					<div class="info-row">
						<span class="label">License Expiration:</span>
						<span class="value"><?php echo esc_html( wp_date( 'F j, Y', strtotime( $member->license_expiration ) ) ); ?></span>
					</div>
					<?php endif; ?>

					<div class="info-row">
						<span class="label">Registration Date:</span>
						<span class="value"><?php echo esc_html( wp_date( 'F j, Y', strtotime( $member->created_at ) ) ); ?></span>
					</div>
				</div>

				<p>If you have any questions or need to make changes to this member's information, please contact us or log in to your member area.</p>

				<p>Thank you for being part of our community!</p>
			</div>

			<div class="footer">
				<p>&copy; <?php echo esc_html( wp_date( 'Y' ) . ' ' . $site_name ); ?>. All rights reserved.</p>
				<p>This is an automated message. Please do not reply to this email.</p>
			</div>
		</body>
		</html>
		<?php
		return ob_get_clean();
	}

	/**
	 * Get admin notification email template
	 *
	 * @param object $member Member object.
	 * @param object $user   User object.
	 * @return string Email HTML content.
	 */
	private function get_admin_email_template( $member, $user ) {
		$site_name = get_bloginfo( 'name' );

		// Get related data.
		$belt_color      = $this->get_belt_color_name( $member->belt_color );
		$club            = $member->club_id ? MACM_Club::get( $member->club_id ) : null;
		$group           = $member->group_id ? MACM_Group::get( $member->group_id ) : null;
		$membership_type = $member->membership_type_id ? MACM_Membership_Type::get( $member->membership_type_id ) : null;

		// Get admin members page URL.
		$members_url = admin_url( 'admin.php?page=kcm-members' );

		ob_start();
		?>
		<!DOCTYPE html>
		<html>
		<head>
			<meta charset="UTF-8">
			<style>
				body {
					font-family: Arial, sans-serif;
					line-height: 1.6;
					color: #111827;
					max-width: 600px;
					margin: 0 auto;
				}
				.header {
					background: linear-gradient(135deg, #B11226 0%, #8F0E1E 100%);
					color: white;
					padding: 30px 20px;
					text-align: center;
				}
				.content {
					padding: 30px 20px;
					background: #F4F4F2;
				}
				.info-section {
					background: white;
					padding: 20px;
					margin: 20px 0;
					border-radius: 5px;
					box-shadow: 0 2px 4px rgba(0,0,0,0.1);
				}
				.info-row {
					padding: 8px 0;
					border-bottom: 1px solid #eee;
				}
				.info-row:last-child {
					border-bottom: none;
				}
				.label {
					font-weight: bold;
					color: #B11226;
					display: inline-block;
					width: 180px;
				}
				.value {
					color: #111827;
				}
				.button {
					display: inline-block;
					padding: 12px 24px;
					background: linear-gradient(135deg, #B11226 0%, #8F0E1E 100%);
					color: white;
					text-decoration: none;
					border-radius: 5px;
					margin: 20px 0;
				}
				.footer {
					padding: 20px;
					text-align: center;
					font-size: 12px;
					color: #6B7280;
				}
			</style>
		</head>
		<body>
			<div class="header">
				<h1><?php echo esc_html( $site_name ); ?></h1>
				<p>New Member Registration</p>
			</div>

			<div class="content">
				<p><strong>A new member has been registered!</strong></p>

				<div class="info-section">
					<h3 style="margin-top: 0; color: #B11226;">User Information</h3>

					<div class="info-row">
						<span class="label">User Name:</span>
						<span class="value"><?php echo esc_html( $user->display_name ); ?> (<?php echo esc_html( $user->user_login ); ?>)</span>
					</div>

					<div class="info-row">
						<span class="label">User Email:</span>
						<span class="value"><?php echo esc_html( $user->user_email ); ?></span>
					</div>

					<?php
					$mobile = get_user_meta( $user->ID, 'macm_mobile_number', true );
					if ( $mobile ) :
						?>
					<div class="info-row">
						<span class="label">User Mobile:</span>
						<span class="value"><?php echo esc_html( $mobile ); ?></span>
					</div>
					<?php endif; ?>
				</div>

				<div class="info-section">
					<h3 style="margin-top: 0; color: #B11226;">Member Information</h3>

					<div class="info-row">
						<span class="label">Full Name:</span>
						<span class="value"><?php echo esc_html( $member->full_name ); ?></span>
					</div>

					<div class="info-row">
						<span class="label">Date of Birth:</span>
						<span class="value"><?php echo esc_html( $member->date_of_birth ? wp_date( 'F j, Y', strtotime( $member->date_of_birth ) ) : 'Not provided' ); ?></span>
					</div>

					<div class="info-row">
						<span class="label">Belt Color:</span>
						<span class="value"><?php echo esc_html( $belt_color ); ?></span>
					</div>

					<?php if ( $membership_type ) : ?>
					<div class="info-row">
						<span class="label">Membership Type:</span>
						<span class="value"><?php echo esc_html( $membership_type->type_name ); ?></span>
					</div>
					<?php endif; ?>

					<?php if ( $club ) : ?>
					<div class="info-row">
						<span class="label">Club:</span>
						<span class="value"><?php echo esc_html( $club->club_name ); ?></span>
					</div>
					<?php endif; ?>

					<?php if ( $group ) : ?>
					<div class="info-row">
						<span class="label">Group:</span>
						<span class="value"><?php echo esc_html( $group->group_name ); ?></span>
					</div>
					<?php endif; ?>

					<?php if ( $member->weight ) : ?>
					<div class="info-row">
						<span class="label">Weight:</span>
						<span class="value">
							<?php
							/* translators: %s: weight value in kilograms */
							printf( esc_html__( '%s kg', 'martial-arts-club-manager' ), esc_html( $member->weight ) );
							?>
						</span>
					</div>
					<?php endif; ?>

					<?php if ( $member->height ) : ?>
					<div class="info-row">
						<span class="label">Height:</span>
						<span class="value">
							<?php
							/* translators: %s: height value in centimeters */
							printf( esc_html__( '%s cm', 'martial-arts-club-manager' ), esc_html( $member->height ) );
							?>
						</span>
					</div>
					<?php endif; ?>

					<?php if ( $member->license_number ) : ?>
					<div class="info-row">
						<span class="label">License Number:</span>
						<span class="value"><?php echo esc_html( $member->license_number ); ?></span>
					</div>
					<?php endif; ?>

					<?php if ( $member->license_expiration ) : ?>
					<div class="info-row">
						<span class="label">License Expiration:</span>
						<span class="value"><?php echo esc_html( wp_date( 'F j, Y', strtotime( $member->license_expiration ) ) ); ?></span>
					</div>
					<?php endif; ?>

					<div class="info-row">
						<span class="label">Member ID:</span>
						<span class="value">#<?php echo esc_html( $member->id ); ?></span>
					</div>

					<div class="info-row">
						<span class="label">Registration Date:</span>
						<span class="value"><?php echo esc_html( wp_date( 'F j, Y g:i A', strtotime( $member->created_at ) ) ); ?></span>
					</div>

					<div class="info-row">
						<span class="label">Status:</span>
						<span class="value"><?php echo esc_html( ucfirst( $member->status ) ); ?></span>
					</div>
				</div>

				<div style="text-align: center;">
					<a href="<?php echo esc_url( $members_url ); ?>" class="button">View Members Panel</a>
				</div>
			</div>

			<div class="footer">
				<p>&copy; <?php echo esc_html( wp_date( 'Y' ) . ' ' . $site_name ); ?>. All rights reserved.</p>
				<p>This is an automated notification from your member management system.</p>
			</div>
		</body>
		</html>
		<?php
		return ob_get_clean();
	}

	/**
	 * Get belt color display name
	 *
	 * @param string $color_key Belt color key.
	 * @return string Belt color name.
	 */
	private function get_belt_color_name( $color_key ) {
		if ( empty( $color_key ) ) {
			return 'Not specified';
		}

		$belt_color = MACM_Belt_Color::get_by_key( $color_key );
		return $belt_color ? $belt_color->color_name : $color_key;
	}

	/**
	 * Set email content type to HTML
	 *
	 * @return string Content type.
	 */
	public function set_html_content_type() {
		return 'text/html';
	}

	/**
	 * Send emails when a member is deleted
	 *
	 * @param int   $member_id   Member ID that was deleted.
	 * @param array $member_data Member data captured before deletion.
	 */
	public function send_member_deletion_emails( $member_id, $member_data ) {
		// Get user who owned the member.
		$user = get_user_by( 'id', $member_data['user_id'] );

		if ( ! $user ) {
			return;
		}

		// Send notification email to admin.
		$this->send_admin_deletion_notification( $member_data, $user );

		// Send confirmation email to user.
		$this->send_user_deletion_confirmation( $member_data, $user );
	}

	/**
	 * Send deletion notification email to admin
	 *
	 * @param array  $member_data Member data.
	 * @param object $user        User object.
	 */
	private function send_admin_deletion_notification( $member_data, $user ) {
		$recipients = macm_parse_notification_emails( get_option( 'macm_member_notification_email', '' ) );
		/* translators: %s: member full name */
		$subject = sprintf( __( 'Member Deleted: %s', 'martial-arts-club-manager' ), $member_data['full_name'] );

		// Build email content.
		$message = $this->get_admin_deletion_email_template( $member_data, $user );

		// Set HTML content type.
		add_filter( 'wp_mail_content_type', array( $this, 'set_html_content_type' ) );

		// Send email.
		$sent = wp_mail( $recipients, $subject, $message );

		// Reset content type.
		remove_filter( 'wp_mail_content_type', array( $this, 'set_html_content_type' ) );
	}

	/**
	 * Send deletion confirmation email to user
	 *
	 * @param array  $member_data Member data.
	 * @param object $user        User object.
	 */
	private function send_user_deletion_confirmation( $member_data, $user ) {
		$to = $user->user_email;
		/* translators: %s: member full name */
		$subject = sprintf( __( 'Member Deletion Confirmation - %s', 'martial-arts-club-manager' ), $member_data['full_name'] );

		// Build email content.
		$message = $this->get_user_deletion_email_template( $member_data, $user );

		// Set HTML content type.
		add_filter( 'wp_mail_content_type', array( $this, 'set_html_content_type' ) );

		// Send email.
		$sent = wp_mail( $to, $subject, $message );

		// Reset content type.
		remove_filter( 'wp_mail_content_type', array( $this, 'set_html_content_type' ) );
	}

	/**
	 * Get admin deletion notification email template
	 *
	 * @param array  $member_data Member data.
	 * @param object $user        User object.
	 * @return string Email HTML content.
	 */
	private function get_admin_deletion_email_template( $member_data, $user ) {
		$site_name = get_bloginfo( 'name' );

		// Get related data.
		$belt_color      = $this->get_belt_color_name( $member_data['belt_color'] );
		$club            = $member_data['club_id'] ? MACM_Club::get( $member_data['club_id'] ) : null;
		$group           = $member_data['group_id'] ? MACM_Group::get( $member_data['group_id'] ) : null;
		$membership_type = $member_data['membership_type_id'] ? MACM_Membership_Type::get( $member_data['membership_type_id'] ) : null;

		// Get admin members page URL.
		$members_url = admin_url( 'admin.php?page=kcm-members' );

		ob_start();
		?>
		<!DOCTYPE html>
		<html>
		<head>
			<meta charset="UTF-8">
			<style>
				body {
					font-family: Arial, sans-serif;
					line-height: 1.6;
					color: #111827;
					max-width: 600px;
					margin: 0 auto;
				}
				.header {
					background: linear-gradient(135deg, #e74c3c 0%, #c0392b 100%);
					color: white;
					padding: 30px 20px;
					text-align: center;
				}
				.content {
					padding: 30px 20px;
					background: #F4F4F2;
				}
				.info-section {
					background: white;
					padding: 20px;
					margin: 20px 0;
					border-radius: 5px;
					box-shadow: 0 2px 4px rgba(0,0,0,0.1);
				}
				.info-row {
					padding: 8px 0;
					border-bottom: 1px solid #eee;
				}
				.info-row:last-child {
					border-bottom: none;
				}
				.label {
					font-weight: bold;
					color: #e74c3c;
					display: inline-block;
					width: 180px;
				}
				.value {
					color: #111827;
				}
				.button {
					display: inline-block;
					padding: 12px 24px;
					background: linear-gradient(135deg, #B11226 0%, #8F0E1E 100%);
					color: white;
					text-decoration: none;
					border-radius: 5px;
					margin: 20px 0;
				}
				.alert-box {
					background: #fff5f5;
					border-left: 4px solid #e74c3c;
					padding: 15px;
					margin: 20px 0;
				}
				.footer {
					padding: 20px;
					text-align: center;
					font-size: 12px;
					color: #6B7280;
				}
			</style>
		</head>
		<body>
			<div class="header">
				<h1><?php echo esc_html( $site_name ); ?></h1>
				<p>Member Deletion Notice</p>
			</div>

			<div class="content">
				<div class="alert-box">
					<strong>A member has been deleted from the system.</strong>
				</div>

				<div class="info-section">
					<h3 style="margin-top: 0; color: #e74c3c;">User Who Deleted</h3>

					<div class="info-row">
						<span class="label">User Name:</span>
						<span class="value"><?php echo esc_html( $user->display_name ); ?> (<?php echo esc_html( $user->user_login ); ?>)</span>
					</div>

					<div class="info-row">
						<span class="label">User Email:</span>
						<span class="value"><?php echo esc_html( $user->user_email ); ?></span>
					</div>

					<?php
					$mobile = get_user_meta( $user->ID, 'macm_mobile_number', true );
					if ( $mobile ) :
						?>
					<div class="info-row">
						<span class="label">User Mobile:</span>
						<span class="value"><?php echo esc_html( $mobile ); ?></span>
					</div>
					<?php endif; ?>
				</div>

				<div class="info-section">
					<h3 style="margin-top: 0; color: #e74c3c;">Deleted Member Information</h3>

					<div class="info-row">
						<span class="label">Member ID:</span>
						<span class="value">#<?php echo esc_html( $member_data['id'] ); ?></span>
					</div>

					<div class="info-row">
						<span class="label">Full Name:</span>
						<span class="value"><?php echo esc_html( $member_data['full_name'] ); ?></span>
					</div>

					<div class="info-row">
						<span class="label">Date of Birth:</span>
						<span class="value"><?php echo esc_html( $member_data['date_of_birth'] ? wp_date( 'F j, Y', strtotime( $member_data['date_of_birth'] ) ) : 'Not provided' ); ?></span>
					</div>

					<div class="info-row">
						<span class="label">Belt Color:</span>
						<span class="value"><?php echo esc_html( $belt_color ); ?></span>
					</div>

					<?php if ( $membership_type ) : ?>
					<div class="info-row">
						<span class="label">Membership Type:</span>
						<span class="value"><?php echo esc_html( $membership_type->type_name ); ?></span>
					</div>
					<?php endif; ?>

					<?php if ( $club ) : ?>
					<div class="info-row">
						<span class="label">Club:</span>
						<span class="value"><?php echo esc_html( $club->club_name ); ?></span>
					</div>
					<?php endif; ?>

					<?php if ( $group ) : ?>
					<div class="info-row">
						<span class="label">Group:</span>
						<span class="value"><?php echo esc_html( $group->group_name ); ?></span>
					</div>
					<?php endif; ?>

					<?php if ( $member_data['weight'] ) : ?>
					<div class="info-row">
						<span class="label">Weight:</span>
						<span class="value">
							<?php
							/* translators: %s: weight value in kilograms */
							printf( esc_html__( '%s kg', 'martial-arts-club-manager' ), esc_html( $member_data['weight'] ) );
							?>
						</span>
					</div>
					<?php endif; ?>

					<?php if ( $member_data['height'] ) : ?>
					<div class="info-row">
						<span class="label">Height:</span>
						<span class="value">
							<?php
							/* translators: %s: height value in centimeters */
							printf( esc_html__( '%s cm', 'martial-arts-club-manager' ), esc_html( $member_data['height'] ) );
							?>
						</span>
					</div>
					<?php endif; ?>

					<?php if ( $member_data['license_number'] ) : ?>
					<div class="info-row">
						<span class="label">License Number:</span>
						<span class="value"><?php echo esc_html( $member_data['license_number'] ); ?></span>
					</div>
					<?php endif; ?>

					<?php if ( $member_data['license_expiration'] ) : ?>
					<div class="info-row">
						<span class="label">License Expiration:</span>
						<span class="value"><?php echo esc_html( wp_date( 'F j, Y', strtotime( $member_data['license_expiration'] ) ) ); ?></span>
					</div>
					<?php endif; ?>

					<div class="info-row">
						<span class="label">Status (before deletion):</span>
						<span class="value"><?php echo esc_html( ucfirst( $member_data['status'] ) ); ?></span>
					</div>

					<div class="info-row">
						<span class="label">Originally Registered:</span>
						<span class="value"><?php echo esc_html( wp_date( 'F j, Y g:i A', strtotime( $member_data['created_at'] ) ) ); ?></span>
					</div>

					<div class="info-row">
						<span class="label">Deletion Time:</span>
						<span class="value"><?php echo esc_html( wp_date( 'F j, Y g:i A' ) ); ?></span>
					</div>
				</div>

				<div style="text-align: center;">
					<a href="<?php echo esc_url( $members_url ); ?>" class="button">View Members Panel</a>
				</div>
			</div>

			<div class="footer">
				<p>&copy; <?php echo esc_html( wp_date( 'Y' ) . ' ' . $site_name ); ?>. All rights reserved.</p>
				<p>This is an automated notification from your member management system.</p>
			</div>
		</body>
		</html>
		<?php
		return ob_get_clean();
	}

	/**
	 * Get user deletion confirmation email template
	 *
	 * @param array  $member_data Member data.
	 * @param object $user        User object.
	 * @return string Email HTML content.
	 */
	private function get_user_deletion_email_template( $member_data, $user ) {
		$site_name = get_bloginfo( 'name' );

		// Get related data.
		$belt_color      = $this->get_belt_color_name( $member_data['belt_color'] );
		$club            = $member_data['club_id'] ? MACM_Club::get( $member_data['club_id'] ) : null;
		$group           = $member_data['group_id'] ? MACM_Group::get( $member_data['group_id'] ) : null;
		$membership_type = $member_data['membership_type_id'] ? MACM_Membership_Type::get( $member_data['membership_type_id'] ) : null;

		ob_start();
		?>
		<!DOCTYPE html>
		<html>
		<head>
			<meta charset="UTF-8">
			<style>
				body {
					font-family: Arial, sans-serif;
					line-height: 1.6;
					color: #111827;
					max-width: 600px;
					margin: 0 auto;
				}
				.header {
					background: linear-gradient(135deg, #B11226 0%, #8F0E1E 100%);
					color: white;
					padding: 30px 20px;
					text-align: center;
				}
				.content {
					padding: 30px 20px;
					background: #F4F4F2;
				}
				.info-section {
					background: white;
					padding: 20px;
					margin: 20px 0;
					border-radius: 5px;
					box-shadow: 0 2px 4px rgba(0,0,0,0.1);
				}
				.info-row {
					padding: 8px 0;
					border-bottom: 1px solid #eee;
				}
				.info-row:last-child {
					border-bottom: none;
				}
				.label {
					font-weight: bold;
					color: #B11226;
					display: inline-block;
					width: 180px;
				}
				.value {
					color: #111827;
				}
				.footer {
					padding: 20px;
					text-align: center;
					font-size: 12px;
					color: #6B7280;
				}
			</style>
		</head>
		<body>
			<div class="header">
				<h1><?php echo esc_html( $site_name ); ?></h1>
				<p>Member Deletion Confirmation</p>
			</div>

			<div class="content">
				<p>Dear <?php echo esc_html( $user->display_name ); ?>,</p>

				<p>This email confirms that the following member has been removed from your account at <?php echo esc_html( $site_name ); ?>.</p>

				<div class="info-section">
					<h3 style="margin-top: 0; color: #B11226;">Deleted Member Information</h3>

					<div class="info-row">
						<span class="label">Full Name:</span>
						<span class="value"><?php echo esc_html( $member_data['full_name'] ); ?></span>
					</div>

					<div class="info-row">
						<span class="label">Date of Birth:</span>
						<span class="value"><?php echo esc_html( $member_data['date_of_birth'] ? wp_date( 'F j, Y', strtotime( $member_data['date_of_birth'] ) ) : 'Not provided' ); ?></span>
					</div>

					<div class="info-row">
						<span class="label">Belt Color:</span>
						<span class="value"><?php echo esc_html( $belt_color ); ?></span>
					</div>

					<?php if ( $membership_type ) : ?>
					<div class="info-row">
						<span class="label">Membership Type:</span>
						<span class="value"><?php echo esc_html( $membership_type->type_name ); ?></span>
					</div>
					<?php endif; ?>

					<?php if ( $club ) : ?>
					<div class="info-row">
						<span class="label">Club:</span>
						<span class="value"><?php echo esc_html( $club->club_name ); ?></span>
					</div>
					<?php endif; ?>

					<?php if ( $group ) : ?>
					<div class="info-row">
						<span class="label">Group:</span>
						<span class="value"><?php echo esc_html( $group->group_name ); ?></span>
					</div>
					<?php endif; ?>

					<div class="info-row">
						<span class="label">Deletion Date:</span>
						<span class="value"><?php echo esc_html( wp_date( 'F j, Y g:i A' ) ); ?></span>
					</div>
				</div>

				<p>If you did not request this deletion or believe this was done in error, please contact us immediately.</p>

				<p>Thank you for being part of our community.</p>
			</div>

			<div class="footer">
				<p>&copy; <?php echo esc_html( wp_date( 'Y' ) . ' ' . $site_name ); ?>. All rights reserved.</p>
				<p>This is an automated message. Please do not reply to this email.</p>
			</div>
		</body>
		</html>
		<?php
		return ob_get_clean();
	}
}

// Initialize.
new MACM_Email_Notifications();
