<?php
/**
 * Admin Videos Management
 *
 * Handles the admin interface for managing training videos
 *
 * @package    Karate_Club_Manager
 * @subpackage Karate_Club_Manager/includes/admin
 * @since      1.0.27
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class MACM_Admin_Videos
 *
 * Manages admin interface for training videos
 *
 * @since 1.0.27
 */
class MACM_Admin_Videos {

	/**
	 * Constructor
	 *
	 * @since 1.0.27
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_menu_page' ), 19 );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
		add_action( 'admin_init', array( $this, 'handle_actions' ) );
	}

	/**
	 * Add menu page
	 *
	 * @since 1.0.27
	 */
	public function add_menu_page() {
		add_submenu_page(
			'martial-arts-club-manager',
			__( 'Training Videos', 'martial-arts-club-manager' ),
			__( 'Training Videos', 'martial-arts-club-manager' ),
			'manage_options',
			'kcm-training-videos',
			array( $this, 'render_page' )
		);
	}

	/**
	 * Enqueue admin assets
	 *
	 * @since 1.0.27
	 * @param string $hook Page hook.
	 */
	public function enqueue_assets( $hook ) {
		// Only load on our admin page.
		$screen = get_current_screen();
		if ( ! $screen || 'martial-arts-club-manager_page_kcm-training-videos' !== $screen->id ) {
			return;
		}

		// Enqueue media uploader.
		wp_enqueue_media();

		// Enqueue CSS.
		wp_enqueue_style(
			'macm-admin-videos',
			MACM_PLUGIN_URL . 'assets/css/admin-videos.css',
			array(),
			MACM_VERSION
		);

		// Enqueue JS.
		wp_enqueue_script(
			'macm-admin-videos',
			MACM_PLUGIN_URL . 'assets/js/admin-videos.js',
			array( 'jquery', 'media-upload', 'media-views' ),
			MACM_VERSION,
			true
		);

		// Localize script.
		wp_localize_script(
			'macm-admin-videos',
			'macmVideos',
			array(
				'ajaxurl'         => admin_url( 'admin-ajax.php' ),
				'nonce'           => wp_create_nonce( 'macm_videos_nonce' ),
				'confirmDelete'   => __( 'Are you sure you want to delete this video? This action cannot be undone.', 'martial-arts-club-manager' ),
				'selectThumbnail' => __( 'Select Thumbnail', 'martial-arts-club-manager' ),
				'useThumbnail'    => __( 'Use as Thumbnail', 'martial-arts-club-manager' ),
			)
		);
	}

	/**
	 * Handle form actions
	 *
	 * @since 1.0.27
	 */
	public function handle_actions() {
		// Only handle on our page.
		if ( ! isset( $_GET['page'] ) || 'kcm-training-videos' !== $_GET['page'] ) {
			return;
		}

		// Handle delete action.
		if ( isset( $_GET['action'] ) && 'delete' === $_GET['action'] && isset( $_GET['id'] ) ) {
			$this->handle_delete();
		}

		// Handle save action - verify nonce before accessing any POST data.
		if ( isset( $_POST['macm_save_video'], $_POST['macm_video_nonce'] ) ) {
			$nonce = sanitize_text_field( wp_unslash( $_POST['macm_video_nonce'] ) );
			if ( wp_verify_nonce( $nonce, 'macm_save_video' ) ) {
				$this->handle_save();
			}
		}
	}

	/**
	 * Handle delete action
	 *
	 * @since 1.0.27
	 */
	private function handle_delete() {
		$video_id = isset( $_GET['id'] ) ? absint( wp_unslash( $_GET['id'] ) ) : 0;

		// Verify nonce.
		$nonce = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'macm_delete_video_' . $video_id ) ) {
			wp_die( esc_html__( 'Security check failed.', 'martial-arts-club-manager' ) );
		}

		// Check capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to delete videos.', 'martial-arts-club-manager' ) );
		}

		// Delete video.
		if ( MACM_Video::delete( $video_id ) ) {
			$redirect = add_query_arg(
				array(
					'page'    => 'kcm-training-videos',
					'deleted' => '1',
				),
				admin_url( 'admin.php' )
			);
		} else {
			$redirect = add_query_arg(
				array(
					'page'  => 'kcm-training-videos',
					'error' => 'delete',
				),
				admin_url( 'admin.php' )
			);
		}

		wp_safe_redirect( $redirect );
		exit;
	}

	/**
	 * Handle save action
	 *
	 * @since 1.0.27
	 */
	private function handle_save() {
		// Verify nonce.
		$nonce = isset( $_POST['macm_video_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['macm_video_nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'macm_save_video' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'martial-arts-club-manager' ) );
		}

		// Check capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to save videos.', 'martial-arts-club-manager' ) );
		}

		// Prepare data - ensure we never pass null to sanitization functions.
		// Use null coalescing and explicit null checks for PHP 8.1+ compatibility.
		$data = array(
			'title'         => isset( $_POST['title'] ) && null !== $_POST['title'] && '' !== $_POST['title'] ? sanitize_text_field( wp_unslash( $_POST['title'] ) ) : '',
			'description'   => isset( $_POST['description'] ) && null !== $_POST['description'] && '' !== $_POST['description'] ? wp_kses_post( wp_unslash( $_POST['description'] ) ) : '',
			'video_url'     => isset( $_POST['video_url'] ) && null !== $_POST['video_url'] && '' !== $_POST['video_url'] ? esc_url_raw( wp_unslash( $_POST['video_url'] ) ) : '',
			'video_type'    => isset( $_POST['video_type'] ) && null !== $_POST['video_type'] && '' !== $_POST['video_type'] ? sanitize_text_field( wp_unslash( $_POST['video_type'] ) ) : 'youtube',
			'duration'      => isset( $_POST['duration'] ) && null !== $_POST['duration'] && '' !== $_POST['duration'] ? sanitize_text_field( wp_unslash( $_POST['duration'] ) ) : '',
			'thumbnail_id'  => isset( $_POST['thumbnail_id'] ) && null !== $_POST['thumbnail_id'] && '' !== $_POST['thumbnail_id'] ? absint( wp_unslash( $_POST['thumbnail_id'] ) ) : 0,
			'belt_level'    => isset( $_POST['belt_level'] ) && null !== $_POST['belt_level'] && '' !== $_POST['belt_level'] ? sanitize_text_field( wp_unslash( $_POST['belt_level'] ) ) : '',
			'category'      => isset( $_POST['category'] ) && null !== $_POST['category'] && '' !== $_POST['category'] ? sanitize_text_field( wp_unslash( $_POST['category'] ) ) : '',
			'difficulty'    => isset( $_POST['difficulty'] ) && null !== $_POST['difficulty'] && '' !== $_POST['difficulty'] ? sanitize_text_field( wp_unslash( $_POST['difficulty'] ) ) : '',
			'instructor_id' => isset( $_POST['instructor_id'] ) && null !== $_POST['instructor_id'] && '' !== $_POST['instructor_id'] ? absint( wp_unslash( $_POST['instructor_id'] ) ) : null,
			'display_order' => isset( $_POST['display_order'] ) && null !== $_POST['display_order'] && '' !== $_POST['display_order'] ? absint( wp_unslash( $_POST['display_order'] ) ) : 0,
			'status'        => isset( $_POST['status'] ) && null !== $_POST['status'] && '' !== $_POST['status'] ? sanitize_text_field( wp_unslash( $_POST['status'] ) ) : 'published',
		);

		// Validate.
		if ( empty( $data['title'] ) ) {
			add_settings_error( 'macm_videos', 'title_required', __( 'Title is required.', 'martial-arts-club-manager' ), 'error' );
			return;
		}

		if ( empty( $data['video_url'] ) ) {
			add_settings_error( 'macm_videos', 'url_required', __( 'Video URL is required.', 'martial-arts-club-manager' ), 'error' );
			return;
		}

		// Save video.
		$video_id = isset( $_POST['video_id'] ) ? absint( wp_unslash( $_POST['video_id'] ) ) : 0;

		if ( $video_id ) {
			// Update existing video.
			$result  = MACM_Video::update( $video_id, $data );
			$message = $result ? 'updated' : 'error';
		} else {
			// Create new video.
			$result  = MACM_Video::create( $data );
			$message = $result ? 'created' : 'error';
			if ( $result ) {
				$video_id = $result;
			}
		}

		// Redirect with message - ensure all values are not null.
		$redirect_args = array(
			'page'    => 'kcm-training-videos',
			'message' => isset( $message ) && null !== $message ? $message : 'error',
		);

		if ( $result && isset( $_POST['action'] ) && null !== $_POST['action'] && 'edit' === $_POST['action'] ) {
			$redirect_args['action'] = 'edit';
			$redirect_args['id']     = absint( $video_id );
		}

		// Remove any null values from the array to prevent PHP 8.1+ warnings.
		$redirect_args = array_filter(
			$redirect_args,
			function ( $value ) {
				return null !== $value && '' !== $value;
			}
		);

		$redirect_url = add_query_arg( $redirect_args, admin_url( 'admin.php' ) );

		wp_safe_redirect( esc_url_raw( $redirect_url ) );
		exit;
	}

	/**
	 * Render admin page
	 *
	 * @since 1.0.27
	 */
	public function render_page() {
		// Check capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to access this page.', 'martial-arts-club-manager' ) );
		}

		$action_raw = filter_input( INPUT_GET, 'action', FILTER_SANITIZE_FULL_SPECIAL_CHARS );
		$action     = ! empty( $action_raw ) ? sanitize_text_field( $action_raw ) : 'list';

		?>
		<div class="wrap kcm-videos-wrap">
			<h1 class="wp-heading-inline">
				<?php esc_html_e( 'Training Videos', 'martial-arts-club-manager' ); ?>
			</h1>

			<?php
			if ( 'list' === $action ) {
				?>
				<a href="
				<?php
				echo esc_url(
					add_query_arg(
						array(
							'page'   => 'kcm-training-videos',
							'action' => 'add',
						),
						admin_url( 'admin.php' )
					)
				);
				?>
							" class="page-title-action">
					<?php esc_html_e( 'Add New Video', 'martial-arts-club-manager' ); ?>
				</a>
				<?php
			}
			?>

			<hr class="wp-header-end">

			<?php $this->display_notices(); ?>

			<?php
			switch ( $action ) {
				case 'add':
					$this->render_add_form();
					break;

				case 'edit':
					$this->render_edit_form();
					break;

				default:
					$this->render_list();
					break;
			}
			?>
		</div>
		<?php
	}

	/**
	 * Display admin notices
	 *
	 * @since 1.0.27
	 */
	private function display_notices() {
		$message_raw = filter_input( INPUT_GET, 'message', FILTER_SANITIZE_FULL_SPECIAL_CHARS );
		if ( ! empty( $message_raw ) ) {
			$message = sanitize_text_field( $message_raw );

			switch ( $message ) {
				case 'created':
					?>
					<div class="notice notice-success is-dismissible">
						<p><?php esc_html_e( 'Video created successfully.', 'martial-arts-club-manager' ); ?></p>
					</div>
					<?php
					break;

				case 'updated':
					?>
					<div class="notice notice-success is-dismissible">
						<p><?php esc_html_e( 'Video updated successfully.', 'martial-arts-club-manager' ); ?></p>
					</div>
					<?php
					break;

				case 'error':
					?>
					<div class="notice notice-error is-dismissible">
						<p><?php esc_html_e( 'An error occurred. Please try again.', 'martial-arts-club-manager' ); ?></p>
					</div>
					<?php
					break;
			}
		}

		$deleted_param = filter_input( INPUT_GET, 'deleted', FILTER_SANITIZE_FULL_SPECIAL_CHARS );
		if ( ! empty( $deleted_param ) ) {
			?>
			<div class="notice notice-success is-dismissible">
				<p><?php esc_html_e( 'Video deleted successfully.', 'martial-arts-club-manager' ); ?></p>
			</div>
			<?php
		}

		// Display settings errors.
		settings_errors( 'macm_videos' );
	}

	/**
	 * Render videos list
	 *
	 * @since 1.0.27
	 */
	private function render_list() {
		require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-videos-list-table.php';
		require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-video.php';

		$list_table = new MACM_Videos_List_Table();
		$list_table->prepare_items();

		?>
		<form method="post">
			<?php
			$list_table->display();
			?>
		</form>
		<?php
	}

	/**
	 * Render add video form
	 *
	 * @since 1.0.27
	 */
	private function render_add_form() {
		$this->render_video_form();
	}

	/**
	 * Render edit video form
	 *
	 * @since 1.0.27
	 */
	private function render_edit_form() {
		$video_id_raw = filter_input( INPUT_GET, 'id', FILTER_SANITIZE_NUMBER_INT );
		$video_id     = ! empty( $video_id_raw ) ? absint( $video_id_raw ) : 0;

		if ( ! $video_id ) {
			wp_die( esc_html__( 'Invalid video ID.', 'martial-arts-club-manager' ) );
		}

		require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-video.php';
		$video = MACM_Video::get( $video_id );

		if ( ! $video ) {
			wp_die( esc_html__( 'Video not found.', 'martial-arts-club-manager' ) );
		}

		$this->render_video_form( $video );
	}

	/**
	 * Render video form
	 *
	 * @since 1.0.27
	 * @param object|null $video Video object for editing, null for adding.
	 */
	private function render_video_form( $video = null ) {
		// Enqueue media uploader directly here.
		wp_enqueue_media();
		wp_enqueue_script( 'macm-admin-videos', MACM_PLUGIN_URL . 'assets/js/admin-videos.js', array( 'jquery', 'media-upload', 'media-views' ), MACM_VERSION, true );
		wp_localize_script(
			'macm-admin-videos',
			'macmVideos',
			array(
				'ajaxurl'         => admin_url( 'admin-ajax.php' ),
				'nonce'           => wp_create_nonce( 'macm_videos_nonce' ),
				'confirmDelete'   => __( 'Are you sure you want to delete this video? This action cannot be undone.', 'martial-arts-club-manager' ),
				'selectThumbnail' => __( 'Select Thumbnail', 'martial-arts-club-manager' ),
				'useThumbnail'    => __( 'Use as Thumbnail', 'martial-arts-club-manager' ),
			)
		);

		$is_edit = ! is_null( $video );
		$action  = $is_edit ? 'edit' : 'add';

		?>
		<form method="post" class="kcm-video-form">
			<?php wp_nonce_field( 'macm_save_video', 'macm_video_nonce' ); ?>
			<input type="hidden" name="macm_save_video" value="1">
			<input type="hidden" name="action" value="<?php echo esc_attr( $action ); ?>">
			<?php if ( $is_edit ) : ?>
				<input type="hidden" name="video_id" value="<?php echo esc_attr( $video->id ); ?>">
			<?php endif; ?>

			<table class="form-table">
				<tbody>
					<tr>
						<th scope="row">
							<label for="title"><?php esc_html_e( 'Title', 'martial-arts-club-manager' ); ?> <span class="required">*</span></label>
						</th>
						<td>
							<input type="text" name="title" id="title" class="regular-text" value="<?php echo $is_edit ? esc_attr( $video->title ?? '' ) : ''; ?>" required>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="description"><?php esc_html_e( 'Description', 'martial-arts-club-manager' ); ?></label>
						</th>
						<td>
							<textarea name="description" id="description" class="large-text" rows="4"><?php echo $is_edit ? esc_textarea( $video->description ?? '' ) : ''; ?></textarea>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="video_type"><?php esc_html_e( 'Video Type', 'martial-arts-club-manager' ); ?> <span class="required">*</span></label>
						</th>
						<td>
							<select name="video_type" id="video_type" required>
								<option value="youtube" <?php selected( $is_edit ? ( $video->video_type ?? 'youtube' ) : '', 'youtube' ); ?>><?php esc_html_e( 'YouTube', 'martial-arts-club-manager' ); ?></option>
								<option value="vimeo" <?php selected( $is_edit ? ( $video->video_type ?? 'youtube' ) : '', 'vimeo' ); ?>><?php esc_html_e( 'Vimeo', 'martial-arts-club-manager' ); ?></option>
								<option value="direct" <?php selected( $is_edit ? ( $video->video_type ?? 'youtube' ) : '', 'direct' ); ?>><?php esc_html_e( 'Direct URL (MP4)', 'martial-arts-club-manager' ); ?></option>
								<option value="embed" <?php selected( $is_edit ? ( $video->video_type ?? 'youtube' ) : '', 'embed' ); ?>><?php esc_html_e( 'Custom Embed Code', 'martial-arts-club-manager' ); ?></option>
							</select>
							<p class="description">
								<?php esc_html_e( 'Select the type of video source.', 'martial-arts-club-manager' ); ?>
							</p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="video_url"><?php esc_html_e( 'Video URL / Embed Code', 'martial-arts-club-manager' ); ?> <span class="required">*</span></label>
						</th>
						<td>
							<input type="text" name="video_url" id="video_url" class="large-text" value="<?php echo $is_edit ? esc_attr( $video->video_url ?? '' ) : ''; ?>" required>
							<p class="description">
								<strong><?php esc_html_e( 'Examples:', 'martial-arts-club-manager' ); ?></strong><br>
								<?php esc_html_e( 'YouTube: https://www.youtube.com/watch?v=...', 'martial-arts-club-manager' ); ?><br>
								<?php esc_html_e( 'Vimeo: https://vimeo.com/...', 'martial-arts-club-manager' ); ?><br>
								<?php esc_html_e( 'Direct: Full URL to MP4 file', 'martial-arts-club-manager' ); ?><br>
								<?php esc_html_e( 'Embed: Paste iframe embed code', 'martial-arts-club-manager' ); ?>
							</p>
							<p class="description" style="margin-top: 10px; padding: 10px; background: #fde8ea; border-left: 4px solid #B11226;">
								<strong><?php esc_html_e( 'To upload your own video:', 'martial-arts-club-manager' ); ?></strong><br>
								<?php
								printf(
									wp_kses(
										/* translators: %s: URL to Media Library */
										__( '1. Go to <a href="%s" target="_blank">Media Library</a> and upload your video file (MP4 recommended)', 'martial-arts-club-manager' ),
										array(
											'a' => array(
												'href'   => true,
												'target' => true,
											),
										)
									),
									esc_url( admin_url( 'upload.php' ) )
								);
								?>
								<br>
								<?php esc_html_e( '2. Click on the uploaded video and copy the "File URL"', 'martial-arts-club-manager' ); ?><br>
								<?php esc_html_e( '3. Paste the URL here and select "Direct URL" as the video type', 'martial-arts-club-manager' ); ?><br>
								<em style="color: #6B7280;">
									<?php esc_html_e( 'Note: Large video files may exceed your hosting upload limits. For best performance, consider using YouTube or Vimeo for hosting.', 'martial-arts-club-manager' ); ?>
								</em>
							</p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="duration"><?php esc_html_e( 'Duration', 'martial-arts-club-manager' ); ?></label>
						</th>
						<td>
							<input type="text" name="duration" id="duration" class="small-text" value="<?php echo $is_edit ? esc_attr( $video->duration ?? '' ) : ''; ?>" placeholder="10:30">
							<p class="description">
								<?php esc_html_e( 'Optional. Format: mm:ss or hh:mm:ss', 'martial-arts-club-manager' ); ?>
							</p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="belt_level"><?php esc_html_e( 'Belt Level', 'martial-arts-club-manager' ); ?></label>
						</th>
						<td>
							<input type="text" name="belt_level" id="belt_level" class="regular-text" value="<?php echo $is_edit && isset( $video->belt_level ) ? esc_attr( $video->belt_level ) : ''; ?>" placeholder="<?php esc_attr_e( 'e.g., White Belt, Yellow Belt, Black Belt', 'martial-arts-club-manager' ); ?>">
							<p class="description">
								<?php esc_html_e( 'Optional. Enter the belt level this video is suitable for.', 'martial-arts-club-manager' ); ?>
							</p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="category"><?php esc_html_e( 'Category', 'martial-arts-club-manager' ); ?></label>
						</th>
						<td>
							<input type="text" name="category" id="category" class="regular-text" value="<?php echo $is_edit && isset( $video->category ) ? esc_attr( $video->category ) : ''; ?>" placeholder="<?php esc_attr_e( 'e.g., Kata, Kumite, Self-Defense, Basics', 'martial-arts-club-manager' ); ?>">
							<p class="description">
								<?php esc_html_e( 'Optional. Enter the category for this video.', 'martial-arts-club-manager' ); ?>
							</p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="difficulty"><?php esc_html_e( 'Difficulty', 'martial-arts-club-manager' ); ?></label>
						</th>
						<td>
							<select name="difficulty" id="difficulty">
								<option value=""><?php esc_html_e( '-- Select Difficulty --', 'martial-arts-club-manager' ); ?></option>
								<option value="Beginner" <?php selected( $is_edit ? ( $video->difficulty ?? '' ) : '', 'Beginner' ); ?>><?php esc_html_e( 'Beginner', 'martial-arts-club-manager' ); ?></option>
								<option value="Intermediate" <?php selected( $is_edit ? ( $video->difficulty ?? '' ) : '', 'Intermediate' ); ?>><?php esc_html_e( 'Intermediate', 'martial-arts-club-manager' ); ?></option>
								<option value="Advanced" <?php selected( $is_edit ? ( $video->difficulty ?? '' ) : '', 'Advanced' ); ?>><?php esc_html_e( 'Advanced', 'martial-arts-club-manager' ); ?></option>
							</select>
							<p class="description">
								<?php esc_html_e( 'Optional. Select the difficulty level for this video.', 'martial-arts-club-manager' ); ?>
							</p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="instructor_id"><?php esc_html_e( 'Instructor', 'martial-arts-club-manager' ); ?></label>
						</th>
						<td>
							<select name="instructor_id" id="instructor_id">
								<option value=""><?php esc_html_e( '-- Select Instructor --', 'martial-arts-club-manager' ); ?></option>
								<?php
								// Get all instructors.
								$instructors = MACM_Instructor::get_all( true );
								foreach ( $instructors as $instructor ) :
									?>
									<option value="<?php echo esc_attr( $instructor->id ); ?>" <?php selected( $is_edit ? ( $video->instructor_id ?? '' ) : '', $instructor->id ); ?>>
										<?php echo esc_html( $instructor->full_name ); ?>
									</option>
								<?php endforeach; ?>
							</select>
							<p class="description">
								<?php esc_html_e( 'Optional. Select the instructor featured in this video.', 'martial-arts-club-manager' ); ?>
							</p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="thumbnail"><?php esc_html_e( 'Custom Thumbnail', 'martial-arts-club-manager' ); ?></label>
						</th>
						<td>
							<div class="kcm-thumbnail-wrap">
								<?php
								$thumbnail_id  = $is_edit ? $video->thumbnail_id : 0;
								$thumbnail_url = $thumbnail_id ? wp_get_attachment_image_url( $thumbnail_id, 'medium' ) : '';
								?>
								<div class="kcm-thumbnail-preview" style="<?php echo $thumbnail_url ? '' : 'display:none;'; ?>">
									<img src="<?php echo esc_url( $thumbnail_url ); ?>" alt="" style="max-width: 300px; height: auto; border: 1px solid #D1D5DB; padding: 5px;">
								</div>
								<input type="hidden" name="thumbnail_id" id="thumbnail_id" value="<?php echo esc_attr( $thumbnail_id ); ?>">
								<button type="button" class="button" id="upload_thumbnail_button">
									<?php esc_html_e( 'Select Thumbnail', 'martial-arts-club-manager' ); ?>
								</button>
								<button type="button" class="button" id="remove_thumbnail_button" style="<?php echo $thumbnail_url ? '' : 'display:none;'; ?>">
									<?php esc_html_e( 'Remove Thumbnail', 'martial-arts-club-manager' ); ?>
								</button>
								<p class="description">
									<?php esc_html_e( 'Optional. If not set, YouTube/Vimeo thumbnails will be used automatically.', 'martial-arts-club-manager' ); ?>
								</p>
							</div>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="display_order"><?php esc_html_e( 'Display Order', 'martial-arts-club-manager' ); ?></label>
						</th>
						<td>
							<input type="number" name="display_order" id="display_order" class="small-text" value="<?php echo $is_edit ? esc_attr( $video->display_order ?? 0 ) : '0'; ?>" min="0">
							<p class="description">
								<?php esc_html_e( 'Lower numbers appear first. Default: 0', 'martial-arts-club-manager' ); ?>
							</p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="status"><?php esc_html_e( 'Status', 'martial-arts-club-manager' ); ?></label>
						</th>
						<td>
							<select name="status" id="status">
								<option value="published" <?php selected( $is_edit ? ( $video->status ?? 'published' ) : 'published', 'published' ); ?>><?php esc_html_e( 'Published', 'martial-arts-club-manager' ); ?></option>
								<option value="draft" <?php selected( $is_edit ? ( $video->status ?? 'published' ) : '', 'draft' ); ?>><?php esc_html_e( 'Draft', 'martial-arts-club-manager' ); ?></option>
							</select>
							<p class="description">
								<?php esc_html_e( 'Only published videos will be visible to users.', 'martial-arts-club-manager' ); ?>
							</p>
						</td>
					</tr>
				</tbody>
			</table>

			<p class="submit">
				<input type="submit" name="submit" id="submit" class="button button-primary" value="<?php echo $is_edit ? esc_attr__( 'Update Video', 'martial-arts-club-manager' ) : esc_attr__( 'Add Video', 'martial-arts-club-manager' ); ?>">
				<a href="<?php echo esc_url( add_query_arg( 'page', 'kcm-training-videos', admin_url( 'admin.php' ) ) ); ?>" class="button">
					<?php esc_html_e( 'Cancel', 'martial-arts-club-manager' ); ?>
				</a>
			</p>
		</form>
		<?php
	}
}

// Initialize - PREMIUM feature.
// This block will be removed from the free version by Freemius.
if ( function_exists( 'macm_fs' ) && macm_fs()->is__premium_only() ) {
	if ( macm_fs()->can_use_premium_code() ) {
		new MACM_Admin_Videos();
	}
}
