<?php
/**
 * Admin Events Management Class
 *
 * Handles the admin interface for managing events and competitions.
 *
 * @package    Karate_Club_Manager
 * @subpackage Karate_Club_Manager/includes/admin
 * @since      1.0.72
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin Events Class
 *
 * @since 1.0.72
 */
class MACM_Admin_Events {

	/**
	 * Initialize the class
	 *
	 * @since 1.0.72
	 */
	public function __construct() {
		add_action( 'admin_init', array( $this, 'handle_csv_export' ) );
		add_action( 'admin_menu', array( $this, 'add_menu_page' ), 21 ); // Right after Trial Classes (20).
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'wp_ajax_macm_save_event', array( $this, 'ajax_save_event' ) );
		add_action( 'wp_ajax_macm_delete_event', array( $this, 'ajax_delete_event' ) );
		add_action( 'wp_ajax_macm_archive_event', array( $this, 'ajax_archive_event' ) );
		add_action( 'wp_ajax_macm_unarchive_event', array( $this, 'ajax_unarchive_event' ) );
		add_action( 'admin_post_macm_remove_event_registration', array( $this, 'handle_remove_registration' ) );
	}

	/**
	 * Handle CSV export on admin_init
	 *
	 * @since 1.0.84
	 */
	public function handle_csv_export() {
		// Check if this is a CSV export request using filter_input for secure access.
		$page = filter_input( INPUT_GET, 'page', FILTER_SANITIZE_FULL_SPECIAL_CHARS );
		if ( empty( $page ) || 'kcm-events' !== sanitize_key( $page ) ) {
			return;
		}

		$action = filter_input( INPUT_GET, 'action', FILTER_SANITIZE_FULL_SPECIAL_CHARS );
		if ( empty( $action ) || 'export_registrations_csv' !== sanitize_text_field( $action ) ) {
			return;
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions.', 'martial-arts-club-manager' ) );
		}

		// Export CSV.
		$this->export_registrations_csv();
	}

	/**
	 * Add admin menu page
	 *
	 * @since 1.0.72
	 */
	public function add_menu_page() {
		add_submenu_page(
			'martial-arts-club-manager',
			__( 'Events & Competitions', 'martial-arts-club-manager' ),
			__( 'Events', 'martial-arts-club-manager' ),
			'manage_options',
			'kcm-events',
			array( $this, 'render_events_page' )
		);
	}

	/**
	 * Enqueue admin scripts and styles
	 *
	 * @since 1.0.72
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_scripts( $hook ) {
		// Check if we're on the events page using filter_input for secure access.
		$screen = get_current_screen();
		$page   = filter_input( INPUT_GET, 'page', FILTER_SANITIZE_FULL_SPECIAL_CHARS );
		if ( ! $screen || empty( $page ) || 'kcm-events' !== sanitize_key( $page ) ) {
			return;
		}

		// Enqueue WordPress media uploader.
		wp_enqueue_media();

		// Enqueue styles.
		wp_enqueue_style(
			'kcm-admin-events',
			MACM_PLUGIN_URL . 'assets/css/admin-events.css',
			array(),
			MACM_VERSION
		);

		// Enqueue scripts.
		wp_enqueue_script(
			'kcm-admin-events',
			MACM_PLUGIN_URL . 'assets/js/admin-events.js',
			array( 'jquery' ),
			MACM_VERSION,
			true
		);

		// Localize script.
		wp_localize_script(
			'kcm-admin-events',
			'macmEvents',
			array(
				'ajaxurl' => admin_url( 'admin-ajax.php' ),
				'nonce'   => wp_create_nonce( 'macm_events_nonce' ),
				'strings' => array(
					'confirmDelete'  => __( 'Are you sure you want to delete this event?', 'martial-arts-club-manager' ),
					'confirmArchive' => __( 'Are you sure you want to archive this event?', 'martial-arts-club-manager' ),
					'error'          => __( 'An error occurred. Please try again.', 'martial-arts-club-manager' ),
				),
			)
		);
	}

	/**
	 * Render events page
	 *
	 * @since 1.0.72
	 */
	public function render_events_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'martial-arts-club-manager' ) );
		}

		// Get current action using filter_input for secure access.
		$action_input = filter_input( INPUT_GET, 'action', FILTER_SANITIZE_FULL_SPECIAL_CHARS );
		$action       = ! empty( $action_input ) ? sanitize_text_field( $action_input ) : 'list';

		// Route to appropriate template.
		switch ( $action ) {
			case 'add':
			case 'edit':
				include MACM_PLUGIN_DIR . 'templates/admin/events-form.php';
				break;

			case 'registrations':
				include MACM_PLUGIN_DIR . 'templates/admin/events-registrations.php';
				break;

			default:
				include MACM_PLUGIN_DIR . 'templates/admin/events-list.php';
				break;
		}
	}

	/**
	 * Get all events
	 *
	 * @since 1.0.72
	 * @param array $args Query arguments.
	 * @return array Events data.
	 */
	public static function get_events( $args = array() ) {
		global $wpdb;

		$defaults = array(
			'status'   => 'active',
			'orderby'  => 'start_date',
			'order'    => 'ASC',
			'per_page' => 20,
			'paged'    => 1,
		);

		$args = wp_parse_args( $args, $defaults );

		// Build ORDER BY clause using only literal strings (validated).
		$is_asc = 'ASC' === strtoupper( $args['order'] );
		switch ( $args['orderby'] ) {
			case 'title':
				$order_clause = $is_asc ? ' ORDER BY title ASC' : ' ORDER BY title DESC';
				break;
			case 'end_date':
				$order_clause = $is_asc ? ' ORDER BY end_date ASC' : ' ORDER BY end_date DESC';
				break;
			case 'status':
				$order_clause = $is_asc ? ' ORDER BY status ASC' : ' ORDER BY status DESC';
				break;
			default:
				$order_clause = $is_asc ? ' ORDER BY start_date ASC' : ' ORDER BY start_date DESC';
				break;
		}

		// Calculate offset for pagination.
		$offset = ( $args['paged'] - 1 ) * $args['per_page'];

		// Build and execute query based on status filter.
		$table_name = $wpdb->prefix . 'macm_events';

		if ( $args['status'] && 'all' !== $args['status'] ) {
			// With status filter.
			return $wpdb->get_results(
				$wpdb->prepare(
					'SELECT * FROM %i WHERE status = %s',
					$table_name,
					$args['status']
				) . $order_clause . $wpdb->prepare( ' LIMIT %d OFFSET %d', $args['per_page'], $offset )
			);
		} else {
			// Without status filter.
			return $wpdb->get_results(
				$wpdb->prepare(
					'SELECT * FROM %i',
					$table_name
				) . $order_clause . $wpdb->prepare( ' LIMIT %d OFFSET %d', $args['per_page'], $offset )
			);
		}
	}

	/**
	 * Get total count of events
	 *
	 * @since 1.0.73
	 * @param array $args Query arguments.
	 * @return int Total count.
	 */
	public static function get_events_count( $args = array() ) {
		global $wpdb;

		$defaults = array(
			'status' => 'active',
		);

		$args = wp_parse_args( $args, $defaults );

		// Build and execute query based on status filter.
		$table_name = $wpdb->prefix . 'macm_events';

		if ( $args['status'] && 'all' !== $args['status'] ) {
			// With status filter.
			return (int) $wpdb->get_var(
				$wpdb->prepare(
					'SELECT COUNT(*) FROM %i WHERE status = %s',
					$table_name,
					$args['status']
				)
			);
		} else {
			// Without status filter.
			return (int) $wpdb->get_var(
				$wpdb->prepare( 'SELECT COUNT(*) FROM %i', $table_name )
			);
		}
	}

	/**
	 * Get single event
	 *
	 * @since 1.0.72
	 * @param int $event_id Event ID.
	 * @return object|null Event data or null.
	 */
	public static function get_event( $event_id ) {
		global $wpdb;

		return $wpdb->get_row(
			$wpdb->prepare(
				"SELECT *
				FROM {$wpdb->prefix}macm_events
				WHERE id = %d",
				$event_id
			)
		);
	}

	/**
	 * Get event registrations
	 *
	 * @since 1.0.72
	 * @param int    $event_id Event ID.
	 * @param string $orderby  Column to order by.
	 * @param string $order    Order direction (ASC or DESC).
	 * @return array Registrations data.
	 */
	public static function get_event_registrations( $event_id, $orderby = 'registration_date', $order = 'DESC' ) {
		global $wpdb;

		// Build ORDER BY clause using only literal strings.
		$is_asc = 'ASC' === strtoupper( $order );
		switch ( $orderby ) {
			case 'member_name':
				$order_clause = $is_asc ? 'ORDER BY m.full_name ASC' : 'ORDER BY m.full_name DESC';
				break;
			default:
				$order_clause = $is_asc ? 'ORDER BY er.registration_date ASC' : 'ORDER BY er.registration_date DESC';
				break;
		}

		// Build and execute query with table identifier placeholders.
		$registrations_table = $wpdb->prefix . 'macm_event_registrations';
		$members_table       = $wpdb->prefix . 'macm_members';

		// Note: $order_clause validated on lines 296-302.
		return $wpdb->get_results(
			$wpdb->prepare(
				'SELECT er.*, m.full_name, u.user_email, u.display_name as user_name
				FROM %i er
				INNER JOIN %i m ON er.member_id = m.id
				INNER JOIN %i u ON er.user_id = u.ID
				WHERE er.event_id = %d',
				$registrations_table,
				$members_table,
				$wpdb->users,
				$event_id
			) . ' ' . $order_clause
		);
	}

	/**
	 * Export event registrations to CSV
	 *
	 * @since 1.0.83
	 */
	private function export_registrations_csv() {
		// Verify nonce.
		$nonce = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';
		if ( ! $nonce || ! wp_verify_nonce( $nonce, 'macm_export_registrations_csv' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'martial-arts-club-manager' ) );
		}

		// Get event ID.
		$event_id = isset( $_GET['event_id'] ) ? absint( wp_unslash( $_GET['event_id'] ) ) : 0;
		if ( ! $event_id ) {
			wp_die( esc_html__( 'Invalid event ID.', 'martial-arts-club-manager' ) );
		}

		// Get event.
		$event = self::get_event( $event_id );
		if ( ! $event ) {
			wp_die( esc_html__( 'Event not found.', 'martial-arts-club-manager' ) );
		}

		// Get registrations.
		$registrations = self::get_event_registrations( $event_id );

		// Generate filename.
		$filename = sanitize_file_name( 'event-registrations-' . $event_id . '-' . wp_date( 'Y-m-d' ) . '.csv' );

		// Set headers for CSV download.
		header( 'Content-Type: text/csv; charset=utf-8' );
		header( 'Content-Disposition: attachment; filename=' . $filename );
		header( 'Pragma: no-cache' );
		header( 'Expires: 0' );

		// Open output stream.
		$output = fopen( 'php://output', 'w' );

		// Add BOM for UTF-8.
		fprintf( $output, chr( 0xEF ) . chr( 0xBB ) . chr( 0xBF ) );

		// Add event information header.
		fputcsv( $output, array( __( 'Event Information', 'martial-arts-club-manager' ) ) );
		fputcsv( $output, array( __( 'Event Title:', 'martial-arts-club-manager' ), $event->title ) );

		// Format event dates.
		$start_date_formatted = date_i18n( get_option( 'date_format' ), strtotime( $event->start_date ) );
		$end_date_formatted   = date_i18n( get_option( 'date_format' ), strtotime( $event->end_date ) );
		$event_time_formatted = date_i18n( get_option( 'time_format' ), strtotime( $event->event_time ) );

		if ( $event->start_date === $event->end_date ) {
			fputcsv( $output, array( __( 'Event Date:', 'martial-arts-club-manager' ), $start_date_formatted ) );
		} else {
			fputcsv( $output, array( __( 'Event Dates:', 'martial-arts-club-manager' ), $start_date_formatted . ' - ' . $end_date_formatted ) );
		}

		fputcsv( $output, array( __( 'Event Time:', 'martial-arts-club-manager' ), $event_time_formatted ) );

		if ( ! empty( $event->location ) ) {
			fputcsv( $output, array( __( 'Location:', 'martial-arts-club-manager' ), $event->location ) );
		}

		fputcsv( $output, array( __( 'Total Registrations:', 'martial-arts-club-manager' ), count( $registrations ) ) );

		// Add blank row separator.
		fputcsv( $output, array() );

		// Add registration list headers.
		fputcsv(
			$output,
			array(
				__( 'Member Name', 'martial-arts-club-manager' ),
				__( 'Registered By', 'martial-arts-club-manager' ),
				__( 'Email', 'martial-arts-club-manager' ),
				__( 'Registration Date', 'martial-arts-club-manager' ),
				__( 'Status', 'martial-arts-club-manager' ),
			)
		);

		// Add data rows.
		foreach ( $registrations as $registration ) {
			fputcsv(
				$output,
				array(
					$registration->full_name,
					$registration->user_name,
					$registration->user_email,
					date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $registration->registration_date ) ),
					ucfirst( $registration->status ),
				)
			);
		}

		exit;
	}

	/**
	 * AJAX handler: Save event
	 *
	 * @since 1.0.72
	 */
	public function ajax_save_event() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'macm_events_nonce', 'nonce', false ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'martial-arts-club-manager' ) ) );
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'martial-arts-club-manager' ) ) );
		}

		global $wpdb;

		// Get and sanitize data.
		$event_id           = isset( $_POST['event_id'] ) ? absint( wp_unslash( $_POST['event_id'] ) ) : 0;
		$title              = isset( $_POST['title'] ) ? sanitize_text_field( wp_unslash( $_POST['title'] ) ) : '';
		$location           = isset( $_POST['location'] ) ? sanitize_text_field( wp_unslash( $_POST['location'] ) ) : '';
		$start_date         = isset( $_POST['start_date'] ) ? sanitize_text_field( wp_unslash( $_POST['start_date'] ) ) : '';
		$end_date           = isset( $_POST['end_date'] ) ? sanitize_text_field( wp_unslash( $_POST['end_date'] ) ) : '';
		$event_time         = isset( $_POST['event_time'] ) ? sanitize_text_field( wp_unslash( $_POST['event_time'] ) ) : '';
		$description        = isset( $_POST['description'] ) ? wp_kses_post( wp_unslash( $_POST['description'] ) ) : '';
		$cost_info          = isset( $_POST['cost_info'] ) ? wp_kses_post( wp_unslash( $_POST['cost_info'] ) ) : '';
		$image_id           = isset( $_POST['image_id'] ) ? absint( wp_unslash( $_POST['image_id'] ) ) : null;
		$closing_date       = isset( $_POST['closing_date'] ) ? sanitize_text_field( wp_unslash( $_POST['closing_date'] ) ) : '';
		$notification_email = isset( $_POST['notification_email'] ) ? sanitize_email( wp_unslash( $_POST['notification_email'] ) ) : '';

		// If end_date is empty, use start_date.
		if ( empty( $end_date ) && ! empty( $start_date ) ) {
			$end_date = $start_date;
		}

		// Validate required fields.
		if ( empty( $title ) || empty( $start_date ) || empty( $end_date ) || empty( $event_time ) || empty( $closing_date ) ) {
			wp_send_json_error( array( 'message' => __( 'Please fill in all required fields.', 'martial-arts-club-manager' ) ) );
		}

		// Validate email.
		if ( ! empty( $notification_email ) && ! is_email( $notification_email ) ) {
			wp_send_json_error( array( 'message' => __( 'Please enter a valid notification email.', 'martial-arts-club-manager' ) ) );
		}

		// Prepare data.
		$data = array(
			'title'              => $title,
			'location'           => $location,
			'start_date'         => $start_date,
			'end_date'           => $end_date,
			'event_time'         => $event_time,
			'description'        => $description,
			'cost_info'          => $cost_info,
			'image_id'           => $image_id ? $image_id : null,
			'closing_date'       => $closing_date,
			'notification_email' => $notification_email,
			'updated_at'         => current_time( 'mysql' ),
		);

		if ( $event_id ) {
			// Update existing event.
			$result = $wpdb->update(
				$wpdb->prefix . 'macm_events',
				$data,
				array( 'id' => $event_id ),
				array( '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%s', '%s', '%s' ),
				array( '%d' )
			);

			if ( false === $result ) {
				wp_send_json_error( array( 'message' => __( 'Failed to update event.', 'martial-arts-club-manager' ) ) );
			}

			wp_send_json_success(
				array(
					'message'  => __( 'Event updated successfully.', 'martial-arts-club-manager' ),
					'event_id' => $event_id,
				)
			);
		} else {
			// Create new event.
			$data['created_at'] = current_time( 'mysql' );

			$result = $wpdb->insert(
				$wpdb->prefix . 'macm_events',
				$data,
				array( '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%s', '%s', '%s', '%s' )
			);

			if ( false === $result ) {
				wp_send_json_error( array( 'message' => __( 'Failed to create event.', 'martial-arts-club-manager' ) ) );
			}

			wp_send_json_success(
				array(
					'message'  => __( 'Event created successfully.', 'martial-arts-club-manager' ),
					'event_id' => $wpdb->insert_id,
				)
			);
		}
	}

	/**
	 * AJAX handler: Delete event
	 *
	 * @since 1.0.72
	 */
	public function ajax_delete_event() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'macm_events_nonce', 'nonce', false ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'martial-arts-club-manager' ) ) );
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'martial-arts-club-manager' ) ) );
		}

		$event_id = isset( $_POST['event_id'] ) ? absint( wp_unslash( $_POST['event_id'] ) ) : 0;

		if ( ! $event_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid event ID.', 'martial-arts-club-manager' ) ) );
		}

		global $wpdb;

		// Delete event registrations first.
		$wpdb->delete(
			$wpdb->prefix . 'macm_event_registrations',
			array( 'event_id' => $event_id ),
			array( '%d' )
		);

		// Delete event.
		$result = $wpdb->delete(
			$wpdb->prefix . 'macm_events',
			array( 'id' => $event_id ),
			array( '%d' )
		);

		if ( false === $result ) {
			wp_send_json_error( array( 'message' => __( 'Failed to delete event.', 'martial-arts-club-manager' ) ) );
		}

		wp_send_json_success( array( 'message' => __( 'Event deleted successfully.', 'martial-arts-club-manager' ) ) );
	}

	/**
	 * AJAX handler: Archive event
	 *
	 * @since 1.0.72
	 */
	public function ajax_archive_event() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'macm_events_nonce', 'nonce', false ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'martial-arts-club-manager' ) ) );
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'martial-arts-club-manager' ) ) );
		}

		$event_id = isset( $_POST['event_id'] ) ? absint( wp_unslash( $_POST['event_id'] ) ) : 0;

		if ( ! $event_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid event ID.', 'martial-arts-club-manager' ) ) );
		}

		global $wpdb;

		$result = $wpdb->update(
			$wpdb->prefix . 'macm_events',
			array( 'status' => 'archived' ),
			array( 'id' => $event_id ),
			array( '%s' ),
			array( '%d' )
		);

		if ( false === $result ) {
			wp_send_json_error( array( 'message' => __( 'Failed to archive event.', 'martial-arts-club-manager' ) ) );
		}

		wp_send_json_success( array( 'message' => __( 'Event archived successfully.', 'martial-arts-club-manager' ) ) );
	}

	/**
	 * AJAX handler: Unarchive event
	 *
	 * @since 1.0.73
	 */
	public function ajax_unarchive_event() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'macm_events_nonce', 'nonce', false ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'martial-arts-club-manager' ) ) );
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'martial-arts-club-manager' ) ) );
		}

		$event_id = isset( $_POST['event_id'] ) ? absint( wp_unslash( $_POST['event_id'] ) ) : 0;

		if ( ! $event_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid event ID.', 'martial-arts-club-manager' ) ) );
		}

		global $wpdb;

		$result = $wpdb->update(
			$wpdb->prefix . 'macm_events',
			array( 'status' => 'active' ),
			array( 'id' => $event_id ),
			array( '%s' ),
			array( '%d' )
		);

		if ( false === $result ) {
			wp_send_json_error( array( 'message' => __( 'Failed to unarchive event.', 'martial-arts-club-manager' ) ) );
		}

		wp_send_json_success( array( 'message' => __( 'Event unarchived successfully.', 'martial-arts-club-manager' ) ) );
	}

	/**
	 * Handle remove event registration
	 *
	 * @since 1.0.105
	 */
	public function handle_remove_registration() {
		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions.', 'martial-arts-club-manager' ) );
		}

		// Get and validate parameters.
		$registration_id = isset( $_GET['registration_id'] ) ? absint( wp_unslash( $_GET['registration_id'] ) ) : 0;
		$event_id        = isset( $_GET['event_id'] ) ? absint( wp_unslash( $_GET['event_id'] ) ) : 0;

		if ( ! $registration_id || ! $event_id ) {
			wp_die( esc_html__( 'Invalid registration or event ID.', 'martial-arts-club-manager' ) );
		}

		// Verify nonce.
		$nonce = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';
		if ( ! $nonce || ! wp_verify_nonce( $nonce, 'macm_remove_event_registration_' . $registration_id ) ) {
			wp_die( esc_html__( 'Security check failed.', 'martial-arts-club-manager' ) );
		}

		// Remove the registration.
		global $wpdb;
		$result = $wpdb->delete(
			$wpdb->prefix . 'macm_event_registrations',
			array( 'id' => $registration_id ),
			array( '%d' )
		);

		// Redirect back to registrations page.
		$redirect_url = add_query_arg(
			array(
				'page'     => 'kcm-events',
				'action'   => 'registrations',
				'event_id' => $event_id,
			),
			admin_url( 'admin.php' )
		);

		if ( false === $result ) {
			// Failed to remove.
			$redirect_url = add_query_arg( 'message', 'remove_failed', $redirect_url );
		} else {
			// Success.
			$redirect_url = add_query_arg( 'message', 'removed', $redirect_url );
		}

		wp_safe_redirect( $redirect_url );
		exit;
	}
}

// Initialize the class - PREMIUM feature.
// This block will be removed from the free version by Freemius.
if ( function_exists( 'macm_fs' ) && macm_fs()->is__premium_only() ) {
	if ( macm_fs()->can_use_premium_code() ) {
		new MACM_Admin_Events();
	}
}
