<?php
/**
 * User roles and capabilities management
 *
 * @package KarateClubManager
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Roles class
 */
class MACM_Roles {
	/**
	 * Create custom roles and add capabilities
	 *
	 * Note: Custom membership roles (macm_premium_member, macm_basic_member) have been removed.
	 * Membership types are now managed via database table (macm_membership_types).
	 * This method now only adds capabilities to the administrator role.
	 */
	public static function create_roles() {
		// Add capabilities to administrator role.
		$admin_role = get_role( 'administrator' );
		if ( $admin_role ) {
			$admin_role->add_cap( 'manage_macm_classes' );
			$admin_role->add_cap( 'manage_macm_members' );
			$admin_role->add_cap( 'manage_macm_attendance' );
			$admin_role->add_cap( 'manage_macm_enrollments' );
			$admin_role->add_cap( 'manage_macm_settings' );
			$admin_role->add_cap( 'view_macm_reports' );
			$admin_role->add_cap( 'send_macm_emails' );
			$admin_role->add_cap( 'manage_macm_locations' );
			$admin_role->add_cap( 'manage_macm_groups' );
			$admin_role->add_cap( 'view_training_videos' );
			$admin_role->add_cap( 'manage_own_members' );
			$admin_role->add_cap( 'view_class_schedule' );
			$admin_role->add_cap( 'purchase_dojo_products' );
		}
	}

	/**
	 * Remove custom roles
	 *
	 * Note: This also removes the deprecated membership roles if they still exist.
	 */
	public static function remove_roles() {
		// Remove deprecated membership roles (if they exist).
		remove_role( 'macm_premium_member' );
		remove_role( 'macm_basic_member' );

		// Remove custom capabilities from administrator.
		$admin_role = get_role( 'administrator' );
		if ( $admin_role ) {
			$admin_role->remove_cap( 'manage_macm_classes' );
			$admin_role->remove_cap( 'manage_macm_members' );
			$admin_role->remove_cap( 'manage_macm_attendance' );
			$admin_role->remove_cap( 'manage_macm_enrollments' );
			$admin_role->remove_cap( 'manage_macm_settings' );
			$admin_role->remove_cap( 'view_macm_reports' );
			$admin_role->remove_cap( 'send_macm_emails' );
			$admin_role->remove_cap( 'manage_macm_locations' );
			$admin_role->remove_cap( 'manage_macm_groups' );
			$admin_role->remove_cap( 'view_training_videos' );
			$admin_role->remove_cap( 'manage_own_members' );
			$admin_role->remove_cap( 'view_class_schedule' );
			$admin_role->remove_cap( 'purchase_dojo_products' );
		}
	}

	/**
	 * Reset roles (useful for updates)
	 */
	public static function reset_roles() {
		self::remove_roles();
		self::create_roles();
	}

	/**
	 * Check if a user can manage the class register
	 *
	 * @param int $user_id User ID to check.
	 * @return bool True if user has permission, false otherwise.
	 */
	public static function can_manage_class_register( $user_id ) {
		// Administrators always have access.
		if ( user_can( $user_id, 'manage_options' ) ) {
			return true;
		}

		// Check if user is in the class register users list.
		$users = get_option( 'macm_class_register_users', array() );
		return in_array( $user_id, $users, true );
	}

	/**
	 * Grant class register permission to a user
	 *
	 * @param int $user_id User ID.
	 * @return bool True on success, false on failure.
	 */
	public static function grant_class_register_permission( $user_id ) {
		$users = get_option( 'macm_class_register_users', array() );

		if ( ! in_array( $user_id, $users, true ) ) {
			$users[] = $user_id;
			return update_option( 'macm_class_register_users', array_unique( $users ) );
		}

		return true;
	}

	/**
	 * Revoke class register permission from a user
	 *
	 * @param int $user_id User ID.
	 * @return bool True on success, false on failure.
	 */
	public static function revoke_class_register_permission( $user_id ) {
		$users = get_option( 'macm_class_register_users', array() );

		$users = array_diff( $users, array( $user_id ) );

		return update_option( 'macm_class_register_users', array_values( $users ) );
	}
}
