<?php
/**
 * Main plugin class
 *
 * @package KarateClubManager
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main plugin class - Singleton pattern
 */
class MACM_Main {
	/**
	 * The single instance of the class
	 *
	 * @var MACM_Main
	 */
	protected static $instance = null;

	/**
	 * Whether premium files exist in this build
	 *
	 * @var bool
	 */
	private $premium_files_exist = false;

	/**
	 * Whether premium email features are active
	 *
	 * @var bool
	 */
	private $is_premium_active = false;

	/**
	 * Get the singleton instance
	 *
	 * @return MACM_Main
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor - Initialize the plugin
	 */
	private function __construct() {
		$this->load_dependencies();
		$this->register_mail_from_filter();
		$this->define_admin_hooks();
		$this->define_public_hooks();
		$this->define_woocommerce_hooks();
	}

	/**
	 * Load required dependencies
	 */
	private function load_dependencies() {
		// Load database classes.
		require_once MACM_PLUGIN_DIR . 'includes/database/class-macm-database.php';

		// Load helper functions.
		require_once MACM_PLUGIN_DIR . 'includes/macm-notification-helpers.php';
		require_once MACM_PLUGIN_DIR . 'includes/macm-wp-user-helpers.php';

		// Load core feature classes (FREE).
		require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-belt-color.php';
		require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-membership-type.php';
		require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-member.php';
		require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-product-member.php';
		require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-data-export.php';
		require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-data-import.php';
		require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-data-mapper.php';
		require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-email-notifications.php';

		// Load PREMIUM core feature classes.
		// Check BOTH license AND file existence (free build doesn't include premium files).
		$this->premium_files_exist = file_exists( MACM_PLUGIN_DIR . 'includes/classes/class-macm-location.php' );
		if ( function_exists( 'macm_fs' ) && macm_fs()->can_use_premium_code() && $this->premium_files_exist ) {
			require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-location.php';
			require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-group.php';
			require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-club.php';
			require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-instructor.php';
			require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-class.php';
			require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-attendance.php';
			require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-page-access.php';
			require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-license-reminders.php';
			require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-video.php';
			require_once MACM_PLUGIN_DIR . 'includes/classes/class-macm-grading-history.php';
		}

		// Load admin classes (also for AJAX requests which may not pass is_admin() check).
		if ( is_admin() || wp_doing_ajax() ) {
			// FREE admin classes.
			require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin.php';
			require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-menu.php';
			require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-settings.php';
			require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-polish.php';
			require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-belt-colors.php';
			require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-membership-types.php';
			require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-trial-bookings.php';
			require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-feedback.php';
			require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-data.php';

			// PREMIUM admin classes.
			// Check BOTH license AND file existence (free build doesn't include premium files).
			if ( macm_fs()->can_use_premium_code() && $this->premium_files_exist ) {
				// Class Management (Premium).
				require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-locations.php';
				require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-groups.php';
				require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-clubs.php';
				require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-instructors.php';
				require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-classes.php';
				require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-attendance.php';
				// Other Premium Features.
				require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-email.php';
				require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-page-access.php';
				require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-license-reminders.php';
				require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-videos.php';
				require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-events.php';
				require_once MACM_PLUGIN_DIR . 'includes/admin/class-macm-admin-grading-history.php';
			}
		}

		// Load public classes (FREE).
		require_once MACM_PLUGIN_DIR . 'includes/public/class-macm-public.php';
		require_once MACM_PLUGIN_DIR . 'includes/public/class-macm-shortcodes.php';
		require_once MACM_PLUGIN_DIR . 'includes/public/class-macm-trial-booking.php';

		// PREMIUM public classes.
		// Check BOTH license AND file existence (free build doesn't include premium files).
		if ( macm_fs()->can_use_premium_code() && $this->premium_files_exist ) {
			require_once MACM_PLUGIN_DIR . 'includes/public/class-macm-class-register.php';
			require_once MACM_PLUGIN_DIR . 'includes/public/class-macm-training-videos.php';
			require_once MACM_PLUGIN_DIR . 'includes/public/class-macm-access-control.php';
			require_once MACM_PLUGIN_DIR . 'includes/public/class-macm-events.php';
			require_once MACM_PLUGIN_DIR . 'includes/public/class-macm-widget-instructors.php';
			require_once MACM_PLUGIN_DIR . 'includes/public/class-macm-shortcode-instructors.php';
		}

		// Load WooCommerce integration (FREE).
		require_once MACM_PLUGIN_DIR . 'includes/woocommerce/class-macm-registration.php';
		require_once MACM_PLUGIN_DIR . 'includes/woocommerce/class-macm-my-account.php';
		require_once MACM_PLUGIN_DIR . 'includes/woocommerce/class-macm-products.php';
		require_once MACM_PLUGIN_DIR . 'includes/woocommerce/class-macm-cart.php';
		require_once MACM_PLUGIN_DIR . 'includes/woocommerce/class-macm-checkout.php';
	}

	/**
	 * Define admin hooks
	 */
	private function define_admin_hooks() {
		if ( is_admin() ) {
			$admin = new MACM_Admin();
			add_action( 'admin_enqueue_scripts', array( $admin, 'enqueue_styles' ) );
			add_action( 'admin_enqueue_scripts', array( $admin, 'enqueue_scripts' ) );

			// Initialize admin menu.
			new MACM_Admin_Menu();

			// Initialize admin settings.
			new MACM_Admin_Settings();

			// Initialize admin polish (action links, notices, help tabs).
			new MACM_Admin_Polish();

			// PREMIUM: Initialize admin email.
			if ( macm_fs()->can_use_premium_code() && $this->premium_files_exist ) {
				new MACM_Admin_Email();
			}

			// Initialize admin data (export/import).
			new MACM_Admin_Data();

			// Note: Admin page classes (Locations, Groups, Classes) are self-initialized.
			// at the bottom of their respective class files when they are required.
		}
	}

	/**
	 * Register wp_mail_from filters for all contexts.
	 *
	 * Uses priority 20 to override WordPress defaults (priority 10) while
	 * allowing SMTP plugins (typically priority 100+) to override our setting.
	 * Registered here in MACM_Main so the filter is active in ALL contexts
	 * including admin, AJAX, frontend, and cron.
	 */
	private function register_mail_from_filter() {
		$this->is_premium_active = function_exists( 'macm_fs' )
			&& macm_fs()->can_use_premium_code()
			&& $this->premium_files_exist;

		add_filter( 'wp_mail_from', array( $this, 'macm_mail_from' ), 20 );
		add_filter( 'wp_mail_from_name', array( $this, 'macm_mail_from_name' ), 20 );
	}

	/**
	 * Filter the From email address.
	 *
	 * In premium mode, uses the configured plugin setting with admin_email fallback.
	 * In free mode, uses the WordPress admin email directly.
	 *
	 * @param string $email The default From email.
	 * @return string The filtered email address.
	 */
	public function macm_mail_from( $email ) {
		if ( $this->is_premium_active ) {
			$from_email = get_option( 'macm_email_from_email', '' );
			if ( ! empty( $from_email ) && is_email( $from_email ) ) {
				return $from_email;
			}
		}

		return get_option( 'admin_email' );
	}

	/**
	 * Filter the From name.
	 *
	 * In premium mode, uses the configured plugin setting with site name fallback.
	 * In free mode, uses the site name directly.
	 *
	 * @param string $name The default From name.
	 * @return string The filtered sender name.
	 */
	public function macm_mail_from_name( $name ) {
		if ( $this->is_premium_active ) {
			$from_name = get_option( 'macm_email_from_name', '' );
			if ( ! empty( $from_name ) ) {
				return $from_name;
			}
		}

		return get_bloginfo( 'name' );
	}

	/**
	 * Define public hooks
	 */
	private function define_public_hooks() {
		$public = new MACM_Public();
		add_action( 'wp_enqueue_scripts', array( $public, 'enqueue_styles' ) );
		add_action( 'wp_enqueue_scripts', array( $public, 'enqueue_scripts' ) );

		// PREMIUM: Initialize access control, events frontend, and instructors widget.
		if ( macm_fs()->can_use_premium_code() && $this->premium_files_exist ) {
			// Initialize access control.
			new MACM_Access_Control();

			// Initialize events frontend.
			new MACM_Events();

			// Register instructors widget.
			add_action(
				'widgets_init',
				function () {
					register_widget( 'MACM_Widget_Instructors' );
				}
			);
		}
	}

	/**
	 * Define WooCommerce integration hooks
	 */
	private function define_woocommerce_hooks() {
		// Initialize WooCommerce registration customization.
		new MACM_Registration();

		// Initialize WooCommerce My Account customization.
		new MACM_My_Account();
	}
}
