<?php
/**
 * Admin Feedback Handler
 *
 * Handles feature requests and bug reports from the dashboard.
 *
 * @package KarateClubManager
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin Feedback Class
 */
class MACM_Admin_Feedback {

	/**
	 * Validate tmp_name paths from an array of paths.
	 *
	 * @param array $tmp_names Array of tmp_name paths from $_FILES.
	 * @return array Validated array of tmp_name paths (empty strings for invalid entries).
	 */
	private function validate_tmp_names( $tmp_names ) {
		if ( ! is_array( $tmp_names ) ) {
			// Single file upload.
			return is_uploaded_file( $tmp_names ) ? array( $tmp_names ) : array( '' );
		}

		// Multi-file upload - validate each path.
		$validated = array();
		foreach ( $tmp_names as $path ) {
			// is_uploaded_file() is the proper validation for tmp_name.
			$validated[] = is_string( $path ) && is_uploaded_file( $path ) ? $path : '';
		}

		return $validated;
	}

	/**
	 * Sanitize a single file from a multi-file $_FILES array.
	 *
	 * @param array $files The $_FILES array with multi-file structure.
	 * @param int   $index The index of the file to sanitize.
	 * @return array|WP_Error Sanitized file array or WP_Error on failure.
	 */
	private function sanitize_multi_file_upload( $files, $index ) {
		// Validate. required keys exist in parent array.
		$required_keys = array( 'name', 'type', 'tmp_name', 'error', 'size' );
		foreach ( $required_keys as $key ) {
			if ( ! isset( $files[ $key ][ $index ] ) ) {
				return new WP_Error( 'invalid_file', __( 'Invalid file upload structure.', 'martial-arts-club-manager' ) );
			}
		}

		// Check for upload errors.
		if ( UPLOAD_ERR_OK !== $files['error'][ $index ] ) {
			return new WP_Error( 'upload_error', __( 'File upload error.', 'martial-arts-club-manager' ) );
		}

		// Skip empty files.
		if ( empty( $files['name'][ $index ] ) ) {
			return new WP_Error( 'empty_file', __( 'Empty file.', 'martial-arts-club-manager' ) );
		}

		// Verify this is an actual uploaded file (security check).
		if ( ! is_uploaded_file( $files['tmp_name'][ $index ] ) ) {
			return new WP_Error( 'not_uploaded', __( 'File is not a valid upload.', 'martial-arts-club-manager' ) );
		}

		// Sanitize the filename.
		$sanitized_name = sanitize_file_name( $files['name'][ $index ] );
		if ( empty( $sanitized_name ) ) {
			return new WP_Error( 'invalid_filename', __( 'Invalid filename.', 'martial-arts-club-manager' ) );
		}

		// Return sanitized file array in single-file format for wp_handle_upload().
		return array(
			'name'     => $sanitized_name,
			'type'     => sanitize_mime_type( $files['type'][ $index ] ),
			'tmp_name' => $files['tmp_name'][ $index ], // Cannot sanitize - system path.
			'error'    => absint( $files['error'][ $index ] ),
			'size'     => absint( $files['size'][ $index ] ),
		);
	}

	/**
	 * Constructor
	 */
	public function __construct() {
		// Enqueue scripts on dashboard page only.
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );

		// AJAX handlers.
		add_action( 'wp_ajax_macm_submit_feature_request', array( $this, 'handle_feature_request' ) );
		add_action( 'wp_ajax_macm_submit_bug_report', array( $this, 'handle_bug_report' ) );
	}

	/**
	 * Enqueue scripts.
	 *
	 * @param string $hook The current admin page hook.
	 */
	public function enqueue_scripts( $hook ) {
		// Only load on dashboard page.
		if ( 'toplevel_page_martial-arts-club-manager' !== $hook ) {
			return;
		}

		// Enqueue Chart.js from local file.
		wp_enqueue_script(
			'chartjs',
			MACM_PLUGIN_URL . 'assets/js/chart.min.js',
			array(),
			'4.4.1',
			true
		);

		wp_enqueue_script(
			'kcm-admin-dashboard',
			MACM_PLUGIN_URL . 'assets/js/admin-dashboard.js',
			array( 'jquery', 'chartjs' ),
			MACM_VERSION,
			true
		);

		// Get chart data.
		$chart_data = $this->get_chart_data();

		wp_localize_script(
			'kcm-admin-dashboard',
			'macmDashboardData',
			array(
				'nonce'     => wp_create_nonce( 'kcm-dashboard-feedback' ),
				'chartData' => $chart_data,
				'i18n'      => array(
					'members'               => __( 'Members', 'martial-arts-club-manager' ),
					'attendance'            => __( 'Attendance', 'martial-arts-club-manager' ),
					'submitting'            => __( 'Submitting...', 'martial-arts-club-manager' ),
					'featureRequestSuccess' => __( 'Thank you! Your feature request has been submitted successfully.', 'martial-arts-club-manager' ),
					'bugReportSuccess'      => __( 'Thank you! Your bug report has been submitted successfully.', 'martial-arts-club-manager' ),
					'submitError'           => __( 'An error occurred. Please try again.', 'martial-arts-club-manager' ),
					'failedFeatureRequest'  => __( 'Failed to submit feature request.', 'martial-arts-club-manager' ),
					'failedBugReport'       => __( 'Failed to submit bug report.', 'martial-arts-club-manager' ),
					'maxScreenshots'        => __( 'You can only upload up to 3 screenshots.', 'martial-arts-club-manager' ),
				),
			)
		);
	}

	/**
	 * Get chart data for dashboard
	 */
	private function get_chart_data() {
		global $wpdb;

		// Belt Distribution - use cache for dashboard performance.
		$cache_key = 'macm_dashboard_belt_data';
		$belt_data = wp_cache_get( $cache_key, 'macm' );

		if ( false === $belt_data ) {
			$belt_data = $wpdb->get_results(
				"SELECT bc.color_name, bc.color_key, COUNT(m.id) as count
				FROM {$wpdb->prefix}macm_belt_colors bc
				LEFT JOIN {$wpdb->prefix}macm_members m ON m.belt_color = bc.color_key
				WHERE bc.is_active = 1
				GROUP BY bc.id, bc.color_name, bc.color_key
				ORDER BY bc.sort_order ASC"
			);
			wp_cache_set( $cache_key, $belt_data, 'macm', 300 );
		}

		$belt_labels = array();
		$belt_counts = array();
		$belt_colors = array();

		// Belt color mapping.
		$color_map = array(
			'white'           => '#f0f0f1',
			'yellow'          => '#ffc107',
			'orange'          => '#ff9800',
			'green'           => '#4caf50',
			'blue'            => '#2196f3',
			'purple'          => '#9c27b0',
			'brown'           => '#795548',
			'brown_1_stripe'  => '#795548',
			'brown_2_stripes' => '#795548',
			'red'             => '#f44336',
			'black'           => '#212121',
			'black_stripe'    => '#212121',
		);

		foreach ( $belt_data as $belt ) {
			if ( (int) $belt->count > 0 ) {
				$belt_labels[] = $belt->color_name;
				$belt_counts[] = (int) $belt->count;
				$belt_colors[] = isset( $color_map[ $belt->color_key ] ) ? $color_map[ $belt->color_key ] : '#6B7280';
			}
		}

		// Attendance Trends (last 6 months) - use cache for dashboard performance.
		$attendance_cache_key = 'macm_dashboard_attendance_data';
		$attendance_data      = wp_cache_get( $attendance_cache_key, 'macm' );

		if ( false === $attendance_data ) {
			$attendance_data = $wpdb->get_results(
				"SELECT
					DATE_FORMAT(attendance_date, '%Y-%m') as month,
					DATE_FORMAT(attendance_date, '%b %Y') as month_label,
					COUNT(*) as count
				FROM {$wpdb->prefix}macm_attendance
				WHERE attendance_date >= DATE_SUB(CURDATE(), INTERVAL 6 MONTH)
				GROUP BY DATE_FORMAT(attendance_date, '%Y-%m')
				ORDER BY month ASC"
			);
			wp_cache_set( $attendance_cache_key, $attendance_data, 'macm', 300 );
		}

		$attendance_labels = array();
		$attendance_counts = array();

		foreach ( $attendance_data as $month ) {
			$attendance_labels[] = $month->month_label;
			$attendance_counts[] = (int) $month->count;
		}

		// Membership Distribution - use cache for dashboard performance.
		$membership_cache_key = 'macm_dashboard_membership_data';
		$membership_data      = wp_cache_get( $membership_cache_key, 'macm' );

		if ( false === $membership_data ) {
			$membership_data = $wpdb->get_results(
				"SELECT mt.type_name, COUNT(m.id) as count
				FROM {$wpdb->prefix}macm_membership_types mt
				LEFT JOIN {$wpdb->prefix}macm_members m ON m.membership_type_id = mt.id
				WHERE mt.is_active = 1
				GROUP BY mt.id, mt.type_name
				ORDER BY mt.type_name ASC"
			);
			wp_cache_set( $membership_cache_key, $membership_data, 'macm', 300 );
		}

		$membership_labels = array();
		$membership_counts = array();
		$membership_colors = array( '#B11226', '#f093fb', '#4facfe', '#fa709a', '#43e97b', '#30cfd0' );

		$i = 0;
		foreach ( $membership_data as $type ) {
			$membership_labels[] = $type->type_name;
			$membership_counts[] = (int) $type->count;
			++$i;
		}

		return array(
			'belt'       => array(
				'labels' => $belt_labels,
				'data'   => $belt_counts,
				'colors' => $belt_colors,
			),
			'attendance' => array(
				'labels' => $attendance_labels,
				'data'   => $attendance_counts,
			),
			'membership' => array(
				'labels' => $membership_labels,
				'data'   => $membership_counts,
				'colors' => array_slice( $membership_colors, 0, count( $membership_labels ) ),
			),
		);
	}

	/**
	 * Handle feature request submission
	 */
	public function handle_feature_request() {
		// Verify nonce.
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'kcm-dashboard-feedback' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'martial-arts-club-manager' ) ) );
		}

		// Check capabilities.
		if ( ! current_user_can( 'manage_macm_members' ) ) {
			wp_send_json_error( array( 'message' => __( 'You do not have permission to do this.', 'martial-arts-club-manager' ) ) );
		}

		// Sanitize input.
		$name        = isset( $_POST['name'] ) ? sanitize_text_field( wp_unslash( $_POST['name'] ) ) : '';
		$email       = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
		$explanation = isset( $_POST['explanation'] ) ? sanitize_textarea_field( wp_unslash( $_POST['explanation'] ) ) : '';

		// Validate.
		if ( empty( $name ) || empty( $email ) || empty( $explanation ) ) {
			wp_send_json_error( array( 'message' => __( 'All fields are required.', 'martial-arts-club-manager' ) ) );
		}

		if ( ! is_email( $email ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid email address.', 'martial-arts-club-manager' ) ) );
		}

		// Get site info.
		$site_url  = get_site_url();
		$site_name = get_bloginfo( 'name' );
		$user      = wp_get_current_user();

		// Prepare email.
		$to      = 'feature-request@martialartsclubmanager.com'; // Feature request email.
		$subject = sprintf( '[MACM Feature Request] %s', $site_name );
		$body    = $this->format_feature_request_email( $name, $email, $explanation, $site_url, $site_name, $user );
		$headers = array(
			'Content-Type: text/html; charset=UTF-8',
			'Reply-To: ' . $email,
		);

		// Send email.
		$sent = wp_mail( $to, $subject, $body, $headers );

		if ( $sent ) {
			wp_send_json_success( array( 'message' => __( 'Feature request submitted successfully.', 'martial-arts-club-manager' ) ) );
		} else {
			wp_send_json_error( array( 'message' => __( 'Failed to send feature request. Please try again.', 'martial-arts-club-manager' ) ) );
		}
	}

	/**
	 * Handle bug report submission
	 */
	public function handle_bug_report() {
		// Verify nonce.
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'kcm-dashboard-feedback' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'martial-arts-club-manager' ) ) );
		}

		// Check capabilities.
		if ( ! current_user_can( 'manage_macm_members' ) ) {
			wp_send_json_error( array( 'message' => __( 'You do not have permission to do this.', 'martial-arts-club-manager' ) ) );
		}

		// Sanitize input.
		$url         = isset( $_POST['url'] ) ? esc_url_raw( wp_unslash( $_POST['url'] ) ) : '';
		$page        = isset( $_POST['page'] ) ? sanitize_text_field( wp_unslash( $_POST['page'] ) ) : '';
		$description = isset( $_POST['description'] ) ? sanitize_textarea_field( wp_unslash( $_POST['description'] ) ) : '';

		// Validate.
		if ( empty( $url ) || empty( $page ) || empty( $description ) ) {
			wp_send_json_error( array( 'message' => __( 'All required fields must be filled.', 'martial-arts-club-manager' ) ) );
		}

		// Get site info.
		$site_url  = get_site_url();
		$site_name = get_bloginfo( 'name' );
		$user      = wp_get_current_user();

		// Handle file uploads with proper sanitization.
		$attachments = array();

		// Validate and store screenshots files array.
		// Nonce verified above at line 325 via wp_verify_nonce().
		$screenshots_files = array();
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated -- Validated with isset checks below.
		if ( isset( $_FILES['screenshots'] )
			&& is_array( $_FILES['screenshots'] )
			&& isset( $_FILES['screenshots']['name'] )
			&& isset( $_FILES['screenshots']['type'] )
			&& isset( $_FILES['screenshots']['tmp_name'] )
			&& isset( $_FILES['screenshots']['error'] )
			&& isset( $_FILES['screenshots']['size'] )
		) {
			// Extract each key separately with proper sanitization.
			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized via sanitize_file_name below.
			$names_raw = isset( $_FILES['screenshots']['name'] ) ? wp_unslash( $_FILES['screenshots']['name'] ) : array();
			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized via sanitize_mime_type below.
			$types_raw = isset( $_FILES['screenshots']['type'] ) ? wp_unslash( $_FILES['screenshots']['type'] ) : array();
			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- tmp_name validated via is_uploaded_file() in validate_tmp_names().
			$tmp_names_raw = isset( $_FILES['screenshots']['tmp_name'] ) ? $_FILES['screenshots']['tmp_name'] : array();
			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized via absint below.
			$errors_raw = isset( $_FILES['screenshots']['error'] ) ? $_FILES['screenshots']['error'] : array();
			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized via absint below.
			$sizes_raw = isset( $_FILES['screenshots']['size'] ) ? $_FILES['screenshots']['size'] : array();

			// Build sanitized structure from extracted arrays.
			// tmp_name is validated via is_uploaded_file(), not sanitized (system-generated path).
			$screenshots_files = array(
				'name'     => is_array( $names_raw )
					? array_map( 'sanitize_file_name', $names_raw )
					: array( sanitize_file_name( $names_raw ) ),
				'type'     => is_array( $types_raw )
					? array_map( 'sanitize_mime_type', $types_raw )
					: array( sanitize_mime_type( $types_raw ) ),
				'tmp_name' => $this->validate_tmp_names( $tmp_names_raw ),
				'error'    => is_array( $errors_raw )
					? array_map( 'absint', $errors_raw )
					: array( absint( $errors_raw ) ),
				'size'     => is_array( $sizes_raw )
					? array_map( 'absint', $sizes_raw )
					: array( absint( $sizes_raw ) ),
			);
		}

		if ( ! empty( $screenshots_files ) && isset( $screenshots_files['name'][0] ) && ! empty( $screenshots_files['name'][0] ) ) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
			require_once ABSPATH . 'wp-admin/includes/image.php';

			$upload_overrides = array(
				'test_form' => false,
				'mimes'     => array(
					'jpg|jpeg|jpe' => 'image/jpeg',
					'png'          => 'image/png',
					'gif'          => 'image/gif',
				),
			);

			// Limit to 3 files maximum.
			$file_count = min( count( $screenshots_files['name'] ), 3 );

			for ( $i = 0; $i < $file_count; $i++ ) {
				// Sanitize each file from the multi-file upload.
				$sanitized_file = $this->sanitize_multi_file_upload( $screenshots_files, $i );

				// Skip files with errors (empty files, invalid uploads, etc.).
				if ( is_wp_error( $sanitized_file ) ) {
					continue;
				}

				// Upload the sanitized file.
				$uploaded = wp_handle_upload( $sanitized_file, $upload_overrides );

				if ( ! isset( $uploaded['error'] ) && isset( $uploaded['file'] ) ) {
					$attachments[] = $uploaded['file'];
				}
			}
		}

		// Prepare email.
		$to      = 'support@martialartsclubmanager.com'; // Developer email.
		$subject = sprintf( '[MACM Bug Report] %s', $site_name );
		$body    = $this->format_bug_report_email( $url, $page, $description, $site_url, $site_name, $user, $attachments );
		$headers = array(
			'Content-Type: text/html; charset=UTF-8',
			'Reply-To: ' . $user->user_email,
		);

		// Send email.
		$sent = wp_mail( $to, $subject, $body, $headers, $attachments );

		if ( $sent ) {
			// Clean up uploaded files after sending.
			foreach ( $attachments as $file ) {
				if ( file_exists( $file ) ) {
					wp_delete_file( $file );
				}
			}
			wp_send_json_success( array( 'message' => __( 'Bug report submitted successfully.', 'martial-arts-club-manager' ) ) );
		} else {
			wp_send_json_error( array( 'message' => __( 'Failed to send bug report. Please try again.', 'martial-arts-club-manager' ) ) );
		}
	}

	/**
	 * Format feature request email.
	 *
	 * @param string   $name        Requester name.
	 * @param string   $email       Requester email.
	 * @param string   $explanation Feature explanation.
	 * @param string   $site_url    Site URL.
	 * @param string   $site_name   Site name.
	 * @param \WP_User $user        Current user object.
	 * @return string The formatted email HTML.
	 */
	private function format_feature_request_email( $name, $email, $explanation, $site_url, $site_name, $user ) {
		$html = '<!DOCTYPE html>
<html>
<head>
	<meta charset="UTF-8">
</head>
<body style="font-family: Arial, sans-serif; line-height: 1.6; color: #111827;">
	<div style="max-width: 600px; margin: 0 auto; padding: 20px;">
		<h2 style="color: #B11226; border-bottom: 2px solid #B11226; padding-bottom: 10px;">
			Feature Request - Martial Arts Club Manager
		</h2>

		<table style="width: 100%; border-collapse: collapse; margin: 20px 0;">
			<tr>
				<td style="padding: 10px; border-bottom: 1px solid #eee; font-weight: bold; width: 30%;">Site:</td>
				<td style="padding: 10px; border-bottom: 1px solid #eee;">' . esc_html( $site_name ) . '</td>
			</tr>
			<tr>
				<td style="padding: 10px; border-bottom: 1px solid #eee; font-weight: bold;">URL:</td>
				<td style="padding: 10px; border-bottom: 1px solid #eee;"><a href="' . esc_url( $site_url ) . '">' . esc_html( $site_url ) . '</a></td>
			</tr>
			<tr>
				<td style="padding: 10px; border-bottom: 1px solid #eee; font-weight: bold;">Submitted By:</td>
				<td style="padding: 10px; border-bottom: 1px solid #eee;">' . esc_html( $name ) . ' (' . esc_html( $user->user_login ) . ')</td>
			</tr>
			<tr>
				<td style="padding: 10px; border-bottom: 1px solid #eee; font-weight: bold;">Email:</td>
				<td style="padding: 10px; border-bottom: 1px solid #eee;"><a href="mailto:' . esc_attr( $email ) . '">' . esc_html( $email ) . '</a></td>
			</tr>
			<tr>
				<td style="padding: 10px; border-bottom: 1px solid #eee; font-weight: bold;">Plugin Version:</td>
				<td style="padding: 10px; border-bottom: 1px solid #eee;">' . esc_html( MACM_VERSION ) . '</td>
			</tr>
		</table>

		<h3 style="color: #B11226; margin-top: 30px;">Feature Explanation:</h3>
		<div style="background: #F4F4F2; padding: 20px; border-left: 4px solid #B11226; border-radius: 4px;">
			' . nl2br( esc_html( $explanation ) ) . '
		</div>

		<p style="margin-top: 30px; font-size: 12px; color: #6B7280;">
			Submitted on ' . wp_date( 'F j, Y \a\t g:i A' ) . '
		</p>
	</div>
</body>
</html>';

		return $html;
	}

	/**
	 * Format bug report email.
	 *
	 * @param string   $url         URL where bug occurred.
	 * @param string   $page        Page name where bug occurred.
	 * @param string   $description Bug description.
	 * @param string   $site_url    Site URL.
	 * @param string   $site_name   Site name.
	 * @param \WP_User $user        Current user object.
	 * @param array    $attachments Array of attachment file paths.
	 * @return string The formatted email HTML.
	 */
	private function format_bug_report_email( $url, $page, $description, $site_url, $site_name, $user, $attachments ) {
		$html = '<!DOCTYPE html>
<html>
<head>
	<meta charset="UTF-8">
</head>
<body style="font-family: Arial, sans-serif; line-height: 1.6; color: #111827;">
	<div style="max-width: 600px; margin: 0 auto; padding: 20px;">
		<h2 style="color: #d63638; border-bottom: 2px solid #d63638; padding-bottom: 10px;">
			Bug Report - Martial Arts Club Manager
		</h2>

		<table style="width: 100%; border-collapse: collapse; margin: 20px 0;">
			<tr>
				<td style="padding: 10px; border-bottom: 1px solid #eee; font-weight: bold; width: 30%;">Site:</td>
				<td style="padding: 10px; border-bottom: 1px solid #eee;">' . esc_html( $site_name ) . '</td>
			</tr>
			<tr>
				<td style="padding: 10px; border-bottom: 1px solid #eee; font-weight: bold;">Site URL:</td>
				<td style="padding: 10px; border-bottom: 1px solid #eee;"><a href="' . esc_url( $site_url ) . '">' . esc_html( $site_url ) . '</a></td>
			</tr>
			<tr>
				<td style="padding: 10px; border-bottom: 1px solid #eee; font-weight: bold;">Bug Page URL:</td>
				<td style="padding: 10px; border-bottom: 1px solid #eee;"><a href="' . esc_url( $url ) . '">' . esc_html( $url ) . '</a></td>
			</tr>
			<tr>
				<td style="padding: 10px; border-bottom: 1px solid #eee; font-weight: bold;">Page/Section:</td>
				<td style="padding: 10px; border-bottom: 1px solid #eee;">' . esc_html( $page ) . '</td>
			</tr>
			<tr>
				<td style="padding: 10px; border-bottom: 1px solid #eee; font-weight: bold;">Reported By:</td>
				<td style="padding: 10px; border-bottom: 1px solid #eee;">' . esc_html( $user->display_name ) . ' (' . esc_html( $user->user_login ) . ')</td>
			</tr>
			<tr>
				<td style="padding: 10px; border-bottom: 1px solid #eee; font-weight: bold;">Email:</td>
				<td style="padding: 10px; border-bottom: 1px solid #eee;"><a href="mailto:' . esc_attr( $user->user_email ) . '">' . esc_html( $user->user_email ) . '</a></td>
			</tr>
			<tr>
				<td style="padding: 10px; border-bottom: 1px solid #eee; font-weight: bold;">Plugin Version:</td>
				<td style="padding: 10px; border-bottom: 1px solid #eee;">' . esc_html( MACM_VERSION ) . '</td>
			</tr>
			<tr>
				<td style="padding: 10px; border-bottom: 1px solid #eee; font-weight: bold;">WordPress Version:</td>
				<td style="padding: 10px; border-bottom: 1px solid #eee;">' . esc_html( get_bloginfo( 'version' ) ) . '</td>
			</tr>
			<tr>
				<td style="padding: 10px; border-bottom: 1px solid #eee; font-weight: bold;">PHP Version:</td>
				<td style="padding: 10px; border-bottom: 1px solid #eee;">' . esc_html( PHP_VERSION ) . '</td>
			</tr>
			<tr>
				<td style="padding: 10px; border-bottom: 1px solid #eee; font-weight: bold;">Screenshots:</td>
				<td style="padding: 10px; border-bottom: 1px solid #eee;">' . count( $attachments ) . ' attached</td>
			</tr>
		</table>

		<h3 style="color: #d63638; margin-top: 30px;">Bug Description:</h3>
		<div style="background: #fcf0f1; padding: 20px; border-left: 4px solid #d63638; border-radius: 4px;">
			' . nl2br( esc_html( $description ) ) . '
		</div>

		<p style="margin-top: 30px; font-size: 12px; color: #6B7280;">
			Submitted on ' . wp_date( 'F j, Y \a\t g:i A' ) . '
		</p>
	</div>
</body>
</html>';

		return $html;
	}
}

// Initialize.
new MACM_Admin_Feedback();
