/**
 * Martial Arts Club Manager - Public Scripts
 *
 * @package KarateClubManager
 */

(function($) {
	'use strict';

	/**
	 * Initialize when document is ready
	 */
	$(document).ready(function() {
		KCM_Public.init();
	});

	/**
	 * Main public object
	 */
	var KCM_Public = {
		/**
		 * Initialize public functionality
		 */
		init: function() {
			this.initMemberArea();
			this.initLicenseValidation();
		},

		/**
		 * Initialize member area functionality
		 */
		initMemberArea: function() {
			var self = this;

			// Add member button
			$(document).on('click', '#kcm-add-member-btn', function(e) {
				e.preventDefault();
				self.openMemberModal();
			});

			// Edit member button
			$(document).on('click', '.kcm-edit-member-btn', function(e) {
				e.preventDefault();
				var memberId = $(this).data('member-id');
				self.openMemberModal(memberId);
			});

			// Delete member button
			$(document).on('click', '.kcm-delete-member-btn', function(e) {
				e.preventDefault();
				var memberId = $(this).data('member-id');
				if (confirm(macmPublic.strings.confirmDelete)) {
					self.deleteMember(memberId);
				}
			});

			// Upload photo button
			$(document).on('click', '.kcm-upload-photo-btn', function(e) {
				e.preventDefault();
				var memberId = $(this).data('member-id');
				$('#kcm-photo-member-id').val(memberId);
				$('#kcm-photo-input').click();
			});

			// Photo input change
			$(document).on('change', '#kcm-photo-input', function() {
				var memberId = $('#kcm-photo-member-id').val();
				if (this.files && this.files[0]) {
					self.uploadPhoto(memberId, this.files[0]);
				}
			});

			// Delete photo button
			$(document).on('click', '.kcm-delete-photo-btn', function(e) {
				e.preventDefault();
				e.stopPropagation();
				var memberId = $(this).data('member-id');
				if (confirm('Are you sure you want to delete this photo?')) {
					self.deletePhoto(memberId);
				}
			});

			// Modal close buttons
			$(document).on('click', '.kcm-modal-close', function(e) {
				e.preventDefault();
				self.closeMemberModal();
			});

			// Click outside modal to close
			$(document).on('click', '.kcm-modal', function(e) {
				if ($(e.target).hasClass('kcm-modal')) {
					self.closeMemberModal();
					$('#kcm-classes-modal').fadeOut();
					$('#kcm-events-modal').fadeOut();
				}
			});

			// Modal close buttons
			$(document).on('click', '#kcm-classes-modal .kcm-modal-close', function(e) {
				e.preventDefault();
				$('#kcm-classes-modal').fadeOut();
			});

			$(document).on('click', '#kcm-events-modal .kcm-modal-close', function(e) {
				e.preventDefault();
				$('#kcm-events-modal').fadeOut();
			});

			// Member form submit
			$(document).on('submit', '#kcm-member-form', function(e) {
				e.preventDefault();
				self.saveMember();
			});

			// View member classes button
			$(document).on('click', '.kcm-view-classes-btn', function(e) {
				e.preventDefault();
				self.viewMemberClasses.call(this);
			});

			// View member events button
			$(document).on('click', '.kcm-view-events-btn', function(e) {
				e.preventDefault();
				self.viewMemberEvents.call(this);
			});

			// Events filter buttons
			$(document).on('click', '.kcm-filter-btn', function(e) {
				e.preventDefault();
				var filter = $(this).data('filter');
				self.renderFilteredEvents(filter);
			});

			// View member belt history button (Premium)
			$(document).on('click', '.kcm-view-belt-history-btn', function(e) {
				e.preventDefault();
				self.viewMemberBeltHistory.call(this);
			});

			// Export belt history CSV
			$(document).on('click', '#kcm-export-belt-history-btn', function(e) {
				e.preventDefault();
				if (self.beltHistoryExportUrl) {
					window.location.href = self.beltHistoryExportUrl;
				}
			});

			// Close belt history modal
			$('#kcm-belt-history-modal .kcm-modal-close').on('click', function() {
				$('#kcm-belt-history-modal').fadeOut(200);
			});
		},

		/**
		 * Open member modal
		 */
		openMemberModal: function(memberId) {
			var self = this;
			var modal = $('#kcm-member-modal');
			var form = $('#kcm-member-form');

			// Reset form
			form[0].reset();
			$('#kcm-member-id').val('');

			if (memberId) {
				// Editing existing member
				$('#kcm-modal-title').text('Edit Member');
				$('#kcm-member-id').val(memberId);

				// Fetch member data from server
				$.ajax({
					url: macmPublic.ajaxurl,
					type: 'GET',
					data: {
						action: 'macm_get_member',
						member_id: memberId,
						nonce: macmPublic.nonce
					},
					success: function(response) {
						if (response.success && response.data.member) {
							var member = response.data.member;
							// Populate form with member data
							$('#kcm-full-name').val(member.full_name);
							$('#kcm-date-of-birth').val(member.date_of_birth);
							$('#kcm-belt-color').val(member.belt_color);
							$('#kcm-club').val(member.club_id || '');
							$('#kcm-weight').val(member.weight || '');
							$('#kcm-height').val(member.height || '');
							$('#kcm-license-number').val(member.license_number || '');
							$('#kcm-license-expiration').val(member.license_expiration || '');

							// Handle membership type field
							if ($('#kcm-membership-type').length) {
								// Admin user - populate dropdown
								$('#kcm-membership-type').val(member.membership_type_id || '');
							} else if ($('#kcm-membership-type-display').length) {
								// Non-admin user - show read-only text
								var membershipDisplay = member.membership_type_name || 'Not set';
								$('#kcm-membership-type-display').val(membershipDisplay);
								// Keep hidden field synced
								$('input[type="hidden"]#kcm-membership-type').val(member.membership_type_id || '');
							}
						}
					}
				});
			} else {
				// Adding new member
				$('#kcm-modal-title').text('Add New Member');
			}

			modal.fadeIn(200);
		},

		/**
		 * Close member modal
		 */
		closeMemberModal: function() {
			$('#kcm-member-modal').fadeOut(200);
		},

		/**
		 * Save member (add or edit)
		 */
		saveMember: function() {
			var self = this;
			var form = $('#kcm-member-form');
			var memberId = $('#kcm-member-id').val();
			var action = memberId ? 'macm_edit_member' : 'macm_add_member';
			var submitBtn = form.find('button[type="submit"]');

			// Get form data
			var formData = {
				full_name: $('#kcm-full-name').val(),
				date_of_birth: $('#kcm-date-of-birth').val(),
				belt_color: $('#kcm-belt-color').val(),
				club_id: $('#kcm-club').val(),
				weight: $('#kcm-weight').val(),
				height: $('#kcm-height').val(),
				license_number: $('#kcm-license-number').val(),
				license_expiration: $('#kcm-license-expiration').val()
			};

			// Include membership type ID (will be from hidden field for non-admins)
			var membershipTypeId = $('#kcm-membership-type').val();
			if (membershipTypeId) {
				formData.membership_type_id = membershipTypeId;
			}

			if (memberId) {
				formData.member_id = memberId;
			}

			// Show loading
			self.showLoading(submitBtn);
			submitBtn.prop('disabled', true);

			// Make AJAX request
			self.ajaxRequest(
				action,
				formData,
				function(data) {
					self.hideLoading(submitBtn);
					submitBtn.prop('disabled', false);
					self.closeMemberModal();
					self.showSuccess(data.message, '#kcm-messages');

					// Reload page to show updated members
					setTimeout(function() {
						location.reload();
					}, 1000);
				},
				function(data) {
					self.hideLoading(submitBtn);
					submitBtn.prop('disabled', false);
					self.showError(data.message || macmPublic.strings.error, '#kcm-messages');
				}
			);
		},

		/**
		 * Delete member
		 */
		deleteMember: function(memberId) {
			var self = this;
			var card = $('.kcm-member-card[data-member-id="' + memberId + '"]');

			// Show loading on card
			card.css('opacity', '0.5');

			self.ajaxRequest(
				'kcm_delete_member',
				{ member_id: memberId },
				function(data) {
					self.showSuccess(data.message, '#kcm-messages');

					// Remove card with animation
					card.fadeOut(400, function() {
						$(this).remove();

						// Show "no members" message if no cards left
						if ($('.kcm-member-card').length === 0) {
							location.reload();
						}
					});
				},
				function(data) {
					card.css('opacity', '1');
					self.showError(data.message || macmPublic.strings.error, '#kcm-messages');
				}
			);
		},

		/**
		 * Upload photo
		 */
		uploadPhoto: function(memberId, file) {
			var self = this;
			var card = $('.kcm-member-card[data-member-id="' + memberId + '"]');
			var photoDiv = card.find('.kcm-member-photo');

			// Show loading
			photoDiv.addClass('kcm-loading');

			// Create FormData
			var formData = new FormData();
			formData.append('action', 'macm_upload_photo');
			formData.append('nonce', macmPublic.nonce);
			formData.append('member_id', memberId);
			formData.append('photo', file);

			// Upload via AJAX
			$.ajax({
				url: macmPublic.ajaxurl,
				type: 'POST',
				data: formData,
				processData: false,
				contentType: false,
				success: function(response) {
					photoDiv.removeClass('kcm-loading');

					if (response.success) {
						self.showSuccess(response.data.message, '#kcm-messages');

						// Reload page to show new photo
						setTimeout(function() {
							location.reload();
						}, 1000);
					} else {
						self.showError(response.data.message || macmPublic.strings.error, '#kcm-messages');
					}
				},
				error: function() {
					photoDiv.removeClass('kcm-loading');
					self.showError(macmPublic.strings.error, '#kcm-messages');
				}
			});

			// Clear the file input
			$('#kcm-photo-input').val('');
		},

		/**
		 * Delete photo
		 */
		deletePhoto: function(memberId) {
			var self = this;
			var card = $('.kcm-member-card[data-member-id="' + memberId + '"]');
			var photoDiv = card.find('.kcm-member-photo');

			// Show loading
			photoDiv.addClass('kcm-loading');

			self.ajaxRequest(
				'kcm_delete_photo',
				{ member_id: memberId },
				function(data) {
					photoDiv.removeClass('kcm-loading');
					self.showSuccess(data.message, '#kcm-messages');

					// Reload page to show placeholder
					setTimeout(function() {
						location.reload();
					}, 1000);
				},
				function(data) {
					photoDiv.removeClass('kcm-loading');
					self.showError(data.message || macmPublic.strings.error, '#kcm-messages');
				}
			);
		},

		/**
		 * Show loading state
		 */
		showLoading: function(element) {
			$(element).addClass('kcm-loading');
			var spinner = $('<span class="kcm-spinner"></span>');
			$(element).prepend(spinner);
		},

		/**
		 * Hide loading state
		 */
		hideLoading: function(element) {
			$(element).removeClass('kcm-loading');
			$(element).find('.kcm-spinner').remove();
		},

		/**
		 * Show success message
		 */
		showSuccess: function(message, container) {
			this.showMessage(message, 'success', container);
		},

		/**
		 * Show error message
		 */
		showError: function(message, container) {
			this.showMessage(message, 'error', container);
		},

		/**
		 * Show info message
		 */
		showInfo: function(message, container) {
			this.showMessage(message, 'info', container);
		},

		/**
		 * Show message
		 */
		showMessage: function(message, type, container) {
			var messageClass = 'kcm-message kcm-message-' + type;
			var messageElement = $('<div class="' + messageClass + '">' + message + '</div>');

			$(container).prepend(messageElement);

			// Scroll to message
			$('html, body').animate({
				scrollTop: messageElement.offset().top - 100
			}, 500);

			// Auto-hide after 5 seconds
			setTimeout(function() {
				messageElement.fadeOut(function() {
					$(this).remove();
				});
			}, 5000);
		},

		/**
		 * Make AJAX request
		 */
		ajaxRequest: function(action, data, successCallback, errorCallback) {
			data.action = action;
			data.nonce = macmPublic.nonce;

			$.ajax({
				url: macmPublic.ajaxurl,
				type: 'POST',
				data: data,
				success: function(response) {
					if (response.success) {
						if (typeof successCallback === 'function') {
							successCallback(response.data);
						}
					} else {
						if (typeof errorCallback === 'function') {
							errorCallback(response.data);
						}
					}
				},
				error: function(xhr, status, error) {
					if (typeof errorCallback === 'function') {
						errorCallback({message: 'AJAX request failed: ' + error});
					}
				}
			});
		},

		/**
		 * Initialize license number/expiration validation
		 * Requires license number before allowing expiration date
		 */
		initLicenseValidation: function() {
			var self = this;

			// Function to update expiration field state
			function updateExpirationField(licenseInput, expirationInput) {
				var hasLicenseNumber = licenseInput.val().trim().length > 0;

				if (hasLicenseNumber) {
					expirationInput.prop('disabled', false);
					expirationInput.closest('.kcm-form-row').find('.kcm-license-note').remove();
				} else {
					expirationInput.prop('disabled', true);
					expirationInput.val(''); // Clear any existing value

					// Add note if not already present
					if (expirationInput.closest('.kcm-form-row').find('.kcm-license-note').length === 0) {
						expirationInput.closest('.kcm-form-row').append(
							'<p class="kcm-license-note" style="color: #d63638; font-size: 12px; margin-top: 5px;">' +
							'Please enter a License Number first before setting the expiration date.' +
							'</p>'
						);
					}
				}
			}

			// Frontend Add/Edit Member Modal
			$(document).on('input', '#kcm-license-number', function() {
				var licenseInput = $(this);
				var expirationInput = $('#kcm-license-expiration');
				updateExpirationField(licenseInput, expirationInput);
			});

			// Initialize on modal open
			$(document).on('kcm-modal-opened', function() {
				var licenseInput = $('#kcm-license-number');
				var expirationInput = $('#kcm-license-expiration');
				if (licenseInput.length && expirationInput.length) {
					updateExpirationField(licenseInput, expirationInput);
				}
			});

			// Also check on page load
			setTimeout(function() {
				var licenseInput = $('#kcm-license-number');
				var expirationInput = $('#kcm-license-expiration');
				if (licenseInput.length && expirationInput.length) {
					updateExpirationField(licenseInput, expirationInput);
				}
			}, 100);
		},

		/**
		 * View Member Classes handler
		 */
		viewMemberClasses: function() {
			var memberId = $(this).data('member-id');
			var memberName = $(this).data('member-name');

			// Open modal
			$('#kcm-classes-modal').fadeIn();
			$('#kcm-classes-modal-title').text(memberName + ' - ' + macmPublic.strings.myClassesTitle);
			$('#kcm-classes-content').html('<div class="kcm-loading">' + macmPublic.strings.loadingClasses + '</div>');

			// Fetch classes via AJAX
			$.ajax({
				url: macmPublic.ajaxurl,
				type: 'POST',
				data: {
					action: 'macm_get_member_classes',
					member_id: memberId,
					nonce: macmPublic.nonce
				},
				success: function(response) {
					if (response.success) {
						var html = '';

						if (response.data.classes.length === 0) {
							html = '<div class="kcm-no-results">';
							html += '<p>' + macmPublic.strings.noClassesEnrolled + '</p>';
							html += '</div>';
						} else {
							html = '<div class="kcm-classes-list">';

							response.data.classes.forEach(function(classData) {
								html += '<div class="kcm-class-item">';
								html += '<div class="kcm-class-header">';
								html += '<h4><span class="dashicons dashicons-calendar-alt"></span> ' + classData.class_name + '</h4>';
								html += '</div>';
								html += '<div class="kcm-class-details">';
								html += '<div class="kcm-detail-row">';
								html += '<span class="kcm-detail-label"><strong>' + macmPublic.strings.schedule + ':</strong></span>';
								html += '<span class="kcm-detail-value">' + classData.day + ' ' + classData.time + '</span>';
								html += '</div>';
								html += '<div class="kcm-detail-row">';
								html += '<span class="kcm-detail-label"><strong>' + macmPublic.strings.location + ':</strong></span>';
								html += '<span class="kcm-detail-value">' + classData.location + '</span>';
								html += '</div>';
								html += '<div class="kcm-detail-row">';
								html += '<span class="kcm-detail-label"><strong>' + macmPublic.strings.instructor + ':</strong></span>';
								html += '<span class="kcm-detail-value">' + classData.instructor + '</span>';
								html += '</div>';
								html += '<div class="kcm-detail-row">';
								html += '<span class="kcm-detail-label"><strong>' + macmPublic.strings.enrolledDate + ':</strong></span>';
								html += '<span class="kcm-detail-value">' + classData.enrolled_date + '</span>';
								html += '</div>';
								html += '</div>';
								html += '</div>';
							});

							html += '</div>';
						}

						$('#kcm-classes-content').html(html);
					} else {
						$('#kcm-classes-content').html('<div class="kcm-error">' + response.data.message + '</div>');
					}
				},
				error: function() {
					$('#kcm-classes-content').html('<div class="kcm-error">' + macmPublic.strings.error + '</div>');
				}
			});
		},

		/**
		 * View Member Events handler
		 */
		viewMemberEvents: function() {
			var memberId = $(this).data('member-id');
			var memberName = $(this).data('member-name');

			// Open modal
			$('#kcm-events-modal').fadeIn();
			$('#kcm-events-modal-title').text(memberName + ' - ' + macmPublic.strings.myEventsTitle);
			$('#kcm-events-content').html('<div class="kcm-loading">' + macmPublic.strings.loadingEvents + '</div>');

			// Fetch events via AJAX
			$.ajax({
				url: macmPublic.ajaxurl,
				type: 'POST',
				data: {
					action: 'macm_get_member_events',
					member_id: memberId,
					nonce: macmPublic.nonce
				},
				success: function(response) {
					if (response.success) {
						// Store all events in the modal for filtering
						$('#kcm-events-modal').data('all-events', response.data.events);
						$('#kcm-events-modal').data('current-filter', 'active');

						// Render events with default filter (active/upcoming)
						KCM_Public.renderFilteredEvents('active');
					} else {
						$('#kcm-events-content').html('<div class="kcm-error">' + response.data.message + '</div>');
					}
				},
				error: function() {
					$('#kcm-events-content').html('<div class="kcm-error">' + macmPublic.strings.error + '</div>');
				}
			});
		},

		/**
		 * Render filtered events
		 */
		renderFilteredEvents: function(filter) {
			var allEvents = $('#kcm-events-modal').data('all-events');
			var html = '';

			if (!allEvents || allEvents.length === 0) {
				html = '<div class="kcm-no-results">';
				html += '<p>' + macmPublic.strings.noEventsRegistered + '</p>';
				html += '</div>';
				$('#kcm-events-content').html(html);
				return;
			}

			// Filter events based on selected filter
			var filteredEvents = allEvents.filter(function(event) {
				if (filter === 'active') {
					return !event.is_past; // Show upcoming and ongoing
				} else {
					return event.is_past; // Show past events
				}
			});

			// Add filter buttons
			html += '<div class="kcm-events-filter">';
			html += '<button type="button" class="kcm-filter-btn ' + (filter === 'active' ? 'active' : '') + '" data-filter="active">';
			html += '<span class="dashicons dashicons-calendar"></span> ' + macmPublic.strings.currentFutureEvents;
			html += '</button>';
			html += '<button type="button" class="kcm-filter-btn ' + (filter === 'past' ? 'active' : '') + '" data-filter="past">';
			html += '<span class="dashicons dashicons-backup"></span> ' + macmPublic.strings.pastEvents;
			html += '</button>';
			html += '</div>';

			if (filteredEvents.length === 0) {
				html += '<div class="kcm-no-results">';
				if (filter === 'active') {
					html += '<p>' + macmPublic.strings.noCurrentEvents + '</p>';
				} else {
					html += '<p>' + macmPublic.strings.noPastEvents + '</p>';
				}
				html += '</div>';
			} else {
				html += '<div class="kcm-events-list">';

				filteredEvents.forEach(function(event) {
					var statusClass = '';
					if (event.is_past) {
						statusClass = 'kcm-status-past';
					} else if (event.is_upcoming) {
						statusClass = 'kcm-status-upcoming';
					} else {
						statusClass = 'kcm-status-ongoing';
					}

					html += '<div class="kcm-event-item ' + statusClass + '">';
					html += '<div class="kcm-event-header">';
					html += '<h4><span class="dashicons dashicons-tickets-alt"></span> ' + event.title + '</h4>';
					html += '<span class="kcm-event-status">' + event.status + '</span>';
					html += '</div>';
					html += '<div class="kcm-event-details">';
					html += '<div class="kcm-detail-row">';
					html += '<span class="kcm-detail-label"><strong>' + macmPublic.strings.eventDates + ':</strong></span>';
					html += '<span class="kcm-detail-value">' + event.start_date;
					if (event.start_date !== event.end_date) {
						html += ' - ' + event.end_date;
					}
					html += '</span>';
					html += '</div>';
					html += '<div class="kcm-detail-row">';
					html += '<span class="kcm-detail-label"><strong>' + macmPublic.strings.location + ':</strong></span>';
					html += '<span class="kcm-detail-value">' + event.location + '</span>';
					html += '</div>';
					if (event.description) {
						html += '<div class="kcm-detail-row">';
						html += '<span class="kcm-detail-label"><strong>' + macmPublic.strings.description + ':</strong></span>';
						html += '<span class="kcm-detail-value">' + event.description + '</span>';
						html += '</div>';
					}
					html += '<div class="kcm-detail-row">';
					html += '<span class="kcm-detail-label"><strong>' + macmPublic.strings.registeredDate + ':</strong></span>';
					html += '<span class="kcm-detail-value">' + event.registered_date + '</span>';
					html += '</div>';
					html += '</div>';
					html += '</div>';
				});

				html += '</div>';
			}

			$('#kcm-events-content').html(html);
		},

		/**
		 * View Member Belt History handler (Premium)
		 */
		viewMemberBeltHistory: function() {
			var memberId = $(this).data('member-id');
			var memberName = $(this).data('member-name');

			// Open modal
			$('#kcm-belt-history-modal').fadeIn();
			$('#kcm-belt-history-modal-title').text(memberName + ' - ' + macmPublic.strings.beltHistoryTitle);
			$('#kcm-belt-history-content').html('<div class="kcm-loading">' + macmPublic.strings.loadingBeltHistory + '</div>');

			// Fetch belt history via AJAX
			$.ajax({
				url: macmPublic.ajaxurl,
				type: 'POST',
				data: {
					action: 'macm_get_member_belt_history',
					member_id: memberId,
					nonce: macmPublic.nonce
				},
				success: function(response) {
					if (response.success) {
						var html = '';
						var history = response.data.history;
						var exportUrl = response.data.export_url;

						if (!history || history.length === 0) {
							html = '<div class="kcm-no-results">';
							html += '<p>' + macmPublic.strings.noBeltHistory + '</p>';
							html += '</div>';
						} else {
							// Export button
							if (exportUrl) {
								html += '<div class="kcm-belt-history-actions">';
								html += '<a href="' + exportUrl + '" class="kcm-btn kcm-btn-secondary kcm-export-btn" target="_blank">';
								html += '<span class="dashicons dashicons-download"></span> ' + macmPublic.strings.exportCsv;
								html += '</a>';
								html += '</div>';
							}

							// Belt history table
							html += '<div class="kcm-belt-history-table-wrapper">';
							html += '<table class="kcm-belt-history-table">';
							html += '<thead>';
							html += '<tr>';
							html += '<th>' + macmPublic.strings.gradingDate + '</th>';
							html += '<th>' + macmPublic.strings.beltColor + '</th>';
							html += '<th>' + macmPublic.strings.examiner + '</th>';
							html += '<th>' + macmPublic.strings.score + '</th>';
							html += '<th>' + macmPublic.strings.notes + '</th>';
							html += '</tr>';
							html += '</thead>';
							html += '<tbody>';

							history.forEach(function(record) {
								html += '<tr>';
								html += '<td>' + record.grading_date + '</td>';
								html += '<td>';
								if (record.belt_color_key) {
									html += '<span class="kcm-belt-badge kcm-belt-' + record.belt_color_key + '">';
									html += record.belt_color;
									html += '</span>';
								} else {
									html += record.belt_color || '-';
								}
								html += '</td>';
								html += '<td>' + (record.examiner_name || '-') + '</td>';
								html += '<td>' + (record.score || '-') + '</td>';
								html += '<td>' + (record.notes || '-') + '</td>';
								html += '</tr>';
							});

							html += '</tbody>';
							html += '</table>';
							html += '</div>';
						}

						$('#kcm-belt-history-content').html(html);
					} else {
						$('#kcm-belt-history-content').html('<div class="kcm-error">' + response.data.message + '</div>');
					}
				},
				error: function() {
					$('#kcm-belt-history-content').html('<div class="kcm-error">' + macmPublic.strings.error + '</div>');
				}
			});
		}
	};

	// Make KCM_Public available globally
	window.KCM_Public = KCM_Public;

})(jQuery);
