<?php
/**
 * Plugin Name:       Markdown Renderer for GitHub
 * Plugin URI:        https://github.com/wakalab/markdown-renderer-for-github
 * Description:       Renders GFM (GitHub Flavored Markdown) content beautifully on the front end using JavaScript libraries. It supports syntax highlighting for code blocks and diagram rendering with Mermaid.js.
 * Version:           2.5.0
 * Requires at least: 6.5
 * Requires PHP:      8.1
 * Tested up to:      6.9
 * Author:            Wakalab
 * Author URI:        https://wakalab.dev/
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       markdown-renderer-for-github
 * Domain Path:       /languages
 */

defined( 'ABSPATH' ) || exit;

// Load Composer autoloader
if ( file_exists( __DIR__ . '/vendor/autoload.php' ) ) {
	require_once __DIR__ . '/vendor/autoload.php';
}

// Define new prefixed constants (WordPress.org requirement compliance)
define( 'GFMR_VERSION', '1.15.0' );
define( 'GFMR_MIN_PRO_VERSION', '1.0.0' );
define( 'GFMR_PLUGIN_FILE', __FILE__ );
define( 'GFMR_SLUG', 'markdown-renderer-for-github' );
define( 'GFMR_URI', plugin_dir_url( __FILE__ ) );
define( 'GFMR_PATH', plugin_dir_path( __FILE__ ) );
define( 'GFMR_PLUGIN_PATH', dirname( plugin_basename( __FILE__ ) ) );
define( 'GFMR_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );
define( 'GFMR_ASSETS_PATH', GFMR_URI . 'assets/' );

/**
 * Development mode - enables Pro features for testing.
 * Automatically enabled when WP_DEBUG is true.
 */
if ( ! defined( 'GFMR_DEV_MODE' ) ) {
	define( 'GFMR_DEV_MODE', defined( 'WP_DEBUG' ) && WP_DEBUG );
}

// Load plugin textdomain for translations
add_action(
	'plugins_loaded',
	function () {
		load_plugin_textdomain(
			'markdown-renderer-for-github',
			false,
			dirname( plugin_basename( __FILE__ ) ) . '/languages'
		);
	},
	-1000
);

// Load main plugin class
require_once __DIR__ . '/includes/class-gfmr-renderer.php';

// Load metadata handler (code block language information management)
require_once __DIR__ . '/includes/class-gfmr-metadata-handler.php';

// Load settings class
require_once __DIR__ . '/includes/class-gfmr-settings.php';

// Load extension API class
require_once __DIR__ . '/includes/class-gfmr-extension-api.php';

// Load Chart Handler (Chart.js integration with Pro feature gating)
require_once __DIR__ . '/includes/class-gfmr-chart-handler.php';

// Load Multilingual Handler
require_once __DIR__ . '/includes/class-gfmr-multilingual.php';

// Load URL Rewriter
require_once __DIR__ . '/includes/class-gfmr-url-rewriter.php';

// Load WP-CLI commands (only when running in CLI mode)
if ( defined( 'WP_CLI' ) && WP_CLI ) {
	require_once __DIR__ . '/includes/class-gfmr-cli.php';
}

// Initialize plugin instance
function gfmr_init() {
	return \Wakalab\WpGfmRenderer\GFMR_Renderer::get_instance();
}

// Execute plugin initialization
gfmr_init();

// Initialize extension API
$gfmr_extension_api = new \Wakalab\WpGfmRenderer\GFMR_Extension_API();
$gfmr_extension_api->init();

// Initialize Chart Handler
\Wakalab\WpGfmRenderer\GFMR_Chart_Handler::get_instance();

// Initialize Multilingual Handler
\Wakalab\WpGfmRenderer\GFMR_Multilingual::get_instance();

// Initialize URL Rewriter
add_action(
	'init',
	function () {
		if ( class_exists( 'Wakalab\\WpGfmRenderer\\GFMR_URL_Rewriter' ) ) {
			\Wakalab\WpGfmRenderer\GFMR_URL_Rewriter::get_instance()->init();
		}
	},
	5
);

// Plugin activation hook
register_activation_hook(
	__FILE__,
	function () {
		if ( class_exists( 'Wakalab\\WpGfmRenderer\\GFMR_URL_Rewriter' ) ) {
			$rewriter = \Wakalab\WpGfmRenderer\GFMR_URL_Rewriter::get_instance();
			$rewriter->add_rewrite_rules();
			flush_rewrite_rules();
		}
	}
);

// Plugin deactivation hook
register_deactivation_hook(
	__FILE__,
	function () {
		flush_rewrite_rules();
	}
);

/**
 * Public API Functions
 *
 * These functions provide a stable API for addons to interact with
 * plugin's core features.
 */

/**
 * Get plugin version
 *
 * @return string Plugin version
 */
function gfmr_get_version() {
	return GFMR_VERSION;
}

/**
 * Get default feature flags
 *
 * @return array Default feature flags
 */
function gfmr_get_default_flags() {
	return array(
		'syntax_highlight' => true,
		'mermaid'          => true,
		'toc'              => false,
		'line_numbers'     => false,
	);
}

/**
 * Check if a feature is enabled
 *
 * This function allows addons to check if a specific feature is enabled.
 * Addons can modify feature flags using the 'gfmr_feature_flags' filter.
 *
 * @param string $flag_name Feature flag name
 * @return bool True if feature is enabled, false otherwise
 */
function gfmr_is_feature_enabled( $flag_name ) {
	/**
	 * Filters feature flags
	 *
	 * Addons can use this filter to enable or disable features.
	 *
	 * @since 2.0.0
	 *
	 * @param array $flags Feature flags array
	 */
	$flags = apply_filters( 'gfmr_feature_flags', gfmr_get_default_flags() );
	return isset( $flags[ $flag_name ] ) && $flags[ $flag_name ];
}
