<?php
/**
 * GFMR URL Rewriter Class
 *
 * Handles URL rewriting for multilingual content with path-based language prefixes.
 * Provides SEO-friendly URLs like /en/post-name/ and /ja/post-name/.
 *
 * @package WpGfmRenderer
 * @since 2.5.0
 */

namespace Wakalab\WpGfmRenderer;

defined( 'ABSPATH' ) || exit;

// Prevent class redeclaration when both Free and Pro versions are active
if ( class_exists( __NAMESPACE__ . '\\GFMR_URL_Rewriter' ) ) {
	return;
}

class GFMR_URL_Rewriter {

	/**
	 * Singleton instance
	 *
	 * @var GFMR_URL_Rewriter|null
	 */
	private static $instance = null;

	/**
	 * Query variable name for language
	 */
	const QUERY_VAR = 'gfmr_lang';

	/**
	 * Get singleton instance
	 *
	 * @return GFMR_URL_Rewriter
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		// Don't call init() here - it will be called by get_instance()
	}

	/**
	 * Initialize the URL rewriter
	 *
	 * Should be called from get_instance() or via action hook
	 */
	public function init() {
		if ( $this->should_enable() ) {
			$this->init_hooks();
		}
	}

	/**
	 * Initialize hooks
	 */
	private function init_hooks() {
		// Register rewrite rules on init
		add_action( 'init', array( $this, 'add_rewrite_rules' ), 10 );

		// Add query variables
		add_filter( 'query_vars', array( $this, 'add_query_vars' ) );

		// Output hreflang tags
		add_action( 'wp_head', array( $this, 'output_hreflang_tags' ) );

		// Output canonical tag
		add_action( 'wp_head', array( $this, 'output_canonical_tag' ) );

		// Handle query to path redirect
		add_action( 'template_redirect', array( $this, 'maybe_redirect_query_to_path' ) );
	}

	/**
	 * Check if URL rewriter should be enabled
	 *
	 * @return bool True if enabled
	 */
	private function should_enable() {
		// Check if WPML is active
		if ( defined( 'ICL_SITEPRESS_VERSION' ) ) {
			return false;
		}

		// Check if Polylang is active
		if ( function_exists( 'pll_current_language' ) ) {
			return false;
		}

		// Check if path mode is enabled in settings
		if ( ! $this->is_path_mode_enabled() ) {
			return false;
		}

		/**
		 * Filter to enable/disable multilingual URL rewriter
		 *
		 * @since 2.5.0
		 *
		 * @param bool $enabled Whether URL rewriter is enabled
		 */
		return apply_filters( 'gfmr_multilingual_url_enabled', true );
	}

	/**
	 * Add rewrite rules for multilingual URLs
	 */
	public function add_rewrite_rules() {
		$languages = $this->get_supported_languages();
		$default_lang = $this->get_default_language();

		// Get non-default languages
		$non_default_langs = array_filter(
			$languages,
			function ( $lang ) use ( $default_lang ) {
				return $lang !== $default_lang;
			}
		);

		if ( empty( $non_default_langs ) ) {
			return;
		}

		// Build language pattern
		$lang_pattern = '(' . implode( '|', array_map( 'preg_quote', $non_default_langs ) ) . ')';

		// Rewrite rule for posts/pages
		add_rewrite_rule(
			"^{$lang_pattern}/(.+?)/?$",
			'index.php?pagename=$matches[2]&' . self::QUERY_VAR . '=$matches[1]',
			'top'
		);

		// Rewrite rule for category archives
		add_rewrite_rule(
			"^{$lang_pattern}/category/(.+?)/?$",
			'index.php?category_name=$matches[2]&' . self::QUERY_VAR . '=$matches[1]',
			'top'
		);

		// Rewrite rule for tag archives
		add_rewrite_rule(
			"^{$lang_pattern}/tag/(.+?)/?$",
			'index.php?tag=$matches[2]&' . self::QUERY_VAR . '=$matches[1]',
			'top'
		);
	}

	/**
	 * Add query variable for language
	 *
	 * @param array $vars Query variables
	 * @return array Modified query variables
	 */
	public function add_query_vars( $vars ) {
		$vars[] = self::QUERY_VAR;
		return $vars;
	}

	/**
	 * Get supported languages from settings
	 *
	 * @return array Supported language codes
	 */
	public function get_supported_languages() {
		$settings = GFMR_Settings::get_instance()->get_settings();
		$languages = $settings['multilingual_supported_languages'] ?? array( 'en', 'ja' );

		// Validate and sanitize language codes
		$valid_languages = array();
		foreach ( (array) $languages as $lang ) {
			$lang = sanitize_key( $lang );
			if ( preg_match( '/^[a-z]{2}$/', $lang ) ) {
				$valid_languages[] = $lang;
			}
		}

		return $valid_languages ? $valid_languages : array( 'en' );
	}

	/**
	 * Get default language from settings
	 *
	 * @return string Default language code
	 */
	public function get_default_language() {
		$settings = GFMR_Settings::get_instance()->get_settings();
		$default = $settings['multilingual_default_language'] ?? '';

		// Fallback to WordPress locale
		if ( ! $default ) {
			$locale = get_locale();
			$default = substr( $locale, 0, 2 );
		}

		return sanitize_key( $default );
	}

	/**
	 * Check if path mode is enabled
	 *
	 * @return bool True if path mode is enabled
	 */
	public function is_path_mode_enabled() {
		$settings = GFMR_Settings::get_instance()->get_settings();
		return ( $settings['multilingual_url_mode'] ?? 'query' ) === 'path';
	}

	/**
	 * Check if default language should be hidden in URL
	 *
	 * @return bool True if default language is hidden
	 */
	public function hide_default_language() {
		$settings = GFMR_Settings::get_instance()->get_settings();
		return (bool) ( $settings['multilingual_hide_default_lang'] ?? true );
	}

	/**
	 * Check if query to path redirect is enabled
	 *
	 * @return bool True if redirect is enabled
	 */
	public function is_redirect_enabled() {
		$settings = GFMR_Settings::get_instance()->get_settings();
		return (bool) ( $settings['multilingual_redirect_query'] ?? true );
	}

	/**
	 * Check if hreflang tags are enabled
	 *
	 * @return bool True if hreflang is enabled
	 */
	public function is_hreflang_enabled() {
		$settings = GFMR_Settings::get_instance()->get_settings();
		return (bool) ( $settings['multilingual_hreflang_enabled'] ?? true );
	}

	/**
	 * Validate language code
	 *
	 * @param string $lang Language code to validate
	 * @return bool True if valid
	 */
	public function is_valid_language( $lang ) {
		$lang = sanitize_key( $lang );
		return in_array( $lang, $this->get_supported_languages(), true );
	}

	/**
	 * Detect current language from URL
	 *
	 * @return string Current language code
	 */
	public function detect_language() {
		// Check query variable
		$lang = get_query_var( self::QUERY_VAR );
		if ( $lang && $this->is_valid_language( $lang ) ) {
			return $lang;
		}

		// Check URL path for language prefix
		$path = trim( $_SERVER['REQUEST_URI'], '/' );
		$segments = explode( '/', $path );
		if ( ! empty( $segments[0] ) && $this->is_valid_language( $segments[0] ) ) {
			return $segments[0];
		}

		// Default language
		return $this->get_default_language();
	}

	/**
	 * Get current language
	 *
	 * @return string Current language code
	 */
	public function get_current_language() {
		return $this->detect_language();
	}

	/**
	 * Generate language URL for a post
	 *
	 * @param int $post_id Post ID
	 * @param string $lang Language code
	 * @return string|false URL or false on failure
	 */
	public function generate_language_url( $post_id, $lang ) {
		if ( ! $this->is_valid_language( $lang ) ) {
			return false;
		}

		$post = get_post( $post_id );
		if ( ! $post ) {
			return false;
		}

		$permalink = get_permalink( $post_id );
		if ( ! $permalink ) {
			return false;
		}

		$default_lang = $this->get_default_language();
		$hide_default = $this->hide_default_language();

		// If default language and hiding is enabled, return original permalink
		if ( $lang === $default_lang && $hide_default ) {
			return $permalink;
		}

		// Parse URL and insert language prefix
		$parsed = wp_parse_url( $permalink );
		$path = isset( $parsed['path'] ) ? $parsed['path'] : '';

		// Remove trailing slash for consistent handling
		$path = untrailingslashit( $path );

		// Add language prefix
		$path = '/' . $lang . $path;

		// Rebuild URL
		$scheme = isset( $parsed['scheme'] ) ? $parsed['scheme'] : 'https';
		$host = isset( $parsed['host'] ) ? $parsed['host'] : $_SERVER['HTTP_HOST'];
		$query = isset( $parsed['query'] ) ? '?' . $parsed['query'] : '';

		$url = $scheme . '://' . $host . $path . '/' . $query;

		return $url;
	}

	/**
	 * Get canonical URL for a post in current language
	 *
	 * @param int $post_id Post ID
	 * @param string|null $lang Language code (null for current)
	 * @return string|false Canonical URL or false on failure
	 */
	public function get_canonical_url( $post_id, $lang = null ) {
		if ( null === $lang ) {
			$lang = $this->get_current_language();
		}
		return $this->generate_language_url( $post_id, $lang );
	}

	/**
	 * Output hreflang tags for current post
	 */
	public function output_hreflang_tags() {
		if ( ! $this->is_hreflang_enabled() ) {
			return;
		}

		if ( ! is_singular() ) {
			return;
		}

		$post_id = get_the_ID();
		if ( ! $post_id ) {
			return;
		}

		$languages = $this->get_supported_languages();
		$default_lang = $this->get_default_language();

		foreach ( $languages as $lang ) {
			$url = $this->generate_language_url( $post_id, $lang );
			if ( $url ) {
				printf(
					'<link rel="alternate" hreflang="%s" href="%s" />' . "\n",
					esc_attr( $lang ),
					esc_url( $url )
				);
			}
		}

		// Output x-default
		$default_url = $this->generate_language_url( $post_id, $default_lang );
		if ( $default_url ) {
			printf(
				'<link rel="alternate" hreflang="x-default" href="%s" />' . "\n",
				esc_url( $default_url )
			);
		}
	}

	/**
	 * Output canonical tag for current post
	 */
	public function output_canonical_tag() {
		if ( ! is_singular() ) {
			return;
		}

		$post_id = get_the_ID();
		if ( ! $post_id ) {
			return;
		}

		$current_lang = $this->get_current_language();
		$canonical_url = $this->get_canonical_url( $post_id, $current_lang );

		if ( $canonical_url ) {
			printf(
				'<link rel="canonical" href="%s" />' . "\n",
				esc_url( $canonical_url )
			);
		}
	}

	/**
	 * Redirect query parameter URLs to path-based URLs
	 */
	public function maybe_redirect_query_to_path() {
		if ( ! $this->is_path_mode_enabled() ) {
			return;
		}

		if ( ! $this->is_redirect_enabled() ) {
			return;
		}

		// Check if ?lang= parameter exists
		if ( isset( $_GET['lang'] ) && $this->is_valid_language( $_GET['lang'] ) ) {
			$lang = sanitize_key( $_GET['lang'] );
			$post_id = get_queried_object_id();

			if ( $post_id ) {
				$new_url = $this->generate_language_url( $post_id, $lang );
				if ( $new_url ) {
					wp_safe_redirect( $new_url, 301 );
					exit;
				}
			}
		}
	}
}
