<?php
/**
 * SSR Renderer - Server-Side Rendering for code blocks and diagrams
 *
 * Central coordinator for SSR processing
 *
 * @package WpGfmRenderer
 * @since 2.0.0
 */

namespace Wakalab\WpGfmRenderer;

defined( 'ABSPATH' ) || exit;

// Prevent class redeclaration when both Free and Pro versions are active
if ( class_exists( __NAMESPACE__ . '\\GFMR_SSR_Renderer' ) ) {
	return;
}

class GFMR_SSR_Renderer {

	/**
	 * Cache manager instance
	 *
	 * @var GFMR_Cache_Manager
	 */
	private $cache_manager;

	/**
	 * Code highlighter instance
	 *
	 * @var GFMR_Code_Highlighter
	 */
	private $code_highlighter;

	/**
	 * Mermaid SSR handler instance
	 *
	 * @var GFMR_Mermaid_SSR_Handler
	 */
	private $mermaid_handler;

	/**
	 * Settings instance
	 *
	 * @var GFMR_Settings
	 */
	private $settings;

	/**
	 * Constructor
	 *
	 * @param GFMR_Cache_Manager         $cache_manager Cache manager instance
	 * @param GFMR_Code_Highlighter      $code_highlighter Code highlighter instance
	 * @param GFMR_Mermaid_SSR_Handler   $mermaid_handler Mermaid SSR handler instance
	 * @param GFMR_Settings              $settings Settings instance
	 */
	public function __construct( $cache_manager, $code_highlighter, $mermaid_handler, $settings ) {
		$this->cache_manager    = $cache_manager;
		$this->code_highlighter = $code_highlighter;
		$this->mermaid_handler  = $mermaid_handler;
		$this->settings         = $settings;
	}

	/**
	 * Process HTML content and apply SSR
	 *
	 * @param string $html HTML content to process
	 * @param string $shiki_theme Shiki theme identifier
	 * @param string $mermaid_bg_color Mermaid background color
	 * @return string Processed HTML
	 */
	public function process_html( $html, $shiki_theme, $mermaid_bg_color ) {
		// Process code blocks
		if ( $this->settings->is_code_block_ssr_enabled() ) {
			$html = $this->process_code_blocks( $html, $shiki_theme );
		}

		// Process Mermaid diagrams
		if ( $this->settings->is_mermaid_ssr_enabled() ) {
			$html = $this->process_mermaid_diagrams( $html, $shiki_theme, $mermaid_bg_color );
		}

		return $html;
	}

	/**
	 * Process code blocks in HTML
	 *
	 * @param string $html HTML content
	 * @param string $theme Shiki theme
	 * @return string Processed HTML
	 */
	private function process_code_blocks( $html, $theme ) {
		// Pattern to match code blocks: <pre><code class="language-xxx">...</code></pre>
		$pattern = '/<pre><code\s+class="language-(\w+)">(.*?)<\/code><\/pre>/s';

		return preg_replace_callback(
			$pattern,
			function ( $matches ) use ( $theme ) {
				$language = $matches[1];
				$code     = html_entity_decode( $matches[2], ENT_QUOTES | ENT_HTML5 );

				// Generate cache key
				$cache_key = $this->generate_cache_key( $code, $language, $theme );

				// Check cache
				$cached = $this->cache_manager->get_ssr_cache( $cache_key, $theme );
				if ( false !== $cached ) {
					$this->log_debug( "Cache hit for code block: {$language}" );
					return $cached;
				}

				// Render with highlight.php
				$highlighted = $this->code_highlighter->highlight( $code, $language );

				// Cache result
				$this->cache_manager->set_ssr_cache( $cache_key, $theme, $highlighted );

				return $highlighted;
			},
			$html
		);
	}

	/**
	 * Process Mermaid diagrams in HTML
	 *
	 * @param string $html HTML content
	 * @param string $theme Theme identifier
	 * @param string $bg_color Background color
	 * @return string Processed HTML
	 */
	private function process_mermaid_diagrams( $html, $theme, $bg_color ) {
		// Pattern to match Mermaid code blocks in gfmr-mermaid-container
		// Looking for: <div class="gfmr-mermaid-container" ...><pre class="gfmr-mermaid-source">...</pre></div>
		$pattern = '/<div class="gfmr-mermaid-container"[^>]*>.*?<pre class="gfmr-mermaid-source"[^>]*>(.*?)<\/pre>.*?<\/div>/s';

		return preg_replace_callback(
			$pattern,
			function ( $matches ) use ( $theme, $bg_color ) {
				$mermaid_code = html_entity_decode( $matches[1], ENT_QUOTES | ENT_HTML5 );
				$diagram_id   = md5( $mermaid_code );

				// Check cache
				$cached_svg = $this->mermaid_handler->get_cached_svg( $diagram_id, $theme, $bg_color );

				if ( false !== $cached_svg ) {
					// Cache hit - return SSR markup with SVG
					$this->log_debug( "Cache hit for Mermaid diagram: {$diagram_id}" );
					return $this->generate_ssr_mermaid_markup( $diagram_id, $theme, $bg_color, $cached_svg );
				}

				// Cache miss - return placeholder for client-side rendering
				$this->log_debug( "Cache miss for Mermaid diagram: {$diagram_id}" );
				return $this->mermaid_handler->generate_placeholder( $diagram_id, $theme, $bg_color, $mermaid_code );
			},
			$html
		);
	}

	/**
	 * Generate SSR-rendered Mermaid markup
	 *
	 * @param string $diagram_id Diagram identifier
	 * @param string $theme Theme identifier
	 * @param string $bg_color Background color
	 * @param string $svg SVG content
	 * @return string Rendered markup
	 */
	private function generate_ssr_mermaid_markup( $diagram_id, $theme, $bg_color, $svg ) {
		$markup  = '<div class="gfmr-mermaid-container" ';
		$markup .= 'data-diagram-id="' . esc_attr( $diagram_id ) . '" ';
		$markup .= 'data-theme="' . esc_attr( $theme ) . '" ';
		$markup .= 'data-mermaid-bg-color="' . esc_attr( $bg_color ) . '" ';
		$markup .= 'data-ssr="true" ';
		$markup .= 'data-gfmr-processed="true">';
		$markup .= $svg; // SVG is already sanitized by mermaid_handler
		$markup .= '</div>';

		return $markup;
	}

	/**
	 * Generate cache key for content
	 *
	 * @param string $content Content to hash
	 * @param string $language Language identifier
	 * @param string $theme Theme identifier
	 * @return string Cache key
	 */
	private function generate_cache_key( $content, $language, $theme ) {
		return sprintf(
			'code_%s_%s_%s',
			$language,
			$theme,
			md5( $content )
		);
	}

	/**
	 * Debug logging
	 *
	 * @param string $message Log message
	 */
	private function log_debug( $message ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( '[GFMR_SSR_Renderer] ' . $message );
		}
	}
}
