<?php
/**
 * GFMR Settings Class
 *
 * Handles plugin settings through WordPress Options API
 * Provides Light/Dark theme selection for Shiki syntax highlighting
 *
 * @package MarkdownRendererForGitHub
 * @since 2.1.0
 */

namespace Wakalab\WpGfmRenderer;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Prevent class redeclaration when both Free and Pro versions are active
if ( class_exists( __NAMESPACE__ . '\\GFMR_Settings' ) ) {
	return;
}

/**
 * Settings management class for GFMR plugin
 */
class GFMR_Settings {

	/**
	 * Singleton instance
	 *
	 * @var GFMR_Settings|null
	 */
	private static $instance = null;

	/**
	 * Flag to track if hooks have been registered
	 *
	 * @var bool
	 */
	private static $hooks_registered = false;

	/**
	 * Option name for theme settings
	 */
	const OPTION_NAME = 'gfmr_theme_settings';

	/**
	 * Get singleton instance
	 *
	 * @return GFMR_Settings
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Available theme options
	 */
	const AVAILABLE_THEMES = array(
		'light'  => 'github-light',
		'dark'   => 'github-dark',
		'system' => 'auto',
	);

	/**
	 * Default theme setting (system = follow OS preference)
	 */
	const DEFAULT_THEME = 'system';

	/**
	 * Settings page slug
	 */
	const SETTINGS_SLUG = 'gfmr-settings';

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->init();
	}

	/**
	 * Initialize the settings
	 */
	public function init() {
		if ( self::$hooks_registered ) {
			return;
		}
		self::$hooks_registered = true;

		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_init', array( $this, 'init_settings' ) );
		add_filter( 'wp_redirect', array( $this, 'preserve_active_tab_on_redirect' ) );
		add_action( 'wp_ajax_gfmr_preview_theme', array( $this, 'ajax_preview_theme' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
	}

	/**
	 * Preserve active tab on settings save redirect
	 *
	 * @param string $location Redirect location URL.
	 * @return string Modified redirect location URL
	 */
	public function preserve_active_tab_on_redirect( $location ) {
		if ( isset( $_POST['gfmr_active_tab'] ) && strpos( $location, 'page=' . self::SETTINGS_SLUG ) !== false ) {
			$tab      = \sanitize_key( $_POST['gfmr_active_tab'] );
			$location = add_query_arg( 'tab', $tab, $location );
		}
		return $location;
	}


	/**
	 * Add admin menu page
	 */
	public function add_admin_menu() {
		add_options_page(
			__( 'Markdown Renderer for GitHub Settings', 'markdown-renderer-for-github' ),
			__( 'Markdown Renderer for GitHub', 'markdown-renderer-for-github' ),
			'manage_options',
			self::SETTINGS_SLUG,
			array( $this, 'render_settings_page' )
		);
	}

	/**
	 * Initialize settings and register fields
	 */
	public function init_settings() {
		register_setting(
			'gfmr_settings_group',
			self::OPTION_NAME,
			array(
				'type'              => 'array',
				'sanitize_callback' => array( $this, 'sanitize_settings' ),
				'default'           => $this->get_default_settings(),
			)
		);

		add_settings_section(
			'gfmr_theme_section',
			__( 'Code Block Theme', 'markdown-renderer-for-github' ),
			array( $this, 'render_theme_section' ),
			self::SETTINGS_SLUG
		);

		add_settings_field(
			'theme_selection',
			__( 'Theme Selection', 'markdown-renderer-for-github' ),
			array( $this, 'render_theme_field' ),
			self::SETTINGS_SLUG,
			'gfmr_theme_section'
		);

		// General settings section.
		add_settings_section(
			'gfmr_general_section',
			__( 'General Settings', 'markdown-renderer-for-github' ),
			array( $this, 'render_general_section' ),
			self::SETTINGS_SLUG
		);

		add_settings_field(
			'frontmatter_auto_title',
			__( 'Auto-apply Frontmatter Title', 'markdown-renderer-for-github' ),
			array( $this, 'render_frontmatter_auto_title_field' ),
			self::SETTINGS_SLUG,
			'gfmr_general_section'
		);

		add_settings_field(
			'show_frontmatter_header',
			__( 'Show Frontmatter Header', 'markdown-renderer-for-github' ),
			array( $this, 'render_show_frontmatter_field' ),
			self::SETTINGS_SLUG,
			'gfmr_general_section'
		);

		// Schema.org structured data section.
		add_settings_section(
			'gfmr_schema_section',
			__( 'Structured Data (Schema.org)', 'markdown-renderer-for-github' ),
			array( $this, 'render_schema_section' ),
			self::SETTINGS_SLUG
		);

		add_settings_field(
			'schema_enabled',
			__( 'Enable Structured Data', 'markdown-renderer-for-github' ),
			array( $this, 'render_schema_enabled_field' ),
			self::SETTINGS_SLUG,
			'gfmr_schema_section'
		);

		add_settings_field(
			'schema_auto_detect_type',
			__( 'Auto-detect Article Type', 'markdown-renderer-for-github' ),
			array( $this, 'render_schema_auto_detect_field' ),
			self::SETTINGS_SLUG,
			'gfmr_schema_section'
		);

		add_settings_field(
			'schema_include_author',
			__( 'Include Author Information', 'markdown-renderer-for-github' ),
			array( $this, 'render_schema_author_field' ),
			self::SETTINGS_SLUG,
			'gfmr_schema_section'
		);

		add_settings_field(
			'schema_include_publisher',
			__( 'Include Publisher Information', 'markdown-renderer-for-github' ),
			array( $this, 'render_schema_publisher_field' ),
			self::SETTINGS_SLUG,
			'gfmr_schema_section'
		);

		// Server-Side Rendering (SSR) section.
		add_settings_section(
			'gfmr_ssr_section',
			__( 'Server-Side Rendering (SSR)', 'markdown-renderer-for-github' ),
			array( $this, 'render_ssr_section' ),
			self::SETTINGS_SLUG
		);

		add_settings_field(
			'ssr_enabled',
			__( 'Enable SSR', 'markdown-renderer-for-github' ),
			array( $this, 'render_ssr_enabled_field' ),
			self::SETTINGS_SLUG,
			'gfmr_ssr_section'
		);

		add_settings_field(
			'ssr_code_blocks',
			__( 'Code Block SSR', 'markdown-renderer-for-github' ),
			array( $this, 'render_ssr_code_blocks_field' ),
			self::SETTINGS_SLUG,
			'gfmr_ssr_section'
		);

		add_settings_field(
			'ssr_mermaid',
			__( 'Mermaid Diagram SSR', 'markdown-renderer-for-github' ),
			array( $this, 'render_ssr_mermaid_field' ),
			self::SETTINGS_SLUG,
			'gfmr_ssr_section'
		);

		// Table of Contents (TOC) section.
		add_settings_section(
			'gfmr_toc_section',
			__( 'Table of Contents (TOC)', 'markdown-renderer-for-github' ),
			array( $this, 'render_toc_section' ),
			self::SETTINGS_SLUG
		);

		add_settings_field(
			'toc_enabled',
			__( 'Enable TOC', 'markdown-renderer-for-github' ),
			array( $this, 'render_toc_enabled_field' ),
			self::SETTINGS_SLUG,
			'gfmr_toc_section'
		);

		add_settings_field(
			'toc_heading_range',
			__( 'Heading Range', 'markdown-renderer-for-github' ),
			array( $this, 'render_toc_heading_range_field' ),
			self::SETTINGS_SLUG,
			'gfmr_toc_section'
		);

		add_settings_field(
			'toc_position',
			__( 'Position', 'markdown-renderer-for-github' ),
			array( $this, 'render_toc_position_field' ),
			self::SETTINGS_SLUG,
			'gfmr_toc_section'
		);

		add_settings_field(
			'toc_scroll_offset',
			__( 'Scroll Offset', 'markdown-renderer-for-github' ),
			array( $this, 'render_toc_scroll_offset_field' ),
			self::SETTINGS_SLUG,
			'gfmr_toc_section'
		);

		add_settings_field(
			'toc_sticky',
			__( 'Sticky Sidebar', 'markdown-renderer-for-github' ),
			array( $this, 'render_toc_sticky_field' ),
			self::SETTINGS_SLUG,
			'gfmr_toc_section'
		);

		add_settings_field(
			'toc_mobile_hidden',
			__( 'Mobile Visibility', 'markdown-renderer-for-github' ),
			array( $this, 'render_toc_mobile_hidden_field' ),
			self::SETTINGS_SLUG,
			'gfmr_toc_section'
		);

		// Extensions section.
		add_settings_section(
			'gfmr_extensions_section',
			__( 'Extensions', 'markdown-renderer-for-github' ),
			array( $this, 'render_extensions_section' ),
			self::SETTINGS_SLUG
		);

		// Multilingual section.
		add_settings_section(
			'gfmr_multilingual_section',
			__( 'Multilingual URLs', 'markdown-renderer-for-github' ),
			array( $this, 'render_multilingual_section' ),
			self::SETTINGS_SLUG
		);

		add_settings_field(
			'multilingual_url_mode',
			__( 'URL Mode', 'markdown-renderer-for-github' ),
			array( $this, 'render_multilingual_url_mode_field' ),
			self::SETTINGS_SLUG,
			'gfmr_multilingual_section'
		);

		add_settings_field(
			'multilingual_default_language',
			__( 'Default Language', 'markdown-renderer-for-github' ),
			array( $this, 'render_multilingual_default_language_field' ),
			self::SETTINGS_SLUG,
			'gfmr_multilingual_section'
		);

		add_settings_field(
			'multilingual_supported_languages',
			__( 'Supported Languages', 'markdown-renderer-for-github' ),
			array( $this, 'render_multilingual_supported_languages_field' ),
			self::SETTINGS_SLUG,
			'gfmr_multilingual_section'
		);

		add_settings_field(
			'multilingual_hide_default_lang',
			__( 'Hide Default Language in URL', 'markdown-renderer-for-github' ),
			array( $this, 'render_multilingual_hide_default_lang_field' ),
			self::SETTINGS_SLUG,
			'gfmr_multilingual_section'
		);

		add_settings_field(
			'multilingual_hreflang_enabled',
			__( 'Output hreflang Tags', 'markdown-renderer-for-github' ),
			array( $this, 'render_multilingual_hreflang_enabled_field' ),
			self::SETTINGS_SLUG,
			'gfmr_multilingual_section'
		);

		add_settings_field(
			'multilingual_redirect_query',
			__( 'Redirect ?lang= to /lang/', 'markdown-renderer-for-github' ),
			array( $this, 'render_multilingual_redirect_query_field' ),
			self::SETTINGS_SLUG,
			'gfmr_multilingual_section'
		);
	}

	/**
	 * Get default settings
	 *
	 * @return array Default settings
	 */
	private function get_default_settings() {
		$locale = \get_locale();
		$default_lang = substr( $locale, 0, 2 );

		return array(
			'theme'                            => self::DEFAULT_THEME,
			// Schema settings (default: OFF to avoid conflicts with SEO plugins).
			'schema_enabled'                   => false,
			'schema_auto_detect_type'          => true,
			'schema_include_author'            => true,
			'schema_include_publisher'         => true,
			// Frontmatter settings.
			'frontmatter_auto_title'           => false,
			'show_frontmatter_header'          => false,
			// SSR settings (default: OFF for gradual adoption).
			'ssr_enabled'                      => false,
			'ssr_code_blocks'                  => true,
			'ssr_mermaid'                      => true,
			// Table of Contents (TOC) settings.
			'toc_enabled'                      => false,
			'toc_heading_min'                  => 2,
			'toc_heading_max'                  => 4,
			'toc_position'                     => 'right',
			'toc_scroll_offset'                => 0,
			'toc_sticky'                       => true,
			'toc_mobile_hidden'                => true,
			// Multilingual URL settings.
			'multilingual_url_mode'            => 'path',
			'multilingual_hide_default_lang'   => true,
			'multilingual_default_language'    => $default_lang,
			'multilingual_supported_languages' => array( 'en', 'ja' ),
			'multilingual_hreflang_enabled'    => true,
			'multilingual_redirect_query'      => true,
		);
	}

	/**
	 * Sanitize settings input
	 *
	 * @param array $input Raw input data
	 * @return array Sanitized settings
	 */
	public function sanitize_settings( $input ) {
		$sanitized = array();

		// Validate theme selection.
		if ( isset( $input['theme'] ) && array_key_exists( $input['theme'], self::AVAILABLE_THEMES ) ) {
			$sanitized['theme'] = \sanitize_key( $input['theme'] );
		} else {
			$sanitized['theme'] = self::DEFAULT_THEME;
			add_settings_error(
				self::OPTION_NAME,
				'invalid_theme',
				__( 'Invalid theme selection. Reset to default.', 'markdown-renderer-for-github' ),
				'warning'
			);
		}

		// Sanitize schema settings (boolean values).
		$sanitized['schema_enabled']           = ! empty( $input['schema_enabled'] );
		$sanitized['schema_auto_detect_type']  = ! empty( $input['schema_auto_detect_type'] );
		$sanitized['schema_include_author']    = ! empty( $input['schema_include_author'] );
		$sanitized['schema_include_publisher'] = ! empty( $input['schema_include_publisher'] );

		// Sanitize frontmatter settings (boolean values).
		$sanitized['frontmatter_auto_title']  = ! empty( $input['frontmatter_auto_title'] );
		$sanitized['show_frontmatter_header'] = ! empty( $input['show_frontmatter_header'] );

		// Sanitize SSR settings (boolean values).
		$sanitized['ssr_enabled']     = ! empty( $input['ssr_enabled'] );
		$sanitized['ssr_code_blocks'] = ! empty( $input['ssr_code_blocks'] );
		$sanitized['ssr_mermaid']     = ! empty( $input['ssr_mermaid'] );

		// Sanitize TOC settings.
		$sanitized['toc_enabled']       = ! empty( $input['toc_enabled'] );
		$sanitized['toc_heading_min']   = isset( $input['toc_heading_min'] ) ? \absint( $input['toc_heading_min'] ) : 2;
		$sanitized['toc_heading_max']   = isset( $input['toc_heading_max'] ) ? \absint( $input['toc_heading_max'] ) : 4;
		$sanitized['toc_position']      = isset( $input['toc_position'] ) && in_array( $input['toc_position'], array( 'left', 'right' ), true ) ? \sanitize_key( $input['toc_position'] ) : 'right';
		$sanitized['toc_scroll_offset'] = isset( $input['toc_scroll_offset'] ) ? \intval( $input['toc_scroll_offset'] ) : 0;
		$sanitized['toc_sticky']        = ! empty( $input['toc_sticky'] );
		$sanitized['toc_mobile_hidden'] = ! empty( $input['toc_mobile_hidden'] );

		// Ensure min is less than max for heading range.
		if ( $sanitized['toc_heading_min'] > $sanitized['toc_heading_max'] ) {
			$sanitized['toc_heading_min'] = $sanitized['toc_heading_max'];
		}

		// Sanitize multilingual URL settings.
		$sanitized['multilingual_url_mode'] = isset( $input['multilingual_url_mode'] ) && in_array( $input['multilingual_url_mode'], array( 'query', 'path' ), true ) ? \sanitize_key( $input['multilingual_url_mode'] ) : 'path';
		$sanitized['multilingual_hide_default_lang'] = ! empty( $input['multilingual_hide_default_lang'] );
		$sanitized['multilingual_default_language'] = isset( $input['multilingual_default_language'] ) ? \sanitize_key( $input['multilingual_default_language'] ) : 'en';
		$sanitized['multilingual_hreflang_enabled'] = ! empty( $input['multilingual_hreflang_enabled'] );
		$sanitized['multilingual_redirect_query'] = ! empty( $input['multilingual_redirect_query'] );

		// Sanitize supported languages array.
		if ( isset( $input['multilingual_supported_languages'] ) && is_array( $input['multilingual_supported_languages'] ) ) {
			$sanitized_languages = array();
			foreach ( $input['multilingual_supported_languages'] as $lang ) {
				$lang = \sanitize_key( $lang );
				if ( preg_match( '/^[a-z]{2}$/', $lang ) ) {
					$sanitized_languages[] = $lang;
				}
			}
			// Ensure at least one language is selected.
			$sanitized['multilingual_supported_languages'] = ! empty( $sanitized_languages ) ? array_values( array_unique( $sanitized_languages ) ) : array( 'en' );
		} else {
			$sanitized['multilingual_supported_languages'] = array( 'en', 'ja' );
		}

		return $sanitized;
	}

	/**
	 * Get tabs configuration for settings page
	 *
	 * @return array Tabs configuration
	 */
	private function get_tabs() {
		$tabs = array(
			'code-block'   => array(
				'title'    => __( 'Code Block', 'markdown-renderer-for-github' ),
				'sections' => array( 'gfmr_theme_section' ),
			),
			'general'      => array(
				'title'    => __( 'General Settings', 'markdown-renderer-for-github' ),
				'sections' => array( 'gfmr_general_section', 'gfmr_ssr_section' ),
			),
			'toc'          => array(
				'title'    => __( 'Table of Contents', 'markdown-renderer-for-github' ),
				'sections' => array( 'gfmr_toc_section' ),
			),
			'multilingual' => array(
				'title'    => __( 'Multilingual', 'markdown-renderer-for-github' ),
				'sections' => array( 'gfmr_multilingual_section' ),
			),
			'schema'       => array(
				'title'    => __( 'Structured Data', 'markdown-renderer-for-github' ),
				'sections' => array( 'gfmr_schema_section' ),
			),
			'extensions'   => array(
				'title'    => __( 'Extensions', 'markdown-renderer-for-github' ),
				'sections' => array( 'gfmr_extensions_section' ),
			),
		);

		/**
		 * Filters the settings tabs list
		 *
		 * Allows addons to add their own settings tabs.
		 *
		 * @since 2.0.0
		 *
		 * @param array $tabs Tabs configuration array
		 */
		return apply_filters( 'gfmr_settings_tabs_list', $tabs );
	}

	/**
	 * Render settings page
	 */
	public function render_settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Handle settings save.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- WordPress Settings API verified nonce in options.php before redirect
		if ( isset( $_GET['settings-updated'] ) && $_GET['settings-updated'] ) {
			add_settings_error(
				self::OPTION_NAME,
				'settings_updated',
				__( 'Settings saved successfully.', 'markdown-renderer-for-github' ),
				'updated'
			);
		}

		$current_settings = $this->get_settings();
		$tabs             = $this->get_tabs();
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only tab selection, sanitized with sanitize_key
		$default_tab = 'code-block';
		$current_tab = isset( $_GET['tab'] ) ? \sanitize_key( $_GET['tab'] ) : $default_tab;

		// Validate current tab.
		if ( ! array_key_exists( $current_tab, $tabs ) ) {
			$current_tab = $default_tab;
		}
		?>
		<div class="wrap">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>

			<?php settings_errors( self::OPTION_NAME ); ?>

			<nav class="nav-tab-wrapper">
				<?php foreach ( $tabs as $tab_id => $tab ) : ?>
					<a href="#"
						class="nav-tab<?php echo $current_tab === $tab_id ? ' nav-tab-active' : ''; ?>"
						data-tab="<?php echo esc_attr( $tab_id ); ?>">
						<?php echo esc_html( $tab['title'] ); ?>
					</a>
				<?php endforeach; ?>
			</nav>

			<form action="options.php" method="post" id="gfmr-settings-form">
				<?php settings_fields( 'gfmr_settings_group' ); ?>

				<?php foreach ( $tabs as $tab_id => $tab ) : ?>
					<div id="tab-<?php echo esc_attr( $tab_id ); ?>" class="gfmr-tab-content" style="<?php echo $current_tab === $tab_id ? '' : 'display:none;'; ?>">
						<?php
						// Render sections for this tab.
						foreach ( $tab['sections'] as $section_id ) {
							$this->render_section( $section_id );
						}

						/**
						 * Fires after rendering sections for each tab
						 *
						 * Allows addons to add custom content for specific tabs.
						 *
						 * @since 1.15.0
						 *
						 * @param string $page_slug  Settings page slug
						 * @param string $tab_id     Current tab ID
						 */
						do_action( 'gfmr_settings_sections', self::SETTINGS_SLUG, $tab_id );

						// Add preview for code-block tab.
						if ( 'code-block' === $tab_id ) :
							?>
							<div id="gfmr-theme-preview" class="gfmr-preview-container" data-current-theme="<?php echo esc_attr( $current_settings['theme'] ); ?>">
								<h3><?php esc_html_e( 'Live Preview', 'markdown-renderer-for-github' ); ?></h3>
								<div class="gfmr-preview-content">
									<div id="gfmr-preview-wrapper" class="gfmr-admin-preview-wrapper">
										<pre id="gfmr-preview-code" class="language-javascript"><code>// Syntax highlighting theme preview
function calculateSum(a, b) {
	// This function adds two numbers
	const result = a + b;
	console.log(`Result: ${result}`);
	return result;
}

// Usage example
const numbers = [1, 2, 3, 4, 5];
const sum = numbers.reduce(calculateSum);
console.log('Total sum:', sum);</code></pre>
									</div>
								</div>
								<div class="gfmr-preview-info">
									<p class="description">
										<?php esc_html_e( 'Preview updates automatically when you select a different theme option above.', 'markdown-renderer-for-github' ); ?>
									</p>
									<p class="gfmr-current-theme">
										<?php
										printf(
											/* translators: %s: current theme name */
											esc_html__( 'Current theme: %s', 'markdown-renderer-for-github' ),
											'<span id="gfmr-theme-display">' . esc_html( ucfirst( $current_settings['theme'] ) ) . '</span>'
										);
										?>
									</p>
								</div>
							</div>
							<?php
						endif;
						?>
					</div>
				<?php endforeach; ?>

				<?php submit_button(); ?>
			</form>
		</div>
		<?php
	}

	/**
	 * Render a single settings section
	 *
	 * @param string $section_id Section ID to render.
	 */
	private function render_section( $section_id ) {
		global $wp_settings_sections, $wp_settings_fields;

		if ( ! isset( $wp_settings_sections[ self::SETTINGS_SLUG ][ $section_id ] ) ) {
			return;
		}

		$section = $wp_settings_sections[ self::SETTINGS_SLUG ][ $section_id ];

		// Section title and callback.
		if ( $section['title'] ) {
			echo '<h2>' . esc_html( $section['title'] ) . '</h2>';
		}

		if ( $section['callback'] ) {
			call_user_func( $section['callback'], $section );
		}

		// Section fields.
		if ( ! isset( $wp_settings_fields ) || ! isset( $wp_settings_fields[ self::SETTINGS_SLUG ][ $section_id ] ) ) {
			return;
		}

		echo '<table class="form-table" role="presentation">';
		do_settings_fields( self::SETTINGS_SLUG, $section_id );
		echo '</table>';
	}

	/**
	 * Render theme section description
	 */
	public function render_theme_section() {
		echo '<p>' . esc_html__( 'Choose the appearance theme for code blocks in your content.', 'markdown-renderer-for-github' ) . '</p>';
	}

	/**
	 * Render general section description
	 */
	public function render_general_section() {
		echo '<p>' . esc_html__( 'Configure general Markdown rendering options.', 'markdown-renderer-for-github' ) . '</p>';
	}

	/**
	 * Render frontmatter auto title field
	 */
	public function render_frontmatter_auto_title_field() {
		$options = $this->get_settings();
		$checked = ! empty( $options['frontmatter_auto_title'] );
		?>
		<label>
			<input type="checkbox"
					name="<?php echo esc_attr( self::OPTION_NAME ); ?>[frontmatter_auto_title]"
					value="1"
					<?php checked( $checked ); ?> />
			<?php esc_html_e( 'Automatically use frontmatter title as page title', 'markdown-renderer-for-github' ); ?>
		</label>
		<p class="description">
			<?php esc_html_e( 'When enabled, if a Markdown block has frontmatter with a title field, it will be used as the page title in the browser tab and SEO metadata.', 'markdown-renderer-for-github' ); ?>
		</p>
		<?php
	}

	/**
	 * Render show frontmatter header field
	 */
	public function render_show_frontmatter_field() {
		$options = $this->get_settings();
		$checked = ! empty( $options['show_frontmatter_header'] );
		?>
		<label>
			<input type="checkbox"
					name="<?php echo esc_attr( self::OPTION_NAME ); ?>[show_frontmatter_header]"
					value="1"
					<?php checked( $checked ); ?> />
			<?php esc_html_e( 'Display frontmatter metadata as header by default', 'markdown-renderer-for-github' ); ?>
		</label>
		<p class="description">
			<?php esc_html_e( 'When enabled, new Markdown blocks will display YAML frontmatter (title, date, author, tags) as a formatted header. This can be overridden per block.', 'markdown-renderer-for-github' ); ?>
		</p>
		<?php
	}

	/**
	 * Render theme selection field
	 */
	public function render_theme_field() {
		$options       = $this->get_settings();
		$current_theme = $options['theme'];
		?>
		<fieldset>
			<div class="gfmr-theme-option">
				<label>
					<input type="radio"
							name="<?php echo esc_attr( self::OPTION_NAME ); ?>[theme]"
							value="light"
							<?php checked( $current_theme, 'light' ); ?> />
					<?php esc_html_e( 'Light Theme', 'markdown-renderer-for-github' ); ?>
					<p class="description"><?php esc_html_e( 'Bright background, suitable for day-time reading', 'markdown-renderer-for-github' ); ?></p>
				</label>
			</div>
			
			<div class="gfmr-theme-option">
				<label>
					<input type="radio"
							name="<?php echo esc_attr( self::OPTION_NAME ); ?>[theme]"
							value="dark"
							<?php checked( $current_theme, 'dark' ); ?> />
					<?php esc_html_e( 'Dark Theme', 'markdown-renderer-for-github' ); ?>
					<p class="description"><?php esc_html_e( 'Dark background, easier on the eyes', 'markdown-renderer-for-github' ); ?></p>
				</label>
			</div>
			
			<div class="gfmr-theme-option">
				<label>
					<input type="radio"
							name="<?php echo esc_attr( self::OPTION_NAME ); ?>[theme]"
							value="system"
							<?php checked( $current_theme, 'system' ); ?> />
					<?php esc_html_e( 'Follow System Setting', 'markdown-renderer-for-github' ); ?>
					<p class="description"><?php esc_html_e( 'Automatically match your device\'s light/dark preference', 'markdown-renderer-for-github' ); ?></p>
				</label>
			</div>
		</fieldset>
		<?php
	}

	/**
	 * Render schema section description
	 */
	public function render_schema_section() {
		echo '<p>' . esc_html__( 'Configure Schema.org structured data (JSON-LD) for SEO optimization and rich snippets.', 'markdown-renderer-for-github' ) . '</p>';
		echo '<p class="description">' . esc_html__( 'Note: If you are generating structured data with another SEO plugin, we recommend disabling this feature to avoid duplication.', 'markdown-renderer-for-github' ) . '</p>';
	}

	/**
	 * Render schema enabled field
	 */
	public function render_schema_enabled_field() {
		$options = $this->get_settings();
		$enabled = ! empty( $options['schema_enabled'] );
		?>
		<label>
			<input type="checkbox"
					name="<?php echo esc_attr( self::OPTION_NAME ); ?>[schema_enabled]"
					value="1"
					<?php checked( $enabled ); ?> />
			<?php esc_html_e( 'Enable automatic generation of Schema.org structured data', 'markdown-renderer-for-github' ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'When enabled, JSON-LD structured data will be automatically added to posts and pages.', 'markdown-renderer-for-github' ); ?></p>
		<?php
	}

	/**
	 * Render schema auto-detect type field
	 */
	public function render_schema_auto_detect_field() {
		$options     = $this->get_settings();
		$auto_detect = ! empty( $options['schema_auto_detect_type'] );
		?>
		<label>
			<input type="checkbox"
					name="<?php echo esc_attr( self::OPTION_NAME ); ?>[schema_auto_detect_type]"
					value="1"
					<?php checked( $auto_detect ); ?> />
			<?php esc_html_e( 'Auto-detect article type (Article, TechArticle, HowTo)', 'markdown-renderer-for-github' ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'When enabled, the plugin analyzes content to determine the optimal schema type.', 'markdown-renderer-for-github' ); ?></p>
		<?php
	}

	/**
	 * Render schema include author field
	 */
	public function render_schema_author_field() {
		$options        = $this->get_settings();
		$include_author = ! empty( $options['schema_include_author'] );
		?>
		<label>
			<input type="checkbox"
					name="<?php echo esc_attr( self::OPTION_NAME ); ?>[schema_include_author]"
					value="1"
					<?php checked( $include_author ); ?> />
			<?php esc_html_e( 'Include author information in structured data', 'markdown-renderer-for-github' ); ?>
		</label>
		<?php
	}

	/**
	 * Render schema include publisher field
	 */
	public function render_schema_publisher_field() {
		$options           = $this->get_settings();
		$include_publisher = ! empty( $options['schema_include_publisher'] );
		?>
		<label>
			<input type="checkbox"
					name="<?php echo esc_attr( self::OPTION_NAME ); ?>[schema_include_publisher]"
					value="1"
					<?php checked( $include_publisher ); ?> />
			<?php esc_html_e( 'Include publisher (site) information in structured data', 'markdown-renderer-for-github' ); ?>
		</label>
		<?php
	}

	/**
	 * Render SSR section description
	 */
	public function render_ssr_section() {
		echo '<p>' . esc_html__( 'Server-Side Rendering (SSR) pre-renders code blocks and diagrams for improved SEO and initial load performance.', 'markdown-renderer-for-github' ) . '</p>';
		echo '<p class="description">' . esc_html__( 'Note: SSR is disabled by default. Enable it gradually to ensure compatibility with your theme and plugins.', 'markdown-renderer-for-github' ) . '</p>';
	}

	/**
	 * Render SSR enabled field
	 */
	public function render_ssr_enabled_field() {
		$options = $this->get_settings();
		$enabled = ! empty( $options['ssr_enabled'] );
		?>
		<label>
			<input type="checkbox"
					name="<?php echo esc_attr( self::OPTION_NAME ); ?>[ssr_enabled]"
					value="1"
					<?php checked( $enabled ); ?> />
			<?php esc_html_e( 'Enable Server-Side Rendering', 'markdown-renderer-for-github' ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'When enabled, code blocks and diagrams are pre-rendered on the server for better SEO.', 'markdown-renderer-for-github' ); ?></p>
		<?php
	}

	/**
	 * Render SSR code blocks field
	 */
	public function render_ssr_code_blocks_field() {
		$options = $this->get_settings();
		$enabled = ! empty( $options['ssr_code_blocks'] );
		?>
		<label>
			<input type="checkbox"
					name="<?php echo esc_attr( self::OPTION_NAME ); ?>[ssr_code_blocks]"
					value="1"
					<?php checked( $enabled ); ?> />
			<?php esc_html_e( 'Enable SSR for code blocks', 'markdown-renderer-for-github' ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'Uses highlight.php for server-side syntax highlighting.', 'markdown-renderer-for-github' ); ?></p>
		<?php
	}

	/**
	 * Render SSR Mermaid field
	 */
	public function render_ssr_mermaid_field() {
		$options = $this->get_settings();
		$enabled = ! empty( $options['ssr_mermaid'] );
		?>
		<label>
			<input type="checkbox"
					name="<?php echo esc_attr( self::OPTION_NAME ); ?>[ssr_mermaid]"
					value="1"
					<?php checked( $enabled ); ?> />
			<?php esc_html_e( 'Enable SSR for Mermaid diagrams', 'markdown-renderer-for-github' ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'Caches rendered Mermaid SVG diagrams for faster subsequent loads.', 'markdown-renderer-for-github' ); ?></p>
		<?php
	}

	/**
	 * Render TOC section description
	 */
	public function render_toc_section() {
		echo '<p>' . esc_html__( 'Configure the automatic Table of Contents (TOC) for your Markdown content.', 'markdown-renderer-for-github' ) . '</p>';
		echo '<p class="description">' . esc_html__( 'The TOC is generated from headings (H1-H6) in your content and appears as a sticky sidebar.', 'markdown-renderer-for-github' ) . '</p>';
	}

	/**
	 * Render TOC enabled field
	 */
	public function render_toc_enabled_field() {
		$options = $this->get_settings();
		$enabled = ! empty( $options['toc_enabled'] );
		?>
		<label>
			<input type="checkbox"
					name="<?php echo esc_attr( self::OPTION_NAME ); ?>[toc_enabled]"
					value="1"
					<?php checked( $enabled ); ?> />
			<?php esc_html_e( 'Enable Table of Contents', 'markdown-renderer-for-github' ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'When enabled, a scroll-tracking TOC sidebar will be generated for posts and pages.', 'markdown-renderer-for-github' ); ?></p>
		<?php
	}

	/**
	 * Render TOC heading range field
	 */
	public function render_toc_heading_range_field() {
		$options = $this->get_settings();
		$min     = $options['toc_heading_min'] ?? 2;
		$max     = $options['toc_heading_max'] ?? 4;
		?>
		<select name="<?php echo esc_attr( self::OPTION_NAME ); ?>[toc_heading_min]">
			<?php for ( $i = 1; $i <= 6; $i++ ) : ?>
				<option value="<?php echo $i; ?>" <?php selected( $min, $i ); ?>>H<?php echo $i; ?></option>
			<?php endfor; ?>
		</select>
		<span>&mdash;</span>
		<select name="<?php echo esc_attr( self::OPTION_NAME ); ?>[toc_heading_max]">
			<?php for ( $i = 1; $i <= 6; $i++ ) : ?>
				<option value="<?php echo $i; ?>" <?php selected( $max, $i ); ?>>H<?php echo $i; ?></option>
			<?php endfor; ?>
		</select>
		<p class="description"><?php esc_html_e( 'Select the range of heading levels to include in the TOC.', 'markdown-renderer-for-github' ); ?></p>
		<?php
	}

	/**
	 * Render TOC position field
	 */
	public function render_toc_position_field() {
		$options  = $this->get_settings();
		$position = $options['toc_position'] ?? 'right';
		?>
		<fieldset>
			<label>
				<input type="radio" name="<?php echo esc_attr( self::OPTION_NAME ); ?>[toc_position]" value="left" <?php checked( $position, 'left' ); ?>>
				<?php esc_html_e( 'Left', 'markdown-renderer-for-github' ); ?>
			</label>
			<br>
			<label>
				<input type="radio" name="<?php echo esc_attr( self::OPTION_NAME ); ?>[toc_position]" value="right" <?php checked( $position, 'right' ); ?>>
				<?php esc_html_e( 'Right', 'markdown-renderer-for-github' ); ?>
			</label>
		</fieldset>
		<p class="description"><?php esc_html_e( 'Choose which side of the content the TOC sidebar should appear on.', 'markdown-renderer-for-github' ); ?></p>
		<?php
	}

	/**
	 * Render TOC scroll offset field
	 */
	public function render_toc_scroll_offset_field() {
		$options = $this->get_settings();
		$offset  = $options['toc_scroll_offset'] ?? 0;
		?>
		<input type="number"
				name="<?php echo esc_attr( self::OPTION_NAME ); ?>[toc_scroll_offset]"
				value="<?php echo esc_attr( $offset ); ?>"
				class="small-text" />
		<span>px</span>
		<p class="description"><?php esc_html_e( 'Adjust the scroll offset (in pixels) for highlighting the active heading. Useful for sites with a sticky header.', 'markdown-renderer-for-github' ); ?></p>
		<?php
	}

	/**
	 * Render TOC sticky field
	 */
	public function render_toc_sticky_field() {
		$options = $this->get_settings();
		$sticky  = ! empty( $options['toc_sticky'] );
		?>
		<label>
			<input type="checkbox"
					name="<?php echo esc_attr( self::OPTION_NAME ); ?>[toc_sticky]"
					value="1"
					<?php checked( $sticky ); ?> />
			<?php esc_html_e( 'Enable Sticky Sidebar', 'markdown-renderer-for-github' ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'When enabled, the TOC sidebar will remain fixed in place while scrolling.', 'markdown-renderer-for-github' ); ?></p>
		<?php
	}

	/**
	 * Render TOC mobile hidden field
	 */
	public function render_toc_mobile_hidden_field() {
		$options = $this->get_settings();
		$hidden  = ! empty( $options['toc_mobile_hidden'] );
		?>
		<label>
			<input type="checkbox"
					name="<?php echo esc_attr( self::OPTION_NAME ); ?>[toc_mobile_hidden]"
					value="1"
					<?php checked( $hidden ); ?> />
			<?php esc_html_e( 'Hide on Mobile Devices', 'markdown-renderer-for-github' ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'When enabled, the TOC will be hidden on screens smaller than 768px wide.', 'markdown-renderer-for-github' ); ?></p>
		<?php
	}

	/**
	 * AJAX handler for theme preview
	 */
	public function ajax_preview_theme() {
		check_ajax_referer( 'gfmr_preview_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'markdown-renderer-for-github' ) );
		}

		$theme = \sanitize_key( $_POST['theme'] ?? self::DEFAULT_THEME );

		if ( ! array_key_exists( $theme, self::AVAILABLE_THEMES ) ) {
			wp_send_json_error( __( 'Invalid theme selected.', 'markdown-renderer-for-github' ) );
		}

		wp_send_json_success(
			array(
				'theme'       => $theme,
				'shiki_theme' => self::AVAILABLE_THEMES[ $theme ],
			)
		);
	}

	/**
	 * Get current settings
	 *
	 * @return array Current settings
	 */
	public function get_settings() {
		$default_settings = $this->get_default_settings();
		$saved_settings   = get_option( self::OPTION_NAME, array() );

		return wp_parse_args( $saved_settings, $default_settings );
	}

	/**
	 * Get a single setting value
	 *
	 * @param string $key Setting key
	 * @param mixed $fallback_value Default value if key not found
	 * @return mixed Setting value
	 */
	public function get( string $key, $fallback_value = null ) {
		$settings = $this->get_settings();
		return $settings[ $key ] ?? $fallback_value;
	}

	/**
	 * Get current theme setting
	 *
	 * @return string Current theme
	 */
	public function get_current_theme() {
		$settings = $this->get_settings();
		return $settings['theme'];
	}

	/**
	 * Get Shiki theme name for current setting
	 *
	 * @return string Shiki theme name
	 */
	public function get_shiki_theme() {
		$current_theme = $this->get_current_theme();
		return self::AVAILABLE_THEMES[ $current_theme ];
	}

	/**
	 * Check if system theme is enabled
	 *
	 * @return bool True if system theme is enabled
	 */
	public function is_system_theme_enabled() {
		return $this->get_current_theme() === 'system';
	}

	/**
	 * Check if Schema.org structured data is enabled
	 *
	 * @return bool True if schema generation is enabled
	 */
	public function is_schema_enabled() {
		$settings = $this->get_settings();
		return ! empty( $settings['schema_enabled'] );
	}

	/**
	 * Get schema-related settings
	 *
	 * @return array Schema settings
	 */
	public function get_schema_settings() {
		$settings = $this->get_settings();
		return array(
			'enabled'           => ! empty( $settings['schema_enabled'] ),
			'auto_detect_type'  => ! empty( $settings['schema_auto_detect_type'] ),
			'include_author'    => ! empty( $settings['schema_include_author'] ),
			'include_publisher' => ! empty( $settings['schema_include_publisher'] ),
		);
	}

	/**
	 * Check if SSR is enabled
	 *
	 * @return bool True if SSR is enabled
	 */
	public function is_ssr_enabled() {
		$settings = $this->get_settings();
		return ! empty( $settings['ssr_enabled'] );
	}

	/**
	 * Check if code block SSR is enabled
	 *
	 * @return bool True if code block SSR is enabled
	 */
	public function is_code_block_ssr_enabled() {
		// Only enabled if both SSR and code block SSR are turned on
		return $this->is_ssr_enabled() && ! empty( $this->get_settings()['ssr_code_blocks'] );
	}

	/**
	 * Check if Mermaid SSR is enabled
	 *
	 * @return bool True if Mermaid SSR is enabled
	 */
	public function is_mermaid_ssr_enabled() {
		// Only enabled if both SSR and Mermaid SSR are turned on
		return $this->is_ssr_enabled() && ! empty( $this->get_settings()['ssr_mermaid'] );
	}

	/**
	 * Enqueue admin assets for settings page
	 *
	 * @param string $hook_suffix Current admin page
	 */
	public function enqueue_admin_assets( $hook_suffix ) {
		// Only load on our settings page.
		if ( 'settings_page_' . self::SETTINGS_SLUG !== $hook_suffix ) {
			return;
		}

		$plugin_url = defined( 'GFMR_URI' ) ? GFMR_URI : plugin_dir_url( __DIR__ );
		$version    = defined( 'GFMR_VERSION' ) ? GFMR_VERSION : '1.0.0';

		// Enqueue settings page CSS.
		wp_enqueue_style(
			'gfmr-admin-settings',
			$plugin_url . 'assets/css/gfmr-admin-settings.css',
			array(),
			$version
		);

		// Enqueue settings page JavaScript.
		wp_enqueue_script(
			'gfmr-admin-settings',
			$plugin_url . 'assets/js/gfmr-admin-settings.js',
			array( 'jquery' ),
			$version,
			true
		);

		// Pass localized data to JavaScript.
		wp_localize_script(
			'gfmr-admin-settings',
			'gfmrAdminSettings',
			array(
				'optionName' => self::OPTION_NAME,
				'i18n'       => array(
					'light'         => __( 'Light', 'markdown-renderer-for-github' ),
					'dark'          => __( 'Dark', 'markdown-renderer-for-github' ),
					'system'        => __( 'System', 'markdown-renderer-for-github' ),
					'applyingTheme' => __( 'Applying theme...', 'markdown-renderer-for-github' ),
				),
			)
		);

		// Use the asset manager for admin script enqueuing (e.g., Shiki).
		$asset_manager = new GFMR_Asset_Manager();
		$asset_manager->enqueue_admin_scripts( $hook_suffix );
	}

	/**
	 * Render Extensions section
	 *
	 * Displays information about extensions and lists registered addons.
	 */
	public function render_extensions_section() {
		// Get extension API instance
		$extension_api = new \Wakalab\WpGfmRenderer\GFMR_Extension_API();
		$addons        = $extension_api->get_registered_addons();

		echo '<p>';
		echo esc_html__( 'Extend the functionality of Markdown Renderer for GitHub by installing addons.', 'markdown-renderer-for-github' );
		echo '</p>';

		/**
		 * Fires before the extensions section content
		 *
		 * Allows addons to add content at the top of the extensions tab.
		 *
		 * @since 2.0.0
		 *
		 * @param string $page_slug Settings page slug
		 * @param array  $tabs      Available tabs
		 */
		do_action( 'gfmr_settings_tabs', self::SETTINGS_SLUG, $this->get_tabs() );

		if ( empty( $addons ) ) {
			echo '<p>';
			echo esc_html__( 'No addons are currently installed. Install addons to add new features and capabilities.', 'markdown-renderer-for-github' );
			echo '</p>';
		} else {
			echo '<h3>' . esc_html__( 'Installed Addons', 'markdown-renderer-for-github' ) . '</h3>';
			echo '<table class="widefat striped">';
			echo '<thead><tr>';
			echo '<th>' . esc_html__( 'Name', 'markdown-renderer-for-github' ) . '</th>';
			echo '<th>' . esc_html__( 'Version', 'markdown-renderer-for-github' ) . '</th>';
			echo '<th>' . esc_html__( 'Author', 'markdown-renderer-for-github' ) . '</th>';
			echo '</tr></thead>';
			echo '<tbody>';

			foreach ( $addons as $addon_id => $addon_info ) {
				echo '<tr>';
				echo '<td>' . esc_html( $addon_info['name'] ?? $addon_id ) . '</td>';
				echo '<td>' . esc_html( $addon_info['version'] ?? 'N/A' ) . '</td>';
				echo '<td>' . esc_html( $addon_info['author'] ?? 'N/A' ) . '</td>';
				echo '</tr>';
			}

			echo '</tbody>';
			echo '</table>';
		}

		/**
		 * Fires in the extensions section
		 *
		 * Allows addons to add their own settings sections.
		 *
		 * @since 2.0.0
		 *
		 * @param string $page_slug  Settings page slug
		 * @param string $active_tab Currently active tab ID
		 */
		do_action( 'gfmr_settings_sections', self::SETTINGS_SLUG, 'extensions' );
	}

	/**
	 * Render Multilingual section description
	 */
	public function render_multilingual_section() {
		echo '<p>' . esc_html__( 'Configure multilingual URL structure for language-specific content.', 'markdown-renderer-for-github' ) . '</p>';
		echo '<p class="description">' . esc_html__( 'Note: Path-based URLs require permalink settings to be set to "Post name". You may need to flush permalink rules after changing these settings.', 'markdown-renderer-for-github' ) . '</p>';
	}

	/**
	 * Render multilingual URL mode field
	 */
	public function render_multilingual_url_mode_field() {
		$options = $this->get_settings();
		$mode = $options['multilingual_url_mode'] ?? 'path';
		?>
		<fieldset>
			<label>
				<input type="radio" name="<?php echo esc_attr( self::OPTION_NAME ); ?>[multilingual_url_mode]" value="path" <?php checked( $mode, 'path' ); ?>>
				<?php esc_html_e( 'Path-based (/en/post-name/)', 'markdown-renderer-for-github' ); ?>
			</label>
			<br>
			<label>
				<input type="radio" name="<?php echo esc_attr( self::OPTION_NAME ); ?>[multilingual_url_mode]" value="query" <?php checked( $mode, 'query' ); ?>>
				<?php esc_html_e( 'Query parameter (?lang=en)', 'markdown-renderer-for-github' ); ?>
			</label>
		</fieldset>
		<p class="description"><?php esc_html_e( 'Choose how language is specified in URLs. Path-based URLs are SEO-friendly.', 'markdown-renderer-for-github' ); ?></p>
		<?php
	}

	/**
	 * Render multilingual default language field
	 */
	public function render_multilingual_default_language_field() {
		$options = $this->get_settings();
		$default = $options['multilingual_default_language'] ?? 'en';
		$languages = $options['multilingual_supported_languages'] ?? array( 'en', 'ja' );
		?>
		<select name="<?php echo esc_attr( self::OPTION_NAME ); ?>[multilingual_default_language]">
			<?php foreach ( (array) $languages as $lang ) : ?>
				<option value="<?php echo esc_attr( $lang ); ?>" <?php selected( $default, $lang ); ?>>
					<?php echo esc_html( strtoupper( $lang ) ); ?>
				</option>
			<?php endforeach; ?>
		</select>
		<p class="description"><?php esc_html_e( 'The default language used when no language is specified in the URL.', 'markdown-renderer-for-github' ); ?></p>
		<?php
	}

	/**
	 * Render multilingual supported languages field
	 */
	public function render_multilingual_supported_languages_field() {
		$options = $this->get_settings();
		$languages = $options['multilingual_supported_languages'] ?? array( 'en', 'ja' );
		$available_languages = array(
			'en' => 'English',
			'ja' => 'Japanese',
			'es' => 'Spanish',
			'fr' => 'French',
			'de' => 'German',
			'it' => 'Italian',
			'pt' => 'Portuguese',
			'ru' => 'Russian',
			'zh' => 'Chinese',
			'ko' => 'Korean',
			'ar' => 'Arabic',
			'hi' => 'Hindi',
		);
		?>
		<fieldset>
			<?php foreach ( $available_languages as $code => $name ) : ?>
				<label>
					<input type="checkbox"
							name="<?php echo esc_attr( self::OPTION_NAME ); ?>[multilingual_supported_languages][]"
							value="<?php echo esc_attr( $code ); ?>"
							<?php checked( in_array( $code, $languages, true ) ); ?> />
					<?php echo esc_html( $name ); ?> (<?php echo esc_html( $code ); ?>)
				</label>
				<br>
			<?php endforeach; ?>
		</fieldset>
		<p class="description"><?php esc_html_e( 'Select the languages available on your site. Use ISO 639-1 two-letter codes.', 'markdown-renderer-for-github' ); ?></p>
		<?php
	}

	/**
	 * Render hide default language field
	 */
	public function render_multilingual_hide_default_lang_field() {
		$options = $this->get_settings();
		$checked = ! empty( $options['multilingual_hide_default_lang'] );
		?>
		<label>
			<input type="checkbox"
					name="<?php echo esc_attr( self::OPTION_NAME ); ?>[multilingual_hide_default_lang]"
					value="1"
					<?php checked( $checked ); ?> />
			<?php esc_html_e( 'Hide default language in URL (e.g., /post-name/ instead of /en/post-name/)', 'markdown-renderer-for-github' ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'When enabled, the default language URL will not show the language prefix.', 'markdown-renderer-for-github' ); ?></p>
		<?php
	}

	/**
	 * Render hreflang enabled field
	 */
	public function render_multilingual_hreflang_enabled_field() {
		$options = $this->get_settings();
		$checked = ! empty( $options['multilingual_hreflang_enabled'] );
		?>
		<label>
			<input type="checkbox"
					name="<?php echo esc_attr( self::OPTION_NAME ); ?>[multilingual_hreflang_enabled]"
					value="1"
					<?php checked( $checked ); ?> />
			<?php esc_html_e( 'Output hreflang tags for SEO', 'markdown-renderer-for-github' ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'Adds hreflang and x-default tags to help search engines understand language variations.', 'markdown-renderer-for-github' ); ?></p>
		<?php
	}

	/**
	 * Render redirect query field
	 */
	public function render_multilingual_redirect_query_field() {
		$options = $this->get_settings();
		$checked = ! empty( $options['multilingual_redirect_query'] );
		?>
		<label>
			<input type="checkbox"
					name="<?php echo esc_attr( self::OPTION_NAME ); ?>[multilingual_redirect_query]"
					value="1"
					<?php checked( $checked ); ?> />
			<?php esc_html_e( 'Redirect ?lang=xx URLs to /xx/ URLs (301 redirect)', 'markdown-renderer-for-github' ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'Automatically redirects old query parameter URLs to new path-based URLs.', 'markdown-renderer-for-github' ); ?></p>
		<?php
	}

	/**
	 * Reset singleton instance for testing
	 * Only available in test environment
	 *
	 * @return void
	 */
	public static function reset_instance_for_testing() {
		if ( defined( 'WP_TESTS_DOMAIN' ) ) {
			self::$instance = null;
			self::$hooks_registered = false;
		}
	}
}
