<?php
/**
 * Schema.org structured data generator for GFMR plugin
 *
 * Generates JSON-LD structured data for posts and pages to enhance SEO
 * and enable rich snippets in search results.
 *
 * @package WpGfmRenderer
 * @since 1.4.0
 */

namespace Wakalab\WpGfmRenderer;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Prevent class redeclaration when both Free and Pro versions are active
if ( class_exists( __NAMESPACE__ . '\\GFMR_Schema_Generator' ) ) {
	return;
}

/**
 * Schema Generator class
 *
 * Handles generation and output of Schema.org JSON-LD markup.
 */
class GFMR_Schema_Generator {

	/**
	 * Post meta key for caching generated schema
	 *
	 * @var string
	 */
	const META_KEY = '_gfmr_schema_cache';

	/**
	 * Schema version for cache invalidation
	 *
	 * @var string
	 */
	const SCHEMA_VERSION = '1.0';

	/**
	 * Settings instance
	 *
	 * @var GFMR_Settings|object
	 */
	private $settings;

	/**
	 * Content analyzer instance
	 *
	 * @var GFMR_Content_Analyzer|null
	 */
	private $content_analyzer = null;

	/**
	 * Constructor
	 *
	 * @param GFMR_Settings|object $settings Settings instance.
	 */
	public function __construct( $settings ) {
		$this->settings = $settings;
	}

	/**
	 * Initialize the schema generator
	 *
	 * Registers WordPress hooks if schema generation is enabled.
	 *
	 * @return void
	 */
	public function init() {
		// Only register hooks if schema is enabled.
		if ( ! $this->settings->is_schema_enabled() ) {
			return;
		}

		add_action( 'wp_head', array( $this, 'maybe_output_schema' ), 1 );
		add_action( 'save_post', array( $this, 'regenerate_schema_on_save' ), 20, 2 );
	}

	/**
	 * Output schema markup on singular pages if applicable
	 *
	 * @return void
	 */
	public function maybe_output_schema() {
		if ( ! is_singular() ) {
			return;
		}

		$post = get_queried_object();
		if ( ! $post || ! is_a( $post, 'WP_Post' ) ) {
			return;
		}

		// Only process posts and pages.
		if ( ! in_array( $post->post_type, array( 'post', 'page' ), true ) ) {
			return;
		}

		// Try to get cached schema.
		$cached = get_post_meta( $post->ID, self::META_KEY, true );
		if ( ! empty( $cached ) && isset( $cached['version'] ) && self::SCHEMA_VERSION === $cached['version'] ) {
			$schema = $cached['schema'];
		} else {
			// Generate and cache schema.
			$schema = $this->generate_schema( $post );
			update_post_meta(
				$post->ID,
				self::META_KEY,
				array(
					'schema'    => $schema,
					'version'   => self::SCHEMA_VERSION,
					'timestamp' => time(),
				)
			);
		}

		$this->output_json_ld( $schema );
	}

	/**
	 * Regenerate schema when post is saved
	 *
	 * @param int      $post_id Post ID.
	 * @param \WP_Post $post    Post object.
	 * @return void
	 */
	public function regenerate_schema_on_save( $post_id, $post ) {
		// Skip autosave and revisions.
		if ( wp_is_post_autosave( $post_id ) ) {
			return;
		}
		if ( wp_is_post_revision( $post_id ) ) {
			return;
		}

		// Only process posts and pages.
		if ( ! in_array( $post->post_type, array( 'post', 'page' ), true ) ) {
			return;
		}

		// Only process published posts.
		if ( 'publish' !== $post->post_status ) {
			return;
		}

		// Generate and cache schema.
		$schema = $this->generate_schema( $post );
		update_post_meta(
			$post_id,
			self::META_KEY,
			array(
				'schema'    => $schema,
				'version'   => self::SCHEMA_VERSION,
				'timestamp' => time(),
			)
		);
	}

	/**
	 * Generate schema for a post
	 *
	 * @param \WP_Post $post Post object.
	 * @return array Schema data.
	 */
	public function generate_schema( $post ) {
		$article_type = $this->get_article_type( $post );

		$schema = array(
			'@context' => 'https://schema.org',
			'@type'    => $article_type,
		);

		// HowTo uses 'name' instead of 'headline'.
		if ( 'HowTo' === $article_type ) {
			$schema['name']        = $post->post_title;
			$schema['description'] = $this->get_excerpt( $post );

			// Add HowTo steps if available.
			if ( null !== $this->content_analyzer ) {
				$steps = $this->content_analyzer->extract_how_to_steps( $post->post_content );
				if ( ! empty( $steps ) ) {
					$schema['step'] = $this->format_how_to_steps( $steps );
				}
			}
		} else {
			$schema['headline']         = $post->post_title;
			$schema['datePublished']    = get_the_date( 'c', $post );
			$schema['dateModified']     = get_the_modified_date( 'c', $post );
			$schema['mainEntityOfPage'] = get_permalink( $post );
		}

		// Add author information.
		$schema_settings = $this->settings->get_schema_settings();
		if ( ! empty( $schema_settings['include_author'] ) ) {
			$schema['author'] = $this->get_author_schema( $post->post_author );
		}

		// Add publisher information (for Article types only, not HowTo).
		if ( 'HowTo' !== $article_type && ! empty( $schema_settings['include_publisher'] ) ) {
			$schema['publisher'] = $this->get_publisher_schema();
		}

		// Add featured image if available.
		if ( has_post_thumbnail( $post ) ) {
			$image_url = get_the_post_thumbnail_url( $post, 'full' );
			if ( $image_url ) {
				$schema['image'] = $image_url;
			}
		}

		/**
		 * Filter the generated schema
		 *
		 * @param array    $schema Generated schema.
		 * @param \WP_Post $post   Post object.
		 */
		return apply_filters( 'gfmr_schema_data', $schema, $post );
	}

	/**
	 * Get post excerpt
	 *
	 * @param \WP_Post $post Post object.
	 * @return string Post excerpt.
	 */
	private function get_excerpt( $post ) {
		if ( ! empty( $post->post_excerpt ) ) {
			return wp_strip_all_tags( $post->post_excerpt );
		}

		// Generate excerpt from content.
		$content = wp_strip_all_tags( $post->post_content );
		$content = preg_replace( '/\s+/', ' ', $content );
		return mb_substr( $content, 0, 160 );
	}

	/**
	 * Format HowTo steps for schema
	 *
	 * @param array $steps Raw steps data.
	 * @return array Formatted HowToStep items.
	 */
	private function format_how_to_steps( $steps ) {
		$formatted = array();

		foreach ( $steps as $step ) {
			$formatted[] = array(
				'@type'    => 'HowToStep',
				'position' => $step['position'],
				'name'     => $step['name'],
			);
		}

		return $formatted;
	}

	/**
	 * Get article type based on content analysis
	 *
	 * @param \WP_Post $post Post object.
	 * @return string Article type (Article, TechArticle, or HowTo).
	 */
	private function get_article_type( $post ) {
		$schema_settings = $this->settings->get_schema_settings();

		// If auto-detect is disabled, return default Article type.
		if ( empty( $schema_settings['auto_detect_type'] ) ) {
			return 'Article';
		}

		// Use content analyzer if available.
		if ( null !== $this->content_analyzer ) {
			return $this->content_analyzer->detect_article_type( $post->post_content );
		}

		return 'Article';
	}

	/**
	 * Get author schema data
	 *
	 * @param int $author_id Author ID.
	 * @return array Author schema.
	 */
	private function get_author_schema( $author_id ) {
		$author = array(
			'@type' => 'Person',
			'name'  => get_the_author_meta( 'display_name', $author_id ),
		);

		$author_url = get_the_author_meta( 'user_url', $author_id );
		if ( $author_url ) {
			$author['url'] = $author_url;
		}

		return $author;
	}

	/**
	 * Get publisher schema data
	 *
	 * @return array Publisher schema.
	 */
	private function get_publisher_schema() {
		$publisher = array(
			'@type' => 'Organization',
			'name'  => get_bloginfo( 'name' ),
		);

		// Add logo if custom logo is set.
		$custom_logo_id = get_theme_mod( 'custom_logo' );
		if ( $custom_logo_id ) {
			$logo_url = wp_get_attachment_image_url( $custom_logo_id, 'full' );
			if ( $logo_url ) {
				$publisher['logo'] = array(
					'@type' => 'ImageObject',
					'url'   => $logo_url,
				);
			}
		}

		return $publisher;
	}

	/**
	 * Output JSON-LD script tag
	 *
	 * @param array $schema Schema data.
	 * @return void
	 */
	public function output_json_ld( $schema ) {
		if ( empty( $schema ) ) {
			return;
		}

		$json = wp_json_encode( $schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT );
		if ( ! $json ) {
			return;
		}

		// JSON-LD is safe because wp_json_encode escapes properly for script context.
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		echo '<script type="application/ld+json">' . $json . '</script>' . "\n";
	}

	/**
	 * Set content analyzer instance
	 *
	 * @param GFMR_Content_Analyzer $analyzer Content analyzer.
	 * @return void
	 */
	public function set_content_analyzer( $analyzer ) {
		$this->content_analyzer = $analyzer;
	}
}
