<?php
/**
 * Main plugin class - Core renderer for GitHub Flavored Markdown
 *
 * Handles plugin initialization, dependency injection, and WordPress hooks.
 *
 * @package WpGfmRenderer
 * @since 0.0.2
 */

namespace Wakalab\WpGfmRenderer;

defined( 'ABSPATH' ) || exit;

// Prevent class redeclaration when both Free and Pro versions are active
if ( class_exists( __NAMESPACE__ . '\\GFMR_Renderer' ) ) {
	return;
}

class GFMR_Renderer {

	private static $instance = null;

	private $plugin_data;

	private $asset_manager;

	private $block_registry;

	private $cache_manager;

	private $settings;

	private $schema_generator;

	private $content_analyzer;

	/**
	 * Constructor with optional dependency injection.
	 *
	 * @param GFMR_Asset_Manager|null    $asset_manager    Asset manager instance.
	 * @param GFMR_Block_Registry|null   $block_registry   Block registry instance.
	 * @param GFMR_Cache_Manager|null    $cache_manager    Cache manager instance.
	 * @param GFMR_Settings|null         $settings         Settings instance.
	 * @param GFMR_Schema_Generator|null $schema_generator Schema generator instance.
	 * @param GFMR_Content_Analyzer|null $content_analyzer Content analyzer instance.
	 */
	private function __construct(
		?GFMR_Asset_Manager $asset_manager = null,
		?GFMR_Block_Registry $block_registry = null,
		?GFMR_Cache_Manager $cache_manager = null,
		?GFMR_Settings $settings = null,
		?GFMR_Schema_Generator $schema_generator = null,
		?GFMR_Content_Analyzer $content_analyzer = null
	) {
		$this->init_dependencies(
			$asset_manager,
			$block_registry,
			$cache_manager,
			$settings,
			$schema_generator,
			$content_analyzer
		);
		$this->init_hooks();
	}

	/**
	 * Initialize dependencies with optional injection.
	 *
	 * @param GFMR_Asset_Manager|null    $asset_manager    Asset manager instance.
	 * @param GFMR_Block_Registry|null   $block_registry   Block registry instance.
	 * @param GFMR_Cache_Manager|null    $cache_manager    Cache manager instance.
	 * @param GFMR_Settings|null         $settings         Settings instance.
	 * @param GFMR_Schema_Generator|null $schema_generator Schema generator instance.
	 * @param GFMR_Content_Analyzer|null $content_analyzer Content analyzer instance.
	 */
	private function init_dependencies(
		?GFMR_Asset_Manager $asset_manager = null,
		?GFMR_Block_Registry $block_registry = null,
		?GFMR_Cache_Manager $cache_manager = null,
		?GFMR_Settings $settings = null,
		?GFMR_Schema_Generator $schema_generator = null,
		?GFMR_Content_Analyzer $content_analyzer = null
	) {
		require_once __DIR__ . '/class-gfmr-asset-manager.php';
		require_once __DIR__ . '/class-gfmr-block-registry.php';
		require_once __DIR__ . '/class-gfmr-cache-manager.php';
		require_once __DIR__ . '/class-gfmr-settings.php';
		require_once __DIR__ . '/class-gfmr-schema-generator.php';
		require_once __DIR__ . '/class-gfmr-content-analyzer.php';

		// SSR-related classes
		require_once __DIR__ . '/class-gfmr-code-highlighter.php';
		require_once __DIR__ . '/class-gfmr-mermaid-ssr-handler.php';
		require_once __DIR__ . '/class-gfmr-ssr-renderer.php';

		$this->asset_manager    = $asset_manager ?? new GFMR_Asset_Manager();
		$this->cache_manager    = $cache_manager ?? new GFMR_Cache_Manager();
		$this->settings         = $settings ?? GFMR_Settings::get_instance();
		$this->content_analyzer = $content_analyzer ?? new GFMR_Content_Analyzer();
		$this->schema_generator = $schema_generator ?? new GFMR_Schema_Generator( $this->settings );

		// Initialize SSR components
		$code_highlighter = new GFMR_Code_Highlighter();
		$mermaid_handler  = new GFMR_Mermaid_SSR_Handler( $this->cache_manager );
		$ssr_renderer     = new GFMR_SSR_Renderer( $this->cache_manager, $code_highlighter, $mermaid_handler, $this->settings );

		// Create block registry with SSR support
		$this->block_registry = $block_registry ?? new GFMR_Block_Registry( $ssr_renderer, $this->settings );

		// Set content analyzer for schema generator (enables TechArticle/HowTo detection).
		$this->schema_generator->set_content_analyzer( $this->content_analyzer );
	}

	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function get_plugin_data() {
		if ( null === $this->plugin_data ) {
			if ( ! function_exists( 'get_plugin_data' ) ) {
				require_once ABSPATH . 'wp-admin/includes/plugin.php';
			}
			$this->plugin_data = get_plugin_data( GFMR_PLUGIN_FILE, false, false );
		}
		return $this->plugin_data;
	}

	private function init_hooks() {
		// File-based logging to debug hook registration
		$log_file = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'debug.log';
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
		file_put_contents(
			$log_file,
			'[' . current_time( 'mysql' ) . '] 📌 init_hooks() called, registering enqueue_block_editor_assets hook' . "\n",
			FILE_APPEND
		);

		add_action( 'plugins_loaded', array( $this, 'init' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_assets' ) );
		add_action( 'init', array( $this, 'register_blocks' ) );
		// Register BOTH enqueue hooks to test which one WordPress uses
		add_action( 'enqueue_block_editor_assets', array( $this, 'enqueue_block_editor_assets' ) );
		add_action( 'enqueue_block_assets', array( $this, 'enqueue_block_assets_test' ) );

		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
		file_put_contents(
			$log_file,
			'[' . current_time( 'mysql' ) . '] ✅ Both enqueue hooks registered' . "\n",
			FILE_APPEND
		);

		add_filter( 'wp_kses_allowed_html', array( $this, 'allow_code_language_classes' ), 10, 2 );
		add_filter( 'wp_kses_allowed_html', array( $this, 'allow_frontmatter_html_tags' ), 10, 2 );
		// Add content filter for processing Markdown in post content
		add_filter( 'the_content', array( $this, 'process_markdown_content' ), 10 );
		// Frontmatter title handling
		add_filter( 'render_block', array( $this, 'save_frontmatter_title' ), 10, 2 );
		add_filter( 'document_title_parts', array( $this, 'apply_frontmatter_title' ) );
		// SSR cache invalidation hooks
		add_action( 'save_post', array( $this, 'invalidate_post_ssr_cache' ) );
		add_action( 'update_option_gfmr_theme_settings', array( $this, 'invalidate_theme_ssr_cache' ), 10, 2 );
		// Settings are initialized in init() method
		register_activation_hook( GFMR_PLUGIN_FILE, array( $this, 'activate' ) );
		register_deactivation_hook( GFMR_PLUGIN_FILE, array( $this, 'deactivate' ) );
	}

	public function init() {
		try {
			// Initialize settings early
			$this->settings->init();
			// Initialize schema generator (registers hooks if enabled)
			$this->schema_generator->init();
		} catch ( \Exception $e ) {
			$this->handle_error( $e );
		}
	}


	public function enqueue_assets() {
		try {
			// Use Asset Manager's condition check
			if ( $this->asset_manager->should_enqueue_assets() ) {
				$this->log_debug( 'Enqueuing assets via Asset Manager' );
				$version = $this->get_version();
				$this->asset_manager->enqueue_scripts( $version );
				$this->asset_manager->enqueue_styles( $version );
				// Setup localization with theme settings
				$this->asset_manager->setup_localization_with_theme_settings( $version, $this->settings );
			} else {
				$this->log_debug( 'Skipping asset enqueue - conditions not met' );
			}
		} catch ( \Exception $e ) {
			$this->handle_error( $e );
		}
	}

	/**
	 * Check asset loading conditions (legacy compatibility)
	 *
	 * @return bool Whether to load assets
	 */
	private function should_enqueue_assets() {
		// Fallback to Asset Manager judgment (maintain legacy compatibility)
		$has_markdown         = $this->asset_manager->has_markdown_blocks();
		$may_have_code_blocks = $this->asset_manager->may_have_code_blocks();

		// Maintain conventional logic (debug and E2E test support)
		$force_enqueue = defined( 'WP_DEBUG' ) && WP_DEBUG;
		$is_e2e_test   = strpos( get_bloginfo( 'name' ), 'E2E Test' ) !== false;

		// Output debug information to log
		$this->log_debug( 'WP GFM Renderer legacy enqueue check:' );
		$this->log_debug( '- Force enqueue (WP_DEBUG): ' . ( $force_enqueue ? 'Yes' : 'No' ) );
		$this->log_debug( '- Is E2E test: ' . ( $is_e2e_test ? 'Yes' : 'No' ) );
		$this->log_debug( '- Has markdown blocks: ' . ( $has_markdown ? 'Yes' : 'No' ) );
		$this->log_debug( '- May have code blocks: ' . ( $may_have_code_blocks ? 'Yes' : 'No' ) );

		return $force_enqueue || $is_e2e_test || $has_markdown || $may_have_code_blocks;
	}


	public function activate() {
		try {
			// Add future initialization processes here
			$this->log_debug( 'Plugin activated successfully' );
		} catch ( \Exception $e ) {
			$this->handle_error( $e );
		}
	}

	public function deactivate() {
		try {
			// Add future cleanup processes here
			$this->log_debug( 'Plugin deactivated successfully' );
		} catch ( \Exception $e ) {
			$this->handle_error( $e );
		}
	}

	private function handle_error( \Exception $e ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			$error_message = sprintf(
				'[WP GFM Renderer] Error: %s in %s on line %d',
				$e->getMessage(),
				$e->getFile(),
				$e->getLine()
			);

			// Use WordPress standard error handling
			if ( function_exists( 'wp_trigger_error' ) ) {
				wp_trigger_error( '', $error_message, E_USER_WARNING );
			}
		}
	}

	private function log_debug( $message ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			$debug_message = '[WP GFM Renderer] ' . $message;

			// Use WordPress standard error handling
			if ( function_exists( 'wp_trigger_error' ) ) {
				wp_trigger_error( '', $debug_message, E_USER_NOTICE );
			}
		}
	}


	public function get_version() {
		// Ensure plugin_data is initialized
		if ( null === $this->plugin_data ) {
			$this->get_plugin_data();
		}
		return $this->plugin_data['Version'] ?? '1.0.0';
	}

	public function get_plugin_name() {
		return $this->plugin_data['Name'] ?? 'WP GFM Renderer';
	}

	public function register_blocks() {
		try {
			// Delegate to Block Registry
			$this->block_registry->register_blocks();
		} catch ( \Exception $e ) {
			$this->handle_error( $e );
		}
	}

	/**
	 * Enqueue block editor assets
	 * This wrapper method ensures the enqueue hook is registered early in init_hooks()
	 */
	public function enqueue_block_editor_assets() {
		// File-based logging to debug hook execution
		$log_file = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'debug.log';
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
		file_put_contents(
			$log_file,
			'[' . current_time( 'mysql' ) . '] 🎯 GFMR_Renderer->enqueue_block_editor_assets() called!' . "\n",
			FILE_APPEND
		);

		try {
			// Delegate to Block Registry
			$this->block_registry->enqueue_editor_assets();
		} catch ( \Exception $e ) {
			$this->handle_error( $e );
		}
	}

	/**
	 * Test method to check which enqueue hook WordPress actually fires
	 */
	public function enqueue_block_assets_test() {
		// File-based logging to debug hook execution
		$log_file = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'debug.log';
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
		file_put_contents(
			$log_file,
			'[' . current_time( 'mysql' ) . '] 🔥 GFMR_Renderer->enqueue_block_assets_test() called!' . "\n",
			FILE_APPEND
		);
	}


	/**
	 * MD-34 Fix: Explicitly register block editor scripts
	 * Supplementary method to resolve block.json relative path issues
	 */
	private function register_block_editor_scripts() {
		$plugin_url = plugin_dir_url( GFMR_PLUGIN_FILE );
		$version    = $this->get_version();

		// Path to editor JS file
		$editor_script_path = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'build/index.js';
		$editor_asset_path  = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'build/index.asset.php';

		if ( ! file_exists( $editor_script_path ) ) {
			$this->log_debug( 'Editor script not found: ' . $editor_script_path );
			return;
		}

		// Load asset information
		$asset_data = array(
			'dependencies' => array(),
			'version'      => $version,
		);
		if ( file_exists( $editor_asset_path ) ) {
			$asset_data = include $editor_asset_path;
		}

		// Editor script dependencies (basic dependencies only)
		$editor_dependencies = $asset_data['dependencies'];

		// Register and enqueue editor script
		wp_register_script(
			'gfmr-renderer-editor',
			$plugin_url . 'build/index.js',
			$editor_dependencies,
			$asset_data['version'],
			false // Editor script loads in head
		);

		// Enqueue only in admin area
		if ( is_admin() ) {
			wp_enqueue_script( 'gfmr-renderer-editor' );

			// Pass block editor settings to JavaScript
			wp_localize_script(
				'gfmr-renderer-editor',
				'wpGfmConfig',
				array(
					'markdownItUrl' => $plugin_url . '/assets/libs/markdown-it/markdown-it.min.js',
					'pluginUrl'     => $plugin_url,
					'version'       => $version,
					'debug'         => defined( 'WP_DEBUG' ) && WP_DEBUG,
					'scriptDebug'   => true,
					'timestamp'     => time(),
					// Theme settings (consistent with frontend structure)
					'theme'         => array(
						'current'     => $this->settings->get_current_theme(),
						'shiki_theme' => $this->settings->get_shiki_theme(),
						'system_auto' => $this->settings->is_system_theme_enabled(),
						'available'   => GFMR_Settings::AVAILABLE_THEMES,
					),
				)
			);

			// Add asset URL resolution function for block editor
			wp_add_inline_script(
				'gfmr-renderer-editor',
				'window.wpGfmBuildLocalAssetUrl = function(relativePath) { 
					var baseUrl = "' . esc_js( $plugin_url ) . '";
					var path = relativePath.startsWith("/") ? relativePath.substring(1) : relativePath;
					console.log("[WP GFM Debug] Asset URL resolved:", baseUrl + path);
					return baseUrl + path;
				};',
				'before'
			);
		}

		// Also register and enqueue block CSS
		if ( file_exists( plugin_dir_path( GFMR_PLUGIN_FILE ) . 'blocks/markdown/editor.css' ) ) {
			wp_register_style(
				'gfmr-renderer-editor-css',
				$plugin_url . 'blocks/markdown/editor.css',
				array( 'gfmr-variables', 'gfmr-table-styles' ), // Clarify dependencies
				$version
			);

			// Enqueue only in admin area
			if ( is_admin() ) {
				wp_enqueue_style( 'gfmr-renderer-editor-css' );
			}
		}

		$this->log_debug( 'Block editor scripts registered successfully' );
	}

	public function render_markdown_block( $attributes ) {
		$content = $attributes['content'] ?? '';
		$html    = $attributes['html'] ?? '';

		if ( empty( $html ) && ! empty( $content ) ) {
			// Fallback: process Markdown on server side
			$html = $this->render_markdown_fallback( $content );
		}

		$output  = '<div class="gfmr-markdown-container">';
		$output .= '<div class="gfmr-markdown-source" style="display: none;">' . esc_html( $content ) . '</div>';
		$output .= '<div class="gfmr-markdown-rendered">' . wp_kses_post( $html ) . '</div>';
		$output .= '</div>';

		return $output;
	}

	/**
	 * Server-side Markdown fallback processing
	 *
	 * Code blocks are left to frontend Shiki, processing only basic Markdown syntax
	 *
	 * @param string $content Markdown content
	 * @return string HTML content
	 */
	private function render_markdown_fallback( $content ) {
		// Minimal fallback processing to allow Shiki processing on frontend
		// Leave code blocks unchanged and delegate to frontend
		$html  = '<div class="gfmr-fallback-content" data-original-markdown="' . esc_attr( $content ) . '">';
		$html .= '<p>📝 ' . esc_html__( 'Processing Markdown on the frontend...', 'markdown-renderer-for-github' ) . '</p>';
		$html .= '</div>';

		return $html;
	}


	/**
	 * Check global Markdown usage (lightweight version)
	 *
	 * @return bool
	 */
	private function check_global_markdown_usage() {
		// Use cached result if available
		$cache_key     = 'gfmr_has_markdown_globally';
		$cached_result = get_transient( $cache_key );

		if ( false !== $cached_result ) {
			$this->log_debug( 'Global markdown check (cached): ' . ( $cached_result ? 'found' : 'not found' ) );
			return (bool) $cached_result;
		}

		// Efficient check using WordPress standard functions
		$posts_with_markdown = get_posts(
			array(
				'post_type'   => array( 'post', 'page' ),
				'post_status' => 'publish',
				'numberposts' => 1,
				's'           => 'gfmr-renderer/markdown',
				'fields'      => 'ids',
			)
		);

		$has_markdown = ! empty( $posts_with_markdown );

		// MD-22 optimization: Extend cache period to 1 hour (reduce frequent DB queries)
		set_transient( $cache_key, $has_markdown ? 1 : 0, 1 * HOUR_IN_SECONDS );

		$this->log_debug( 'Global markdown check (fresh): ' . ( $has_markdown ? 'found' : 'not found' ) );
		return $has_markdown;
	}

	/**
	 * Allow language classes in code blocks for syntax highlighting
	 * Prevents WordPress from stripping language-* classes from <pre> and <code> elements
	 *
	 * @param array  $allowed_html Allowed HTML elements and attributes
	 * @param string $context Context for which the filter is being applied
	 * @return array Modified allowed HTML array
	 * @since 1.0.0
	 */
	public function allow_code_language_classes( $allowed_html, $context ) {
		// Only modify for post context to avoid affecting other areas
		if ( 'post' !== $context ) {
			return $allowed_html;
		}

		// Ensure pre and code elements exist in allowed HTML
		if ( ! isset( $allowed_html['pre'] ) ) {
			$allowed_html['pre'] = array();
		}
		if ( ! isset( $allowed_html['code'] ) ) {
			$allowed_html['code'] = array();
		}

		// Allow class attribute for both pre and code elements
		$code_attributes = array(
			'class'  => true,
			'id'     => true,
			'style'  => true,
			'data-*' => true,
		);

		$allowed_html['pre']  = array_merge( $allowed_html['pre'], $code_attributes );
		$allowed_html['code'] = array_merge( $allowed_html['code'], $code_attributes );

		return $allowed_html;
	}

	/**
	 * Process Markdown content in post content
	 * Detects Markdown patterns and wraps them in proper containers for JavaScript processing
	 *
	 * @param string $content The post content
	 * @return string Modified content with Markdown containers
	 * @since 1.0.0
	 */
	public function process_markdown_content( $content ) {
		// Skip processing in admin area or during AJAX requests
		if ( is_admin() || wp_doing_ajax() ) {
			return $content;
		}

		// Skip if content is empty
		if ( empty( trim( $content ) ) ) {
			return $content;
		}

		// Check if content contains Markdown patterns (code blocks, etc.)
		if ( ! $this->has_markdown_patterns( $content ) ) {
			return $content;
		}

		$this->log_debug( 'Processing Markdown content with patterns detected' );

		// Build context for extension hooks
		$context = array(
			'post_id'     => get_the_ID(),
			'block_attrs' => array(),
		);

		/**
		 * Filters Markdown content before rendering
		 *
		 * Allows addons to modify the Markdown content before it's processed.
		 *
		 * @since 2.0.0
		 *
		 * @param string $content The original Markdown content
		 * @param array  $context Context information
		 */
		$content = apply_filters( 'gfmr_pre_render_markdown', $content, $context );

		// Create container structure for JavaScript processing
		$output  = '<div class="gfmr-markdown-container">';
		$output .= '<div class="gfmr-markdown-source" style="display: none;">' . esc_html( $content ) . '</div>';
		$output .= '<div class="gfmr-markdown-rendered">' . wp_kses_post( $content ) . '</div>';
		$output .= '</div>';

		/**
		 * Filters the rendered Markdown HTML output
		 *
		 * Allows addons to modify the final HTML output after rendering.
		 *
		 * @since 2.0.0
		 *
		 * @param string $output   The rendered HTML output
		 * @param string $content  The original Markdown content
		 * @param array  $context  Context information
		 */
		$output = apply_filters( 'gfmr_render_markdown', $output, $content, $context );

		/**
		 * Fires after Markdown rendering is complete
		 *
		 * Allows addons to perform actions after rendering.
		 *
		 * @since 2.0.0
		 *
		 * @param string $output   The rendered HTML output
		 * @param string $content  The original Markdown content
		 * @param array  $context  Context information
		 */
		do_action( 'gfmr_after_render', $output, $content, $context );

		return $output;
	}

	/**
	 * Check if content contains Markdown patterns
	 *
	 * @param string $content Content to check
	 * @return bool True if Markdown patterns found
	 */
	private function has_markdown_patterns( $content ) {
		// Check for fenced code blocks
		if ( preg_match( '/```[\w]*\n.*?\n```/s', $content ) ) {
			return true;
		}

		// Check for inline code
		if ( strpos( $content, '`' ) !== false ) {
			return true;
		}

		// Check for Mermaid diagrams
		$mermaid_patterns = array(
			'graph',
			'sequenceDiagram',
			'classDiagram',
			'flowchart',
			'pie',
			'stateDiagram',
			'erDiagram',
			'gitGraph',
			'gantt',
			'journey',
		);
		foreach ( $mermaid_patterns as $pattern ) {
			if ( strpos( $content, $pattern ) !== false ) {
				return true;
			}
		}

		return false;
	}

	private function __clone() {}

	public function __wakeup() {
		throw new \Exception( esc_html__( 'Cannot unserialize singleton', 'markdown-renderer-for-github' ) );
	}

	/**
	 * Allow frontmatter HTML tags in wp_kses
	 *
	 * @param array  $allowed_html Allowed HTML tags.
	 * @param string $context      Context.
	 * @return array Modified allowed HTML tags.
	 */
	public function allow_frontmatter_html_tags( $allowed_html, $context ) {
		if ( 'post' !== $context ) {
			return $allowed_html;
		}

		$allowed_html['header'] = array(
			'class' => true,
			'id'    => true,
			'style' => true,
		);
		$allowed_html['time']   = array(
			'class'    => true,
			'datetime' => true,
			'style'    => true,
		);

		return $allowed_html;
	}

	/**
	 * Save frontmatter title to post meta
	 *
	 * @param string $block_content Block content.
	 * @param array  $block         Block data.
	 * @return string Block content.
	 */
	public function save_frontmatter_title( $block_content, $block ) {
		if ( ! isset( $block['blockName'] ) || 'gfm-renderer/markdown' !== $block['blockName'] ) {
			return $block_content;
		}

		$frontmatter = $block['attrs']['frontmatterData'] ?? array();
		if ( ! empty( $frontmatter['title'] ) ) {
			$post_id = get_the_ID();
			if ( $post_id ) {
				update_post_meta( $post_id, '_gfmr_frontmatter_title', sanitize_text_field( $frontmatter['title'] ) );
			}
		}

		return $block_content;
	}

	/**
	 * Apply frontmatter title to document title
	 *
	 * @param array $title_parts Title parts.
	 * @return array Modified title parts.
	 */
	public function apply_frontmatter_title( $title_parts ) {
		if ( ! is_singular() ) {
			return $title_parts;
		}

		if ( ! $this->settings->get( 'frontmatter_auto_title' ) ) {
			return $title_parts;
		}

		$fm_title = get_post_meta( get_the_ID(), '_gfmr_frontmatter_title', true );
		if ( $fm_title ) {
			$title_parts['title'] = esc_html( $fm_title );
		}

		return $title_parts;
	}

	/**
	 * Invalidate SSR cache for a specific post
	 *
	 * @param int $post_id Post ID.
	 */
	public function invalidate_post_ssr_cache( $post_id ) {
		// Skip for autosave and revisions
		if ( wp_is_post_autosave( $post_id ) || wp_is_post_revision( $post_id ) ) {
			return;
		}

		// Get post content to check if it contains Markdown blocks
		$post = get_post( $post_id );
		if ( ! $post || ! has_blocks( $post->post_content ) ) {
			return;
		}

		// Parse blocks to find Markdown blocks
		$blocks = parse_blocks( $post->post_content );
		foreach ( $blocks as $block ) {
			if ( isset( $block['blockName'] ) && 'gfm-renderer/markdown' === $block['blockName'] ) {
				// Get content hash from block attributes
				$content = $block['attrs']['content'] ?? '';
				if ( ! empty( $content ) ) {
					$content_hash = hash( 'md5', $content );

					// Clear SSR cache for all themes
					foreach ( array( 'github-light', 'github-dark', 'auto' ) as $theme ) {
						$this->cache_manager->delete_cache( "ssr_{$content_hash}_{$theme}" );
					}

					$this->log_debug( "Invalidated SSR cache for post {$post_id}" );
				}
			}
		}
	}

	/**
	 * Invalidate all SSR cache when theme settings change
	 *
	 * @param mixed $old_value Old option value.
	 * @param mixed $value     New option value.
	 */
	public function invalidate_theme_ssr_cache( $old_value, $value ) {
		// Only clear if theme actually changed
		if ( isset( $old_value['shiki_theme'] ) && isset( $value['shiki_theme'] ) ) {
			if ( $old_value['shiki_theme'] !== $value['shiki_theme'] ) {
				// Clear all code block cache for the changed theme
				$this->cache_manager->clear_theme_cache( $value['shiki_theme'] );
				$this->log_debug( "Invalidated SSR cache for theme change: {$value['shiki_theme']}" );
			}
		}

		// Clear all SSR cache if system theme setting changed
		if ( isset( $old_value['system_theme'] ) && isset( $value['system_theme'] ) ) {
			if ( $old_value['system_theme'] !== $value['system_theme'] ) {
				$this->cache_manager->clear_theme_cache( 'all' );
				$this->log_debug( 'Invalidated all SSR cache for system theme setting change' );
			}
		}
	}

	/**
	 * Create instance for testing with injected dependencies.
	 *
	 * This method uses reflection to bypass the private constructor,
	 * allowing tests to inject mock dependencies.
	 *
	 * @param GFMR_Asset_Manager|null    $asset_manager    Asset manager instance.
	 * @param GFMR_Block_Registry|null   $block_registry   Block registry instance.
	 * @param GFMR_Cache_Manager|null    $cache_manager    Cache manager instance.
	 * @param GFMR_Settings|null         $settings         Settings instance.
	 * @param GFMR_Schema_Generator|null $schema_generator Schema generator instance.
	 * @param GFMR_Content_Analyzer|null $content_analyzer Content analyzer instance.
	 * @return self New instance with injected dependencies.
	 */
	public static function create_for_testing(
		?GFMR_Asset_Manager $asset_manager = null,
		?GFMR_Block_Registry $block_registry = null,
		?GFMR_Cache_Manager $cache_manager = null,
		?GFMR_Settings $settings = null,
		?GFMR_Schema_Generator $schema_generator = null,
		?GFMR_Content_Analyzer $content_analyzer = null
	): self {
		$reflection  = new \ReflectionClass( self::class );
		$constructor = $reflection->getConstructor();
		$constructor->setAccessible( true );

		$instance = $reflection->newInstanceWithoutConstructor();
		$constructor->invoke(
			$instance,
			$asset_manager,
			$block_registry,
			$cache_manager,
			$settings,
			$schema_generator,
			$content_analyzer
		);

		return $instance;
	}
}
