<?php
/**
 * Multilingual Feature for Markdown Block
 *
 * Provides multilingual content support for the Markdown block.
 * Free version - can be controlled via filters for future Pro version integration.
 *
 * @package WpGfmRenderer
 */

namespace Wakalab\WpGfmRenderer;

defined( 'ABSPATH' ) || exit;

/**
 * GFMR_Multilingual class
 *
 * Handles multilingual feature registration and asset loading.
 */
class GFMR_Multilingual {

	/**
	 * Singleton instance
	 *
	 * @var GFMR_Multilingual|null
	 */
	private static $instance = null;

	/**
	 * Get singleton instance
	 *
	 * @return GFMR_Multilingual
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		// Only initialize if feature is enabled
		if ( $this->is_enabled() ) {
			$this->init_hooks();
		}
	}

	/**
	 * Check if multilingual feature is enabled
	 *
	 * Pro version can disable this via filter:
	 * add_filter( 'gfmr_multilingual_enabled', '__return_false' );
	 *
	 * @return bool
	 */
	public function is_enabled(): bool {
		/**
		 * Filter to enable/disable multilingual feature
		 *
		 * @since 2.4.0
		 *
		 * @param bool $enabled Whether multilingual is enabled
		 */
		return apply_filters( 'gfmr_multilingual_enabled', true );
	}

	/**
	 * Initialize hooks
	 */
	private function init_hooks() {
		// Register feature flag
		add_filter( 'gfmr_feature_flags', array( $this, 'register_feature_flag' ) );

		// Enqueue frontend assets
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_frontend_assets' ) );

		// Enqueue editor assets
		add_action( 'enqueue_block_editor_assets', array( $this, 'enqueue_editor_assets' ) );

		// Add site language to wpGfmConfig
		add_filter( 'gfmr_frontend_config', array( $this, 'add_site_language_config' ) );

		// Add multilingual config to frontend
		add_filter( 'gfmr_frontend_config', array( $this, 'add_multilingual_config' ) );

		// Output hreflang tags
		add_action( 'wp_head', array( $this, 'output_hreflang_meta' ) );
	}

	/**
	 * Register multilingual feature flag
	 *
	 * @param array $flags Existing feature flags.
	 * @return array Modified feature flags.
	 */
	public function register_feature_flag( $flags ) {
		$flags['multilingual'] = true;
		return $flags;
	}

	/**
	 * Add site language to frontend config
	 *
	 * @param array $config Existing config.
	 * @return array Modified config.
	 */
	public function add_site_language_config( $config ) {
		// Get site language (e.g., 'en-US' -> 'en')
		$locale = get_locale();
		$lang_code = substr( $locale, 0, 2 );

		$config['siteLanguage'] = $lang_code;

		return $config;
	}

	/**
	 * Enqueue frontend assets
	 */
	public function enqueue_frontend_assets() {
		/**
		 * Filter to control asset enqueuing
		 *
		 * Pro version can override with custom assets:
		 * add_filter( 'gfmr_multilingual_enqueue_assets', '__return_false' );
		 *
		 * @since 2.4.0
		 *
		 * @param bool $enqueue Whether to enqueue assets
		 */
		if ( ! apply_filters( 'gfmr_multilingual_enqueue_assets', true ) ) {
			return;
		}

		// Check if we have multilingual blocks on the page
		// Only enqueue if needed (performance optimization)
		if ( ! $this->page_has_multilingual_blocks() ) {
			return;
		}

		// Enqueue CSS
		wp_enqueue_style(
			'gfmr-multilingual',
			( defined( 'GFMR_URI' ) ? GFMR_URI : plugin_dir_url( __DIR__ ) ) . 'assets/css/gfmr-multilingual.css',
			array(),
			GFMR_VERSION
		);

		// Enqueue JavaScript
		wp_enqueue_script(
			'gfmr-multilingual',
			( defined( 'GFMR_URI' ) ? GFMR_URI : plugin_dir_url( __DIR__ ) ) . 'assets/js/gfmr-multilingual.js',
			array(),
			GFMR_VERSION,
			true // Load in footer
		);
	}

	/**
	 * Enqueue editor assets
	 */
	public function enqueue_editor_assets() {
		if ( ! apply_filters( 'gfmr_multilingual_enqueue_assets', true ) ) {
			return;
		}

		// Enqueue editor CSS
		wp_enqueue_style(
			'gfmr-multilingual-editor',
			( defined( 'GFMR_URI' ) ? GFMR_URI : plugin_dir_url( __DIR__ ) ) . 'assets/css/gfmr-multilingual.css',
			array(),
			GFMR_VERSION
		);

		// Add site language to editor config
		wp_add_inline_script(
			'gfmr-renderer-editor',
			sprintf(
				'window.wpGfmConfig = window.wpGfmConfig || {}; window.wpGfmConfig.siteLanguage = %s;',
				wp_json_encode( substr( get_locale(), 0, 2 ) )
			),
			'before'
		);
	}

	/**
	 * Check if current page has multilingual blocks
	 *
	 * @return bool
	 */
	private function page_has_multilingual_blocks(): bool {
		global $post;

		if ( ! $post || ! is_singular() ) {
			return false;
		}

		$content = $post->post_content;

		// Check for multilingual markers in post content
		// 1. data-languages attribute in save.js output
		// 2. gfmr-multilingual class in save.js output
		// 3. "availableLanguages" in block comment JSON (regex for robustness)
		return (
			strpos( $content, 'data-languages' ) !== false ||
			strpos( $content, 'gfmr-multilingual' ) !== false ||
			preg_match( '/"availableLanguages"\s*:\s*\[/', $content ) === 1
		);
	}

	/**
	 * Get current language from URL Rewriter
	 *
	 * @return string Current language code
	 */
	public function get_current_language(): string {
		if ( class_exists( __NAMESPACE__ . '\\GFMR_URL_Rewriter' ) ) {
			$rewriter = GFMR_URL_Rewriter::get_instance();
			return $rewriter->get_current_language();
		}

		// Fallback to site language
		$locale = get_locale();
		return substr( $locale, 0, 2 );
	}

	/**
	 * Get available languages for a post
	 *
	 * @param int $post_id Post ID
	 * @return array Available language codes
	 */
	public function get_post_available_languages( $post_id ): array {
		$post = get_post( $post_id );
		if ( ! $post ) {
			return array();
		}

		$content = $post->post_content;

		// Check for multilingual block data
		if ( preg_match( '/data-available-languages=["\']([^"\']+)["\']/', $content, $matches ) ) {
			try {
				$languages = json_decode( $matches[1], true );
				if ( is_array( $languages ) ) {
					return $languages;
				}
			} catch ( \Exception $e ) {
				// Invalid JSON, continue
			}
		}

		// Check for "availableLanguages" in block comment JSON
		if ( preg_match( '/"availableLanguages"\s*:\s*\[([^\]]+)\]/', $content, $matches ) ) {
			$json_str = '[' . $matches[1] . ']';
			try {
				$languages = json_decode( $json_str, true );
				if ( is_array( $languages ) ) {
					return $languages;
				}
			} catch ( \Exception $e ) {
				// Invalid JSON, continue
			}
		}

		return array();
	}

	/**
	 * Generate hreflang links for a post
	 *
	 * @param int $post_id Post ID
	 * @return array Array of hreflang links with language codes as keys
	 */
	public function generate_hreflang_links( $post_id ): array {
		if ( ! class_exists( __NAMESPACE__ . '\\GFMR_URL_Rewriter' ) ) {
			return array();
		}

		$rewriter = GFMR_URL_Rewriter::get_instance();
		$languages = $rewriter->get_supported_languages();
		$links = array();

		foreach ( $languages as $lang ) {
			$url = $rewriter->generate_language_url( $post_id, $lang );
			if ( $url ) {
				$links[ $lang ] = $url;
			}
		}

		return $links;
	}

	/**
	 * Output hreflang meta tags
	 */
	public function output_hreflang_meta() {
		if ( ! is_singular() ) {
			return;
		}

		$post_id = get_the_ID();
		if ( ! $post_id ) {
			return;
		}

		if ( ! class_exists( __NAMESPACE__ . '\\GFMR_URL_Rewriter' ) ) {
			return;
		}

		$rewriter = GFMR_URL_Rewriter::get_instance();

		if ( ! $rewriter->is_hreflang_enabled() ) {
			return;
		}

		$available_langs = $this->get_post_available_languages( $post_id );
		if ( empty( $available_langs ) ) {
			// Fallback to configured languages
			$available_langs = $rewriter->get_supported_languages();
		}

		$default_lang = $rewriter->get_default_language();

		foreach ( $available_langs as $lang ) {
			$url = $rewriter->generate_language_url( $post_id, $lang );
			if ( $url ) {
				printf(
					'<link rel="alternate" hreflang="%s" href="%s" />' . "\n",
					esc_attr( $lang ),
					esc_url( $url )
				);
			}
		}

		// x-default
		$default_url = $rewriter->generate_language_url( $post_id, $default_lang );
		if ( $default_url ) {
			printf(
				'<link rel="alternate" hreflang="x-default" href="%s" />' . "\n",
				esc_url( $default_url )
			);
		}
	}

	/**
	 * Add multilingual config to frontend
	 */
	public function add_multilingual_config( $config ) {
		if ( class_exists( __NAMESPACE__ . '\\GFMR_URL_Rewriter' ) ) {
			$rewriter = GFMR_URL_Rewriter::get_instance();

			$config['multilingualUrlMode'] = $rewriter->is_path_mode_enabled() ? 'path' : 'query';
			$config['multilingualDefaultLang'] = $rewriter->get_default_language();
			$config['multilingualHideDefaultLang'] = $rewriter->hide_default_language();
			$config['multilingualSupportedLangs'] = $rewriter->get_supported_languages();
		}

		return $config;
	}
}
