<?php
/**
 * WP GFM Metadata Handler
 *
 * Handles server-side metadata persistence for code block language information
 * Integrates with WordPress post meta to store language data across content lifecycle
 *
 * @package WpGfmRenderer
 * @since 1.1.0
 */

namespace Wakalab\WpGfmRenderer;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

// Prevent class redeclaration when both Free and Pro versions are active
if ( class_exists( __NAMESPACE__ . '\\GFMR_Metadata_Handler' ) ) {
	return;
}

/**
 * Class WP_GFM_Renderer_Metadata_Handler
 */
class GFMR_Metadata_Handler {

	/**
	 * Meta key for storing code metadata
	 */
	const META_KEY = '_gfmr_code_metadata';

	/**
	 * Version for metadata format
	 */
	const METADATA_VERSION = '1.0';

	/**
	 * Constructor
	 */
	public function __construct() {
		// Hook into WordPress save process
		add_action( 'save_post', array( $this, 'maybe_extract_and_save_metadata' ), 10, 2 );

		// Hook into content processing to inject metadata
		add_filter( 'gfmr_renderer/before_process_content', array( $this, 'inject_metadata_to_content' ), 10, 2 );

		// AJAX endpoint for editor-side metadata saving
		add_action( 'wp_ajax_gfmr_save_metadata', array( $this, 'ajax_save_metadata' ) );
		add_action( 'wp_ajax_gfmr_get_metadata', array( $this, 'ajax_get_metadata' ) );

		// Enqueue nonce for AJAX requests in admin
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_metadata_nonce' ) );
	}

	/**
	 * Enqueue nonce for AJAX requests
	 */
	public function enqueue_metadata_nonce() {
		if ( ! is_admin() ) {
			return;
		}

		wp_localize_script(
			'wp-blocks',
			'wpGfmMetadata',
			array(
				'nonce'   => wp_create_nonce( 'gfmr_metadata_nonce' ),
				'ajaxUrl' => admin_url( 'admin-ajax.php' ),
			)
		);
	}

	/**
	 * Save code block metadata for a post
	 *
	 * @param int $post_id Post ID
	 * @param array $code_blocks Array of code block data
	 * @return bool Success status
	 */
	public function save_post_code_metadata( $post_id, $code_blocks ) {
		if ( ! $post_id || ! is_array( $code_blocks ) ) {
			return false;
		}

		$metadata = array();
		foreach ( $code_blocks as $index => $block ) {
			if ( ! isset( $block['code'] ) || empty( $block['code'] ) ) {
				continue;
			}

			$metadata[ $index ] = array(
				'language'          => sanitize_text_field( $block['language'] ?? 'plaintext' ),
				'original_language' => sanitize_text_field( $block['original_language'] ?? ( $block['language'] ?? 'plaintext' ) ),
				'code_hash'         => $this->generate_code_hash( $block['code'] ),
				'block_id'          => sanitize_text_field( $block['block_id'] ?? null ),
				'timestamp'         => current_time( 'timestamp' ), // phpcs:ignore WordPress.DateTime.CurrentTimeTimestamp.Requested -- Stored for debug/comparison purposes
				'version'           => self::METADATA_VERSION,
			);
		}

		$result = update_post_meta( $post_id, self::META_KEY, $metadata );

		if ( false !== $result ) {
			do_action( 'gfmr_metadata_saved', $post_id, $metadata );
		}

		return false !== $result;
	}

	/**
	 * Get code block metadata for a post
	 *
	 * @param int $post_id Post ID
	 * @return array Metadata array
	 */
	public function get_post_code_metadata( $post_id ) {
		if ( ! $post_id ) {
			return array();
		}

		$metadata = get_post_meta( $post_id, self::META_KEY, true );

		if ( ! is_array( $metadata ) ) {
			return array();
		}

		// Apply filters to allow customization
		return apply_filters( 'gfmr_metadata_get_metadata', $metadata, $post_id );
	}

	/**
	 * Generate hash for code content (matches JS implementation)
	 *
	 * @param string $code Code content
	 * @return string|null Hash or null if invalid
	 */
	private function generate_code_hash( $code ) {
		if ( empty( $code ) || ! is_string( $code ) ) {
			return null;
		}

		// Normalize code for hashing (matches JS implementation)
		$normalized = trim( $code );
		$normalized = str_replace( "\r\n", "\n", $normalized );
		$normalized = str_replace( "\r", "\n", $normalized );

		if ( empty( $normalized ) ) {
			return null;
		}

		// Simple hash (matches JS implementation approach)
		return base_convert( abs( crc32( $normalized ) ), 10, 36 );
	}

	/**
	 * Extract code blocks from post content and save metadata
	 * Called during post save
	 *
	 * @param int $post_id Post ID
	 * @param WP_Post $post Post object
	 */
	public function maybe_extract_and_save_metadata( $post_id, $post ) {
		// Skip autosaves and revisions
		if ( wp_is_post_autosave( $post_id ) || wp_is_post_revision( $post_id ) ) {
			return;
		}

		// Skip if not a supported post type
		if ( ! in_array( $post->post_type, array( 'post', 'page' ), true ) ) {
			return;
		}

		$content = $post->post_content;
		if ( empty( $content ) ) {
			return;
		}

		// Extract code blocks from content
		$code_blocks = $this->extract_code_blocks_from_content( $content );

		if ( ! empty( $code_blocks ) ) {
			$this->save_post_code_metadata( $post_id, $code_blocks );
		}
	}

	/**
	 * Extract code blocks from post content
	 *
	 * @param string $content Post content
	 * @return array Array of code blocks
	 */
	private function extract_code_blocks_from_content( $content ) {
		$code_blocks = array();

		// Pattern 1: Markdown fenced code blocks
		$pattern1 = '/```(\w+)?\s*\n(.*?)\n```/s';
		if ( preg_match_all( $pattern1, $content, $matches, PREG_SET_ORDER ) ) {
			foreach ( $matches as $index => $match ) {
				$language = ! empty( $match[1] ) ? $match[1] : 'plaintext';
				$code     = $match[2];

				$code_blocks[] = array(
					'language'          => $this->resolve_language_alias( $language ),
					'original_language' => $language,
					'code'              => $code,
					'type'              => 'fenced',
					'index'             => $index,
				);
			}
		}

		// Pattern 2: HTML code blocks (from Gutenberg)
		$pattern2 = '/<pre(?:[^>]*class="[^"]*language-(\w+)[^"]*"[^>]*)?><code[^>]*>(.*?)<\/code><\/pre>/s';
		if ( preg_match_all( $pattern2, $content, $matches, PREG_SET_ORDER ) ) {
			foreach ( $matches as $index => $match ) {
				$language = ! empty( $match[1] ) ? $match[1] : 'plaintext';
				$code     = html_entity_decode( $match[2], ENT_QUOTES | ENT_HTML5 );

				$code_blocks[] = array(
					'language'          => $this->resolve_language_alias( $language ),
					'original_language' => $language,
					'code'              => $code,
					'type'              => 'html',
					'index'             => count( $code_blocks ), // Continue numbering
				);
			}
		}

		return $code_blocks;
	}

	/**
	 * Resolve language aliases to canonical names
	 *
	 * @param string $language Original language
	 * @return string Resolved language
	 */
	private function resolve_language_alias( $language ) {
		$aliases = array(
			'js'    => 'javascript',
			'ts'    => 'typescript',
			'py'    => 'python',
			'sh'    => 'bash',
			'shell' => 'bash',
			'yml'   => 'yaml',
			'md'    => 'markdown',
		);

		$normalized = strtolower( trim( $language ) );
		return isset( $aliases[ $normalized ] ) ? $aliases[ $normalized ] : $normalized;
	}

	/**
	 * Inject metadata into content processing
	 *
	 * @param string $content Content
	 * @param int|null $post_id Post ID
	 * @return string Modified content
	 */
	public function inject_metadata_to_content( $content, $post_id = null ) {
		if ( ! $post_id ) {
			$post_id = get_the_ID();
		}

		if ( ! $post_id ) {
			return $content;
		}

		$metadata = $this->get_post_code_metadata( $post_id );

		if ( empty( $metadata ) ) {
			return $content;
		}

		// Add metadata as JavaScript variables
		wp_localize_script(
			'gfmr-renderer',
			'wpGfmPostMetadata',
			array(
				'post_id'       => $post_id,
				'code_metadata' => array_values( $metadata ), // Ensure it's a proper array
				'timestamp'     => time(),
				'version'       => self::METADATA_VERSION,
			)
		);

		return $content;
	}

	/**
	 * AJAX handler for saving metadata from editor
	 */
	public function ajax_save_metadata() {
		// Verify nonce
		$nonce = sanitize_text_field( wp_unslash( $_POST['nonce'] ?? '' ) );
		if ( ! wp_verify_nonce( $nonce, 'gfmr_metadata_nonce' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'markdown-renderer-for-github' ) );
		}

		// Check permissions
		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_die( esc_html__( 'Insufficient permissions.', 'markdown-renderer-for-github' ) );
		}

		$post_id = intval( $_POST['post_id'] ?? 0 );

		// Sanitize code blocks data
		$code_blocks = array();
		if ( isset( $_POST['code_blocks'] ) && is_array( $_POST['code_blocks'] ) ) {
			$code_blocks = array_map( 'sanitize_text_field', wp_unslash( $_POST['code_blocks'] ) );
		}

		if ( ! $post_id || ! is_array( $code_blocks ) ) {
			wp_send_json_error( 'Invalid parameters' );
		}

		$success = $this->save_post_code_metadata( $post_id, $code_blocks );

		if ( $success ) {
			wp_send_json_success(
				array(
					'message' => 'Metadata saved successfully',
					'post_id' => $post_id,
					'count'   => count( $code_blocks ),
				)
			);
		} else {
			wp_send_json_error( 'Failed to save metadata' );
		}
	}

	/**
	 * AJAX handler for getting metadata
	 */
	public function ajax_get_metadata() {
		// Verify nonce
		$nonce = sanitize_text_field( wp_unslash( $_POST['nonce'] ?? '' ) );
		if ( ! wp_verify_nonce( $nonce, 'gfmr_metadata_nonce' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'markdown-renderer-for-github' ) );
		}

		$post_id = intval( $_POST['post_id'] ?? 0 );

		if ( ! $post_id ) {
			wp_send_json_error( 'Invalid post ID' );
		}

		$metadata = $this->get_post_code_metadata( $post_id );

		wp_send_json_success(
			array(
				'post_id'  => $post_id,
				'metadata' => $metadata,
				'count'    => count( $metadata ),
			)
		);
	}
}

// Initialize the handler
new GFMR_Metadata_Handler();
