<?php
/**
 * Mermaid SSR Handler
 *
 * Handles server-side caching of Mermaid SVG diagrams
 *
 * @package WpGfmRenderer
 * @since 2.0.0
 */

namespace Wakalab\WpGfmRenderer;

defined( 'ABSPATH' ) || exit;

// Prevent class redeclaration when both Free and Pro versions are active
if ( class_exists( __NAMESPACE__ . '\\GFMR_Mermaid_SSR_Handler' ) ) {
	return;
}

class GFMR_Mermaid_SSR_Handler {

	/**
	 * Cache manager instance
	 *
	 * @var GFMR_Cache_Manager
	 */
	private $cache_manager;

	/**
	 * Constructor
	 *
	 * @param GFMR_Cache_Manager $cache_manager Cache manager instance.
	 */
	public function __construct( $cache_manager ) {
		$this->cache_manager = $cache_manager;
		$this->init_hooks();
	}

	/**
	 * Initialize WordPress hooks
	 */
	private function init_hooks() {
		// AJAX endpoint for saving Mermaid SVG (logged-in users only)
		add_action( 'wp_ajax_gfmr_save_mermaid_svg', array( $this, 'ajax_save_mermaid_svg' ) );
	}

	/**
	 * Get cached Mermaid SVG
	 *
	 * @param string $diagram_id Diagram identifier (content hash).
	 * @param string $theme      Theme identifier.
	 * @param string $bg_color   Background color.
	 * @return string|false Cached SVG or false if not found.
	 */
	public function get_cached_svg( $diagram_id, $theme, $bg_color ) {
		return $this->cache_manager->get_mermaid_svg_cache( $diagram_id, $theme, $bg_color );
	}

	/**
	 * Save Mermaid SVG to cache
	 *
	 * @param string $diagram_id Diagram identifier (content hash).
	 * @param string $theme      Theme identifier.
	 * @param string $bg_color   Background color.
	 * @param string $svg        SVG content to cache.
	 * @return bool True on success.
	 */
	public function save_svg( $diagram_id, $theme, $bg_color, $svg ) {
		// Sanitize SVG before caching
		$sanitized_svg = $this->sanitize_svg( $svg );

		if ( empty( $sanitized_svg ) ) {
			$this->log_debug( 'SVG sanitization failed or resulted in empty content' );
			return false;
		}

		// Save to cache (7 days expiration)
		return $this->cache_manager->set_mermaid_svg_cache( $diagram_id, $theme, $bg_color, $sanitized_svg );
	}

	/**
	 * Generate placeholder HTML for initial display
	 *
	 * @param string $diagram_id   Diagram identifier.
	 * @param string $theme        Theme identifier.
	 * @param string $bg_color     Background color.
	 * @param string $mermaid_code Original Mermaid code.
	 * @return string Placeholder HTML.
	 */
	public function generate_placeholder( $diagram_id, $theme, $bg_color, $mermaid_code ) {
		$placeholder  = '<div class="gfmr-mermaid-container" ';
		$placeholder .= 'data-diagram-id="' . esc_attr( $diagram_id ) . '" ';
		$placeholder .= 'data-theme="' . esc_attr( $theme ) . '" ';
		$placeholder .= 'data-mermaid-bg-color="' . esc_attr( $bg_color ) . '" ';
		$placeholder .= 'data-ssr="pending">';
		$placeholder .= '<div class="gfmr-mermaid-loading">';
		$placeholder .= '<span class="gfmr-spinner"></span>';
		$placeholder .= '<span>' . esc_html__( 'Rendering diagram...', 'markdown-renderer-for-github' ) . '</span>';
		$placeholder .= '</div>';
		$placeholder .= '<pre class="gfmr-mermaid-source" style="display:none;">' . esc_html( $mermaid_code ) . '</pre>';
		$placeholder .= '</div>';

		return $placeholder;
	}

	/**
	 * Sanitize SVG content to prevent XSS attacks
	 *
	 * Removes JavaScript event handlers and script elements
	 *
	 * @param string $svg SVG content.
	 * @return string Sanitized SVG.
	 */
	public function sanitize_svg( $svg ) {
		// Use wp_kses with SVG-specific allowed tags
		$allowed_svg_tags = array(
			'svg'      => array(
				'class'       => true,
				'viewbox'     => true,
				'xmlns'       => true,
				'width'       => true,
				'height'      => true,
				'aria-hidden' => true,
				'role'        => true,
				'style'       => true,
				'id'          => true,
			),
			'g'        => array(
				'transform' => true,
				'class'     => true,
				'id'        => true,
				'style'     => true,
			),
			'path'     => array(
				'd'            => true,
				'fill'         => true,
				'stroke'       => true,
				'stroke-width' => true,
				'class'        => true,
				'style'        => true,
			),
			'rect'     => array(
				'x'            => true,
				'y'            => true,
				'width'        => true,
				'height'       => true,
				'fill'         => true,
				'stroke'       => true,
				'stroke-width' => true,
				'class'        => true,
				'rx'           => true,
				'ry'           => true,
				'style'        => true,
			),
			'text'     => array(
				'x'                 => true,
				'y'                 => true,
				'class'             => true,
				'text-anchor'       => true,
				'font-size'         => true,
				'font-family'       => true,
				'fill'              => true,
				'style'             => true,
				'dominant-baseline' => true,
			),
			'tspan'    => array(
				'x'     => true,
				'y'     => true,
				'class' => true,
				'style' => true,
			),
			'line'     => array(
				'x1'           => true,
				'y1'           => true,
				'x2'           => true,
				'y2'           => true,
				'stroke'       => true,
				'stroke-width' => true,
				'class'        => true,
				'style'        => true,
				'marker-end'   => true,
				'marker-start' => true,
			),
			'polygon'  => array(
				'points' => true,
				'fill'   => true,
				'stroke' => true,
				'class'  => true,
				'style'  => true,
			),
			'polyline' => array(
				'points'       => true,
				'fill'         => true,
				'stroke'       => true,
				'stroke-width' => true,
				'class'        => true,
				'style'        => true,
				'marker-end'   => true,
			),
			'circle'   => array(
				'cx'     => true,
				'cy'     => true,
				'r'      => true,
				'fill'   => true,
				'stroke' => true,
				'class'  => true,
				'style'  => true,
			),
			'ellipse'  => array(
				'cx'     => true,
				'cy'     => true,
				'rx'     => true,
				'ry'     => true,
				'fill'   => true,
				'stroke' => true,
				'class'  => true,
				'style'  => true,
			),
			'defs'     => array(),
			'marker'   => array(
				'id'           => true,
				'markerWidth'  => true,
				'markerHeight' => true,
				'refX'         => true,
				'refY'         => true,
				'orient'       => true,
				'class'        => true,
			),
			'title'    => array(),
			'desc'     => array(),
		);

		// Remove any event handlers using regex (defense in depth)
		$svg = preg_replace( '/\son\w+\s*=\s*["\'][^"\']*["\']/i', '', $svg );

		// Apply wp_kses
		$sanitized = wp_kses( $svg, $allowed_svg_tags );

		return $sanitized;
	}

	/**
	 * AJAX handler for saving Mermaid SVG
	 */
	public function ajax_save_mermaid_svg() {
		// Verify nonce
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'gfmr_mermaid_ssr' ) ) {
			wp_send_json_error( array( 'message' => 'Security check failed' ), 403 );
		}

		// Verify user is logged in
		if ( ! is_user_logged_in() ) {
			wp_send_json_error( array( 'message' => 'Unauthorized' ), 401 );
		}

		// Get parameters
		$diagram_id = isset( $_POST['diagram_id'] ) ? sanitize_text_field( wp_unslash( $_POST['diagram_id'] ) ) : '';
		$theme      = isset( $_POST['theme'] ) ? sanitize_text_field( wp_unslash( $_POST['theme'] ) ) : '';
		$bg_color   = isset( $_POST['bg_color'] ) ? sanitize_text_field( wp_unslash( $_POST['bg_color'] ) ) : '';
		$svg        = isset( $_POST['svg'] ) ? wp_unslash( $_POST['svg'] ) : '';

		if ( empty( $diagram_id ) || empty( $svg ) ) {
			wp_send_json_error( array( 'message' => 'Missing required parameters' ), 400 );
		}

		// Save SVG to cache
		$success = $this->save_svg( $diagram_id, $theme, $bg_color, $svg );

		if ( $success ) {
			$this->log_debug( "Mermaid SVG cached: {$diagram_id} (theme: {$theme}, bg: {$bg_color})" );
			wp_send_json_success( array( 'message' => 'SVG cached successfully' ) );
		} else {
			wp_send_json_error( array( 'message' => 'Failed to cache SVG' ), 500 );
		}
	}

	/**
	 * Output debug log
	 *
	 * @param string $message Log message.
	 */
	private function log_debug( $message ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging when WP_DEBUG_LOG is enabled
			error_log( '[GFMR_Mermaid_SSR_Handler] ' . $message );
		}
	}
}
