<?php
/**
 * YAML Frontmatter Parser
 *
 * Simple YAML parser for frontmatter (key: value only, no complex nesting)
 *
 * @package WpGfmRenderer
 */

namespace Wakalab\WpGfmRenderer;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Prevent class redeclaration when both Free and Pro versions are active
if ( class_exists( __NAMESPACE__ . '\\GFMR_Frontmatter_Parser' ) ) {
	return;
}

/**
 * GFMR_Frontmatter_Parser class
 */
class GFMR_Frontmatter_Parser {

	/**
	 * Parse Markdown content with YAML frontmatter
	 *
	 * @param string $content Markdown content with optional frontmatter.
	 * @return array Array with 'frontmatter' and 'body' keys.
	 */
	public function parse( $content ) {
		if ( empty( $content ) ) {
			return array(
				'frontmatter' => array(),
				'body'        => '',
			);
		}

		// Match frontmatter pattern (---\n...YAML...\n---\n)
		if ( ! preg_match( '/^---\s*\n(.*?)\n---\s*\n(.*)$/s', $content, $matches ) ) {
			return array(
				'frontmatter' => array(),
				'body'        => $content,
			);
		}

		$yaml_str = $matches[1];
		$body     = $matches[2];

		return array(
			'frontmatter' => $this->parse_yaml( $yaml_str ),
			'body'        => $body,
		);
	}

	/**
	 * Parse simple YAML (key: value only)
	 *
	 * @param string $yaml_str YAML content.
	 * @return array Parsed key-value pairs.
	 */
	private function parse_yaml( $yaml_str ) {
		$result = array();

		if ( empty( $yaml_str ) ) {
			return $result;
		}

		$lines = explode( "\n", $yaml_str );

		foreach ( $lines as $line ) {
			$line = trim( $line );

			// Skip empty lines and comments
			if ( empty( $line ) || substr( $line, 0, 1 ) === '#' ) {
				continue;
			}

			// Parse key: value pattern
			if ( preg_match( '/^(\w+):\s*(.*)$/', $line, $matches ) ) {
				$key   = $matches[1];
				$value = trim( $matches[2], '"\'  ' );

				// Handle array syntax: [item1, item2, item3]
				if ( preg_match( '/^\[(.*)\]$/', $value, $arr_matches ) ) {
					$items = array_map( 'trim', explode( ',', $arr_matches[1] ) );
					$value = array_filter( $items ); // Remove empty items
				}

				$result[ $key ] = $value;
			}
		}

		return $result;
	}

	/**
	 * Render frontmatter as HTML header
	 *
	 * @param array $frontmatter Frontmatter data.
	 * @return string HTML string.
	 */
	public function render_header( $frontmatter ) {
		if ( empty( $frontmatter ) ) {
			return '';
		}

		$html = '<header class="gfmr-frontmatter">';

		// Title
		if ( ! empty( $frontmatter['title'] ) ) {
			$html .= '<h1 class="gfmr-fm-title">' . esc_html( $frontmatter['title'] ) . '</h1>';
		}

		// Date
		if ( ! empty( $frontmatter['date'] ) ) {
			$date  = esc_attr( $frontmatter['date'] );
			$html .= '<time class="gfmr-fm-date" datetime="' . $date . '">' . esc_html( $date ) . '</time>';
		}

		// Author
		if ( ! empty( $frontmatter['author'] ) ) {
			$html .= '<span class="gfmr-fm-author">' . esc_html( $frontmatter['author'] ) . '</span>';
		}

		// Tags
		if ( ! empty( $frontmatter['tags'] ) && is_array( $frontmatter['tags'] ) ) {
			$html .= '<div class="gfmr-fm-tags">';
			foreach ( $frontmatter['tags'] as $tag ) {
				$html .= '<span class="gfmr-fm-tag">' . esc_html( $tag ) . '</span>';
			}
			$html .= '</div>';
		}

		$html .= '</header>';

		return $html;
	}
}
