<?php
/**
 * GFMR Extension API Class
 *
 * Provides open extension API for third-party addons
 * Manages addon registration and lifecycle hooks
 *
 * @package MarkdownRendererForGitHub
 * @since 2.0.0
 */

namespace Wakalab\WpGfmRenderer;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Prevent class redeclaration when both Free and Pro versions are active
if ( class_exists( __NAMESPACE__ . '\\GFMR_Extension_API' ) ) {
	return;
}

/**
 * Extension API management class
 *
 * This class provides extension points for third-party addons including:
 * - Rendering pipeline hooks
 * - Settings screen extensions
 * - Feature flag system
 * - Asset registration hooks
 * - Block extension hooks
 * - Addon registration and lifecycle management
 */
class GFMR_Extension_API {

	/**
	 * Registered addons list
	 *
	 * @var array
	 */
	private $registered_addons = array();

	/**
	 * Initialize the extension API
	 */
	public function init() {
		// Fire gfmr_loaded action on plugins_loaded hook
		add_action( 'plugins_loaded', array( $this, 'fire_loaded_action' ), 10 );
	}

	/**
	 * Fire gfmr_loaded action
	 *
	 * This action is fired after the free version is loaded, allowing addons
	 * to initialize themselves using the extension API.
	 */
	public function fire_loaded_action() {
		/**
		 * Fires when Markdown Renderer for GitHub is fully loaded
		 *
		 * This action allows addons to initialize themselves and register their
		 * features using the extension API provided by the free version.
		 *
		 * @since 2.0.0
		 *
		 * @param string $version The version of the free plugin
		 */
		do_action( 'gfmr_loaded', GFMR_VERSION );
	}

	/**
	 * Get registered addons
	 *
	 * Returns the list of registered addons. This list is populated by addons
	 * using the gfmr_registered_addons filter.
	 *
	 * @return array Array of registered addons
	 */
	public function get_registered_addons() {
		/**
		 * Filters the list of registered addons
		 *
		 * Addons should use this filter to register themselves in the system.
		 *
		 * @since 2.0.0
		 *
		 * @param array $addons Array of registered addons in format:
		 *                      ['addon_id' => [
		 *                          'name' => 'Addon Name',
		 *                          'version' => '1.0.0',
		 *                          'author' => 'Author Name',
		 *                      ]]
		 */
		$this->registered_addons = apply_filters( 'gfmr_registered_addons', $this->registered_addons );
		return $this->registered_addons;
	}

	/**
	 * Check if a specific addon is registered
	 *
	 * @param string $addon_id The addon identifier
	 * @return bool True if the addon is registered, false otherwise
	 */
	public function is_addon_registered( $addon_id ) {
		$addons = $this->get_registered_addons();
		return isset( $addons[ $addon_id ] );
	}

	/**
	 * Get addon information
	 *
	 * @param string $addon_id The addon identifier
	 * @return array|null Addon information array or null if not found
	 */
	public function get_addon_info( $addon_id ) {
		$addons = $this->get_registered_addons();
		return $addons[ $addon_id ] ?? null;
	}
}
