<?php
/**
 * Code Highlighter using highlight.php
 *
 * Wraps highlight.php library for server-side syntax highlighting
 *
 * @package WpGfmRenderer
 * @since 2.0.0
 */

namespace Wakalab\WpGfmRenderer;

use Highlight\Highlighter;

defined( 'ABSPATH' ) || exit;

// Prevent class redeclaration when both Free and Pro versions are active
if ( class_exists( __NAMESPACE__ . '\\GFMR_Code_Highlighter' ) ) {
	return;
}

class GFMR_Code_Highlighter {

	/**
	 * Highlighter instance
	 *
	 * @var Highlighter
	 */
	private $highlighter;

	/**
	 * Theme mapping: Shiki theme -> highlight.php theme
	 *
	 * @var array
	 */
	private $theme_mapping = array(
		'github-light' => 'github',
		'github-dark'  => 'github-dark',
		'auto'         => 'github', // Default to light theme
	);

	/**
	 * Constructor
	 */
	public function __construct() {
		// Load highlight.php via Composer autoload
		if ( ! class_exists( 'Highlight\Highlighter' ) ) {
			require_once GFMR_PATH . 'vendor/autoload.php';
		}

		$this->highlighter = new Highlighter();
	}

	/**
	 * Highlight code with specified language
	 *
	 * @param string $code Code to highlight
	 * @param string $language Language identifier
	 * @return string Highlighted HTML
	 */
	public function highlight( $code, $language ) {
		try {
			// Resolve language alias
			$language = $this->resolve_language_alias( $language );

			// Check if language is supported
			if ( ! $this->is_language_supported( $language ) ) {
				$language = 'plaintext';
			}

			// Highlight code
			$highlighted = $this->highlighter->highlight( $language, $code );

			// Wrap in pre/code with data-ssr="true" and data-gfmr-processed attributes
			$output = sprintf(
				'<pre data-ssr="true" data-gfmr-processed="true"><code class="hljs language-%s">%s</code></pre>',
				esc_attr( $language ),
				$highlighted->value
			);

			return $output;
		} catch ( \Exception $e ) {
			// Fallback: return plain code
			$this->log_debug( 'Highlighting failed: ' . $e->getMessage() );
			return $this->generate_fallback( $code, $language );
		}
	}

	/**
	 * Set theme (maps Shiki theme to highlight.php theme)
	 *
	 * @param string $shiki_theme Shiki theme identifier
	 */
	public function set_theme( $shiki_theme ) {
		// Theme mapping is handled via CSS, not programmatically
		// highlight.php uses CSS classes, so we just note the theme preference
		$this->log_debug( "Theme set to: {$shiki_theme}" );
	}

	/**
	 * Check if language is supported
	 *
	 * @param string $language Language identifier
	 * @return bool True if supported
	 */
	public function is_language_supported( $language ) {
		$supported = $this->highlighter->listLanguages();
		return in_array( $language, $supported, true );
	}

	/**
	 * Resolve language alias to standard name
	 *
	 * @param string $language Language identifier or alias
	 * @return string Resolved language name
	 */
	private function resolve_language_alias( $language ) {
		$aliases = array(
			'js'   => 'javascript',
			'ts'   => 'typescript',
			'py'   => 'python',
			'rb'   => 'ruby',
			'yml'  => 'yaml',
			'sh'   => 'bash',
			'md'   => 'markdown',
			'html' => 'xml', // highlight.php uses 'xml' for HTML
		);

		return $aliases[ $language ] ?? $language;
	}

	/**
	 * Generate fallback HTML for client-side rendering
	 *
	 * @param string $code Code content
	 * @param string $language Language identifier
	 * @return string Fallback HTML
	 */
	private function generate_fallback( $code, $language ) {
		return sprintf(
			'<pre data-ssr="false"><code class="language-%s">%s</code></pre>',
			esc_attr( $language ),
			esc_html( $code )
		);
	}

	/**
	 * Debug logging
	 *
	 * @param string $message Log message
	 */
	private function log_debug( $message ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( '[GFMR_Code_Highlighter] ' . $message );
		}
	}
}
