<?php
/**
 * WP-CLI commands for GFMR plugin
 *
 * Note: Since Shiki is a JavaScript library, PHP cannot regenerate highlighted HTML.
 * These commands only update block attributes and clear caches.
 * Actual HTML regeneration happens on the frontend when the page is next viewed.
 */

namespace Wakalab\WpGfmRenderer;

defined( 'ABSPATH' ) || exit;

if ( ! defined( 'WP_CLI' ) || ! WP_CLI ) {
	return;
}

// Prevent class redeclaration when both Free and Pro versions are active
if ( class_exists( __NAMESPACE__ . '\\GFMR_CLI' ) ) {
	return;
}

/**
 * Manage GFMR Markdown blocks and caches.
 */
class GFMR_CLI extends \WP_CLI_Command {

	/**
	 * Update shikiTheme attribute on existing Markdown blocks.
	 *
	 * Since Shiki is a JavaScript library, this command cannot regenerate
	 * highlighted HTML. It only updates the shikiTheme attribute so that
	 * the frontend can properly compare and skip unnecessary re-rendering.
	 *
	 * ## OPTIONS
	 *
	 * [--theme=<theme>]
	 * : The Shiki theme to set. Default: github-dark
	 * ---
	 * default: github-dark
	 * options:
	 *   - github-dark
	 *   - github-light
	 *   - auto
	 * ---
	 *
	 * [--post-type=<type>]
	 * : Post type to process. Default: post,page
	 *
	 * [--dry-run]
	 * : Preview changes without saving.
	 *
	 * [--limit=<number>]
	 * : Limit the number of posts to process.
	 *
	 * ## EXAMPLES
	 *
	 *     # Update all posts with github-dark theme
	 *     wp gfmr update-theme --theme=github-dark
	 *
	 *     # Preview changes without saving
	 *     wp gfmr update-theme --theme=github-light --dry-run
	 *
	 *     # Process only pages
	 *     wp gfmr update-theme --post-type=page
	 *
	 * @param array $args       Positional arguments.
	 * @param array $assoc_args Associative arguments.
	 */
	public function update_theme( $args, $assoc_args ) {
		$theme     = $assoc_args['theme'] ?? 'github-dark';
		$post_type = $assoc_args['post-type'] ?? 'post,page';
		$dry_run   = isset( $assoc_args['dry-run'] );
		$limit     = isset( $assoc_args['limit'] ) ? absint( $assoc_args['limit'] ) : -1;

		$post_types = array_map( 'trim', explode( ',', $post_type ) );

		\WP_CLI::log( sprintf( 'Updating shikiTheme attribute to "%s"...', $theme ) );
		if ( $dry_run ) {
			\WP_CLI::log( '(Dry run - no changes will be saved)' );
		}

		$query_args = array(
			'post_type'      => $post_types,
			'post_status'    => 'any',
			'posts_per_page' => $limit,
			's'              => 'wp:gfm-renderer/markdown',
		);

		$query = new \WP_Query( $query_args );
		$posts = $query->posts;

		if ( empty( $posts ) ) {
			\WP_CLI::success( 'No posts with Markdown blocks found.' );
			return;
		}

		\WP_CLI::log( sprintf( 'Found %d posts with potential Markdown blocks.', count( $posts ) ) );

		$updated_count = 0;
		$block_count   = 0;

		foreach ( $posts as $post ) {
			$blocks = parse_blocks( $post->post_content );
			$result = $this->update_blocks_theme( $blocks, $theme );

			if ( $result['updated'] ) {
				$block_count += $result['count'];

				if ( ! $dry_run ) {
					$new_content = serialize_blocks( $result['blocks'] );
					wp_update_post(
						array(
							'ID'           => $post->ID,
							'post_content' => $new_content,
						)
					);
				}

				\WP_CLI::log(
					sprintf(
						'  [%s] %s - %d block(s) updated',
						$dry_run ? 'DRY' : 'OK',
						$post->post_title,
						$result['count']
					)
				);
				++$updated_count;
			}
		}

		if ( $dry_run ) {
			\WP_CLI::success(
				sprintf(
					'Dry run complete. Would update %d block(s) in %d post(s).',
					$block_count,
					$updated_count
				)
			);
		} else {
			\WP_CLI::success(
				sprintf(
					'Updated %d block(s) in %d post(s).',
					$block_count,
					$updated_count
				)
			);
			\WP_CLI::log( 'Note: HTML will be regenerated when each page is next viewed.' );
		}
	}

	/**
	 * Recursively update shikiTheme in blocks.
	 *
	 * @param array  $blocks Array of blocks.
	 * @param string $theme  Theme to set.
	 * @return array Result with 'updated', 'count', and 'blocks'.
	 */
	private function update_blocks_theme( $blocks, $theme ) {
		$updated = false;
		$count   = 0;

		foreach ( $blocks as &$block ) {
			// Check for our Markdown block
			if ( 'gfm-renderer/markdown' === $block['blockName'] ) {
				$current_theme = $block['attrs']['shikiTheme'] ?? '';
				if ( $current_theme !== $theme ) {
					$block['attrs']['shikiTheme'] = $theme;
					$updated                      = true;
					++$count;
				}
			}

			// Process inner blocks recursively
			if ( ! empty( $block['innerBlocks'] ) ) {
				$inner_result         = $this->update_blocks_theme( $block['innerBlocks'], $theme );
				$block['innerBlocks'] = $inner_result['blocks'];
				$updated              = $updated || $inner_result['updated'];
				$count               += $inner_result['count'];
			}
		}

		return array(
			'updated' => $updated,
			'count'   => $count,
			'blocks'  => $blocks,
		);
	}

	/**
	 * Clear GFMR transient caches.
	 *
	 * ## OPTIONS
	 *
	 * [--all]
	 * : Clear all GFMR related caches.
	 *
	 * ## EXAMPLES
	 *
	 *     # Clear all GFMR caches
	 *     wp gfmr clear-cache --all
	 *
	 * @param array $args       Positional arguments.
	 * @param array $assoc_args Associative arguments.
	 */
	public function clear_cache( $args, $assoc_args ) {
		global $wpdb;

		\WP_CLI::log( 'Clearing GFMR caches...' );

		// Delete transients with gfmr_ prefix
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Required for transient cleanup
		$count = $wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
				'_transient_gfmr_%',
				'_transient_timeout_gfmr_%'
			)
		);

		// Also clear any wp_gfm_ prefixed transients (legacy)
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Required for legacy transient cleanup
		$count += $wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
				'_transient_wp_gfm_%',
				'_transient_timeout_wp_gfm_%'
			)
		);

		\WP_CLI::success( sprintf( 'Cleared %d cache entries.', $count / 2 ) );
	}

	/**
	 * Show statistics about Markdown blocks in the site.
	 *
	 * ## EXAMPLES
	 *
	 *     # Show block statistics
	 *     wp gfmr stats
	 *
	 * @param array $args       Positional arguments.
	 * @param array $assoc_args Associative arguments.
	 */
	public function stats( $args, $assoc_args ) {
		$query = new \WP_Query(
			array(
				'post_type'      => array( 'post', 'page' ),
				'post_status'    => 'any',
				'posts_per_page' => -1,
				's'              => 'wp:gfm-renderer/markdown',
			)
		);

		$posts = $query->posts;

		$total_blocks = 0;
		$theme_counts = array();

		foreach ( $posts as $post ) {
			$blocks = parse_blocks( $post->post_content );
			$this->count_blocks_stats( $blocks, $total_blocks, $theme_counts );
		}

		\WP_CLI::log( '=== GFMR Block Statistics ===' );
		\WP_CLI::log( sprintf( 'Posts with Markdown blocks: %d', count( $posts ) ) );
		\WP_CLI::log( sprintf( 'Total Markdown blocks: %d', $total_blocks ) );
		\WP_CLI::log( '' );
		\WP_CLI::log( 'Blocks by shikiTheme:' );

		if ( empty( $theme_counts ) ) {
			\WP_CLI::log( '  (no blocks found)' );
		} else {
			foreach ( $theme_counts as $theme => $count ) {
				$theme_label = empty( $theme ) ? '(not set)' : $theme;
				\WP_CLI::log( sprintf( '  %s: %d', $theme_label, $count ) );
			}
		}
	}

	/**
	 * Recursively count block statistics.
	 *
	 * @param array $blocks       Array of blocks.
	 * @param int   $total_blocks Total block count (by reference).
	 * @param array $theme_counts Theme counts (by reference).
	 */
	private function count_blocks_stats( $blocks, &$total_blocks, &$theme_counts ) {
		foreach ( $blocks as $block ) {
			if ( 'gfm-renderer/markdown' === $block['blockName'] ) {
				++$total_blocks;
				$theme = $block['attrs']['shikiTheme'] ?? '';
				if ( ! isset( $theme_counts[ $theme ] ) ) {
					$theme_counts[ $theme ] = 0;
				}
				++$theme_counts[ $theme ];
			}

			if ( ! empty( $block['innerBlocks'] ) ) {
				$this->count_blocks_stats( $block['innerBlocks'], $total_blocks, $theme_counts );
			}
		}
	}

	/**
	 * Debug block registration status.
	 *
	 * Displays comprehensive information about block and script registration
	 * to help diagnose registration failures.
	 *
	 * ## EXAMPLES
	 *
	 *     # Show block registration debug information
	 *     wp gfmr debug-blocks
	 *
	 * @subcommand debug-blocks
	 * @param array $args       Positional arguments.
	 * @param array $assoc_args Associative arguments.
	 */
	public function debug_blocks( $args, $assoc_args ) {
		\WP_CLI::log( '=== GFMR Block Registration Debug ===' );

		// Check if block is registered
		$registry = \WP_Block_Type_Registry::get_instance();
		$block    = $registry->get_registered( 'gfm-renderer/markdown' );

		if ( $block ) {
			\WP_CLI::success( 'Block "gfm-renderer/markdown" is registered' );
			\WP_CLI::log( 'Editor script: ' . ( $block->editor_script ? $block->editor_script : 'N/A' ) );
			\WP_CLI::log( 'Editor style: ' . ( $block->editor_style ? print_r( $block->editor_style, true ) : 'N/A' ) );
		} else {
			\WP_CLI::error( 'Block "gfm-renderer/markdown" is NOT registered' );
		}

		// Check script registration
		global $wp_scripts;
		$script_registered = isset( $wp_scripts->registered['gfmr-renderer-editor'] );
		\WP_CLI::log( 'Script "gfmr-renderer-editor" registered: ' . ( $script_registered ? 'YES' : 'NO' ) );

		if ( $script_registered ) {
			$script = $wp_scripts->registered['gfmr-renderer-editor'];
			\WP_CLI::log( '  Source: ' . $script->src );
			\WP_CLI::log( '  Version: ' . $script->ver );
			\WP_CLI::log( '  Dependencies: ' . implode( ', ', $script->deps ) );
		}

		// Check debug log
		$debug_log = get_transient( 'gfmr_debug_log' );
		if ( $debug_log ) {
			\WP_CLI::log( '' );
			\WP_CLI::log( 'Debug log entries: ' . count( $debug_log ) );
			foreach ( $debug_log as $entry ) {
				\WP_CLI::log( '  - ' . $entry['time'] . ': ' . $entry['action'] . ' (hook: ' . $entry['hook'] . ')' );
			}
		} else {
			\WP_CLI::warning( 'No debug log found (register_blocks may not have been called)' );
		}

		// Check block.json file
		$block_json_path = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'blocks/markdown/block.json';
		\WP_CLI::log( '' );
		\WP_CLI::log( 'Block JSON file: ' . ( file_exists( $block_json_path ) ? 'EXISTS' : 'NOT FOUND' ) );

		// Check build files
		$build_files = array(
			'build/index.js'        => plugin_dir_path( GFMR_PLUGIN_FILE ) . 'build/index.js',
			'build/index.asset.php' => plugin_dir_path( GFMR_PLUGIN_FILE ) . 'build/index.asset.php',
		);

		\WP_CLI::log( '' );
		\WP_CLI::log( 'Build files:' );
		foreach ( $build_files as $name => $path ) {
			$exists = file_exists( $path );
			$size   = $exists ? filesize( $path ) : 0;
			\WP_CLI::log( sprintf( '  %s: %s (%s)', $name, $exists ? 'EXISTS' : 'NOT FOUND', $exists ? human_filesize( $size ) : 'N/A' ) );
		}
	}
}

/**
 * Helper function to format file size
 *
 * @param int $bytes File size in bytes.
 * @return string Formatted file size.
 */
function human_filesize( $bytes ) {
	$units  = array( 'B', 'KB', 'MB', 'GB' );
	$bytes  = max( $bytes, 0 );
	$pow    = floor( ( $bytes ? log( $bytes ) : 0 ) / log( 1024 ) );
	$pow    = min( $pow, count( $units ) - 1 );
	$bytes /= pow( 1024, $pow );
	return round( $bytes, 2 ) . ' ' . $units[ $pow ];
}

\WP_CLI::add_command( 'gfmr', __NAMESPACE__ . '\\GFMR_CLI' );
