<?php
/**
 * GFMR Chart Handler
 *
 * Chart.js integration for data visualization
 * MD-27 Implementation: Advanced data visualization with Chart.js
 *
 * All chart types are available for free.
 *
 * @package WpGfmRenderer
 * @since 2.1.0
 */

namespace Wakalab\WpGfmRenderer;

defined( 'ABSPATH' ) || exit;

// Prevent class redeclaration
if ( class_exists( __NAMESPACE__ . '\\GFMR_Chart_Handler' ) ) {
	return;
}

class GFMR_Chart_Handler {

	/**
	 * Supported chart types (all free)
	 *
	 * @var array
	 */
	const SUPPORTED_CHART_TYPES = array(
		'line',
		'bar',
		'pie',
		'doughnut',
		'scatter',
		'bubble',
		'radar',
		'polarArea',
	);

	/**
	 * Future chart types (planned)
	 *
	 * @var array
	 */
	const FUTURE_CHART_TYPES = array( 'candlestick', 'ohlc' );

	/**
	 * Singleton instance
	 *
	 * @var GFMR_Chart_Handler|null
	 */
	private static $instance = null;

	/**
	 * Get singleton instance
	 *
	 * @return GFMR_Chart_Handler
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->setup_hooks();
	}

	/**
	 * Setup WordPress hooks
	 *
	 * @return void
	 */
	private function setup_hooks() {
		// Server-side gate: Pre-process content before rendering
		add_filter( 'the_content', array( $this, 'process_chart_blocks' ), 8 );
		add_filter( 'gfmr_render_content', array( $this, 'process_chart_blocks' ), 8 );
	}

	/**
	 * Check if chart type is supported
	 *
	 * @param string $type Chart type
	 * @return bool
	 */
	public function is_supported_type( $type ) {
		return in_array( $type, self::SUPPORTED_CHART_TYPES, true );
	}

	/**
	 * Process chart blocks
	 *
	 * Note: All chart types are now free, so no gating is applied.
	 * This method is kept for potential future processing needs.
	 *
	 * @param string $content Post content
	 * @return string Processed content
	 */
	public function process_chart_blocks( $content ) {
		// All chart types are free, no processing needed
		return $content;
	}

	/**
	 * Check if chart type is allowed
	 *
	 * All supported chart types are allowed (no Pro/Free distinction).
	 *
	 * @param string $type Chart type
	 * @param bool   $is_pro_fence Whether using chart-pro fence (ignored, kept for API compatibility)
	 * @return bool
	 */
	public function is_chart_type_allowed( $type, $is_pro_fence = false ) {
		// All supported chart types are allowed
		return $this->is_supported_type( $type );
	}

	/**
	 * Validate mixed chart configuration
	 *
	 * Checks if dataset types are supported.
	 *
	 * @param array $config Chart configuration
	 * @param bool  $is_pro_fence Whether using chart-pro fence (ignored, kept for API compatibility)
	 * @return array Array of error messages (empty if valid)
	 */
	public function validate_mixed_chart_config( $config, $is_pro_fence = false ) {
		$errors = array();

		if ( ! isset( $config['data']['datasets'] ) ) {
			return $errors;
		}

		foreach ( $config['data']['datasets'] as $index => $dataset ) {
			if ( isset( $dataset['type'] ) ) {
				$dataset_type = $dataset['type'];

				// Check if this dataset type is supported
				if ( ! $this->is_supported_type( $dataset_type ) ) {
					$errors[] = sprintf(
						'Dataset %d uses unsupported chart type "%s"',
						$index,
						$dataset_type
					);
				}
			}
		}

		return $errors;
	}

	/**
	 * Get chart type categories for JavaScript
	 *
	 * @return array
	 */
	public function get_chart_categories() {
		return array(
			'supported' => self::SUPPORTED_CHART_TYPES,
			'future'    => self::FUTURE_CHART_TYPES,
		);
	}
}
