<?php
/**
 * Cache management class
 *
 * Class specialized in cache management for temporary data
 *
 * @package WpGfmRenderer
 * @since 1.0.0
 */

namespace Wakalab\WpGfmRenderer;

defined( 'ABSPATH' ) || exit;

// Prevent class redeclaration when both Free and Pro versions are active
if ( class_exists( __NAMESPACE__ . '\\GFMR_Cache_Manager' ) ) {
	return;
}

class GFMR_Cache_Manager {

	/**
	 * Cache key prefix
	 *
	 * @var string
	 */
	private $cache_prefix = 'gfmr_';

	/**
	 * Default expiration time (seconds)
	 *
	 * @var int
	 */
	private $default_expiration = 3600; // 1 hour

	/**
	 * Get cache
	 *
	 * @param string $key Cache key
	 * @return mixed Cached value, or false if not exists
	 */
	public function get_cache( $key ) {
		$cache_key = $this->build_cache_key( $key );

		// Check if WordPress function exists
		if ( function_exists( 'get_transient' ) ) {
			$cached_data = get_transient( $cache_key );
		} else {
			// Always return false in test environment
			$cached_data = false;
		}

		if ( false !== $cached_data ) {
			$this->log_debug( "Cache hit for key: {$key}" );
			return $cached_data;
		}

		$this->log_debug( "Cache miss for key: {$key}" );
		return false;
	}

	/**
	 * Set cache
	 *
	 * @param string $key Cache key
	 * @param mixed $value Value to cache
	 * @param int $expiration Expiration time in seconds
	 * @return bool True on success, false on failure
	 */
	public function set_cache( $key, $value, $expiration = null ) {
		if ( null === $expiration ) {
			$expiration = $this->default_expiration;
		}

		$cache_key = $this->build_cache_key( $key );

		// Check if WordPress function exists
		if ( function_exists( 'set_transient' ) ) {
			$result = set_transient( $cache_key, $value, $expiration );
		} else {
			// Always return true in test environment
			$result = true;
		}

		if ( $result ) {
			$this->log_debug( "Cache set for key: {$key}, expiration: {$expiration}s" );
		} else {
			$this->log_debug( "Failed to set cache for key: {$key}" );
		}

		return $result;
	}

	/**
	 * Delete cache
	 *
	 * @param string $key Cache key
	 * @return bool True on success, false on failure
	 */
	public function delete_cache( $key ) {
		$cache_key = $this->build_cache_key( $key );

		// Check if WordPress function exists
		if ( function_exists( 'delete_transient' ) ) {
			$result = delete_transient( $cache_key );
		} else {
			// Always return true in test environment
			$result = true;
		}

		if ( $result ) {
			$this->log_debug( "Cache deleted for key: {$key}" );
		} else {
			$this->log_debug( "Failed to delete cache for key: {$key} (may not exist)" );
		}

		return $result;
	}

	/**
	 * Clear all cache
	 *
	 * @return bool True on success, false on failure
	 */
	public function clear_all_cache() {
		// Simple implementation for test environment
		if ( ! function_exists( 'delete_transient' ) ) {
			$this->log_debug( 'Test environment: cache clearing simulated' );
			return true;
		}

		global $wpdb;

		try {
			// WordPress environment implementation: Delete all transients for this plugin
			if ( ! isset( $wpdb ) || ! method_exists( $wpdb, 'get_col' ) ) {
				$this->log_debug( 'WordPress database not available' );
				return true;
			}

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Required for transient cleanup, no caching needed
			$transient_keys = $wpdb->get_col(
				$wpdb->prepare(
					"SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
					'_transient_' . $this->cache_prefix . '%',
					'_transient_timeout_' . $this->cache_prefix . '%'
				)
			);

			if ( empty( $transient_keys ) ) {
				$this->log_debug( 'No cache entries found to clear' );
				return true;
			}

			$cleared_count = 0;
			foreach ( $transient_keys as $key ) {
				// Remove '_transient_' or '_transient_timeout_' prefix
				$transient_name = str_replace( array( '_transient_timeout_', '_transient_' ), '', $key );

				if ( delete_transient( $transient_name ) ) {
					++$cleared_count;
				}
			}

			$this->log_debug( "Cleared {$cleared_count} cache entries" );
			return true;

		} catch ( \Exception $e ) {
			$this->handle_error( $e );
			return false;
		}
	}

	/**
	 * Get cache statistics
	 *
	 * @return array Cache statistics information
	 */
	public function get_cache_stats() {
		global $wpdb;

		try {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Required for cache statistics, no caching needed
			$transient_count = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM {$wpdb->options} WHERE option_name LIKE %s",
					'_transient_' . $this->cache_prefix . '%'
				)
			);

			return array(
				'total_entries'      => (int) $transient_count,
				'cache_prefix'       => $this->cache_prefix,
				'default_expiration' => $this->default_expiration,
			);

		} catch ( \Exception $e ) {
			$this->handle_error( $e );
			return array(
				'total_entries'      => 0,
				'cache_prefix'       => $this->cache_prefix,
				'default_expiration' => $this->default_expiration,
				'error'              => $e->getMessage(),
			);
		}
	}

	/**
	 * Generate cache key for Markdown content
	 *
	 * @param string $content Markdown content
	 * @param array $options Rendering options
	 * @return string Cache key
	 */
	public function generate_content_cache_key( $content, $options = array() ) {
		// Generate hash of content and options
		$content_hash = hash( 'md5', $content );
		$options_hash = hash( 'md5', wp_json_encode( $options ) );

		return "content_{$content_hash}_{$options_hash}";
	}

	/**
	 * Generate cache key for code block highlighting.
	 * Includes theme information to ensure cache invalidation on theme change.
	 *
	 * @param string $content Code content
	 * @param string $language Programming language
	 * @param string $theme Shiki theme name (e.g., 'github-dark', 'github-light')
	 * @return string Cache key
	 */
	public function generate_code_cache_key( $content, $language, $theme = '' ) {
		// Normalize inputs
		$content  = trim( $content );
		$language = strtolower( trim( $language ) );
		$theme    = strtolower( trim( $theme ) );

		// Generate hash including theme for cache key uniqueness
		$content_hash = hash( 'md5', $content );

		// Include language and theme in the key for granular cache control
		return "code_{$language}_{$theme}_{$content_hash}";
	}

	/**
	 * Get cached code block HTML.
	 *
	 * @param string $content Original code content
	 * @param string $language Programming language
	 * @param string $theme Shiki theme name
	 * @return string|false Cached HTML or false if not found
	 */
	public function get_code_cache( $content, $language, $theme = '' ) {
		$cache_key = $this->generate_code_cache_key( $content, $language, $theme );
		return $this->get_cache( $cache_key );
	}

	/**
	 * Set cached code block HTML.
	 *
	 * @param string $content Original code content
	 * @param string $language Programming language
	 * @param string $theme Shiki theme name
	 * @param string $html Highlighted HTML to cache
	 * @param int $expiration Cache expiration time in seconds
	 * @return bool True on success
	 */
	public function set_code_cache( $content, $language, $theme, $html, $expiration = null ) {
		// Use longer expiration for code blocks (24 hours by default)
		if ( null === $expiration ) {
			$expiration = 86400; // 24 hours
		}

		$cache_key = $this->generate_code_cache_key( $content, $language, $theme );
		return $this->set_cache( $cache_key, $html, $expiration );
	}

	/**
	 * Clear cache for a specific theme.
	 * Useful when theme settings change.
	 *
	 * @param string $theme Theme name to clear cache for (or 'all' for all themes)
	 * @return int Number of cache entries cleared
	 */
	public function clear_theme_cache( $theme = 'all' ) {
		global $wpdb;

		if ( ! function_exists( 'delete_transient' ) || ! isset( $wpdb ) ) {
			return 0;
		}

		try {
			$pattern = $this->cache_prefix . 'code_';

			// Add theme-specific pattern if not clearing all
			if ( 'all' !== $theme ) {
				$pattern .= '%_' . strtolower( trim( $theme ) ) . '_%';
			} else {
				$pattern .= '%';
			}

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Required for theme-specific cache cleanup
			$transient_keys = $wpdb->get_col(
				$wpdb->prepare(
					"SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE %s",
					'_transient_' . $pattern
				)
			);

			if ( empty( $transient_keys ) ) {
				$this->log_debug( "No code cache entries found for theme: {$theme}" );
				return 0;
			}

			$cleared_count = 0;
			foreach ( $transient_keys as $key ) {
				$transient_name = str_replace( '_transient_', '', $key );
				if ( delete_transient( $transient_name ) ) {
					++$cleared_count;
				}
			}

			$this->log_debug( "Cleared {$cleared_count} code cache entries for theme: {$theme}" );
			return $cleared_count;

		} catch ( \Exception $e ) {
			$this->handle_error( $e );
			return 0;
		}
	}

	/**
	 * Get SSR cache (Server-Side Rendering)
	 *
	 * @param string $content_hash Hash of content
	 * @param string $theme Theme identifier
	 * @return string|false Cached HTML or false if not found
	 */
	public function get_ssr_cache( $content_hash, $theme ) {
		$cache_key = "ssr_{$content_hash}_{$theme}";
		return $this->get_cache( $cache_key );
	}

	/**
	 * Set SSR cache (Server-Side Rendering)
	 *
	 * @param string $content_hash Hash of content
	 * @param string $theme Theme identifier
	 * @param string $html Rendered HTML
	 * @param int $expiration Cache expiration time in seconds (default: 24 hours)
	 * @return bool True on success
	 */
	public function set_ssr_cache( $content_hash, $theme, $html, $expiration = 86400 ) {
		$cache_key = "ssr_{$content_hash}_{$theme}";
		return $this->set_cache( $cache_key, $html, $expiration );
	}

	/**
	 * Get Mermaid SVG cache
	 *
	 * @param string $diagram_id Diagram identifier
	 * @param string $theme Theme identifier
	 * @param string $bg_color Background color
	 * @return string|false Cached SVG or false if not found
	 */
	public function get_mermaid_svg_cache( $diagram_id, $theme, $bg_color ) {
		$cache_key = "mermaid_{$diagram_id}_{$theme}_{$bg_color}";
		return $this->get_cache( $cache_key );
	}

	/**
	 * Set Mermaid SVG cache
	 *
	 * @param string $diagram_id Diagram identifier
	 * @param string $theme Theme identifier
	 * @param string $bg_color Background color
	 * @param string $svg SVG content
	 * @param int $expiration Cache expiration time in seconds (default: 7 days)
	 * @return bool True on success
	 */
	public function set_mermaid_svg_cache( $diagram_id, $theme, $bg_color, $svg, $expiration = 604800 ) {
		$cache_key = "mermaid_{$diagram_id}_{$theme}_{$bg_color}";
		return $this->set_cache( $cache_key, $svg, $expiration );
	}

	/**
	 * Build cache key
	 *
	 * @param string $key Original key
	 * @return string Key with prefix
	 */
	private function build_cache_key( $key ) {
		return $this->cache_prefix . $key;
	}

	/**
	 * Build cache key with optional language suffix
	 *
	 * @param string $key Base cache key
	 * @param string $lang Language code (optional)
	 * @return string Full cache key
	 */
	public function build_cache_key_with_lang( $key, $lang = '' ) {
		$lang_suffix = $lang ? "_{$lang}" : '';
		return $this->cache_prefix . $key . $lang_suffix;
	}

	/**
	 * Output debug log
	 *
	 * @param string $message Log message
	 * @return void
	 */
	private function log_debug( $message ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging when WP_DEBUG_LOG is enabled
			error_log( '[GFMR_Cache_Manager] ' . $message );
		}
	}

	/**
	 * Error handling
	 *
	 * @param \Exception $exception Exception object
	 * @return void
	 */
	private function handle_error( $exception ) {
		$message = 'Cache Manager Error: ' . $exception->getMessage();
		$this->log_debug( $message );

		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging when WP_DEBUG_LOG is enabled
			error_log( $message );
		}
	}
}
