<?php
/** Block registration management class */

namespace Wakalab\WpGfmRenderer;

defined( 'ABSPATH' ) || exit;

// Prevent class redeclaration when both Free and Pro versions are active
if ( class_exists( __NAMESPACE__ . '\\GFMR_Block_Registry' ) ) {
	return;
}

class GFMR_Block_Registry {

	/**
	 * SSR renderer instance
	 *
	 * @var GFMR_SSR_Renderer|null
	 */
	private $ssr_renderer = null;

	/**
	 * Settings instance
	 *
	 * @var GFMR_Settings|null
	 */
	private $settings = null;

	/**
	 * Constructor
	 *
	 * @param GFMR_SSR_Renderer|null $ssr_renderer SSR renderer instance (optional).
	 * @param GFMR_Settings|null     $settings Settings instance (optional).
	 */
	public function __construct( $ssr_renderer = null, $settings = null ) {
		$this->ssr_renderer = $ssr_renderer;
		$this->settings     = $settings;
	}

	/** Register blocks */
	public function register_blocks() {
		// Allow script tags with type="application/json" in the_content filter
		add_filter( 'wp_kses_allowed_html', array( $this, 'allow_json_script_tags' ), 10, 2 );

		try {
			// Block registration based on block.json (MD-34: Fixed with manual editorScript registration)
			$block_json_path = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'blocks/markdown/block.json';

			if ( function_exists( 'current_time' ) ) {
				$log_file = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'debug.log';
				// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
				file_put_contents(
					$log_file,
					'[' . current_time( 'mysql' ) . '] Checking block.json: ' . $block_json_path . "\n",
					FILE_APPEND
				);
			}

			if ( ! file_exists( $block_json_path ) ) {
				$this->log_debug( 'Block JSON file not found: ' . $block_json_path );
				error_log( 'GFMR Block Registry: Block JSON file not found: ' . $block_json_path );
				if ( function_exists( 'current_time' ) ) {
					$log_file = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'debug.log';
					// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
					file_put_contents(
						$log_file,
						'[' . current_time( 'mysql' ) . '] ERROR: block.json not found!' . "\n",
						FILE_APPEND
					);
				}
				return;
			}

			if ( function_exists( 'current_time' ) ) {
				$log_file = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'debug.log';
				// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
				file_put_contents(
					$log_file,
					'[' . current_time( 'mysql' ) . '] block.json found, registering styles...' . "\n",
					FILE_APPEND
				);
			}

			error_log( 'GFMR Block Registry: Starting block registration' );
			error_log( 'GFMR Block Registry: Block JSON path: ' . $block_json_path );

			// Register required styles before block registration (block.json references these handles)
			$this->register_block_styles();

			if ( function_exists( 'current_time' ) ) {
				$log_file = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'debug.log';
				// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
				file_put_contents(
					$log_file,
					'[' . current_time( 'mysql' ) . '] Styles registered, registering scripts...' . "\n",
					FILE_APPEND
				);
			}

			// MD-34 fix: Explicitly register editor script
			$this->register_block_editor_scripts();

			if ( function_exists( 'current_time' ) ) {
				$log_file = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'debug.log';
				// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
				file_put_contents(
					$log_file,
					'[' . current_time( 'mysql' ) . '] Scripts registered, calling register_block_type()...' . "\n",
					FILE_APPEND
				);
			}

			error_log( 'GFMR Block Registry: Scripts and styles registered' );

			// Load block.json to allow attribute customization
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents -- Reading block.json configuration
			$block_json_content = file_get_contents( $block_json_path );
			$block_config       = json_decode( $block_json_content, true );

			// Get default attributes from block.json
			$attributes = $block_config['attributes'] ?? array();

			/**
			 * Filters block attributes
			 *
			 * Allows addons to add custom attributes to the Markdown block.
			 *
			 * @since 2.0.0
			 *
			 * @param array $attributes Block attribute definitions
			 */
			$attributes = apply_filters( 'gfmr_block_attributes', $attributes );

			// WordPress standard block.json registration
			// block.json now uses file: path for editorScript, so WordPress handles registration automatically
			$result = register_block_type(
				$block_json_path,
				array(
					'render_callback' => array( $this, 'render_markdown_block' ),
					'attributes'      => $attributes,
				)
			);

			if ( function_exists( 'current_time' ) ) {
				$log_file = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'debug.log';
				// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
				file_put_contents(
					$log_file,
					'[' . current_time( 'mysql' ) . '] register_block_type() returned: ' . ( $result ? 'SUCCESS' : 'FALSE/NULL' ) . "\n",
					FILE_APPEND
				);

				if ( $result ) {
					// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
					file_put_contents(
						$log_file,
						'[' . current_time( 'mysql' ) . '] Block name: ' . $result->name . ', title: ' . $result->title . "\n",
						FILE_APPEND
					);
				} else {
					// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
					file_put_contents(
						$log_file,
						'[' . current_time( 'mysql' ) . '] ERROR: register_block_type() failed!' . "\n",
						FILE_APPEND
					);
				}
			}

			if ( $result ) {
				error_log( 'GFMR Block Registry: Block registered successfully - ' . print_r( $result, true ) );
			} else {
				error_log( 'GFMR Block Registry: Block registration returned false/null' );
			}

			$this->log_debug( 'Markdown block registered successfully via block.json with explicit editor script' );

		} catch ( \Exception $e ) {
			error_log( 'GFMR Block Registry ERROR: ' . $e->getMessage() );
			error_log( 'GFMR Block Registry ERROR Stack: ' . $e->getTraceAsString() );
			if ( function_exists( 'current_time' ) ) {
				$log_file = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'debug.log';
				// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
				file_put_contents(
					$log_file,
					'[' . current_time( 'mysql' ) . '] EXCEPTION: ' . $e->getMessage() . "\n" .
					'Stack: ' . $e->getTraceAsString() . "\n",
					FILE_APPEND
				);
			}
			$this->handle_error( $e );
			$this->log_debug( 'Block registration failed: ' . $e->getMessage() );
		}
	}

	/** Register styles required by block.json */
	private function register_block_styles() {
		// Skip if WordPress functions are not available (unit test environment)
		if ( ! function_exists( 'wp_style_is' ) ) {
			return;
		}

		$plugin_url = plugin_dir_url( GFMR_PLUGIN_FILE );
		$version    = $this->get_version();

		// Register gfmr-variables (foundation for other styles)
		if ( ! wp_style_is( 'gfmr-variables', 'registered' ) ) {
			wp_register_style(
				'gfmr-variables',
				$plugin_url . 'assets/css/gfmr-variables.css',
				array(),
				$version
			);
		}

		// Register gfmr-theme-variables (theme-specific color variables)
		if ( ! wp_style_is( 'gfmr-theme-variables', 'registered' ) ) {
			wp_register_style(
				'gfmr-theme-variables',
				$plugin_url . 'assets/css/gfmr-theme-variables.css',
				array( 'gfmr-variables' ),
				$version
			);
		}

		// Register gfmr-table-styles
		if ( ! wp_style_is( 'gfmr-table-styles', 'registered' ) ) {
			wp_register_style(
				'gfmr-table-styles',
				$plugin_url . 'assets/css/gfmr-table-styles.css',
				array( 'gfmr-variables' ),
				$version
			);
		}

		// Register gfmr-code-blocks
		if ( ! wp_style_is( 'gfmr-code-blocks', 'registered' ) ) {
			wp_register_style(
				'gfmr-code-blocks',
				$plugin_url . 'assets/css/gfmr-code-blocks.css',
				array( 'gfmr-variables' ),
				$version
			);
		}

		// Register gfmr-mermaid-styles
		if ( ! wp_style_is( 'gfmr-mermaid-styles', 'registered' ) ) {
			wp_register_style(
				'gfmr-mermaid-styles',
				$plugin_url . 'assets/css/mermaid-styles.css',
				array( 'gfmr-variables' ),
				$version
			);
		}

		// Register gfmr-frontmatter (depends on variables for CSS custom properties)
		if ( ! wp_style_is( 'gfmr-frontmatter', 'registered' ) ) {
			wp_register_style(
				'gfmr-frontmatter',
				$plugin_url . 'assets/css/gfmr-frontmatter.css',
				array( 'gfmr-variables' ),
				$version
			);
		}

		// Register gfmr-multilingual-editor (multilingual language tab switcher CSS)
		if ( ! wp_style_is( 'gfmr-multilingual-editor', 'registered' ) ) {
			wp_register_style(
				'gfmr-multilingual-editor',
				$plugin_url . 'assets/css/gfmr-multilingual.css',
				array(),
				$version
			);
		}

		// Register gfmr-block-style (block-specific style.css)
		if ( ! wp_style_is( 'gfmr-block-style', 'registered' ) ) {
			wp_register_style(
				'gfmr-block-style',
				$plugin_url . 'blocks/markdown/style.css',
				array( 'gfmr-variables', 'gfmr-table-styles' ),
				$version
			);
		}

		$this->log_debug( 'Block styles registered for block.json' );
	}

	/** Register scripts for the block editor */
	public function register_block_editor_scripts() {
		error_log( 'GFMR Block Registry: register_block_editor_scripts() called' );

		$plugin_url = plugin_dir_url( GFMR_PLUGIN_FILE );
		$version    = $this->get_version();

		// Path to editor JS files
		$editor_script_path = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'build/index.js';
		$editor_asset_path  = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'build/index.asset.php';

		error_log( 'GFMR Block Registry: Editor script path: ' . $editor_script_path );
		error_log( 'GFMR Block Registry: Editor script exists: ' . ( file_exists( $editor_script_path ) ? 'YES' : 'NO' ) );

		if ( ! file_exists( $editor_script_path ) ) {
			$this->log_debug( 'Editor script not found: ' . $editor_script_path );
			error_log( 'GFMR Block Registry ERROR: Editor script not found!' );
			return;
		}

		// Load asset information
		$asset_data = array(
			'dependencies' => array(),
			'version'      => $version,
		);
		if ( file_exists( $editor_asset_path ) ) {
			$asset_data = include $editor_asset_path;
		}

		// Editor script dependencies (basic dependencies only)
		$editor_dependencies = $asset_data['dependencies'];

		error_log( 'GFMR Block Registry: Registering script with handle: gfmr-renderer-editor' );
		error_log( 'GFMR Block Registry: Script URL: ' . $plugin_url . 'build/index.js' );
		error_log( 'GFMR Block Registry: Dependencies: ' . print_r( $editor_dependencies, true ) );

		// Register editor script with new prefix (WordPress.org requirements compliance)
		wp_register_script(
			'gfmr-renderer-editor',
			$plugin_url . 'build/index.js',
			$editor_dependencies,
			$asset_data['version'],
			false // Load editor script in head
		);

		// File-based logging
		if ( function_exists( 'current_time' ) && function_exists( 'wp_script_is' ) ) {
			$log_file = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'debug.log';
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
			file_put_contents(
				$log_file,
				'[' . current_time( 'mysql' ) . '] wp_register_script() called for handle: gfmr-renderer-editor' . "\n" .
				'  Script URL: ' . $plugin_url . 'build/index.js' . "\n" .
				'  Dependencies: ' . print_r( $editor_dependencies, true ) . "\n" .
				'  Is registered: ' . ( wp_script_is( 'gfmr-renderer-editor', 'registered' ) ? 'YES' : 'NO' ) . "\n",
				FILE_APPEND
			);
		}

		error_log( 'GFMR Block Registry: wp_register_script called successfully' );

		// Backward compatibility: Register the same script with legacy prefix name
		wp_register_script(
			'gfmr-renderer-editor-legacy',
			$plugin_url . 'build/index.js',
			$editor_dependencies,
			$asset_data['version'],
			false
		);

		// Register and enqueue CSS for blocks
		if ( file_exists( plugin_dir_path( GFMR_PLUGIN_FILE ) . 'blocks/markdown/editor.css' ) ) {
			wp_register_style(
				'gfmr-renderer-editor-css',
				$plugin_url . 'blocks/markdown/editor.css',
				array(
					'gfmr-variables',
					'gfmr-theme-variables',
					'gfmr-table-styles',
					'gfmr-code-blocks',
					'gfmr-mermaid-styles',
				),
				$version
			);
		}

		// Note: enqueue_block_editor_assets hook is registered in GFMR_Renderer->init_hooks()
		// to ensure it's registered before the hook fires

		$this->log_debug( 'Block editor scripts registered successfully' );
	}

	/** Enqueue block editor assets */
	public function enqueue_editor_assets() {
		error_log( 'GFMR Block Registry: enqueue_editor_assets() called!' );

		// File-based logging (only runs in WordPress environment)
		if ( function_exists( 'current_time' ) ) {
			$log_file = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'debug.log';
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
			file_put_contents(
				$log_file,
				'[' . current_time( 'mysql' ) . '] ✅ enqueue_editor_assets() called!' . "\n",
				FILE_APPEND
			);
		}

		$plugin_url = plugin_dir_url( GFMR_PLUGIN_FILE );
		$version    = $this->get_version();

		if ( function_exists( 'current_time' ) ) {
			$log_file = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'debug.log';
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
			file_put_contents(
				$log_file,
				'[' . current_time( 'mysql' ) . '] 📦 Plugin URL and version obtained' . "\n",
				FILE_APPEND
			);
		}

		// Get settings for theme configuration
		$settings = GFMR_Settings::get_instance();

		if ( function_exists( 'current_time' ) ) {
			$log_file = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'debug.log';
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
			file_put_contents(
				$log_file,
				'[' . current_time( 'mysql' ) . '] ⚙️ Settings object created' . "\n",
				FILE_APPEND
			);
		}

		// Check if script is registered
		if ( function_exists( 'wp_script_is' ) ) {
			$is_registered = wp_script_is( 'gfmr-renderer-editor', 'registered' );
			if ( function_exists( 'current_time' ) ) {
				$log_file = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'debug.log';
				// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
				file_put_contents(
					$log_file,
					'[' . current_time( 'mysql' ) . '] 🔍 Script registered: ' . ( $is_registered ? 'YES' : 'NO' ) . "\n",
					FILE_APPEND
				);
			}
		}

		error_log( 'GFMR Block Registry: About to enqueue gfmr-renderer-editor' );

		// Enqueue editor script (using new prefix)
		wp_enqueue_script( 'gfmr-renderer-editor' );

		if ( function_exists( 'current_time' ) ) {
			$log_file = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'debug.log';
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
			file_put_contents(
				$log_file,
				'[' . current_time( 'mysql' ) . '] 📤 wp_enqueue_script() called for gfmr-renderer-editor' . "\n",
				FILE_APPEND
			);
		}

		// Check if script is enqueued
		if ( function_exists( 'wp_script_is' ) ) {
			$is_enqueued = wp_script_is( 'gfmr-renderer-editor', 'enqueued' );
			if ( function_exists( 'current_time' ) ) {
				$log_file = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'debug.log';
				// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Debug logging only
				file_put_contents(
					$log_file,
					'[' . current_time( 'mysql' ) . '] ✅ Script enqueued: ' . ( $is_enqueued ? 'YES' : 'NO' ) . "\n",
					FILE_APPEND
				);
			}
		}

		error_log( 'GFMR Block Registry: wp_enqueue_script called for gfmr-renderer-editor' );

		// Set script translations for block editor
		// For Pro version: includes/core/ is a symlink to Free version
		// Use realpath() to resolve actual Free version path and reference languages from there
		if ( defined( 'GFMR_PRO_VERSION' ) ) {
			$real_includes_dir = realpath( __DIR__ );
			$languages_path    = dirname( $real_includes_dir ) . '/languages';
		} else {
			$languages_path = plugin_dir_path( GFMR_PLUGIN_FILE ) . 'languages';
		}

		wp_set_script_translations(
			'gfmr-renderer-editor',
			'markdown-renderer-for-github',
			$languages_path
		);

		// Pass block editor settings to JavaScript (using new prefix)
		wp_localize_script(
			'gfmr-renderer-editor',
			'wpGfmConfig',
			array(
				'markdownItUrl' => $plugin_url . '/assets/libs/markdown-it/markdown-it.min.js',
				'pluginUrl'     => $plugin_url,
				'version'       => $version,
				'debug'         => defined( 'WP_DEBUG' ) && WP_DEBUG,
				'scriptDebug'   => true,
				'timestamp'     => time(),
				// Theme settings (consistent with frontend)
				'theme'         => array(
					'current'     => $settings->get_current_theme(),
					'shiki_theme' => $settings->get_shiki_theme(),
					'system_auto' => $settings->is_system_theme_enabled(),
					'available'   => GFMR_Settings::AVAILABLE_THEMES,
				),
				// Frontmatter settings
				'frontmatter'   => array(
					'showHeader' => $settings->get( 'show_frontmatter_header', false ),
				),
				// Feature flags (for Pro version migration in the future)
				'features'      => array(
					'imageInsert' => true, // Currently enabled for all users
				),
			)
		);

		// Add asset URL resolver function for block editor
		wp_add_inline_script(
			'gfmr-renderer-editor',
			'window.wpGfmBuildLocalAssetUrl = function(relativePath) {
				var baseUrl = "' . esc_js( $plugin_url ) . '";
				var path = relativePath.startsWith("/") ? relativePath.substring(1) : relativePath;
				console.log("[WP GFM Debug] Asset URL resolved:", baseUrl + path);
				return baseUrl + path;
			};',
			'before'
		);

		// Debug: Output registration status to console
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			wp_add_inline_script(
				'gfmr-renderer-editor',
				'console.log("[GFMR Debug] Script loaded successfully");
				 console.log("[GFMR Debug] wpGfmConfig:", window.wpGfmConfig);
				 if (window.wp && window.wp.blocks) {
					 var ourBlock = wp.blocks.getBlockType("gfm-renderer/markdown");
					 console.log("[GFMR Debug] Block registered:", !!ourBlock);
					 if (ourBlock) {
						 console.log("[GFMR Debug] Block details:", ourBlock);
					 }
				 }',
				'after'
			);
		}

		// Enqueue editor CSS
		wp_enqueue_style( 'gfmr-renderer-editor-css' );

		$this->log_debug( 'Block editor assets enqueued successfully' );
	}

	/** Render the Markdown block */
	public function render_markdown_block( $attributes ) {
		$content          = $attributes['content'] ?? '';
		$html             = $attributes['html'] ?? '';
		$mermaid_bg_color = $attributes['mermaidBgColor'] ?? 'transparent';
		$shiki_theme      = $attributes['shikiTheme'] ?? '';

		// Multilingual attributes
		$languages               = $attributes['languages'] ?? array();
		$default_language        = $attributes['defaultLanguage'] ?? 'en';
		$available_languages     = $attributes['availableLanguages'] ?? array();
		$show_language_switcher  = $attributes['showLanguageSwitcher'] ?? true;

		// Determine if multilingual mode is active (match save.js logic)
		$is_multilingual = count( $available_languages ) > 1 ||
			( count( $available_languages ) === 1 && ! empty( $languages ) );

		// For multilingual blocks, use the default language content/html
		if ( $is_multilingual && ! empty( $languages ) ) {
			$default_lang_data = $languages[ $default_language ] ?? array();
			$html              = $default_lang_data['html'] ?? $html;
			$content           = $default_lang_data['content'] ?? $content;
		}

		if ( empty( $html ) && ! empty( $content ) ) {
			// Fallback: Server-side Markdown processing
			$html = $this->render_markdown_fallback( $content );
		}

		// Apply SSR if enabled
		if ( $this->ssr_renderer && $this->settings && $this->settings->is_ssr_enabled() ) {
			$html = $this->ssr_renderer->process_html( $html, $shiki_theme, $mermaid_bg_color );
			$this->log_debug( 'SSR processing applied to block content' );
		}

		// Build container class
		$container_class = 'gfmr-markdown-container';
		if ( $is_multilingual ) {
			$container_class .= ' gfmr-multilingual';
		}

		$output  = '<div class="' . esc_attr( $container_class ) . '">';
		$output .= '<div class="gfmr-markdown-source" style="display: none;">' . esc_html( $content ) . '</div>';

		// Build data attributes
		$data_attrs = sprintf(
			'data-mermaid-bg-color="%s" data-shiki-theme="%s"',
			esc_attr( $mermaid_bg_color ),
			esc_attr( $shiki_theme )
		);

		// Generate unique ID for multilingual data
		$multilingual_id     = '';
		$multilingual_script = '';

		if ( $is_multilingual ) {
			$multilingual_id = 'gfmr-ml-' . wp_unique_id();

			// Small attributes stay in data-* (safe for HTML attributes)
			$data_attrs .= sprintf(
				' data-multilingual-id="%s" data-default-language="%s" data-available-languages="%s" data-show-switcher="%s"',
				esc_attr( $multilingual_id ),
				esc_attr( $default_language ),
				esc_attr( wp_json_encode( $available_languages ) ),
				$show_language_switcher ? 'true' : 'false'
			);

			// Large JSON data goes in <script> tag (safe, no escaping issues)
			$multilingual_script = sprintf(
				'<script type="application/json" id="%s">%s</script>',
				esc_attr( $multilingual_id ),
				wp_json_encode( $languages, JSON_UNESCAPED_UNICODE | JSON_HEX_TAG )
			);
		}

		$output .= '<div class="gfmr-markdown-rendered" ' . $data_attrs . '>';
		// Use custom kses for SSR content (allows span and SVG tags)
		$output .= wp_kses( $html, $this->get_ssr_allowed_html() );
		$output .= '</div>';
		$output .= $multilingual_script; // Script tag after container
		$output .= '</div>';

		/**
		 * Filters the block render output
		 *
		 * Allows addons to modify the rendered output of the Markdown block.
		 *
		 * @since 2.0.0
		 *
		 * @param string $output     The rendered HTML output
		 * @param array  $attributes Block attributes
		 * @param string $content    Original Markdown content
		 */
		$output = apply_filters( 'gfmr_block_render_output', $output, $attributes, $content );

		return $output;
	}

	/** Server-side Markdown fallback processing */
	private function render_markdown_fallback( $content ) {
		// Basic Markdown processing (simple implementation)
		$html = $content;

		// Convert line breaks to br tags
		$html = nl2br( esc_html( $html ) );

		// Code block detection and basic processing
		$html = preg_replace( '/```(\w+)?\n(.*?)\n```/s', '<pre><code class="language-$1">$2</code></pre>', $html );

		// Inline code
		$html = preg_replace( '/`([^`]+)`/', '<code>$1</code>', $html );

		$this->log_debug( 'Fallback markdown rendering completed' );

		return $html;
	}

	/** Output debug log */
	private function log_debug( $message ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging when WP_DEBUG_LOG is enabled
			error_log( '[WP_GFM_Block_Registry] ' . $message );
		}
	}

	/**
	 * Get SSR allowed HTML tags for wp_kses
	 *
	 * Extends default allowed HTML to include highlight.php and SVG elements
	 *
	 * @return array Allowed HTML tags and attributes
	 */
	private function get_ssr_allowed_html() {
		return array_merge(
			wp_kses_allowed_html( 'post' ),
			array(
				// highlight.php span elements
				'span'    => array(
					'class' => true,
					'style' => true,
				),
				// SVG elements for Mermaid diagrams
				'svg'     => array(
					'class'       => true,
					'viewbox'     => true,
					'xmlns'       => true,
					'width'       => true,
					'height'      => true,
					'aria-hidden' => true,
					'role'        => true,
					'style'       => true,
				),
				'path'    => array(
					'd'            => true,
					'fill'         => true,
					'stroke'       => true,
					'stroke-width' => true,
					'class'        => true,
				),
				'g'       => array(
					'transform' => true,
					'class'     => true,
					'id'        => true,
				),
				'rect'    => array(
					'x'      => true,
					'y'      => true,
					'width'  => true,
					'height' => true,
					'fill'   => true,
					'stroke' => true,
					'class'  => true,
				),
				'text'    => array(
					'x'           => true,
					'y'           => true,
					'class'       => true,
					'text-anchor' => true,
					'font-size'   => true,
					'font-family' => true,
					'fill'        => true,
				),
				'line'    => array(
					'x1'           => true,
					'y1'           => true,
					'x2'           => true,
					'y2'           => true,
					'stroke'       => true,
					'stroke-width' => true,
					'class'        => true,
				),
				'polygon' => array(
					'points' => true,
					'fill'   => true,
					'stroke' => true,
					'class'  => true,
				),
				'circle'  => array(
					'cx'     => true,
					'cy'     => true,
					'r'      => true,
					'fill'   => true,
					'stroke' => true,
					'class'  => true,
				),
				'ellipse' => array(
					'cx'     => true,
					'cy'     => true,
					'rx'     => true,
					'ry'     => true,
					'fill'   => true,
					'stroke' => true,
					'class'  => true,
				),
			)
		);
	}

	/** Error handling */
	private function handle_error( $exception ) {
		$message = 'Block Registry Error: ' . $exception->getMessage();
		$this->log_debug( $message );

		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging when WP_DEBUG_LOG is enabled
			error_log( $message );
		}
	}

	/** Get plugin version */
	private function get_version() {
		// Get version from plugin main file
		$plugin_data = get_file_data( GFMR_PLUGIN_FILE, array( 'Version' => 'Version' ) );
		return $plugin_data['Version'] ?? '1.0.0';
	}

	/** Get list of registered block types */
	public function get_block_types() {
		return array(
			'gfm-renderer/markdown',
		);
	}

	/**
	 * Allow script tags with type="application/json" in wp_kses
	 *
	 * This filter allows script tags to pass through the_content filter
	 * for multilingual JSON data storage.
	 *
	 * @param array  $tags    Allowed HTML tags.
	 * @param string $context Context name.
	 * @return array Modified allowed tags.
	 */
	public function allow_json_script_tags( $tags, $context ) {
		if ( $context === 'post' ) {
			$tags['script'] = array(
				'type' => true,
				'id'   => true,
			);
		}
		return $tags;
	}
}
