<?php
/**
 * Asset management class
 *
 * Class specialized in managing JavaScript and CSS loading
 *
 * @package WpGfmRenderer
 * @since 1.0.0
 */

namespace Wakalab\WpGfmRenderer;

defined( 'ABSPATH' ) || exit;

// Prevent class redeclaration when both Free and Pro versions are active
if ( class_exists( __NAMESPACE__ . '\\GFMR_Asset_Manager' ) ) {
	return;
}

class GFMR_Asset_Manager {

	/**
	 * Load scripts
	 *
	 * @param string $version Version number
	 * @return void
	 */
	public function enqueue_scripts( $version ) {
		$plugin_url = plugin_dir_url( GFMR_PLUGIN_FILE );

		// Cache busting: use file modification time in development, version in production
		if ( ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) || ( defined( 'WP_DEBUG' ) && WP_DEBUG ) ) {
			$cache_buster = filemtime( plugin_dir_path( GFMR_PLUGIN_FILE ) . 'assets/js/gfmr-main.js' );
		} else {
			$cache_buster = $version;
		}

		// Language detection module (must load before main)
		wp_enqueue_script(
			'gfmr-language-detection',
			$plugin_url . 'assets/js/gfmr-language-detection.js',
			array(),
			$cache_buster,
			true
		);

		// Main functionality: direct operation-based feature provision (highest priority loading)
		wp_enqueue_script(
			'gfmr-main',
			$plugin_url . 'assets/js/gfmr-main.js',
			array( 'gfmr-language-detection' ),
			$cache_buster,
			true
		);

		// Mermaid Lightbox module - conditional loading
		if ( $this->may_have_mermaid_content() ) {
			wp_enqueue_script(
				'gfmr-mermaid-lightbox',
				$plugin_url . 'assets/js/gfmr-mermaid-lightbox.js',
				array( 'gfmr-main' ),
				$cache_buster,
				true
			);
		}

		// Preload critical libraries to reduce FOUC
		$this->preload_libraries( $plugin_url, $version );

		// SSR client-side complement script
		wp_enqueue_script(
			'gfmr-ssr-client',
			$plugin_url . 'assets/js/gfmr-ssr-client.js',
			array(),
			$cache_buster,
			true
		);

		// TOC script - conditional loading
		if ( $this->should_load_toc() ) {
			wp_enqueue_script(
				'gfmr-toc',
				$plugin_url . 'assets/js/gfmr-toc.js',
				array( 'gfmr-main' ),
				$cache_buster,
				true
			);
		}

		// Architecture explanation:
		// Currently using only main file to achieve simple configuration
		// Single file integration performs better than multiple file distribution
		// Future migration to modular configuration is under consideration

		// Localization setup
		$this->setup_localization( $version );

		/**
		 * Fires after frontend assets are enqueued
		 *
		 * Allows addons to enqueue their own frontend scripts and styles.
		 *
		 * @since 2.0.0
		 *
		 * @param string $context Asset context ('frontend')
		 */
		do_action( 'gfmr_enqueue_assets', 'frontend' );
	}

	/**
	 * Load styles
	 *
	 * @param string $version Version number
	 * @return void
	 */
	public function enqueue_styles( $version ) {
		$plugin_url = plugin_dir_url( GFMR_PLUGIN_FILE );

		// CSS variable definitions (foundation for other styles)
		wp_enqueue_style(
			'gfmr-variables',
			$plugin_url . 'assets/css/gfmr-variables.css',
			array(),
			$version
		);

		// Theme variables for dynamic theme switching
		wp_enqueue_style(
			'gfmr-theme-variables',
			$plugin_url . 'assets/css/gfmr-theme-variables.css',
			array( 'gfmr-variables' ),
			$version
		);

		// Table styles
		wp_enqueue_style(
			'gfmr-table-styles',
			$plugin_url . 'assets/css/gfmr-table-styles.css',
			array( 'gfmr-theme-variables' ),
			$version
		);

		// Code block styles
		wp_enqueue_style(
			'gfmr-code-blocks',
			$plugin_url . 'assets/css/gfmr-code-blocks.css',
			array( 'gfmr-theme-variables' ),
			$version
		);

		// highlight.php theme styles (for SSR)
		wp_enqueue_style(
			'gfmr-hljs-themes',
			$plugin_url . 'assets/css/gfmr-hljs-themes.css',
			array( 'gfmr-theme-variables' ),
			$version
		);

		// Frontmatter styles
		wp_enqueue_style(
			'gfmr-frontmatter',
			$plugin_url . 'assets/css/gfmr-frontmatter.css',
			array( 'gfmr-variables' ),
			$version
		);

		// Copy button styles
		wp_enqueue_style(
			'gfmr-copy-button',
			$plugin_url . 'assets/css/gfmr-copy-button.css',
			array( 'gfmr-theme-variables' ),
			$version
		);

		// Mermaid styles
		wp_enqueue_style(
			'gfmr-mermaid-styles',
			$plugin_url . 'assets/css/mermaid-styles.css',
			array( 'gfmr-theme-variables' ),
			$version
		);

		// Mermaid zoom modal styles
		wp_enqueue_style(
			'gfmr-mermaid-zoom',
			$plugin_url . 'assets/css/mermaid-zoom-modal.css',
			array( 'gfmr-mermaid-styles' ),
			$version
		);

		// Chart styles
		wp_enqueue_style(
			'gfmr-charts',
			$plugin_url . 'assets/css/gfmr-charts.css',
			array( 'gfmr-theme-variables' ),
			$version
		);

		// TOC styles - conditional loading
		if ( $this->should_load_toc() ) {
			wp_enqueue_style(
				'gfmr-toc',
				$plugin_url . 'assets/css/gfmr-toc.css',
				array( 'gfmr-variables' ),
				$version
			);
		}
	}

	/**
	 * Setup translation data
	 *
	 * @return array Translation data
	 */
	public function localize_translations() {
		$plugin_url  = plugin_dir_url( GFMR_PLUGIN_FILE );
		$plugin_data = get_file_data( GFMR_PLUGIN_FILE, array( 'Version' => 'Version' ) );
		$version     = $plugin_data['Version'] ?? '1.0.0';

		/**
		 * Filters available Shiki languages
		 *
		 * Allows addons to add custom language support for syntax highlighting.
		 *
		 * @since 2.0.0
		 *
		 * @param array $languages Array of language identifiers
		 */
		$supported_languages = apply_filters( 'gfmr_shiki_languages', $this->get_default_languages() );

		// Chart configuration
		$is_pro            = apply_filters( 'gfmr_is_pro_active', false );
		$is_dev_env        = ( defined( 'GFMR_DEV_MODE' ) && GFMR_DEV_MODE ) || ( defined( 'WP_DEBUG' ) && WP_DEBUG );
		$chart_handler     = class_exists( 'Wakalab\\WpGfmRenderer\\GFMR_Chart_Handler' ) ? \Wakalab\WpGfmRenderer\GFMR_Chart_Handler::get_instance() : null;
		$chart_categories  = $chart_handler ? $chart_handler->get_chart_categories() : array(
			'free'   => array( 'line', 'bar', 'pie', 'doughnut' ),
			'pro'    => array( 'scatter', 'bubble', 'radar', 'polarArea' ),
			'future' => array(),
		);

		return array(
			'translation_data' => array(
				'loading'            => __( 'Loading...', 'markdown-renderer-for-github' ),
				'error'              => __( 'Error occurred', 'markdown-renderer-for-github' ),
				'pluginUrl'          => $plugin_url,
				'version'            => $version,
				'debug'              => defined( 'WP_DEBUG' ) && WP_DEBUG,
				'timestamp'          => time(),
				// AJAX configuration for SSR client
				'ajaxUrl'            => function_exists( 'admin_url' ) ? admin_url( 'admin-ajax.php' ) : '',
				'mermaidSsrNonce'    => ( function_exists( 'is_user_logged_in' ) && is_user_logged_in() ) ? wp_create_nonce( 'gfmr_mermaid_ssr' ) : '',
				// Supported languages for syntax highlighting
				'supportedLanguages' => $supported_languages,
				// UI translations for gfmr-ui.js.
				'translations'       => array(
					'copy'        => __( 'Copy', 'markdown-renderer-for-github' ),
					'copied'      => __( 'Copied!', 'markdown-renderer-for-github' ),
					'copy_code'   => __( 'Copy code', 'markdown-renderer-for-github' ),
					'code_copied' => __( 'Code copied to clipboard', 'markdown-renderer-for-github' ),
					'copy_failed' => __( 'Failed to copy', 'markdown-renderer-for-github' ),
					'loading'     => __( 'Loading...', 'markdown-renderer-for-github' ),
					'error'       => __( 'Error', 'markdown-renderer-for-github' ),
					'retry'       => __( 'Retry', 'markdown-renderer-for-github' ),
					'close'       => __( 'Close', 'markdown-renderer-for-github' ),
					'expand'      => __( 'Expand', 'markdown-renderer-for-github' ),
					'collapse'    => __( 'Collapse', 'markdown-renderer-for-github' ),
				),
				// Chart.js configuration
				'isPro'              => $is_pro,
				'isDevEnvironment'   => $is_dev_env,
				'charts'             => $chart_categories,
			),
		);
	}

	/**
	 * Get default supported languages for Shiki
	 *
	 * Returns a comprehensive list of programming languages supported by default.
	 * Addons can extend this list using the 'gfmr_shiki_languages' filter.
	 *
	 * @since 2.0.0
	 * @return array Array of language identifiers
	 */
	private function get_default_languages() {
		return array(
			'javascript',
			'typescript',
			'jsx',
			'tsx',
			'php',
			'python',
			'ruby',
			'java',
			'c',
			'cpp',
			'csharp',
			'go',
			'rust',
			'swift',
			'kotlin',
			'scala',
			'sql',
			'html',
			'css',
			'scss',
			'sass',
			'less',
			'json',
			'yaml',
			'xml',
			'markdown',
			'bash',
			'shell',
			'powershell',
			'dockerfile',
			'makefile',
			'graphql',
			'vue',
			'svelte',
			'astro',
		);
	}

	/**
	 * Determine if TOC assets should be loaded.
	 *
	 * @return bool
	 */
	private function should_load_toc() {
		if ( is_admin() ) {
			return false;
		}
		$settings = GFMR_Settings::get_instance();
		return (bool) $settings->get( 'toc_enabled', false );
	}

	/**
	 * Determine asset loading conditions
	 *
	 * @return bool Whether to load assets
	 */
	public function should_enqueue_assets() {
		// Emergency fix: load when debug mode or Markdown blocks exist
		$force_enqueue        = defined( 'WP_DEBUG' ) && WP_DEBUG;
		$is_e2e_test          = strpos( get_bloginfo( 'name' ), 'E2E Test' ) !== false;
		$has_markdown         = $this->has_markdown_blocks();
		$may_have_code_blocks = $this->may_have_code_blocks();

		// Optimize asset loading conditions (enhanced code block support)
		return $force_enqueue || $is_e2e_test || $has_markdown || $may_have_code_blocks;
	}

	/**
	 * Check if Markdown blocks exist
	 *
	 * @return bool Existence of Markdown blocks
	 */
	public function has_markdown_blocks() {
		global $post;

		if ( ! $post || ! has_blocks( $post->post_content ) ) {
			return false;
		}

		$blocks = parse_blocks( $post->post_content );
		foreach ( $blocks as $block ) {
			if ( 'gfm-renderer/markdown' === $block['blockName'] ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Check possibility of code blocks existing
	 *
	 * @return bool Possibility of code blocks existing
	 */
	public function may_have_code_blocks() {
		global $post;

		if ( ! $post ) {
			return false;
		}

		// Check if saved content contains code block notation
		$content = $post->post_content;

		// Check Markdown code block notation
		if ( strpos( $content, '```' ) !== false ) {
			return true;
		}

		// Check HTML code and pre tags
		if ( strpos( $content, '<code' ) !== false || strpos( $content, '<pre' ) !== false ) {
			return true;
		}

		// Check Gutenberg code blocks
		if ( strpos( $content, 'wp:code' ) !== false ) {
			return true;
		}

		return false;
	}

	/**
	 * Check if content may contain Mermaid diagrams
	 *
	 * @return bool Possibility of Mermaid diagrams existing
	 */
	public function may_have_mermaid_content() {
		global $post;

		if ( ! $post ) {
			return false;
		}

		$content = $post->post_content;

		// Check Mermaid code block notation
		if ( preg_match( '/```mermaid/i', $content ) ) {
			return true;
		}

		// Check Mermaid class in HTML
		if ( strpos( $content, 'language-mermaid' ) !== false ) {
			return true;
		}

		// Check Mermaid diagram types
		$mermaid_types = array( 'flowchart', 'sequenceDiagram', 'classDiagram', 'stateDiagram', 'erDiagram', 'gantt', 'pie', 'gitGraph', 'journey' );
		foreach ( $mermaid_types as $type ) {
			if ( strpos( $content, $type ) !== false ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Preload critical libraries to reduce FOUC
	 *
	 * @param string $plugin_url Plugin URL
	 * @param string $version Version number
	 * @return void
	 */
	private function preload_libraries( $plugin_url, $version ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter -- Reserved for hook compatibility
		// Add preload links for critical libraries
		add_action(
			'wp_head',
			function () use ( $plugin_url ) {
				// Preload Shiki (smaller library, higher priority)
				echo '<link rel="preload" href="' . esc_url( $plugin_url . 'assets/libs/shiki/shiki.min.js' ) . '" as="script" crossorigin="anonymous">' . "\n";

				// Preload Mermaid (larger library, lower priority)
				echo '<link rel="prefetch" href="' . esc_url( $plugin_url . 'assets/libs/mermaid/mermaid.min.js' ) . '" as="script" crossorigin="anonymous">' . "\n";
			},
			1
		);
	}

	/**
	 * Perform localization setup
	 *
	 * @param string $version Version number
	 * @return void
	 */
	private function setup_localization( $version ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter -- Reserved for hook compatibility
		$translations = $this->localize_translations();

		/**
		 * Filters the frontend configuration data
		 *
		 * Allows addons to extend frontend configuration.
		 *
		 * @since 2.4.0
		 *
		 * @param array $config_data Frontend configuration data
		 */
		$config_data = apply_filters( 'gfmr_frontend_config', $translations['translation_data'] );

		wp_localize_script(
			'gfmr-main',
			'wpGfmConfig',
			$config_data
		);
	}

	/**
	 * Setup localization with theme settings
	 *
	 * @param string        $version Plugin version
	 * @param GFMR_Settings $settings Settings instance
	 */
	public function setup_localization_with_theme_settings( $version, $settings ) {
		$translations = $this->localize_translations();
		$theme_config = $this->get_theme_config( $settings );
		$toc_config   = $this->get_toc_config( $settings );

		$config_data = array_merge(
			$translations['translation_data'],
			$theme_config,
			$toc_config
		);

		/** This filter is documented in includes/class-gfmr-asset-manager.php */
		$config_data = apply_filters( 'gfmr_frontend_config', $config_data );

		wp_localize_script(
			'gfmr-main',
			'wpGfmSettings', // Keep this handle for backward compatibility
			$config_data
		);
	}

	/**
	 * Get theme configuration for JavaScript
	 *
	 * @param GFMR_Settings $settings Settings instance
	 * @return array Theme configuration
	 */
	private function get_theme_config( $settings ) {
		/**
		 * Filters available Shiki themes
		 *
		 * Allows addons to add custom Shiki themes.
		 *
		 * @since 2.0.0
		 *
		 * @param array $themes Available themes array ['light' => 'github-light', ...]
		 */
		$available_themes = apply_filters( 'gfmr_shiki_themes', GFMR_Settings::AVAILABLE_THEMES );

		return array(
			'theme' => array(
				'current'     => $settings->get_current_theme(),
				'shiki_theme' => $settings->get_shiki_theme(),
				'system_auto' => $settings->is_system_theme_enabled(),
				'available'   => $available_themes,
			),
		);
	}

	/**
	 * Get TOC configuration for JavaScript
	 *
	 * @param GFMR_Settings $settings Settings instance
	 * @return array TOC configuration
	 */
	private function get_toc_config( $settings ) {
		return array(
			'toc' => array(
				'enabled'      => (bool) $settings->get( 'toc_enabled' ),
				'headingMin'   => (int) $settings->get( 'toc_heading_min', 2 ),
				'headingMax'   => (int) $settings->get( 'toc_heading_max', 4 ),
				'position'     => (string) $settings->get( 'toc_position', 'right' ),
				'scrollOffset' => (int) $settings->get( 'toc_scroll_offset', 0 ),
				'sticky'       => (bool) $settings->get( 'toc_sticky', true ),
				'mobileHidden' => (bool) $settings->get( 'toc_mobile_hidden', true ),
			),
		);
	}

	/**
	 * Enqueue admin-specific scripts and styles
	 *
	 * @param string $hook_suffix Current admin page hook suffix
	 * @return void
	 */
	public function enqueue_admin_scripts( $hook_suffix ) {
		// Only load on our settings page
		if ( 'settings_page_gfmr-settings' !== $hook_suffix ) {
			return;
		}

		$plugin_url  = plugin_dir_url( GFMR_PLUGIN_FILE );
		$plugin_data = get_file_data( GFMR_PLUGIN_FILE, array( 'Version' => 'Version' ) );
		$version     = $plugin_data['Version'] ?? '1.0.0';

		// Enqueue jQuery (required for settings page)
		wp_enqueue_script( 'jquery' );

		// Enqueue foundation utilities first (required for core scripts)
		wp_enqueue_script(
			'gfmr-utils',
			$plugin_url . 'assets/js/gfmr-utils.js',
			array( 'jquery' ),
			$version,
			true
		);

		// Enqueue core scripts for preview functionality (with proper dependencies)
		wp_enqueue_script(
			'gfmr-core',
			$plugin_url . 'assets/js/gfmr-core.js',
			array( 'jquery', 'gfmr-utils' ),
			$version,
			true
		);

		// Enqueue theme variables CSS
		wp_enqueue_style(
			'gfmr-theme-variables',
			$plugin_url . 'assets/css/gfmr-theme-variables.css',
			array(),
			$version
		);

		// Setup localization for admin preview
		$this->setup_admin_localization( $version );

		/**
		 * Fires after admin assets are enqueued
		 *
		 * Allows addons to enqueue their own admin scripts and styles.
		 *
		 * @since 2.0.0
		 *
		 * @param string $context Asset context ('admin')
		 */
		do_action( 'gfmr_enqueue_assets', 'admin' );
	}

	/**
	 * Setup localization for admin preview
	 *
	 * @param string $version Plugin version
	 * @return void
	 */
	private function setup_admin_localization( $version ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter -- Reserved for hook compatibility
		$translations = $this->localize_translations();

		// Get theme settings for admin preview
		$settings_instance = GFMR_Settings::get_instance();
		$theme_config      = $this->get_theme_config( $settings_instance );

		$config_data = array_merge(
			$translations['translation_data'],
			$theme_config,
			array(
				'isAdmin'      => true,
				'adminPreview' => true,
			)
		);

		wp_localize_script(
			'gfmr-core',
			'wpGfmConfig',
			$config_data
		);
	}
}
