<?php
/**
 * WP GFM Asset Detector - PHP Pre-detection Class
 *
 * MD-22 Implementation: Server-side Markdown content analysis for
 * pre-detecting required libraries and performance optimization
 *
 * @package WpGfmRenderer
 * @since 0.2.0
 */

namespace Wakalab\WpGfmRenderer;

defined( 'ABSPATH' ) || exit;

// Prevent class redeclaration when both Free and Pro versions are active
if ( class_exists( __NAMESPACE__ . '\\GFMR_Asset_Detector' ) ) {
	return;
}

class GFMR_Asset_Detector {

	/**
	 * Detection result cache
	 *
	 * @var array
	 */
	private static $detection_cache = array();

	/**
	 * Detection pattern definitions
	 *
	 * @var array
	 */
	private static $detection_patterns = array(
		'shiki'       => array(
			'blocks'   => array( 'core/code' ),
			'patterns' => array(
				'/<pre[^>]*><code[^>]*class="[^"]*language-[^"]*"/',
				'/<code[^>]*class="[^"]*language-[^"]*">/',
				'/```[a-zA-Z]+\s/',
				'/\bhighlight\b/',
				'/\bsyntax\b/',
			),
		),
		'mermaid'     => array(
			'blocks'   => array( 'core/code' ),
			'patterns' => array(
				'/<pre[^>]*><code[^>]*class="[^"]*language-mermaid[^"]*"/',
				'/```mermaid\s/',
				'/\bmermaid\b/',
				'/<div[^>]*class="[^"]*mermaid[^"]*"/',
				'/\bdiagram\b.*\b(graph|flowchart|sequence|gantt|pie|gitgraph)\b/',
			),
		),
		'katex'       => array(
			'blocks'   => array( 'core/latex' ),
			'patterns' => array(
				// Block math: double dollar signs
				'/\$\$[\s\S]*?\$\$/',
				// Inline math: single dollar signs (excluding double)
				'/(?<!\$)\$(?!\$)[^$\n]+\$(?!\$)/',
				// WordPress LaTeX shortcode format
				'/\[latex\][\s\S]*?\[\/latex\]/i',
				// LaTeX parentheses notation with nesting support
				'/\\\\\([^)]*(?:\([^)]*\)[^)]*)*\\\\\)/',
				// LaTeX bracket notation
				'/\\\\\[[\s\S]*?\\\\\]/',
				// LaTeX begin/end environment blocks
				'/\\\\begin\{[^}]+\}[\s\S]*?\\\\end\{[^}]+\}/',
				// Already-rendered KaTeX span elements
				'/<span[^>]*class="[^"]*katex[^"]*"/',
			),
		),
		'gfm-plugins' => array(
			'blocks'   => array(),
			'patterns' => array(
				'/\|[^|]*\|[^|]*\|/', // Tables
				'/- \[[x ]\]/', // Task lists
				'/~~[^~]+~~/', // Strikethrough
				'/==[^=]+==/', // Mark/highlight
				'/\[\^[^\]]+\]/', // Footnotes
				'/\bhttps?:\/\/[^\s]+/', // Auto-links
				'/\bgfm\b.*\b(table|checklist|footnote)\b/',
			),
		),
		'chartjs'     => array(
			'blocks'   => array(),
			'patterns' => array(
				'/(?m)^```chart\b/',
				'/(?m)^```chart-pro\b/',
				'/<code[^>]*class="[^"]*language-chart[^"]*"/',
				'/<code[^>]*class="[^"]*language-chart-pro[^"]*"/',
			),
		),
	);

	/**
	 * Detect required assets from post content
	 *
	 * @param string|int $post_id_or_content Post ID or content string
	 * @return array Array of required libraries
	 */
	public static function detect_required_assets( $post_id_or_content ) {
		// Process as post ID if numeric
		if ( is_numeric( $post_id_or_content ) ) {
			$post_id   = (int) $post_id_or_content;
			$cache_key = 'gfmr_assets_' . $post_id;

			// Check cache
			if ( isset( self::$detection_cache[ $cache_key ] ) ) {
				return self::$detection_cache[ $cache_key ];
			}

			$post = get_post( $post_id );
			if ( ! $post ) {
				return array();
			}

			$content = $post->post_content;
		} else {
			$content   = (string) $post_id_or_content;
			$cache_key = 'gfmr_assets_' . md5( $content );

			// Content-based cache check
			if ( isset( self::$detection_cache[ $cache_key ] ) ) {
				return self::$detection_cache[ $cache_key ];
			}
		}

		$required_assets = array();

		// 1. Fast block detection using has_block()
		if ( function_exists( 'has_block' ) && is_numeric( $post_id_or_content ) ) {
			foreach ( self::$detection_patterns as $library => $config ) {
				if ( ! empty( $config['blocks'] ) ) {
					foreach ( $config['blocks'] as $block_type ) {
						if ( has_block( $block_type, $post_id_or_content ) ) {
							$required_assets[] = $library;
							break; // This library is needed if even one is found
						}
					}
				}
			}
		}

		// 2. Pattern matching detection (more detailed detection)
		foreach ( self::$detection_patterns as $library => $config ) {
			if ( in_array( $library, $required_assets, true ) ) {
				continue; // Already detected
			}

			foreach ( $config['patterns'] as $pattern ) {
				if ( preg_match( $pattern, $content ) ) {
					$required_assets[] = $library;
					break; // This library is needed if even one is found
				}
			}
		}

		// 3. markdown-it is basically always needed (when Markdown content exists)
		if ( ! empty( $required_assets ) || self::has_markdown_syntax( $content ) ) {
			if ( ! in_array( 'markdownit', $required_assets, true ) ) {
				array_unshift( $required_assets, 'markdownit' ); // Place at the beginning
			}
		}

		// Remove duplicates and sort
		$required_assets = array_unique( $required_assets );

		// Sort considering dependencies
		$required_assets = self::sort_by_dependencies( $required_assets );

		// Cache results (30 minutes)
		self::$detection_cache[ $cache_key ] = $required_assets;
		if ( function_exists( 'set_transient' ) ) {
			set_transient( $cache_key, $required_assets, 30 * MINUTE_IN_SECONDS );
		}

		return $required_assets;
	}

	/**
	 * Detect the presence of Markdown syntax
	 *
	 * @param string $content Content to check
	 * @return bool Whether Markdown syntax exists
	 */
	private static function has_markdown_syntax( $content ) {
		$markdown_patterns = array(
			'/^#{1,6}\s+/m',           // Headers
			'/```[\s\S]*?```/',        // Code blocks
			'/\*\*.*?\*\*/',           // Bold
			'/\*.*?\*/',               // Italic
			'/\|.*?\|.*?\|/',          // Tables
			'/^\s*[-*+]\s+/m',         // Lists
			'/!\[.*?\]\(.*?\)/',       // Images
			'/\[.*?\]\(.*?\)/',        // Links
			'/^\s*\d+\.\s+/m',         // Numbered lists
			'/^>\s+/m',                // Blockquotes
			'/`[^`]+`/',               // Inline code
			'/---+/',                  // Horizontal rules
		);

		foreach ( $markdown_patterns as $pattern ) {
			if ( preg_match( $pattern, $content ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Sort libraries considering dependencies
	 *
	 * @param array $libraries Library list
	 * @return array Sorted library list
	 */
	private static function sort_by_dependencies( $libraries ) {
		$dependencies = array(
			'markdownit'  => array(),
			'gfm-plugins' => array( 'markdownit' ),
			'mermaid'     => array( 'markdownit' ),
			'shiki'       => array(),
			'katex'       => array(),
			'chartjs'     => array(),
		);

		$sorted    = array();
		$processed = array();

		$process_library = function ( $library ) use ( &$process_library, &$sorted, &$processed, $dependencies, $libraries ) {
			if ( in_array( $library, $processed, true ) || ! in_array( $library, $libraries, true ) ) {
				return;
			}

			// Process dependencies first
			if ( isset( $dependencies[ $library ] ) ) {
				foreach ( $dependencies[ $library ] as $dependency ) {
					$process_library( $dependency );
				}
			}

			$sorted[]    = $library;
			$processed[] = $library;
		};

		foreach ( $libraries as $library ) {
			$process_library( $library );
		}

		return $sorted;
	}

	/**
	 * Get detailed information about asset detection results
	 *
	 * @param string|int $post_id_or_content Post ID or content string
	 * @return array Detailed information about detection results
	 */
	public static function get_detection_details( $post_id_or_content ) {
		$required_assets = self::detect_required_assets( $post_id_or_content );

		// Size calculation
		$library_sizes = array(
			'markdownit'  => 180000,  // ~180KB
			'shiki'       => 200000,       // ~200KB
			'mermaid'     => 500000,     // ~500KB
			'gfm-plugins' => 50000,  // ~50KB
			'katex'       => 350000,        // ~350KB (JS + CSS)
			'chartjs'     => 200000,  // ~200KB
		);

		$total_size = 0;
		$all_size   = array_sum( $library_sizes );

		foreach ( $required_assets as $library ) {
			if ( isset( $library_sizes[ $library ] ) ) {
				$total_size += $library_sizes[ $library ];
			}
		}

		$saved_size = $all_size - $total_size;

		return array(
			'required_libraries' => $required_assets,
			'total_libraries'    => count( array_keys( $library_sizes ) ),
			'required_count'     => count( $required_assets ),
			'total_size'         => $total_size,
			'saved_size'         => $saved_size,
			'savings_percentage' => $all_size > 0 ? round( ( $saved_size / $all_size ) * 100, 1 ) : 0,
			'library_details'    => array_map(
				function ( $lib ) use ( $library_sizes ) {
					return array(
						'name' => $lib,
						'size' => isset( $library_sizes[ $lib ] ) ? $library_sizes[ $lib ] : 0,
					);
				},
				$required_assets
			),
		);
	}

	/**
	 * Output debug information
	 *
	 * @param string|int $post_id_or_content Post ID or content string
	 * @return void
	 */
	public static function debug_detection( $post_id_or_content ) {
		if ( ! ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) ) {
			return;
		}

		$details = self::get_detection_details( $post_id_or_content );

		// Use WordPress standard error handling
		if ( function_exists( 'wp_trigger_error' ) ) {
			wp_trigger_error( '', '[WP GFM Asset Detector] Detection Results:', E_USER_NOTICE );
			wp_trigger_error( '', '  Required Libraries: ' . implode( ', ', $details['required_libraries'] ), E_USER_NOTICE );
			wp_trigger_error( '', '  Total Size: ' . number_format( $details['total_size'] ) . ' bytes', E_USER_NOTICE );
			wp_trigger_error( '', '  Saved Size: ' . number_format( $details['saved_size'] ) . ' bytes', E_USER_NOTICE );
			wp_trigger_error( '', '  Savings: ' . $details['savings_percentage'] . '%', E_USER_NOTICE );
		}
	}

	/**
	 * Clear cache
	 *
	 * @return void
	 */
	public static function clear_cache() {
		self::$detection_cache = array();

		// Also clear transient cache - using WordPress standard API
		// Avoid direct database queries, use WordPress standard functions
		// Delete only plugin-related transients
		$transient_prefixes = array(
			'gfmr_assets_',
			'gfmr_has_markdown_globally',
		);

		foreach ( $transient_prefixes as $prefix ) {
			// Delete transients for each prefix
			// Use WordPress standard delete_transient()
			delete_transient( $prefix . '*' );
		}

		// Additional safe cleanup: known specific transient keys
		$known_transients = array(
			'gfmr_has_markdown_globally',
		);

		foreach ( $known_transients as $transient ) {
			delete_transient( $transient );
		}
	}
}
