#!/bin/bash
# Docker entrypoint script for Pro version development
# Copies Pro version files and creates symlinks from Free version directories

set -e

# Pro版プラグインディレクトリ
PRO_DIR="/var/www/html/wp-content/plugins/markdown-renderer-for-github-pro"
PRO_SOURCE="/tmp/gfmr-pro-source"
FREE_SOURCE="/tmp/gfmr-free"

# WordPressの初期化を待つ
echo "Waiting for WordPress initialization..."

# 元のWordPress entrypointを実行（バックグラウンドで）
docker-entrypoint.sh "$@" &
WORDPRESS_PID=$!

# wp-content/pluginsディレクトリが作成されるまで待つ
TIMEOUT=60
ELAPSED=0
while [ ! -d "/var/www/html/wp-content/plugins" ] && [ $ELAPSED -lt $TIMEOUT ]; do
    sleep 1
    ELAPSED=$((ELAPSED + 1))
done

if [ ! -d "/var/www/html/wp-content/plugins" ]; then
    echo "ERROR: WordPress plugins directory not created after ${TIMEOUT}s"
    exit 1
fi

echo "WordPress plugins directory is ready"

# Pro版ディレクトリが既に存在する場合は削除
if [ -d "$PRO_DIR" ]; then
    echo "Removing existing Pro version directory..."
    rm -rf "$PRO_DIR"
fi

# Pro版ディレクトリを作成
echo "Setting up Pro version directory..."
mkdir -p "$PRO_DIR"

# Pro版固有ファイルをコピー
echo "Copying Pro version files..."
cp -r "$PRO_SOURCE"/* "$PRO_DIR/" 2>/dev/null || true

# includes と languages ディレクトリを削除（Free版のリソースと統合するため）
rm -rf "$PRO_DIR/includes" "$PRO_DIR/languages"

# includes ディレクトリを再作成してPro版includesをコピー
mkdir -p "$PRO_DIR/includes"
if [ -d "$PRO_SOURCE/includes" ]; then
    # .gitkeepを含む空のディレクトリを除外してコピー
    find "$PRO_SOURCE/includes" -mindepth 1 -maxdepth 1 -type f -exec cp {} "$PRO_DIR/includes/" \;
fi

# languages ディレクトリを再作成してPro版languagesをコピー
mkdir -p "$PRO_DIR/languages"
if [ -d "$PRO_SOURCE/languages" ]; then
    # .gitkeepを含む空のディレクトリを除外してコピー
    find "$PRO_SOURCE/languages" -mindepth 1 -maxdepth 1 -type f -exec cp {} "$PRO_DIR/languages/" \;
fi

# Free版へのシンボリックリンクを作成
echo "Creating symlinks to Free version resources..."

# 空のディレクトリがあれば削除してからシンボリックリンク作成
rm -rf "$PRO_DIR/includes/core"
ln -sf "$FREE_SOURCE/includes" "$PRO_DIR/includes/core"
echo "✓ includes/core -> Free version"

rm -rf "$PRO_DIR/languages/free"
ln -sf "$FREE_SOURCE/languages" "$PRO_DIR/languages/free"
echo "✓ languages/free -> Free version"

# 空のディレクトリを削除してFree版にリンク
for dir in assets blocks build vendor; do
    rm -rf "$PRO_DIR/$dir"
    ln -sf "$FREE_SOURCE/$dir" "$PRO_DIR/$dir"
    echo "✓ $dir -> Free version"
done

echo "Pro version setup complete!"
echo "Pro version location: $PRO_DIR"

# WordPressプロセスをフォアグラウンドに戻す
wait $WORDPRESS_PID
