import yaml from 'js-yaml';

/**
 * Parse YAML frontmatter from markdown content
 *
 * @param {string} content - Markdown content with optional frontmatter
 * @returns {Object} Object with frontmatter and body properties
 */
export const parseFrontMatter = (content) => {
	if (!content || typeof content !== 'string') {
		return { frontmatter: {}, body: content || '' };
	}

	const match = content.match(/^---\s*\n([\s\S]*?)\n---\s*\n([\s\S]*)$/);
	if (!match) {
		return { frontmatter: {}, body: content };
	}

	try {
		const parsed = yaml.load(match[1]);
		return {
			frontmatter: parsed || {},
			body: match[2]
		};
	} catch (e) {
		console.warn('[WP GFM] YAML parse error:', e);
		return { frontmatter: {}, body: content };
	}
};

/**
 * Render frontmatter as HTML header
 *
 * @param {Object} fm - Frontmatter object
 * @param {Object} options - Rendering options
 * @returns {string} HTML string
 */
export const renderFrontmatterHeader = (fm, options = {}) => {
	if (!fm || Object.keys(fm).length === 0) return '';

	const escape = (str) => String(str)
		.replace(/&/g, '&amp;')
		.replace(/</g, '&lt;')
		.replace(/>/g, '&gt;')
		.replace(/"/g, '&quot;');

	let html = '<header class="gfmr-frontmatter">';

	if (fm.title) {
		html += `<h1 class="gfmr-fm-title">${escape(fm.title)}</h1>`;
	}
	if (fm.date) {
		html += `<time class="gfmr-fm-date" datetime="${escape(fm.date)}">${escape(fm.date)}</time>`;
	}
	if (fm.author) {
		html += `<span class="gfmr-fm-author">${escape(fm.author)}</span>`;
	}
	if (fm.tags && Array.isArray(fm.tags)) {
		html += '<div class="gfmr-fm-tags">';
		fm.tags.forEach(tag => {
			html += `<span class="gfmr-fm-tag">${escape(tag)}</span>`;
		});
		html += '</div>';
	}

	html += '</header>';
	return html;
};
