/**
 * Markdown Block Editor Component
 *
 * @package WpGfmRenderer
 */

import { __ } from '@wordpress/i18n';
import {
	useBlockProps,
	BlockControls,
	InspectorControls,
	MediaUpload,
	MediaUploadCheck
} from '@wordpress/block-editor';
import {
	Panel,
	PanelBody,
	PanelRow,
	ToolbarGroup,
	ToolbarButton,
	ResizableBox,
	ColorPicker,
	ToggleControl,
	SelectControl,
	TextControl,
	Button,
	Modal
} from '@wordpress/components';
import { useState, useEffect, useRef, useCallback, useMemo } from '@wordpress/element';
import { RawHTML } from '@wordpress/element';

// Import frontmatter parser
import { parseFrontMatter, renderFrontmatterHeader } from './frontmatter-parser';

// Import markdown-it
import MarkdownIt from 'markdown-it';
import markdownItTaskLists from 'markdown-it-task-lists';
import markdownItFootnote from 'markdown-it-footnote';
import markdownItMark from 'markdown-it-mark';
import markdownItStrikethrough from 'markdown-it-strikethrough-alt';

/**
 * Initialize markdown-it instance (with Shiki integration support)
 * Default highlight function is provided for pre-Shiki fallback
 */
let md = new MarkdownIt({
	html: true,        // Allow HTML tags
	breaks: true,      // Convert newlines to <br>
	linkify: true,     // Automatically convert URLs to links
	typographer: true, // Enable typography features
	highlight: function(code, lang) {
		// Pre-Shiki fallback: render code blocks as plain text with escaping
		const escapedCode = code
			.replace(/&/g, '&amp;')
			.replace(/</g, '&lt;')
			.replace(/>/g, '&gt;');
		const langClass = lang ? ` class="language-${lang}"` : '';
		return `<pre class="gfmr-code-fallback"><code${langClass}>${escapedCode}</code></pre>`;
	}
})
.use(markdownItTaskLists, { enabled: true })  // Task lists
.use(markdownItFootnote)                      // Footnotes
.use(markdownItMark)                          // Marking (==text==)
.use(markdownItStrikethrough);                // Strikethrough

// Cache for Shiki highlighter (for editor)
let editorShikiHighlighter = null;
let shikiLoadPromise = null;

// Shiki retry configuration
const SHIKI_RETRY_CONFIG = {
	maxRetries: 3,
	baseDelay: 1000, // 1 second
	maxDelay: 8000,  // 8 seconds max
};
let shikiRetryCount = 0;
let shikiLoadStatus = 'idle'; // 'idle' | 'loading' | 'loaded' | 'failed'

/**
 * Calculate exponential backoff delay
 * @param {number} retryCount Current retry attempt number
 * @returns {number} Delay in milliseconds
 */
const calculateBackoffDelay = (retryCount) => {
	const delay = SHIKI_RETRY_CONFIG.baseDelay * Math.pow(2, retryCount);
	return Math.min(delay, SHIKI_RETRY_CONFIG.maxDelay);
};

/**
 * Sleep helper function
 * @param {number} ms Milliseconds to sleep
 * @returns {Promise<void>}
 */
const sleep = (ms) => new Promise(resolve => setTimeout(resolve, ms));

// ============================================
// Multilingual Constants & Helpers
// ============================================

// Language code validation regex (ISO 639-1 format)
const VALID_LANG_REGEX = /^[a-z]{2}(-[A-Z]{2})?$/;
const MAX_LANGUAGES = 10;

// Common languages list
const COMMON_LANGUAGES = [
	{ value: 'en', label: 'English (en)' },
	{ value: 'ja', label: '日本語 (ja)' },
	{ value: 'zh', label: '中文 (zh)' },
	{ value: 'ko', label: '한국어 (ko)' },
	{ value: 'es', label: 'Español (es)' },
	{ value: 'fr', label: 'Français (fr)' },
	{ value: 'de', label: 'Deutsch (de)' },
	{ value: 'pt', label: 'Português (pt)' },
	{ value: 'it', label: 'Italiano (it)' },
	{ value: 'ru', label: 'Русский (ru)' },
];

/**
 * Validate language code format
 * @param {string} code - Language code to validate
 * @returns {boolean}
 */
const isValidLanguageCode = ( code ) => {
	if ( ! code || typeof code !== 'string' ) return false;
	// Normalize to lowercase for validation
	const normalized = code.toLowerCase().trim();
	return VALID_LANG_REGEX.test( normalized ) && normalized.length <= 5;
};

/**
 * Sanitize language code
 * @param {string} code - Language code to sanitize
 * @returns {string|null}
 */
const sanitizeLangCode = ( code ) => {
	if ( ! code ) return null;
	const sanitized = code.toLowerCase().trim().slice( 0, 5 );
	return isValidLanguageCode( sanitized ) ? sanitized : null;
};

// ============================================
// Mermaid Detection Constants & Helpers
// ============================================

// Mermaid diagram type patterns (content must start with one of these)
const MERMAID_START_PATTERNS = [
	'graph', 'flowchart', 'sequenceDiagram', 'classDiagram',
	'pie', 'gantt', 'stateDiagram', 'erDiagram', 'gitGraph', 'gitgraph',
	'journey'
];

// Mermaid inline patterns (content must include one of these)
const MERMAID_INCLUDE_PATTERNS = [ 'graph TD', 'graph LR' ];

// Patterns that indicate already-rendered SVG (not Mermaid source)
const SVG_EXCLUSION_PATTERNS = [ '#gfmr-mermaid-', 'font-family:', '<svg', '</svg>' ];

// Class-based Mermaid selectors (Phase 1 detection)
const MERMAID_CLASS_SELECTORS = [
	'pre code.language-mermaid',
	'pre code[class*="language-mermaid"]',
	'code.language-mermaid',
	'.shiki .language-mermaid',
	'[data-language="mermaid"]',
	'pre[class*="language-mermaid"]'
];

// Fallback selectors for content-based detection (Phase 2)
// NOTE: Selectors are ordered by specificity. Avoid overly generic selectors like
// bare 'code' to prevent performance issues when many inline code elements exist.
const MERMAID_FALLBACK_SELECTORS = [
	'pre.shiki code',
	'.shiki code',
	'pre code',
	'.gfmr-markdown-rendered-preview pre code',
	'.gfmr-markdown-rendered-preview code'
];

/**
 * Check if debug mode is enabled via config or URL parameter
 * @returns {boolean} True if debug mode is enabled
 */
const isDebugEnabled = () => {
	return window.wpGfmConfig?.debug ||
		window.location.search.includes( 'debug=true' ) ||
		window.location.search.includes( 'gfmr-debug=1' ) ||
		window.location.search.includes( 'debug=1' ) ||
		false;
};

/**
 * Debug logger - outputs to console only when debug mode is enabled
 * @param {...any} args - Arguments to log
 */
const logDebug = ( ...args ) => {
	if ( isDebugEnabled() ) {
		console.log( '[WP GFM Editor]', ...args );
	}
};

/**
 * Debug warning logger - outputs to console only when debug mode is enabled
 * @param {...any} args - Arguments to log
 */
const logDebugWarn = ( ...args ) => {
	if ( isDebugEnabled() ) {
		console.warn( '[WP GFM Editor]', ...args );
	}
};

/**
 * Escape special characters in Markdown alt text
 *
 * @param {string} text - Text to escape
 * @returns {string} Escaped text
 */
const escapeMarkdownAlt = ( text ) => {
	if ( ! text ) {
		return '';
	}
	return text.replace( /[\[\]\\]/g, '\\$&' );
};

/**
 * Encode URL for Markdown image syntax
 * Handles spaces and special characters
 *
 * @param {string} url - URL to encode
 * @returns {string} Encoded URL wrapped in angle brackets if needed
 */
const encodeMarkdownUrl = ( url ) => {
	if ( ! url ) {
		return '';
	}
	// If URL contains spaces or parentheses, wrap in angle brackets
	if ( /[\s()]/.test( url ) ) {
		return `<${ url }>`;
	}
	return url;
};

/**
 * Validate image URL format
 * @param {string} url - URL to validate
 * @returns {boolean} - True if valid image URL
 */
const isValidImageUrl = ( url ) => {
	if ( ! url || typeof url !== 'string' ) {
		return false;
	}
	try {
		const parsedUrl = new URL( url );
		return [ 'http:', 'https:' ].includes( parsedUrl.protocol );
	} catch {
		// Relative URL check (WordPress internal paths)
		return url.startsWith( '/' ) && ! url.includes( '..' );
	}
};

/**
 * Check if content is Mermaid diagram source (not rendered SVG)
 * @param {string} content - Content to check
 * @returns {boolean} True if content is Mermaid source
 */
const isMermaidContent = ( content ) => {
	if ( ! content ) return false;
	const hasPattern = MERMAID_START_PATTERNS.some( p => content.startsWith( p ) ) ||
	                   MERMAID_INCLUDE_PATTERNS.some( p => content.includes( p ) );
	const isRenderedSvg = SVG_EXCLUSION_PATTERNS.some( e => content.includes( e ) );
	return hasPattern && ! isRenderedSvg;
};

/**
 * Generate Mermaid container CSS style
 * @param {string} bgColor - Background color
 * @returns {string} CSS style string
 */
const getMermaidContainerStyle = ( bgColor = 'transparent' ) =>
	`text-align: center; margin: 20px 0; padding: 15px; background: ${bgColor}; border-radius: 6px; overflow-x: auto; border: none;`;

/**
 * Build Mermaid error HTML template
 * @param {string} errorMessage - Error message
 * @param {string} code - Original Mermaid code
 * @returns {string} Error HTML
 */
const buildMermaidErrorHTML = ( errorMessage, code ) => `
	<div class="gfmr-mermaid-error-unified" style="color: #d1242f; background: #fff8f8; border: 1px solid #ffcdd2; border-radius: 6px; padding: 15px; margin: 10px 0; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif; font-size: 14px; line-height: 1.5;">
		<div style="font-weight: 600; margin-bottom: 8px;">🚨 Mermaid Rendering Error (editor)</div>
		<div style="margin-bottom: 12px; font-size: 13px;">${errorMessage || 'Unknown error occurred'}</div>
		<details style="margin-top: 10px;">
			<summary style="cursor: pointer; font-size: 13px; margin-bottom: 8px;">View Original Code</summary>
			<pre style="background: #f6f8fa; border: 1px solid #d1d9e0; border-radius: 4px; padding: 10px; margin: 5px 0; overflow-x: auto; font-size: 12px; font-family: 'SFMono-Regular', Monaco, 'Cascadia Code', 'Roboto Mono', Consolas, monospace; white-space: pre-wrap; word-wrap: break-word;">${code}</pre>
		</details>
	</div>
`;

/**
 * Helper function to get Shiki script URL
 * @returns {string} Shiki script URL
 */
const getShikiScriptUrl = () => {
	if (window.wpGfmBuildLocalAssetUrl) {
		return window.wpGfmBuildLocalAssetUrl('assets/libs/shiki/shiki.min.js');
	}

	if (window.wpGfmConfig && window.wpGfmConfig.pluginUrl) {
		return window.wpGfmConfig.pluginUrl + 'assets/libs/shiki/shiki.min.js';
	}

	// Fallback URL
	return '/wp-content/plugins/markdown-renderer-for-github/assets/libs/shiki/shiki.min.js';
};

/**
 * Attempt to load Shiki script once
 * @returns {Promise<object|null>} Highlighter instance or null on failure
 */
const attemptShikiLoad = async () => {
	// Check if Shiki is already loaded
	if (window.shiki) {
		console.log('[WP GFM Editor] Shiki already loaded globally');
		return window.shiki;
	}

	// Load Shiki locally (WordPress.org compliant)
	const script = document.createElement('script');
	script.src = getShikiScriptUrl();
	script.async = true;

	console.log('[WP GFM Editor] Loading Shiki locally:', script.src);

	await new Promise((resolve, reject) => {
		script.onload = resolve;
		script.onerror = (error) => reject(new Error(`Script load failed: ${script.src}`));
		document.head.appendChild(script);
	});

	// Access from window.shiki after local load
	if (!window.shiki || !window.shiki.getHighlighter) {
		throw new Error('Shiki global object not found after local asset load');
	}

	console.log('[WP GFM Editor] Shiki loaded successfully from local assets');
	const { getHighlighter } = window.shiki;

	// Create highlighter (new Shiki API)
	const highlighter = await getHighlighter({
		themes: ['github-dark', 'github-light'],
		langs: ['javascript', 'typescript', 'python', 'ruby', 'go', 'rust', 'bash', 'css', 'html', 'json', 'yaml', 'xml', 'sql', 'php', 'java', 'csharp', 'cpp', 'diff']
	});

	return highlighter;
};

/**
 * Configure markdown-it with Shiki highlighter
 * @param {object} highlighter Shiki highlighter instance
 */
const configureMarkdownItWithShiki = (highlighter) => {
	md = md.set({
		highlight: function(code, lang) {
			try {
				// Indent code blocks (no language specified) are treated as plain text
				if (!lang || lang.trim() === '') {
					const escapedCode = code
						.replace(/&/g, '&amp;')
						.replace(/</g, '&lt;')
						.replace(/>/g, '&gt;');
					return `<pre><code>${escapedCode}</code></pre>`;
				}

				// Normalize language name (only when language is specified)
				const normalizedLang = lang.toLowerCase().trim();

				// Language alias support
				const langAliases = {
					'js': 'javascript',
					'ts': 'typescript',
					'py': 'python',
					'rb': 'ruby',
					'sh': 'bash',
					'shell': 'bash',
					'yml': 'yaml',
					'c++': 'cpp',
					'c#': 'csharp',
					'cs': 'csharp',
					'patch': 'diff'
				};

				const finalLang = langAliases[normalizedLang] || normalizedLang;

				// Skip chart blocks - handled by gfmr-charts.js
				if (finalLang === 'chart' || finalLang === 'chart-pro') {
					const escapedCode = code
						.replace(/&/g, '&amp;')
						.replace(/</g, '&lt;')
						.replace(/>/g, '&gt;');
					return `<pre><code class="language-${finalLang}">${escapedCode}</code></pre>`;
				}

				// Check if language is supported
				const supportedLangs = highlighter.getLoadedLanguages();
				// Special handling for Mermaid (highlight as plain text)
				const langToUse = finalLang === 'mermaid' ? 'plaintext' :
					(supportedLangs.includes(finalLang) ? finalLang : 'plaintext');

				// Get theme from plugin settings (consistent with frontend)
				let theme = window.wpGfmConfig?.theme?.shiki_theme ??
					(window.matchMedia?.('(prefers-color-scheme: dark)').matches ? 'github-dark' : 'github-light');
				if (theme === 'auto') {
					const isDark = window.matchMedia &&
						window.matchMedia('(prefers-color-scheme: dark)').matches;
					theme = isDark ? 'github-dark' : 'github-light';
				}

				return highlighter.codeToHtml(code, {
					lang: langToUse,
					theme: theme
				});
			} catch (error) {
				console.warn('[WP GFM Editor] Highlighting failed:', error);

				// Fallback
				const escapedCode = code
					.replace(/&/g, '&amp;')
					.replace(/</g, '&lt;')
					.replace(/>/g, '&gt;');

				return `<pre><code class="language-${lang || 'plaintext'}">${escapedCode}</code></pre>`;
			}
		}
	});
};

/**
 * Shiki loading function (for editor) - with retry logic and exponential backoff
 * @returns {Promise<object|null>} Highlighter instance or null on failure
 */
const loadShikiForEditor = async () => {
	if (editorShikiHighlighter) {
		return editorShikiHighlighter;
	}

	if (shikiLoadPromise) {
		return shikiLoadPromise;
	}

	shikiLoadStatus = 'loading';

	shikiLoadPromise = (async () => {
		while (shikiRetryCount <= SHIKI_RETRY_CONFIG.maxRetries) {
			try {
				const highlighter = await attemptShikiLoad();

				if (highlighter) {
					editorShikiHighlighter = highlighter;
					configureMarkdownItWithShiki(highlighter);
					shikiLoadStatus = 'loaded';
					shikiRetryCount = 0; // Reset for future use
					return editorShikiHighlighter;
				}
			} catch (error) {
				const isLastAttempt = shikiRetryCount >= SHIKI_RETRY_CONFIG.maxRetries;

				if (isLastAttempt) {
					console.error(`[WP GFM Editor] Failed to load Shiki after ${shikiRetryCount + 1} attempts:`, error);
					console.log('[WP GFM Editor] Available globals:', {
						wpGfmBuildLocalAssetUrl: typeof window.wpGfmBuildLocalAssetUrl,
						wpGfmConfig: typeof window.wpGfmConfig,
						wpGfmConfigContent: window.wpGfmConfig
					});
					shikiLoadStatus = 'failed';
					shikiLoadPromise = null; // Allow future retry attempts
					return null;
				}

				// Calculate backoff delay and retry
				const delay = calculateBackoffDelay(shikiRetryCount);
				console.warn(
					`[WP GFM Editor] Shiki load attempt ${shikiRetryCount + 1} failed, retrying in ${delay}ms...`,
					error.message
				);

				shikiRetryCount++;
				await sleep(delay);
			}
		}

		return null;
	})();

	return shikiLoadPromise;
};

/**
 * Get current Shiki load status
 * @returns {string} Current status: 'idle' | 'loading' | 'loaded' | 'failed'
 */
const getShikiLoadStatus = () => shikiLoadStatus;

/**
 * Reset Shiki loader state (useful for manual retry)
 */
const resetShikiLoader = () => {
	shikiLoadPromise = null;
	shikiRetryCount = 0;
	shikiLoadStatus = 'idle';
	editorShikiHighlighter = null;
};

/**
 * Markdown block editor component
 */
export default function Edit( { attributes, setAttributes } ) {
	const {
		content,
		html,
		mermaidBgColor,
		showFrontmatter = false,
		frontmatterData = {},
		// Multilingual attributes
		languages = {},
		defaultLanguage = 'en',
		availableLanguages = [],
		showLanguageSwitcher = true,
	} = attributes;

	const [ isPreview, setIsPreview ] = useState( false );
	const [ renderedHtml, setRenderedHtml ] = useState( html || '' );
	const [ processedHtml, setProcessedHtml ] = useState( null );
	const [ shikiLoaded, setShikiLoaded ] = useState( false );
	const textareaRef = useRef( null );
	const cursorPositionRef = useRef( null );
	const previewRef = useRef( null );
	const processMermaidBlocksRef = useRef( null );
	const isMermaidProcessing = useRef( false );
	const lastRenderedHtmlRef = useRef( '' );

	// Ref to track latest content (avoiding stale closure in handleImageInsert)
	const contentRef = useRef( content );
	useEffect( () => {
		contentRef.current = content;
	}, [ content ] );

	// Multilingual state
	const [ currentEditingLang, setCurrentEditingLang ] = useState( defaultLanguage );
	const [ isAddLanguageModalOpen, setIsAddLanguageModalOpen ] = useState( false );

	// Check if multilingual mode is active (more than one language or has languages data)
	const isMultilingualMode = availableLanguages.length > 0 || Object.keys( languages ).length > 0;

	// 多言語データの整合性を保証
	useEffect( () => {
		const langKeys = Object.keys( languages );

		// languagesオブジェクトがあるが、availableLanguagesが空または不一致の場合は修復
		if ( langKeys.length > 0 ) {
			const needsUpdate = availableLanguages.length === 0 ||
				! langKeys.every( k => availableLanguages.includes( k ) );

			if ( needsUpdate ) {
				const newDefaultLang = langKeys.includes( defaultLanguage )
					? defaultLanguage
					: langKeys[ 0 ];

				setAttributes( {
					availableLanguages: langKeys,
					defaultLanguage: newDefaultLang,
					showLanguageSwitcher: langKeys.length > 1,
				} );

				setCurrentEditingLang( newDefaultLang );
			}
		}
	}, [ languages ] );

	// Get current content based on mode
	const getCurrentContent = useCallback( () => {
		if ( isMultilingualMode && languages[ currentEditingLang ] ) {
			return languages[ currentEditingLang ].content || '';
		}
		return content || '';
	}, [ isMultilingualMode, languages, currentEditingLang, content ] );

	// Get current HTML based on mode
	const getCurrentHtml = useCallback( () => {
		if ( isMultilingualMode && languages[ currentEditingLang ] ) {
			return languages[ currentEditingLang ].html || '';
		}
		return html || '';
	}, [ isMultilingualMode, languages, currentEditingLang, html ] );

	// Feature flags from PHP (for Pro version migration in the future)
	const isImageInsertEnabled = window.wpGfmConfig?.features?.imageInsert ?? true;

	/**
	 * Update content and trigger re-render (unified for both modes)
	 */
	const updateContentWithRender = useCallback( async ( newContent ) => {
		if ( isMultilingualMode ) {
			const updatedLanguages = {
				...languages,
				[ currentEditingLang ]: {
					...( languages[ currentEditingLang ] || {} ),
					content: newContent,
				},
			};
			setAttributes( { languages: updatedLanguages } );
			try {
				await renderMarkdownForLanguage( newContent, currentEditingLang, updatedLanguages );
			} catch ( error ) {
				console.warn( '[updateContentWithRender] Render failed:', error.message );
			}
		} else {
			setAttributes( { content: newContent } );
			try {
				await renderMarkdown( newContent );
			} catch ( error ) {
				console.warn( '[updateContentWithRender] Render failed:', error.message );
			}
		}
	}, [
		isMultilingualMode, languages, currentEditingLang,
		setAttributes, renderMarkdown, renderMarkdownForLanguage
	] );

	/**
	 * Handle image insertion from media library
	 *
	 * @param {Object} media - WordPress media object
	 */
	const handleImageInsert = useCallback( async ( media ) => {
		// Handle array case (multiple selection) - use first item
		const selectedMedia = Array.isArray( media ) ? media[ 0 ] : media;

		if ( ! selectedMedia ) {
			console.warn( '[handleImageInsert] No media selected' );
			return;
		}

		// Prefer source_url over url for reliability
		const imageUrl = selectedMedia?.source_url || selectedMedia?.url;
		if ( ! imageUrl || ! isValidImageUrl( imageUrl ) ) {
			console.warn( '[handleImageInsert] Invalid or missing URL. Media ID:', selectedMedia?.id );
			return;
		}

		// Build alt text: respect empty string as intentional empty alt
		let altText = '';
		if ( selectedMedia.alt !== '' && selectedMedia.alt != null ) {
			altText = selectedMedia.alt;
		} else if ( selectedMedia.alt !== '' && selectedMedia.title ) {
			altText = selectedMedia.title;
		}
		// Note: If both alt and title are empty/undefined, altText remains ''

		// Escape special characters
		const escapedAlt = escapeMarkdownAlt( altText );
		const encodedUrl = encodeMarkdownUrl( imageUrl );
		const imageMarkdown = `![${ escapedAlt }](${ encodedUrl })`;

		// If in preview mode, switch to edit mode first
		if ( isPreview ) {
			setIsPreview( false );
		}

		// Get current content based on mode
		const currentContent = getCurrentContent();
		const textarea = textareaRef.current;
		const saved = cursorPositionRef.current;

		// Determine insertion point and build new content
		let newContent;
		const separator = currentContent && currentContent.length > 0 ? '\n' : '';

		if ( saved && saved.start >= 0 ) {
			// Insert at saved cursor position
			newContent =
				currentContent.substring( 0, saved.start ) +
				imageMarkdown +
				currentContent.substring( saved.end );
		} else {
			// Append to end
			newContent = currentContent + separator + imageMarkdown;
		}

		// Update content (unified)
		await updateContentWithRender( newContent );

		// Clear saved cursor position
		cursorPositionRef.current = null;

		// Focus and restore cursor position if possible
		setTimeout( () => {
			if ( textarea ) {
				// Focus without scrolling (fallback for older browsers)
				try {
					textarea.focus( { preventScroll: true } );
				} catch ( e ) {
					console.warn( 'preventScroll not supported, using fallback:', e );
					textarea.focus();
				}

				// Set cursor position
				const newPos = saved && saved.start >= 0
					? saved.start + imageMarkdown.length
					: newContent.length;
				textarea.setSelectionRange( newPos, newPos );
			}
		}, 0 );
	}, [
		setAttributes,
		isPreview,
		setIsPreview,
		getCurrentContent,
		updateContentWithRender
	] );

	// Compute theme CSS class for preview container (consistent with frontend gfmr-main.js)
	const themeClass = useMemo( () => {
		let theme = window.wpGfmConfig?.theme?.shiki_theme ??
			(window.matchMedia?.('(prefers-color-scheme: dark)').matches ? 'github-dark' : 'github-light');
		if ( theme === 'auto' ) {
			const isDark = window.matchMedia &&
				window.matchMedia( '(prefers-color-scheme: dark)' ).matches;
			theme = isDark ? 'github-dark' : 'github-light';
		}
		return theme === 'github-dark' ? 'gfmr-dark' : 'gfmr-light';
	}, [] );

	// Initialize showFrontmatter from global settings for new blocks
	useEffect( () => {
		// Only apply global setting if showFrontmatter has never been set (undefined in saved data)
		// Check if this is a new block (no content yet) and showFrontmatter is default false
		if ( ! content && showFrontmatter === false ) {
			const globalShowHeader = window.wpGfmConfig?.frontmatter?.showHeader ?? false;
			if ( globalShowHeader ) {
				setAttributes( { showFrontmatter: true } );
			}
		}
	}, [] ); // Run only on mount

	// Dynamically load Mermaid
	// Note: Empty dependency array to stabilize function reference across renders.
	// Uses window.mermaid as the source of truth instead of React state.
	const loadMermaid = useCallback( async () => {
		// Check window.mermaid directly for stability (avoids dependency chain issues)
		if ( window.mermaid ) {
			return true;
		}

		try {
			// Load Mermaid.js locally (WordPress.org distribution compliant)
			const script = document.createElement( 'script' );
			script.src = window.wpGfmBuildAssetUrl ?
				window.wpGfmBuildAssetUrl('assets/libs/mermaid/mermaid.min.js') :
				(( window.wpGfmConfig?.pluginUrl || '' ) + '/assets/libs/mermaid/mermaid.min.js');
			script.async = true;

			await new Promise( ( resolve, reject ) => {
				script.onload = resolve;
				script.onerror = reject;
				document.head.appendChild( script );
			} );

			if ( window.mermaid ) {
				// Use unified initialization function (ensures frontend consistency)
				if ( window.wpGfmInitializeMermaidUnified ) {
					const initResult = await window.wpGfmInitializeMermaidUnified( true );
					if ( initResult ) {
						return true;
					}
				}

				// Fallback: Full configuration matching frontend (gfmr-main.js)
				window.mermaid.initialize( {
					startOnLoad: false,
					theme: 'default',
					securityLevel: 'loose',
					fontFamily: 'monospace',
					themeVariables: {
						background: 'transparent'
					},
					flowchart: {
						useMaxWidth: false,
						htmlLabels: true,
						curve: 'basis',
					},
					sequence: {
						useMaxWidth: true,
						height: 65,
					},
					class: {
						useMaxWidth: true,
					},
					state: {
						useMaxWidth: false,
					},
					er: {
						useMaxWidth: false,
					},
					gantt: {
						useMaxWidth: true,
					},
					gitGraph: {
						useMaxWidth: false,
						mainLineWidth: 2,
						mainBranchOrder: 0,
					},
				} );
				return true;
			}
		} catch ( error ) {
			console.error( '[WP GFM Editor] Failed to load Mermaid:', error );
		}
		return false;
	}, [] );  // Empty dependency array for stable reference

	/**
	 * Extract plain text from Shiki-highlighted code for Chart.js
	 * Removes span tags and decodes HTML entities
	 */
	const extractChartCodeFromShiki = ( element ) => {
		// Clone element and replace all spans with text nodes
		const clone = element.cloneNode( true );
		const spans = clone.querySelectorAll( 'span' );
		spans.forEach( ( span ) => {
			const textNode = document.createTextNode( span.textContent );
			span.parentNode.replaceChild( textNode, span );
		} );

		let text = clone.textContent || clone.innerText || '';

		// Decode HTML entities
		text = text
			.replace( /&lt;/g, '<' )
			.replace( /&gt;/g, '>' )
			.replace( /&amp;/g, '&' )
			.replace( /&quot;/g, '"' )
			.replace( /&#39;/g, "'" )
			.replace( /&nbsp;/g, ' ' );

		return text.trim();
	};

	/**
	 * Process Chart.js blocks in editor preview
	 * Converts rendered charts to images for React state preservation
	 */
	const processChartBlocks = useCallback( async ( container ) => {
		if ( ! container ) return 0;

		// Find chart code blocks - only target elements with language-chart class
		// Chart blocks are NOT processed by Shiki, so they won't have pre.shiki wrapper
		const chartBlocks = container.querySelectorAll(
			'code.language-chart:not([data-gfmr-chart-processed]), ' +
			'code.language-chart-pro:not([data-gfmr-chart-processed])'
		);

		if ( chartBlocks.length === 0 ) {
			return 0;
		}

		logDebug( `[Chart.js] Found ${ chartBlocks.length } chart blocks` );

		// Load Chart.js library if not available
		if ( ! window.Chart ) {
			try {
				const pluginUrl = window.wpGfmConfig?.pluginUrl || '/wp-content/plugins/markdown-renderer-for-github/';
				await new Promise( ( resolve, reject ) => {
					const script = document.createElement( 'script' );
					script.src = pluginUrl + 'assets/libs/chartjs/chart.umd.min.js';
					script.onload = resolve;
					script.onerror = reject;
					document.head.appendChild( script );
				} );
				logDebug( '[Chart.js] Library loaded' );
			} catch ( error ) {
				console.warn( '[WP GFM Editor] Failed to load Chart.js:', error );
				return 0;
			}
		}

		// Theme helper (charts always use light mode for consistency)
		const getChartTheme = () => {
			// Charts always use light theme colors
			const colors = { text: '#24292f', grid: '#d0d7de', border: '#d0d7de', bg: '#ffffff' };
			return { isDark: false, colors };
		};

		// Deep merge helper - preserves user-specified values
		const deepMerge = ( target, source ) => {
			const result = { ...target };
			for ( const key in source ) {
				if ( source[ key ] && typeof source[ key ] === 'object' && ! Array.isArray( source[ key ] ) ) {
					result[ key ] = deepMerge( target[ key ] || {}, source[ key ] );
				} else if ( target[ key ] === undefined ) {
					// Only apply source value if target doesn't have it
					result[ key ] = source[ key ];
				}
			}
			return result;
		};

		// Get theme info for all charts
		const themeInfo = getChartTheme();

		// Apply to Chart.js defaults (for any charts that don't get explicit options)
		window.Chart.defaults.color = themeInfo.colors.text;
		window.Chart.defaults.borderColor = themeInfo.colors.border;
		if ( window.Chart.defaults.scale ) {
			window.Chart.defaults.scale.grid = window.Chart.defaults.scale.grid || {};
			window.Chart.defaults.scale.grid.color = themeInfo.colors.grid;
			window.Chart.defaults.scale.ticks = window.Chart.defaults.scale.ticks || {};
			window.Chart.defaults.scale.ticks.color = themeInfo.colors.text;
		}
		if ( window.Chart.defaults.plugins?.legend?.labels ) {
			window.Chart.defaults.plugins.legend.labels.color = themeInfo.colors.text;
		}
		if ( window.Chart.defaults.plugins?.title ) {
			window.Chart.defaults.plugins.title.color = themeInfo.colors.text;
		}

		// Process each chart block and convert to image
		let processedCount = 0;
		for ( const block of chartBlocks ) {
			try {
				block.setAttribute( 'data-gfmr-chart-processed', 'true' );

				// Extract plain text from potentially Shiki-highlighted code
				const content = extractChartCodeFromShiki( block );
				logDebug( '[Chart.js] Extracted content:', content.substring( 0, 100 ) );

				const config = JSON.parse( content );

				// Theme options for chart (applied only if not user-specified)
				const chartThemeOptions = {
					responsive: true,
					maintainAspectRatio: true,
					animation: false, // Editor-specific: render immediately
					color: themeInfo.colors.text,
					borderColor: themeInfo.colors.border,
					scales: {
						x: {
							grid: { color: themeInfo.colors.grid },
							ticks: { color: themeInfo.colors.text }
						},
						y: {
							grid: { color: themeInfo.colors.grid },
							ticks: { color: themeInfo.colors.text }
						}
					},
					plugins: {
						legend: { labels: { color: themeInfo.colors.text } },
						title: { color: themeInfo.colors.text },
						tooltip: {
							backgroundColor: 'rgba(255, 255, 255, 0.95)',
							titleColor: themeInfo.colors.text,
							bodyColor: themeInfo.colors.text,
							borderColor: themeInfo.colors.border,
							borderWidth: 1
						}
					}
				};

				// Merge: user options override theme options
				config.options = deepMerge( chartThemeOptions, config.options || {} );

				// Create temporary off-screen canvas for rendering
				const tempContainer = document.createElement( 'div' );
				tempContainer.style.width = '600px';
				tempContainer.style.height = '400px';
				tempContainer.style.position = 'absolute';
				tempContainer.style.left = '-9999px';
				document.body.appendChild( tempContainer );

				const canvas = document.createElement( 'canvas' );
				tempContainer.appendChild( canvas );

				// Render chart with Chart.js
				const chart = new window.Chart( canvas.getContext( '2d' ), config );

				// Convert to image (similar to Mermaid's approach)
				const imageDataUrl = chart.toBase64Image( 'image/png', 1.0 );

				// Clean up chart and temporary DOM
				chart.destroy();
				document.body.removeChild( tempContainer );

				// Create image element to replace code block (light theme fixed)
				const imgContainer = document.createElement( 'div' );
				imgContainer.className = 'gfmr-chart-container gfmr-chart-image';
				imgContainer.innerHTML = `<img src="${ imageDataUrl }" alt="Chart" style="max-width: 100%; height: auto;" />`;

				// Replace pre element with image
				const targetElement = block.closest( 'pre' ) || block.parentElement;
				if ( targetElement?.parentNode ) {
					targetElement.parentNode.replaceChild( imgContainer, targetElement );
				}

				processedCount++;
			} catch ( error ) {
				console.warn( '[WP GFM Editor] Chart render error:', error );
				// Show error message (this will be preserved in processedHtml)
				const errorDiv = document.createElement( 'div' );
				errorDiv.className = 'gfmr-chart-error';
				errorDiv.innerHTML = `<div class="gfmr-chart-error-content"><strong>Chart Error:</strong> ${ error.message }</div>`;
				const targetElement = block.closest( 'pre' ) || block.parentElement;
				if ( targetElement?.parentNode ) {
					targetElement.parentNode.replaceChild( errorDiv, targetElement );
				}
				processedCount++; // Count errors too (DOM was modified)
			}
		}

		return processedCount;
	}, [] );

	// Extract plain text from Shiki-highlighted code, preserving newlines
	// Uses same approach as frontend extractPlainTextFromHighlighted for consistency
	const extractMermaidCodeFromShiki = ( element ) => {
		// Clone element and replace all spans with text nodes (preserves newlines in HTML structure)
		const clone = element.cloneNode( true );
		const spans = clone.querySelectorAll( 'span' );
		spans.forEach( ( span ) => {
			const textNode = document.createTextNode( span.textContent );
			span.parentNode.replaceChild( textNode, span );
		} );

		let text = clone.textContent || clone.innerText || '';

		// Decode HTML entities (consistent with front-end)
		text = text
			.replace( /&lt;/g, '<' )
			.replace( /&gt;/g, '>' )
			.replace( /&amp;/g, '&' )
			.replace( /&quot;/g, '"' )
			.replace( /&#39;/g, "'" );

		return text;
	};

	// Unified Mermaid processing function for block editor (alternative to frontend version)
	const processMermaidBlocksLocal = useCallback( async ( container, context = 'editor', bgColor = null ) => {
		logDebug( 'processMermaidBlocksLocal called' );
		logDebug( 'Container HTML preview:', container?.innerHTML?.substring( 0, 200 ) );

		if ( ! container ) {
			return 0;
		}

		// Check if Mermaid is available
		const mermaidAvailable = await loadMermaid();
		if ( ! mermaidAvailable ) {
			logDebug( 'Mermaid not available' );
			return 0;
		}
		logDebug( 'Mermaid is available' );

		// Phase 1: Class-based detection
		logDebug( 'Phase 1: Starting class-based detection' );
		let mermaidBlocks = [];
		for ( const selector of MERMAID_CLASS_SELECTORS ) {
			const blocks = container.querySelectorAll( selector );
			if ( blocks.length > 0 ) {
				mermaidBlocks = Array.from( blocks );
				logDebug( `Phase 1: Found ${ blocks.length } blocks with selector "${ selector }"` );
				break;
			}
		}
		logDebug( `Phase 1 result: ${ mermaidBlocks.length } blocks` );

		// Phase 2: Content-based detection (when class-based detection finds nothing)
		if ( mermaidBlocks.length === 0 ) {
			logDebug( 'Phase 2: Starting content-based detection' );
			for ( const selector of MERMAID_FALLBACK_SELECTORS ) {
				const codeElements = container.querySelectorAll( selector );
				logDebug( `Selector "${ selector }" found ${ codeElements.length } elements` );
				for ( const codeElement of codeElements ) {
					const cleanContent = ( codeElement.textContent || '' ).trim().replace( /\s+/g, ' ' );
					logDebug( 'Content preview:', cleanContent.substring( 0, 60 ) );
					if ( isMermaidContent( cleanContent ) ) {
						logDebug( '✅ Mermaid content detected!' );
						mermaidBlocks.push( codeElement );
					}
				}
				if ( mermaidBlocks.length > 0 ) break;
			}
			logDebug( `Phase 2 result: ${ mermaidBlocks.length } blocks found` );
		}

		if ( mermaidBlocks.length === 0 ) {
			return 0;
		}

		let processedCount = 0;
		for ( let index = 0; index < mermaidBlocks.length; index++ ) {
			const codeBlock = mermaidBlocks[ index ];
			
			try {
				const mermaidCode = extractMermaidCodeFromShiki( codeBlock );
				const preElement = codeBlock.parentElement;
				
				if ( ! preElement || ! mermaidCode?.trim() ) {
					continue;
				}

				// Create Mermaid container with unified container style
				const mermaidContainer = document.createElement( 'div' );
				mermaidContainer.className = 'gfmr-mermaid-container';
				mermaidContainer.style.cssText = getMermaidContainerStyle( bgColor || 'transparent' );

				const mermaidId = `gfmr-editor-mermaid-${Date.now()}-${index}`;
				const mermaidDiv = document.createElement( 'div' );
				mermaidDiv.id = mermaidId;
				mermaidDiv.className = 'gfmr-mermaid';

				mermaidContainer.appendChild( mermaidDiv );
				preElement.parentNode.replaceChild( mermaidContainer, preElement );

				// Unified rendering process (local version)
				try {
					const renderResult = await window.mermaid.render( mermaidId + '-svg', mermaidCode.trim() );

					let svgContent = '';
					if ( renderResult && typeof renderResult === 'object' && renderResult.svg ) {
						// Mermaid v10+ new API
						svgContent = renderResult.svg;
					} else if ( typeof renderResult === 'string' ) {
						// Old API or string response
						svgContent = renderResult;
					} else {
						throw new Error( 'Invalid render result format' );
					}

					// Insert SVG
					mermaidDiv.innerHTML = svgContent;

					// SVG optimization is handled via CSS (.gfmr-mermaid svg in mermaid-styles.css)
					// which applies: display: block, margin: 0 auto, max-width: 100%, height: auto
					// This separation of concerns keeps styling in CSS rather than JS.

					mermaidDiv.setAttribute( 'data-mermaid-rendered', 'true' );
					mermaidDiv.setAttribute( 'data-mermaid-context', context );
					processedCount++;

				} catch ( renderError ) {
					console.error( `[WP GFM Editor] ❌ Render error for block ${index}:`, renderError );
					mermaidDiv.innerHTML = buildMermaidErrorHTML( renderError.message, mermaidCode );
				}
				
			} catch ( error ) {
				console.error( `[WP GFM Editor] ❌ Processing error for block ${index}:`, error );
			}
		}

		return processedCount;
	}, [ loadMermaid, mermaidBgColor ] );

	// Mermaid processing function (unified version: ensures complete consistency between preview and content display)
	const processMermaidBlocks = useCallback( async ( container, bgColor = null ) => {
		logDebug( 'processMermaidBlocks called, container:', container?.tagName, container?.className );

		if ( ! container ) {
			logDebugWarn( 'No container provided for Mermaid processing' );
			return;
		}

		// Use unified Mermaid block processing function if available
		if ( window.wpGfmProcessMermaidBlocksUnified ) {
			logDebug( 'Using unified Mermaid processing for editor' );

			try {
				// Verify Mermaid loading
				const loaded = await loadMermaid();

				if ( ! loaded ) {
					logDebugWarn( 'Mermaid not available, skipping processing' );
					return;
				}

				// Execute unified processing
				const processedCount = await window.wpGfmProcessMermaidBlocksUnified( container, 'editor' );

				logDebug( `✅ Unified Mermaid processing completed: ${ processedCount } blocks processed` );

				return processedCount;
			} catch ( error ) {
				console.error( '[WP GFM Editor] Unified Mermaid processing failed:', error );
				// Continue with fallback processing
			}
		}

		// Local processing: block editor-specific unified processing
		return processMermaidBlocksLocal( container, 'editor', bgColor );
	}, [ loadMermaid, mermaidBgColor, processMermaidBlocksLocal ] );

	// Mermaid error display (unified version: ensures complete consistency between preview and content display)
	const showMermaidError = ( container, error, code, index ) => {
		// Use unified error handler if available
		if ( window.showEnhancedMermaidError ) {
			window.showEnhancedMermaidError( container, error, code, index );
			return;
		}

		// Fallback: emulate unified error display using helper
		container.innerHTML = buildMermaidErrorHTML( error.message, code );
	};

	// Keep processMermaidBlocksRef updated with latest function reference
	// This allows useEffect to use stable dependency array while accessing latest function
	useEffect( () => {
		processMermaidBlocksRef.current = processMermaidBlocks;
	}, [ processMermaidBlocks ] );

	// Shiki initialization (on component mount)
	useEffect( () => {
		if ( ! shikiLoaded ) {
			loadShikiForEditor().then( ( highlighter ) => {
				if ( highlighter ) {
					setShikiLoaded( true );
					console.log( '[WP GFM Editor] Shiki initialized for editor' );

					// Re-render existing content after Shiki loading completes
					if ( content ) {
						renderMarkdown( content );
					}
				}
			} ).catch( ( error ) => {
				console.warn( '[WP GFM Editor] Shiki initialization failed:', error );
			} );
		}
	}, [] );

	// Initial rendering
	useEffect( () => {
		if ( content ) {
			renderMarkdown( content );
		}
	}, [] );

	// Re-render when showFrontmatter changes
	useEffect( () => {
		if ( content ) {
			renderMarkdown( content );
		}
	}, [ showFrontmatter ] );

	/**
	 * Apply diff highlighting classes to HTML string
	 * Adds CSS classes to lines based on their diff prefix (+, -, @@)
	 *
	 * @param {string} html - HTML string containing Shiki-highlighted code blocks
	 * @return {string} - Modified HTML with diff classes applied
	 */
	const applyDiffHighlightingToHtml = ( html ) => {
		// Skip if no Shiki code blocks
		if ( ! html || ! html.includes( 'class="shiki"' ) ) {
			return html;
		}

		const tempDiv = document.createElement( 'div' );
		tempDiv.innerHTML = html;

		const preElements = tempDiv.querySelectorAll( 'pre.shiki' );

		preElements.forEach( ( pre ) => {
			const codeElement = pre.querySelector( 'code' );
			if ( ! codeElement ) {
				return;
			}

			const lines = codeElement.querySelectorAll( '.line' );

			let hasDiff = false;

			lines.forEach( ( line ) => {
			const text = ( line.textContent || '' ).trim();

			// Remove empty lines (including whitespace-only lines)
			if ( ! text ) {
				line.remove();
				return;
			}

			// Skip file header lines (+++, ---, diff --git, index)
			if ( text.startsWith( '+' ) && ! text.startsWith( '+++' ) ) {
				line.classList.add( 'diff', 'add' );
				hasDiff = true;
			} else if ( text.startsWith( '-' ) && ! text.startsWith( '---' ) ) {
				line.classList.add( 'diff', 'remove' );
				hasDiff = true;
			} else if ( text.startsWith( '@@' ) ) {
				line.classList.add( 'diff', 'hunk' );
				hasDiff = true;
			}
		} );

			if ( hasDiff ) {
				pre.classList.add( 'has-diff' );
			}
		} );

		return tempDiv.innerHTML;
	};

	// Markdown rendering (indent block preprocessing + Shiki integration support)
	const renderMarkdown = useCallback( async ( markdownContent ) => {
		if ( ! markdownContent ) {
			setRenderedHtml( '' );
			setAttributes( { html: '', shikiTheme: '', frontmatterData: {} } );
			setProcessedHtml( null );
			return;
		}

		try {
			// Parse frontmatter before rendering
			const { frontmatter, body } = parseFrontMatter( markdownContent );

			// Update frontmatter attributes
			setAttributes( { frontmatterData: frontmatter } );

			// Wait for Shiki to load if not loaded yet
			if ( ! shikiLoaded && ! editorShikiHighlighter ) {
				await loadShikiForEditor();
				setShikiLoaded( true );
			}

			// Render body only (without frontmatter section)
			let bodyHtml;
			if ( window.wpGfmRenderWithIndentPreprocessing ) {
				console.log( '[WP GFM Editor] Using indent block preprocessing for editor' );
				bodyHtml = window.wpGfmRenderWithIndentPreprocessing( body, md );
			} else {
				console.warn( '[WP GFM Editor] Indent preprocessor not available, using fallback' );
				bodyHtml = md.render( body );
			}

			// Generate frontmatter header HTML
			const fmHtml = showFrontmatter ? renderFrontmatterHeader( frontmatter ) : '';

			// Combine frontmatter header and body
			let html = fmHtml + bodyHtml;

			// Apply diff highlighting to code blocks
			html = applyDiffHighlightingToHtml( html );

			// Get the current Shiki theme used for rendering
			let currentShikiTheme = window.wpGfmConfig?.theme?.shiki_theme ??
				(window.matchMedia?.('(prefers-color-scheme: dark)').matches ? 'github-dark' : 'github-light');
			if ( currentShikiTheme === 'auto' ) {
				const isDark = window.matchMedia &&
					window.matchMedia( '(prefers-color-scheme: dark)' ).matches;
				currentShikiTheme = isDark ? 'github-dark' : 'github-light';
			}

			setRenderedHtml( html );
			setAttributes( { html, shikiTheme: currentShikiTheme } );

			// Only reset processedHtml if HTML actually changed
			// This prevents losing Mermaid processing results on unrelated re-renders
			if ( lastRenderedHtmlRef.current !== html ) {
				setProcessedHtml( null );
				lastRenderedHtmlRef.current = html;
			}

			// Note: Mermaid processing is handled by useEffect on renderedHtml change
			// No need for additional setTimeout here as it causes duplicate processing
		} catch ( error ) {
			console.error( '[WP GFM Renderer] Rendering error:', error );
			setRenderedHtml( '<p>An error occurred while rendering Markdown</p>' );
			setProcessedHtml( null );
		}
	}, [ shikiLoaded, showFrontmatter, setRenderedHtml, setProcessedHtml ] );

	// Mermaid processing on preview mode switch
	// Note: processMermaidBlocks is accessed via ref to avoid dependency chain issues
	useEffect( () => {
		let cleanup = false;
		let timeoutId = null;

		logDebug( '🔍 useEffect triggered:', { isPreview, renderedHtml: !!renderedHtml, previewRefExists: !!previewRef.current } );

		if ( isPreview && renderedHtml && previewRef.current ) {
			logDebug( '✅ All conditions met for Mermaid processing' );

			// Execute Mermaid processing when switched to preview mode
			// Wait more reliably for DOM update (wait for Shiki processing completion)
			timeoutId = setTimeout( async () => {
				// Skip if cleanup was called or already processing
				if ( cleanup || isMermaidProcessing.current ) {
					logDebug( '⏭️ Skipping: cleanup=', cleanup, 'isMermaidProcessing=', isMermaidProcessing.current );
					return;
				}

				isMermaidProcessing.current = true;
				logDebug( '⏰ previewRef.current exists in setTimeout:', !!previewRef.current );

				try {
					if ( previewRef.current && processMermaidBlocksRef.current ) {
						logDebug( '🎯 previewRef.current type:', typeof previewRef.current );
						logDebug( '🎯 previewRef.current tagName:', previewRef.current?.tagName );
						logDebug( '🎯 previewRef.current innerHTML length:', previewRef.current?.innerHTML?.length || 0 );

						// Use ref to access latest function without dependency
						const mermaidResult = await processMermaidBlocksRef.current( previewRef.current, mermaidBgColor );
						logDebug( '🎯 processMermaidBlocks call completed successfully, result:', mermaidResult );

						// Process Chart.js blocks after Mermaid (before saving processed HTML)
						let chartResult = 0;
						if ( ! cleanup && previewRef.current ) {
							try {
								chartResult = await processChartBlocks( previewRef.current );
								logDebug( '[Chart.js] Processing completed, result:', chartResult );
							} catch ( chartError ) {
								console.warn( '[WP GFM Editor] Chart processing error:', chartError );
							}
						}

						// Save processed HTML only once after both Mermaid and Chart.js are done
						// This preserves both Mermaid SVGs and Chart.js images in React state
						if ( ! cleanup && ( mermaidResult > 0 || chartResult > 0 ) && previewRef.current ) {
							setProcessedHtml( previewRef.current.innerHTML );
						}
					} else {
						logDebug( '❌ previewRef.current or processMermaidBlocksRef.current is null/undefined' );
					}
				} catch ( error ) {
					console.error( '[WP GFM Editor] ❌ processMermaidBlocks error:', error );
					console.error( '[WP GFM Editor] ❌ Error stack:', error.stack );
				} finally {
					isMermaidProcessing.current = false;
				}
			}, 400 );
		} else {
			logDebug( '❌ Conditions not met:', {
				isPreview,
				renderedHtml: !!renderedHtml,
				previewRefExists: !!previewRef.current
			} );
		}

		// Cleanup function: clear timeout and reset flags on unmount or dependency change
		return () => {
			cleanup = true;
			if ( timeoutId ) {
				clearTimeout( timeoutId );
			}
			isMermaidProcessing.current = false;
		};
	}, [ isPreview, renderedHtml, mermaidBgColor ] );  // Removed processMermaidBlocks from deps

	// Content change handler (async support, multilingual aware)
	const handleChange = async ( event ) => {
		const newContent = event.target.value;

		if ( isMultilingualMode ) {
			// Update content in the current language
			const updatedLanguages = {
				...languages,
				[ currentEditingLang ]: {
					...( languages[ currentEditingLang ] || {} ),
					content: newContent,
				},
			};

			// Render markdown and update HTML for current language
			await renderMarkdownForLanguage( newContent, currentEditingLang, updatedLanguages );
		} else {
			// Legacy single-language mode
			setAttributes( { content: newContent } );
			await renderMarkdown( newContent );
		}
	};

	// Render markdown for a specific language
	const renderMarkdownForLanguage = async ( markdownContent, lang, updatedLanguages ) => {
		if ( ! markdownContent ) {
			const newLanguages = {
				...updatedLanguages,
				[ lang ]: { content: '', html: '' },
			};
			setAttributes( { languages: newLanguages } );
			setRenderedHtml( '' );
			setProcessedHtml( null );
			return;
		}

		try {
			// Parse frontmatter
			const { frontmatter, body } = parseFrontMatter( markdownContent );
			setAttributes( { frontmatterData: frontmatter } );

			// Wait for Shiki
			if ( ! shikiLoaded && ! editorShikiHighlighter ) {
				await loadShikiForEditor();
				setShikiLoaded( true );
			}

			// Render markdown
			let bodyHtml;
			if ( window.wpGfmRenderWithIndentPreprocessing ) {
				bodyHtml = window.wpGfmRenderWithIndentPreprocessing( body, md );
			} else {
				bodyHtml = md.render( body );
			}

			const fmHtml = showFrontmatter ? renderFrontmatterHeader( frontmatter ) : '';
			let generatedHtml = fmHtml + bodyHtml;
			generatedHtml = applyDiffHighlightingToHtml( generatedHtml );

			// Get Shiki theme
			let currentShikiTheme = window.wpGfmConfig?.theme?.shiki_theme ??
				(window.matchMedia?.('(prefers-color-scheme: dark)').matches ? 'github-dark' : 'github-light');
			if ( currentShikiTheme === 'auto' ) {
				const isDark = window.matchMedia &&
					window.matchMedia( '(prefers-color-scheme: dark)' ).matches;
				currentShikiTheme = isDark ? 'github-dark' : 'github-light';
			}

			// Update languages with rendered HTML
			const newLanguages = {
				...updatedLanguages,
				[ lang ]: {
					content: markdownContent,
					html: generatedHtml,
				},
			};

			setAttributes( {
				languages: newLanguages,
				shikiTheme: currentShikiTheme,
				// Also keep top-level content/html updated with default language for compatibility
				content: newLanguages[ defaultLanguage ]?.content || '',
				html: newLanguages[ defaultLanguage ]?.html || '',
			} );

			setRenderedHtml( generatedHtml );

			if ( lastRenderedHtmlRef.current !== generatedHtml ) {
				setProcessedHtml( null );
				lastRenderedHtmlRef.current = generatedHtml;
			}
		} catch ( error ) {
			console.error( '[WP GFM Renderer] Rendering error:', error );
			setRenderedHtml( '<p>An error occurred while rendering Markdown</p>' );
			setProcessedHtml( null );
		}
	};

	// Initialize multilingual mode (convert single-language to multilingual)
	const initializeMultilingualMode = useCallback( () => {
		const siteLang = window.wpGfmConfig?.siteLanguage || 'en';

		// If already in multilingual mode, don't reinitialize
		if ( availableLanguages.length > 0 ) {
			return;
		}

		// Convert current content to multilingual format
		const newLanguages = {
			[ siteLang ]: {
				content: content || '',
				html: html || '',
			},
		};

		setAttributes( {
			languages: newLanguages,
			defaultLanguage: siteLang,
			availableLanguages: [ siteLang ],
			showLanguageSwitcher: false,
		} );

		setCurrentEditingLang( siteLang );
	}, [ content, html, availableLanguages.length, setAttributes ] );

	// Add a new language
	const handleAddLanguage = useCallback( ( langCode, copyFromLang = '' ) => {
		if ( ! isValidLanguageCode( langCode ) ) {
			console.error( '[GFMR] Invalid language code:', langCode );
			return;
		}

		if ( availableLanguages.includes( langCode ) ) {
			console.warn( '[GFMR] Language already exists:', langCode );
			return;
		}

		if ( availableLanguages.length >= MAX_LANGUAGES ) {
			console.warn( '[GFMR] Maximum language limit reached:', MAX_LANGUAGES );
			return;
		}

		// Initialize multilingual mode if needed
		if ( ! isMultilingualMode ) {
			initializeMultilingualMode();
		}

		// Create new language content (optionally copy from existing)
		let newLangContent = { content: '', html: '' };
		if ( copyFromLang && languages[ copyFromLang ] ) {
			newLangContent = { ...languages[ copyFromLang ] };
		}

		const updatedLanguages = {
			...languages,
			[ langCode ]: newLangContent,
		};

		const updatedAvailable = [ ...availableLanguages, langCode ];

		setAttributes( {
			languages: updatedLanguages,
			availableLanguages: updatedAvailable,
			showLanguageSwitcher: updatedAvailable.length > 1,
		} );

		// Switch to the new language
		setCurrentEditingLang( langCode );
		setIsAddLanguageModalOpen( false );
	}, [ availableLanguages, languages, isMultilingualMode, initializeMultilingualMode, setAttributes ] );

	// Delete a language
	const handleDeleteLanguage = useCallback( ( langCode ) => {
		if ( langCode === defaultLanguage ) {
			alert( __( 'Cannot delete the default language. Please change the default first.', 'markdown-renderer-for-github' ) );
			return;
		}

		if ( ! window.confirm(
			__( 'Are you sure you want to delete this translation? This cannot be undone.', 'markdown-renderer-for-github' )
		) ) {
			return;
		}

		const { [ langCode ]: removed, ...remainingLanguages } = languages;
		const updatedAvailable = availableLanguages.filter( l => l !== langCode );

		setAttributes( {
			languages: remainingLanguages,
			availableLanguages: updatedAvailable,
			showLanguageSwitcher: updatedAvailable.length > 1,
		} );

		// Switch to default language if we were editing the deleted language
		if ( currentEditingLang === langCode ) {
			setCurrentEditingLang( defaultLanguage );
		}
	}, [ defaultLanguage, languages, availableLanguages, currentEditingLang, setAttributes ] );

	// Switch editing language
	const handleLanguageSwitch = useCallback( ( langCode ) => {
		// availableLanguagesまたはlanguagesオブジェクトにキーが存在すればOK
		if ( ! availableLanguages.includes( langCode ) && ! languages[ langCode ] ) return;

		setCurrentEditingLang( langCode );

		// Update rendered HTML for the new language
		const langContent = languages[ langCode ];
		if ( langContent ) {
			setRenderedHtml( langContent.html || '' );
			setProcessedHtml( null );
		}
	}, [ availableLanguages, languages ] );

	// Block properties
	const blockProps = useBlockProps( {
		className: 'gfmr-markdown-block'
	} );

	const previewHtml = processedHtml || renderedHtml;

	return (
		<>
			<BlockControls>
				<ToolbarGroup>
					<ToolbarButton
						icon={ isPreview ? 'edit' : 'visibility' }
						label={ isPreview ? __( 'Edit', 'markdown-renderer-for-github' ) : __( 'Preview', 'markdown-renderer-for-github' ) }
						onClick={ () => setIsPreview( ! isPreview ) }
						isActive={ isPreview }
					/>
					{ isImageInsertEnabled && (
						<MediaUploadCheck>
							<MediaUpload
								onSelect={ handleImageInsert }
								allowedTypes={ [ 'image' ] }
								multiple={ false }
								render={ ( { open } ) => (
								<ToolbarButton
									icon="format-image"
									label={ __( 'Insert Image', 'markdown-renderer-for-github' ) }
									disabled={ isPreview }
									onClick={ () => {
										if ( isPreview ) {
											return;
										}
										// Save cursor position before opening media library
										const textarea = textareaRef.current;
										if ( textarea ) {
											cursorPositionRef.current = {
												start: textarea.selectionStart,
												end: textarea.selectionEnd
											};
										}
										open();
									} }
								/>
								) }
							/>
						</MediaUploadCheck>
					) }
				</ToolbarGroup>
			</BlockControls>

			<InspectorControls>
				{ /* Language Settings Panel */ }
				<PanelBody
					title={ __( 'Language Settings', 'markdown-renderer-for-github' ) }
					initialOpen={ isMultilingualMode }
				>
					{ ! isMultilingualMode ? (
						<>
							<p className="components-base-control__help" style={ { marginBottom: '12px' } }>
								{ __( 'Enable multilingual mode to write content in multiple languages.', 'markdown-renderer-for-github' ) }
							</p>
							<Button
								variant="secondary"
								onClick={ () => {
									initializeMultilingualMode();
									setIsAddLanguageModalOpen( true );
								} }
							>
								{ __( 'Enable Multilingual', 'markdown-renderer-for-github' ) }
							</Button>
						</>
					) : (
						<>
							<SelectControl
								label={ __( 'Default Language', 'markdown-renderer-for-github' ) }
								value={ defaultLanguage }
								options={ availableLanguages.map( l => ( {
									label: l.toUpperCase(),
									value: l,
								} ) ) }
								onChange={ ( value ) => {
									setAttributes( {
										defaultLanguage: value,
										// Update top-level content/html with new default
										content: languages[ value ]?.content || '',
										html: languages[ value ]?.html || '',
									} );
								} }
							/>
							<ToggleControl
								label={ __( 'Show Language Switcher', 'markdown-renderer-for-github' ) }
								checked={ showLanguageSwitcher }
								onChange={ ( value ) => setAttributes( { showLanguageSwitcher: value } ) }
								help={ __( 'Display language buttons on the frontend', 'markdown-renderer-for-github' ) }
							/>
							{ availableLanguages.length > 0 && (
								<div className="gfmr-language-list" style={ { marginTop: '16px' } }>
									<p className="components-base-control__label" style={ { marginBottom: '8px' } }>
										{ __( 'Languages', 'markdown-renderer-for-github' ) } ({ availableLanguages.length }/{ MAX_LANGUAGES })
									</p>
									{ availableLanguages.map( lang => (
										<div
											key={ lang }
											style={ {
												display: 'flex',
												justifyContent: 'space-between',
												alignItems: 'center',
												padding: '4px 0',
												borderBottom: '1px solid #ddd',
											} }
										>
											<span>
												{ lang.toUpperCase() }
												{ lang === defaultLanguage && (
													<span style={ { color: '#757575', marginLeft: '8px', fontSize: '12px' } }>
														({ __( 'default', 'markdown-renderer-for-github' ) })
													</span>
												) }
											</span>
											{ lang !== defaultLanguage && (
												<Button
													isDestructive
													isSmall
													onClick={ () => handleDeleteLanguage( lang ) }
												>
													{ __( 'Delete', 'markdown-renderer-for-github' ) }
												</Button>
											) }
										</div>
									) ) }
									{ availableLanguages.length < MAX_LANGUAGES && (
										<Button
											variant="secondary"
											isSmall
											onClick={ () => setIsAddLanguageModalOpen( true ) }
											style={ { marginTop: '12px' } }
										>
											{ __( '+ Add Language', 'markdown-renderer-for-github' ) }
										</Button>
									) }
								</div>
							) }
						</>
					) }
				</PanelBody>

				<PanelBody
					title={ __( 'Frontmatter Settings', 'markdown-renderer-for-github' ) }
					initialOpen={ false }
				>
					<ToggleControl
						label={ __( 'Show Frontmatter Header', 'markdown-renderer-for-github' ) }
						checked={ showFrontmatter }
						onChange={ ( value ) => setAttributes( { showFrontmatter: value } ) }
						help={ __( 'Display YAML frontmatter metadata as a formatted header', 'markdown-renderer-for-github' ) }
					/>
				</PanelBody>

				<PanelBody
					title={ __( 'Mermaid Settings', 'markdown-renderer-for-github' ) }
					initialOpen={ false }
				>
					<PanelRow>
						<div style={ { width: '100%' } }>
							<label
								htmlFor="mermaid-bg-color"
								style={ {
									display: 'block',
									marginBottom: '8px',
									fontWeight: '500'
								} }
							>
								{ __( 'Background Color', 'markdown-renderer-for-github' ) }
							</label>
							<ColorPicker
								color={ (() => {
									const currentColor = mermaidBgColor || 'transparent';
									// ColorPicker constraint: fully transparent (#RRGGBB00) locks alpha to 0, so for 9-digit HEX8 ending in 00, replace with FF for display
									if (currentColor.length === 9 && currentColor.endsWith('00')) {
										const colorWithoutAlpha = currentColor.slice(0, 7);
										const fixedColor = colorWithoutAlpha + 'FF';
										return fixedColor;
									}
									return currentColor;
								})() }
								onChange={ ( color ) => {
									// ColorPicker returns HEX8 format (#RRGGBBAA) when enableAlpha is enabled. Normalize fully transparent (#00000000) to 'transparent' string
									const colorValue = color === '#00000000' ? 'transparent' : color;
									setAttributes( { mermaidBgColor: colorValue } );
								} }
								enableAlpha
							/>
							<div
								className="components-base-control__help"
								style={ {
									marginTop: '8px',
									fontSize: '12px',
									color: '#757575',
									fontStyle: 'italic'
								} }
							>
								💡 { __( 'After changing colors, return to edit mode and preview again', 'markdown-renderer-for-github' ) }
							</div>
						</div>
					</PanelRow>
				</PanelBody>
			</InspectorControls>

			{ /* Add Language Modal */ }
			{ isAddLanguageModalOpen && (
				<AddLanguageModal
					onAdd={ handleAddLanguage }
					onClose={ () => setIsAddLanguageModalOpen( false ) }
					existingLanguages={ availableLanguages }
				/>
			) }

			<div { ...blockProps }>
				<div className="gfmr-markdown-header">
					<span className="gfmr-markdown-label">
						{ __( 'Markdown', 'markdown-renderer-for-github' ) }
					</span>
					<span className="gfmr-markdown-mode">
						{ isPreview ? __( 'Preview', 'markdown-renderer-for-github' ) : __( 'Edit', 'markdown-renderer-for-github' ) }
					</span>
				</div>

				{ /* Language Tabs (shown when multilingual mode is active) */ }
				{ ( isMultilingualMode || Object.keys( languages ).length > 0 ) && (
					<div className="gfmr-language-tabs">
						{ ( availableLanguages.length > 0 ? availableLanguages : Object.keys( languages ) ).map( lang => (
							<button
								key={ lang }
								type="button"
								className={ `gfmr-lang-tab ${ currentEditingLang === lang ? 'active' : '' }` }
								onClick={ () => handleLanguageSwitch( lang ) }
								disabled={ isPreview }
							>
								{ lang.toUpperCase() }
							</button>
						) ) }
						{ availableLanguages.length < MAX_LANGUAGES && (
							<button
								type="button"
								className="gfmr-lang-add"
								onClick={ () => setIsAddLanguageModalOpen( true ) }
								disabled={ isPreview }
								title={ __( 'Add Language', 'markdown-renderer-for-github' ) }
							>
								+
							</button>
						) }
					</div>
				) }

				{ ! isPreview ? (
					<ResizableBox
						size={ { height: 300 } }
						minHeight="100"
						enable={ {
							top: false,
							right: false,
							bottom: true,
							left: false,
							topRight: false,
							bottomRight: false,
							bottomLeft: false,
							topLeft: false,
						} }
						className="gfmr-markdown-editor-wrapper"
					>
						<textarea
							ref={ textareaRef }
							className="gfmr-markdown-editor"
							value={ getCurrentContent() }
							onChange={ handleChange }
							placeholder={ __( 'Enter Markdown here...\n\n# Heading\n**Bold** *Italic*\n- List item', 'markdown-renderer-for-github' ) }
							spellCheck="false"
						/>
					</ResizableBox>
				) : (
					<div className={ `gfmr-markdown-preview gfmr-markdown-rendered-preview ${ themeClass }` } ref={ previewRef }>
						{ previewHtml ? (
							<RawHTML>
								{ previewHtml }
							</RawHTML>
						) : (
							<p className="gfmr-markdown-empty">
								{ __( 'No Markdown content to preview', 'markdown-renderer-for-github' ) }
							</p>
						) }
					</div>
				) }
			</div>
		</>
	);
}

/**
 * Add Language Modal Component
 */
function AddLanguageModal( { onAdd, onClose, existingLanguages } ) {
	const [ selectedLang, setSelectedLang ] = useState( '' );
	const [ customLang, setCustomLang ] = useState( '' );
	const [ copyFrom, setCopyFrom ] = useState( '' );
	const [ error, setError ] = useState( '' );

	const handleAdd = () => {
		const langCode = selectedLang === 'custom'
			? customLang.toLowerCase().trim()
			: selectedLang;

		// Validate
		if ( ! langCode ) {
			setError( __( 'Please select a language', 'markdown-renderer-for-github' ) );
			return;
		}

		if ( ! isValidLanguageCode( langCode ) ) {
			setError( __( 'Invalid language code format', 'markdown-renderer-for-github' ) );
			return;
		}

		if ( existingLanguages.includes( langCode ) ) {
			setError( __( 'This language already exists', 'markdown-renderer-for-github' ) );
			return;
		}

		onAdd( langCode, copyFrom );
	};

	// Filter out already-used languages
	const availableCommonLangs = COMMON_LANGUAGES.filter(
		l => ! existingLanguages.includes( l.value )
	);

	return (
		<Modal
			title={ __( 'Add Language', 'markdown-renderer-for-github' ) }
			onRequestClose={ onClose }
			className="gfmr-add-language-modal"
		>
			<div style={ { minWidth: '300px' } }>
				<SelectControl
					label={ __( 'Language', 'markdown-renderer-for-github' ) }
					value={ selectedLang }
					options={ [
						{ value: '', label: __( 'Select...', 'markdown-renderer-for-github' ) },
						...availableCommonLangs,
						{ value: 'custom', label: __( 'Other (enter code)', 'markdown-renderer-for-github' ) },
					] }
					onChange={ ( value ) => {
						setSelectedLang( value );
						setError( '' );
					} }
				/>

				{ selectedLang === 'custom' && (
					<TextControl
						label={ __( 'Language Code (e.g., pt, nl, ar)', 'markdown-renderer-for-github' ) }
						value={ customLang }
						onChange={ ( value ) => {
							setCustomLang( value );
							setError( '' );
						} }
						maxLength={ 5 }
						help={ __( 'Use ISO 639-1 format (2 letters, e.g., "pt" for Portuguese)', 'markdown-renderer-for-github' ) }
					/>
				) }

				{ existingLanguages.length > 0 && (
					<SelectControl
						label={ __( 'Copy content from', 'markdown-renderer-for-github' ) }
						value={ copyFrom }
						options={ [
							{ value: '', label: __( 'Start empty', 'markdown-renderer-for-github' ) },
							...existingLanguages.map( l => ( {
								value: l,
								label: l.toUpperCase(),
							} ) ),
						] }
						onChange={ setCopyFrom }
					/>
				) }

				{ error && (
					<p style={ { color: '#d63638', marginTop: '8px' } }>{ error }</p>
				) }

				<div style={ { marginTop: '16px', display: 'flex', gap: '8px', justifyContent: 'flex-end' } }>
					<Button variant="secondary" onClick={ onClose }>
						{ __( 'Cancel', 'markdown-renderer-for-github' ) }
					</Button>
					<Button variant="primary" onClick={ handleAdd }>
						{ __( 'Add', 'markdown-renderer-for-github' ) }
					</Button>
				</div>
			</div>
		</Modal>
	);
}
