/** Markdown block deprecated version definitions
 *
 * Backward Compatibility Strategy:
 * - v6: Before multilingual feature (content/html as top-level)
 * - v5: With data-show-frontmatter in HTML
 * - v4: Before showFrontmatter/frontmatterData addition
 * - v3: Before shikiTheme addition
 * - v2: Before mermaidBgColor addition
 * - v1: Initial version
 *
 * WordPress Gutenberg automatically applies migrations in sequence when
 * block structure changes. Each version includes a migrate() function
 * that transforms old block data to the current schema.
 *
 * Key Migration Point (v6 - Multilingual):
 * The multilingual feature introduces a new data structure:
 * - Old: content/html as top-level attributes
 * - New: languages object containing per-language content/html
 *
 * Migration detects old blocks by checking for missing languages attribute
 * and converts content/html to the new languages structure.
 *
 * Key Migration Point (v5):
 * WordPress stores block attributes in two ways:
 * 1. Block comment: Used by default for new attributes
 * 2. HTML attributes (data-*): Legacy storage when source: 'attribute' is set
 *
 * When data-show-frontmatter was removed from save.js, existing blocks
 * had this attribute in their HTML but NOT in the block comment.
 * Without source: 'attribute' in v5, WordPress would read from comment
 * (defaulting to false) instead of HTML (actual value).
 * This caused validation errors when frontmatter was true in HTML.
 */

import { useBlockProps } from '@wordpress/block-editor';

/**
 * v6: Before multilingual feature
 * This deprecated version represents blocks before the languages attribute was added.
 *
 * Migration Strategy:
 * - Detects old blocks by checking for missing or empty languages attribute
 * - Converts top-level content/html to languages object structure
 * - Uses site's default language from wpGfmConfig or falls back to 'en'
 */
const v6 = {
	attributes: {
		content: {
			type: 'string',
			default: '',
		},
		html: {
			type: 'string',
			default: '',
		},
		mermaidBgColor: {
			type: 'string',
			default: 'transparent',
		},
		shikiTheme: {
			type: 'string',
			default: '',
		},
		showFrontmatter: {
			type: 'boolean',
			default: false,
		},
		frontmatterData: {
			type: 'object',
			default: {},
		},
		// Note: languages, defaultLanguage, availableLanguages, showLanguageSwitcher
		// were not present in v6 (pre-multilingual)
	},

	/**
	 * Determine if block needs migration
	 * Returns true for blocks that have content but no languages attribute
	 */
	isEligible( attributes ) {
		// Block needs migration if:
		// 1. Has no languages attribute OR languages is empty object
		// 2. AND has content or html (not a brand new empty block)
		const hasNoLanguages = ! attributes.languages ||
			( typeof attributes.languages === 'object' &&
			  Object.keys( attributes.languages ).length === 0 );
		const hasContent = attributes.content || attributes.html;

		return hasNoLanguages && hasContent;
	},

	/** save function (pre-multilingual format) */
	save( { attributes } ) {
		const { content, html, mermaidBgColor, shikiTheme, showFrontmatter = false } = attributes;

		const bgColor = mermaidBgColor || 'transparent';
		const theme = shikiTheme || '';

		const blockProps = useBlockProps.save( {
			className: 'gfmr-markdown-container',
		} );

		return (
			<div { ...blockProps }>
				<div
					className="gfmr-markdown-source"
					style={ {
						display: 'none',
						visibility: 'hidden',
						position: 'absolute',
						left: '-9999px',
						top: '-9999px',
						width: '1px',
						height: '1px',
						overflow: 'hidden',
					} }
					aria-hidden="true"
				>
					{ content }
				</div>

				<div
					className="gfmr-markdown-rendered"
					data-mermaid-bg-color={ bgColor }
					data-shiki-theme={ theme }
					dangerouslySetInnerHTML={ { __html: html || '' } }
				/>
			</div>
		);
	},

	/**
	 * Migrate to multilingual format
	 * Converts content/html to languages object structure
	 */
	migrate( attributes ) {
		// Get site's default language from config or fall back to 'en'
		const defaultLang = ( typeof window !== 'undefined' &&
			window.wpGfmConfig?.siteLanguage ) || 'en';

		const { content, html, ...rest } = attributes;

		// Log migration in debug mode
		if ( typeof window !== 'undefined' && window.WP_DEBUG ) {
			console.log( '[GFMR] Migrating block to multilingual format:', {
				defaultLang,
				contentLength: ( content || '' ).length,
			} );
		}

		return {
			...rest,
			// Keep content/html for backward compatibility during transition
			content: content || '',
			html: html || '',
			// New multilingual attributes
			languages: {
				[ defaultLang ]: {
					content: content || '',
					html: html || '',
				},
			},
			defaultLanguage: defaultLang,
			availableLanguages: [ defaultLang ],
			showLanguageSwitcher: false, // Single language, no need for switcher
		};
	},
};

/**
 * v5: Version with data-show-frontmatter attribute in HTML
 * This deprecated version represents the save.js before data-show-frontmatter was removed.
 *
 * Critical Implementation:
 * - Uses source: 'attribute' to read showFrontmatter from HTML data attribute
 * - This allows reading the true value from legacy blocks instead of block comment
 * - Without this, validation errors occur on blocks where frontmatter=true
 */
const v5 = {
	attributes: {
		content: {
			type: 'string',
			default: '',
		},
		html: {
			type: 'string',
			default: '',
		},
		mermaidBgColor: {
			type: 'string',
			default: 'transparent',
		},
		shikiTheme: {
			type: 'string',
			default: '',
		},
		showFrontmatter: {
			type: 'boolean',
			default: false,
			source: 'attribute',
			selector: '.gfmr-markdown-rendered',
			attribute: 'data-show-frontmatter',
		},
		frontmatterData: {
			type: 'object',
			default: {},
		},
	},

	/** save function */
	save( { attributes } ) {
		const { content, html, mermaidBgColor, shikiTheme, showFrontmatter = false, frontmatterData = {} } = attributes;

		// Fallback processing for mermaidBgColor (undefined protection)
		const bgColor = mermaidBgColor || 'transparent';
		// Shiki theme used during HTML generation (for frontend comparison)
		const theme = shikiTheme || '';

		const blockProps = useBlockProps.save( {
			className: 'gfmr-markdown-container',
		} );

		return (
			<div { ...blockProps }>
				{ /* Preserve original Markdown (hidden) */ }
				<div
					className="gfmr-markdown-source"
					style={ {
						display: 'none',
						visibility: 'hidden',
						position: 'absolute',
						left: '-9999px',
						top: '-9999px',
						width: '1px',
						height: '1px',
						overflow: 'hidden',
					} }
					aria-hidden="true"
				>
					{ content }
				</div>

				{ /* Rendered HTML (with data-show-frontmatter) */ }
				<div
					className="gfmr-markdown-rendered"
					data-mermaid-bg-color={ bgColor }
					data-shiki-theme={ theme }
					data-show-frontmatter={ showFrontmatter }
					dangerouslySetInnerHTML={ { __html: html || '' } }
				/>
			</div>
		);
	},

	/** Automatic migration */
	migrate( attributes ) {
		return {
			content: attributes.content || '',
			html: attributes.html || '',
			mermaidBgColor: attributes.mermaidBgColor || 'transparent',
			shikiTheme: attributes.shikiTheme || '',
			showFrontmatter: attributes.showFrontmatter || false,
			frontmatterData: attributes.frontmatterData || {}, // Explicitly set default
		};
	},
};

/**
 * v4: Before showFrontmatter and frontmatterData attributes addition
 * This version has shikiTheme but no showFrontmatter/frontmatterData attributes.
 */
const v4 = {
	attributes: {
		content: {
			type: 'string',
			default: '',
		},
		html: {
			type: 'string',
			default: '',
		},
		mermaidBgColor: {
			type: 'string',
			default: 'transparent',
		},
		shikiTheme: {
			type: 'string',
			default: '',
		},
		// Note: showFrontmatter and frontmatterData attributes were not present in v4
	},

	/** save function */
	save( { attributes } ) {
		const { content, html, mermaidBgColor, shikiTheme } = attributes;

		// Fallback processing for mermaidBgColor (undefined protection)
		const bgColor = mermaidBgColor || 'transparent';
		// Shiki theme used during HTML generation (for frontend comparison)
		const theme = shikiTheme || '';

		const blockProps = useBlockProps.save( {
			className: 'gfmr-markdown-container',
		} );

		return (
			<div { ...blockProps }>
				{ /* Preserve original Markdown (hidden) */ }
				<div
					className="gfmr-markdown-source"
					style={ {
						display: 'none',
						visibility: 'hidden',
						position: 'absolute',
						left: '-9999px',
						top: '-9999px',
						width: '1px',
						height: '1px',
						overflow: 'hidden',
					} }
					aria-hidden="true"
				>
					{ content }
				</div>

				{ /* Rendered HTML (without data-show-frontmatter) */ }
				<div
					className="gfmr-markdown-rendered"
					data-mermaid-bg-color={ bgColor }
					data-shiki-theme={ theme }
					dangerouslySetInnerHTML={ { __html: html || '' } }
				/>
			</div>
		);
	},

	/** Automatic migration */
	migrate( attributes ) {
		return {
			content: attributes.content || '',
			html: attributes.html || '',
			mermaidBgColor: attributes.mermaidBgColor || 'transparent',
			shikiTheme: attributes.shikiTheme || '',
			showFrontmatter: false, // New attribute with false default
			frontmatterData: {}, // New attribute with empty object default
		};
	},
};

/**
 * v3: Before shikiTheme attribute addition
 * This version has mermaidBgColor but no shikiTheme attribute.
 */
const v3 = {
	attributes: {
		content: {
			type: 'string',
			default: '',
		},
		html: {
			type: 'string',
			default: '',
		},
		mermaidBgColor: {
			type: 'string',
			default: 'transparent',
		},
		// Note: shikiTheme attribute was not present in v3
	},

	/** save function */
	save( { attributes } ) {
		const { content, html, mermaidBgColor } = attributes;

		// Fallback processing for mermaidBgColor (undefined protection)
		const bgColor = mermaidBgColor || 'transparent';

		const blockProps = useBlockProps.save( {
			className: 'gfmr-markdown-container',
		} );

		return (
			<div { ...blockProps }>
				{ /* Preserve original Markdown (hidden) */ }
				<div
					className="gfmr-markdown-source"
					style={ {
						display: 'none',
						visibility: 'hidden',
						position: 'absolute',
						left: '-9999px',
						top: '-9999px',
						width: '1px',
						height: '1px',
						overflow: 'hidden',
					} }
					aria-hidden="true"
				>
					{ content }
				</div>

				{ /* Rendered HTML (without data-shiki-theme) */ }
				<div
					className="gfmr-markdown-rendered"
					data-mermaid-bg-color={ bgColor }
					dangerouslySetInnerHTML={ { __html: html || '' } }
				/>
			</div>
		);
	},

	/** Automatic migration */
	migrate( attributes ) {
		return {
			content: attributes.content || '',
			html: attributes.html || '',
			mermaidBgColor: attributes.mermaidBgColor || 'transparent',
			shikiTheme: '', // New attribute with empty default
		};
	},
};

/** v2: Before mermaidBgColor attribute addition */
const v2 = {
	attributes: {
		content: {
			type: 'string',
			default: ''
		},
		html: {
			type: 'string',
			default: ''
		}
	},

	/** save function */
	save( { attributes } ) {
		const { content, html } = attributes;
		const blockProps = useBlockProps.save( {
			className: 'gfmr-markdown-container'
		} );

		return (
			<div { ...blockProps }>
				{/* Preserve original Markdown (hidden) */}
				<div
					className="gfmr-markdown-source"
					style={ {
						display: 'none',
						visibility: 'hidden',
						position: 'absolute',
						left: '-9999px',
						top: '-9999px',
						width: '1px',
						height: '1px',
						overflow: 'hidden'
					} }
					aria-hidden="true"
				>
					{ content }
				</div>

				{/* Rendered HTML (without data-mermaid-bg-color attribute) */}
				<div
					className="gfmr-markdown-rendered"
					dangerouslySetInnerHTML={ { __html: html || '' } }
				/>
			</div>
		);
	},

	/** Automatic migration */
	migrate( attributes ) {
		return {
			content: attributes.content || '',
			html: attributes.html || '',
			mermaidBgColor: 'transparent',
			shikiTheme: '', // New attribute with empty default
		};
	},
};

/** v1: Initial version */
const v1 = {
	attributes: {
		content: {
			type: 'string',
			default: ''
		},
		html: {
			type: 'string',
			default: ''
		}
	},

	/** save function */
	save( { attributes } ) {
		const { content, html } = attributes;
		const blockProps = useBlockProps.save( {
			className: 'gfmr-markdown-container'
		} );

		return (
			<div { ...blockProps }>
				{/* Preserve original Markdown (hidden) */}
				<div className="gfmr-markdown-source" style={ { display: 'none' } }>
					{ content }
				</div>

				{/* Rendered HTML */}
				<div
					className="gfmr-markdown-rendered"
					dangerouslySetInnerHTML={ { __html: html || '' } }
				/>
			</div>
		);
	},

	/** Automatic migration */
	migrate( attributes ) {
		return {
			content: attributes.content || '',
			html: attributes.html || '',
			mermaidBgColor: 'transparent',
			shikiTheme: '', // New attribute with empty default
		};
	},
};

export default [
	v6, // Before multilingual feature (content/html as top-level)
	v5, // With data-show-frontmatter in HTML
	v4, // Before showFrontmatter/frontmatterData addition
	v3, // Before shikiTheme addition
	v2, // Before mermaidBgColor addition
	v1, // Initial version (display: none only)
];