/**
 * GFMR Utils - Unified Foundation Module
 *
 * Unified utility, configuration, debug, and error handling foundation
 * Provides common functionality as the base for all GFMR modules
 *
 * Consolidated 6 files into 1:
 * - gfmr-utils.js
 * - gfmr-config-accessor.js  
 * - gfmr-debug-config.js
 * - gfmr-debug-helper.js
 * - markdown-config.js
 * - error-handler.js
 * 
 * @package WpGfmRenderer
 * @since 2.0.0
 */

(function(global) {
    'use strict';

    /**
     * WP GFM Utils - Unified Utility Class
     */
    class WPGFMUtils {
        constructor() {
            // Configuration management
            this.config = new Map();
            this.defaultConfig = new Map();

            // Debug functionality
            this.debugMode = false;
            this.debugLevel = 'info'; // 'debug', 'info', 'warn', 'error'
            this.debugOutput = [];

            // Error handling
            this.errorHandlers = [];
            this.errorHistory = [];
            this.maxErrorHistory = 100;

            // Performance measurement
            this.metrics = new Map();
            this.timers = new Map();

            // Initialization
            this.initializeConfig();
            this.initializeDebugMode();
            this.setupGlobalErrorHandling();
        }

        // =============================================
        // Configuration Management System
        // =============================================

        /**
         * Initialize configuration
         */
        initializeConfig() {
            // Get WordPress configuration
            const wpConfig = global.wpGfmConfig || {};
            const markdownConfig = global.wpGfmMarkdownConfig || {};

            // Default configuration
            this.setDefaultConfig({
                // Debug settings
                'debug.enabled': false,
                'debug.level': 'info',
                'debug.console': true,
                'debug.storage': false,

                // Performance settings
                'performance.tracking': true,
                'performance.maxMetrics': 1000,

                // Error handling settings
                'error.reporting': true,
                'error.maxHistory': 100,
                'error.autoRecovery': true,

                // Internationalization settings
                'i18n.locale': 'en',
                'i18n.fallback': 'en',

                // Highlight settings
                'highlight.theme': 'github-light',
                'highlight.languages': ['javascript', 'python', 'php', 'html', 'css'],

                // Mermaid settings
                'mermaid.theme': 'default',
                'mermaid.maxWidth': 1200,

                // UI settings
                'ui.copyButton': true,
                'ui.accessibility': true,

                // Merged settings (retrieved from WordPress)
                ...wpConfig,
                ...markdownConfig
            });
            
            this.debug('Configuration initialized', { 
                wpConfig: Object.keys(wpConfig).length,
                markdownConfig: Object.keys(markdownConfig).length 
            });
        }

        /**
         * Set default configuration
         */
        setDefaultConfig(defaults) {
            Object.entries(defaults).forEach(([key, value]) => {
                this.defaultConfig.set(key, value);
                if (!this.config.has(key)) {
                    this.config.set(key, value);
                }
            });
        }

        /**
         * Get configuration value
         */
        getConfig(key, defaultValue = null) {
            if (this.config.has(key)) {
                return this.config.get(key);
            }

            if (this.defaultConfig.has(key)) {
                return this.defaultConfig.get(key);
            }

            // Get nested configuration using dot notation
            if (key.includes('.')) {
                const value = this.getNestedConfig(key);
                if (value !== undefined) {
                    return value;
                }
            }
            
            return defaultValue;
        }

        /**
         * Get nested configuration (dot notation support)
         */
        getNestedConfig(key) {
            const keys = key.split('.');
            let current = global.wpGfmConfig || {};
            
            for (const k of keys) {
                if (current && typeof current === 'object' && k in current) {
                    current = current[k];
                } else {
                    return undefined;
                }
            }
            
            return current;
        }

        /**
         * Set configuration value
         */
        setConfig(key, value) {
            this.config.set(key, value);
            this.debug(`Config set: ${key} = ${value}`);
        }

        // =============================================
        // Debug System
        // =============================================

        /**
         * Initialize debug mode
         */
        initializeDebugMode() {
            // Check URL parameter
            const urlDebug = global.location && global.location.search.includes('debug=true');

            // Check WordPress configuration
            const wpDebug = this.getConfig('debug.enabled', false);

            // Check global variable
            const globalDebug = global.wpGfmDebug === true;

            // Determine debug mode
            this.debugMode = urlDebug || wpDebug || globalDebug;
            this.debugLevel = this.getConfig('debug.level', 'info');
            
            if (this.debugMode) {
                this.debug('Debug mode activated', {
                    source: urlDebug ? 'URL' : wpDebug ? 'Config' : 'Global',
                    level: this.debugLevel
                });
            }
        }

        /**
         * Debug output
         */
        debug(message, data = null, level = 'info') {
            if (!this.debugMode) return;
            
            const logEntry = {
                timestamp: new Date().toISOString(),
                level: level,
                message: message,
                data: data,
                stack: this.getConfig('debug.includeStack', false) ? new Error().stack : null
            };
            
            // Console output
            if (this.getConfig('debug.console', true)) {
                const prefix = '[WP GFM Utils]';
                switch (level) {
                    case 'error':
                        console.error(prefix, message, data);
                        break;
                    case 'warn':
                        console.warn(prefix, message, data);
                        break;
                    case 'info':
                        console.log(prefix, message, data);
                        break;
                    case 'debug':
                        console.debug(prefix, message, data);
                        break;
                }
            }
            
            // Storage save
            if (this.getConfig('debug.storage', false)) {
                this.debugOutput.push(logEntry);

                // Maximum storage limit
                const maxEntries = this.getConfig('debug.maxEntries', 1000);
                if (this.debugOutput.length > maxEntries) {
                    this.debugOutput = this.debugOutput.slice(-maxEntries);
                }
            }
        }

        /**
         * Get debug information
         */
        getDebugInfo() {
            return {
                enabled: this.debugMode,
                level: this.debugLevel,
                entries: this.debugOutput.length,
                config: Object.fromEntries(this.config),
                errors: this.errorHistory.length,
                metrics: Object.fromEntries(this.metrics)
            };
        }

        // =============================================
        // Error Handling System
        // =============================================

        /**
         * Setup global error handling
         */
        setupGlobalErrorHandling() {
            if (!this.getConfig('error.reporting', true)) return;

            // Capture JavaScript errors
            const originalErrorHandler = global.onerror;
            global.onerror = (message, source, lineno, colno, error) => {
                this.handleError({
                    type: 'javascript',
                    message: message,
                    source: source,
                    line: lineno,
                    column: colno,
                    error: error
                });
                
                if (originalErrorHandler) {
                    return originalErrorHandler.call(global, message, source, lineno, colno, error);
                }
                return false;
            };
            
            // Capture Promise rejections
            global.addEventListener('unhandledrejection', (event) => {
                this.handleError({
                    type: 'promise',
                    message: 'Unhandled Promise Rejection',
                    error: event.reason
                });
            });
        }

        /**
         * Register error handler
         */
        addErrorHandler(handler, priority = 10) {
            this.errorHandlers.push({ handler, priority });
            this.errorHandlers.sort((a, b) => a.priority - b.priority);
        }

        /**
         * Handle error
         */
        handleError(error, context = null) {
            const errorEntry = {
                timestamp: new Date().toISOString(),
                error: error,
                context: context,
                id: this.generateErrorId()
            };
            
            // Add to error history
            this.errorHistory.push(errorEntry);
            if (this.errorHistory.length > this.maxErrorHistory) {
                this.errorHistory = this.errorHistory.slice(-this.maxErrorHistory);
            }

            // Debug output
            this.debug('Error occurred', errorEntry, 'error');

            // Execute registered handlers
            this.errorHandlers.forEach(({ handler }) => {
                try {
                    handler(errorEntry);
                } catch (handlerError) {
                    console.error('[WP GFM Utils] Error in error handler:', handlerError);
                }
            });
            
            // Attempt auto recovery
            if (this.getConfig('error.autoRecovery', true)) {
                this.attemptAutoRecovery(errorEntry);
            }
            
            return errorEntry.id;
        }

        /**
         * Generate error ID
         */
        generateErrorId() {
            return 'wpgfm_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
        }

        /**
         * Attempt auto recovery
         */
        attemptAutoRecovery(errorEntry) {
            // Basic recovery strategy
            const error = errorEntry.error;

            if (error && error.message) {
                // Recovery for Shiki-related errors
                if (error.message.includes('shiki') || error.message.includes('highlight')) {
                    this.debug('Attempting Shiki recovery');
                    // Execute emergency fix script if available
                    if (global.wpGfmEmergencyHighlightFix) {
                        setTimeout(() => global.wpGfmEmergencyHighlightFix(), 100);
                    }
                }

                // Recovery for Mermaid-related errors
                if (error.message.includes('mermaid')) {
                    this.debug('Attempting Mermaid recovery');
                    // Re-initialize Mermaid, etc.
                }
            }
        }

        // =============================================
        // Performance Measurement
        // =============================================

        /**
         * Start timer
         */
        startTimer(name) {
            this.timers.set(name, performance.now());
        }

        /**
         * End timer and get measured value
         */
        endTimer(name) {
            const startTime = this.timers.get(name);
            if (!startTime) {
                this.debug(`Timer ${name} not found`, null, 'warn');
                return null;
            }
            
            const elapsed = performance.now() - startTime;
            this.timers.delete(name);
            
            // Record in metrics
            this.recordMetric(name, elapsed);
            
            return elapsed;
        }

        /**
         * Record metric
         */
        recordMetric(name, value, unit = 'ms') {
            if (!this.getConfig('performance.tracking', true)) return;
            
            const metric = {
                value: value,
                unit: unit,
                timestamp: Date.now()
            };
            
            // Add to existing metrics
            if (!this.metrics.has(name)) {
                this.metrics.set(name, []);
            }
            
            const metrics = this.metrics.get(name);
            metrics.push(metric);
            
            // Maximum storage limit
            const maxMetrics = this.getConfig('performance.maxMetrics', 1000);
            if (metrics.length > maxMetrics) {
                metrics.splice(0, metrics.length - maxMetrics);
            }
        }

        /**
         * Get metric statistics
         */
        getMetricStats(name) {
            const metrics = this.metrics.get(name);
            if (!metrics || metrics.length === 0) {
                return null;
            }
            
            const values = metrics.map(m => m.value);
            const sum = values.reduce((a, b) => a + b, 0);
            
            return {
                count: values.length,
                sum: sum,
                average: sum / values.length,
                min: Math.min(...values),
                max: Math.max(...values),
                latest: values[values.length - 1]
            };
        }

        // =============================================
        // General Utilities
        // =============================================

        /**
         * Deep clone object
         */
        deepClone(obj) {
            if (obj === null || typeof obj !== 'object') return obj;
            if (obj instanceof Date) return new Date(obj);
            if (obj instanceof Array) return obj.map(item => this.deepClone(item));
            if (typeof obj === 'object') {
                const cloned = {};
                Object.keys(obj).forEach(key => {
                    cloned[key] = this.deepClone(obj[key]);
                });
                return cloned;
            }
        }

        /**
         * Safe merge objects
         */
        mergeObjects(...objects) {
            const result = {};
            objects.forEach(obj => {
                if (obj && typeof obj === 'object') {
                    Object.keys(obj).forEach(key => {
                        result[key] = obj[key];
                    });
                }
            });
            return result;
        }

        /**
         * Sanitize string
         */
        sanitizeString(str) {
            if (typeof str !== 'string') return '';
            return str
                .replace(/[<>]/g, '') // Remove HTML tags
                .replace(/javascript:/gi, '') // Remove JavaScript URLs
                .trim();
        }

        /**
         * Delayed function execution (debounce)
         */
        debounce(func, wait, immediate = false) {
            let timeout;
            return function executedFunction(...args) {
                const later = () => {
                    timeout = null;
                    if (!immediate) func.apply(this, args);
                };
                const callNow = immediate && !timeout;
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
                if (callNow) func.apply(this, args);
            };
        }

        /**
         * Limit function execution frequency (throttle)
         */
        throttle(func, limit) {
            let inThrottle;
            return function(...args) {
                if (!inThrottle) {
                    func.apply(this, args);
                    inThrottle = true;
                    setTimeout(() => inThrottle = false, limit);
                }
            };
        }

        /**
         * Asset URL builder helper function
         * Build local asset URLs compliant with WordPress.org distribution
         *
         * @param {string} path - Relative path (e.g., 'assets/libs/shiki/shiki.min.js')
         * @returns {string} - Complete URL
         */
        buildAssetUrl(path) {
            const baseUrl = global.wpGfmConfig?.pluginUrl || '/wp-content/plugins/markdown-renderer-for-github/';
            // Normalize trailing slash from base URL
            const normalizedBase = baseUrl.replace(/\/+$/, '');
            // Normalize leading slash from path
            const normalizedPath = path.replace(/^\/+/, '');
            return `${normalizedBase}/${normalizedPath}`;
        }

        // =============================================
        // External API (Global Exposure)
        // =============================================

        /**
         * Module initialization complete
         */
        initialize() {
            // Final initialization process
            this.debug('WP GFM Utils initialized', this.getDebugInfo());

            // Initialization complete event
            if (global.document) {
                global.document.dispatchEvent(new CustomEvent('wpGfmUtilsReady', {
                    detail: { utils: this }
                }));
            }
            
            return this;
        }

        /**
         * Health check
         */
        healthCheck() {
            return {
                status: 'ok',
                version: '2.0.0',
                initialized: true,
                debugMode: this.debugMode,
                configLoaded: this.config.size > 0,
                errorsCount: this.errorHistory.length,
                metricsCount: this.metrics.size
            };
        }

        /**
         * Get statistics
         */
        getStats() {
            return {
                initialized: this.initialized,
                debugMode: this.debugMode,
                configEntries: this.config.size,
                errorCount: this.errorHandlers.length,
                metricsCount: this.metrics.size,
                timestamp: Date.now()
            };
        }
    }

    // Create and expose global instance
    const wpGfmUtils = new WPGFMUtils();

    // Expose as global variable
    global.wpGfmUtils = wpGfmUtils;
    global.WPGFMUtils = WPGFMUtils; // Also expose class (for testing)

    // Legacy API for backward compatibility
    global.wpGfmIsDebugMode = () => wpGfmUtils.debugMode;
    global.wpGfmGetConfig = (key, defaultValue) => wpGfmUtils.getConfig(key, defaultValue);
    global.wpGfmDebugLog = (message, data) => wpGfmUtils.debug(message, data);

    // Unified Error Handler API
    // Provides consistent error/warning logging across all GFMR modules
    global.wpGfmErrorHandler = {
        /**
         * Handle an error with context and details
         *
         * @param {Error|string} error - The error object or message
         * @param {string} context - Module or function context (e.g., 'Shiki', 'Mermaid')
         * @param {Object} details - Additional details for debugging
         */
        handle: function(error, context, details) {
            const errorMessage = error instanceof Error ? error.message : String(error);
            const errorStack = error instanceof Error ? error.stack : null;

            console.error('[GFMR Error]', {
                context: context || 'Unknown',
                message: errorMessage,
                details: details || null,
                stack: errorStack
            });

            // Also record in wpGfmUtils error history if available
            if (wpGfmUtils && typeof wpGfmUtils.handleError === 'function') {
                wpGfmUtils.handleError({
                    type: 'gfmr',
                    message: errorMessage,
                    source: context,
                    error: error
                }, details);
            }
        },

        /**
         * Log a warning message with context
         *
         * @param {string} message - The warning message
         * @param {string} context - Module or function context
         * @param {Object} details - Additional details for debugging
         */
        warn: function(message, context, details) {
            console.warn('[GFMR Warn][' + (context || 'Unknown') + ']', message, details || '');

            // Also record in debug log if available
            if (wpGfmUtils && typeof wpGfmUtils.debug === 'function') {
                wpGfmUtils.debug(message, details, 'warn');
            }
        },

        /**
         * Log an info message with context (for non-critical issues)
         *
         * @param {string} message - The info message
         * @param {string} context - Module or function context
         */
        info: function(message, context) {
            if (wpGfmUtils && wpGfmUtils.debugMode) {
                console.info('[GFMR Info][' + (context || 'Unknown') + ']', message);
            }
        }
    };

    // Execute initialization
    if (global.document && global.document.readyState === 'loading') {
        global.document.addEventListener('DOMContentLoaded', () => wpGfmUtils.initialize());
    } else {
        // If DOM is already loaded
        setTimeout(() => wpGfmUtils.initialize(), 0);
    }
    
    // CommonJS / AMD support
    if (typeof module !== 'undefined' && module.exports) {
        module.exports = WPGFMUtils;
    }

})(typeof window !== 'undefined' ? window : global);