/**
 * WP GFM UI v2.0 - Unified User Interface Engine
 *
 * Unified module for UI elements, copy buttons, and internationalization
 * Integrates accessibility, responsive design, and user experience enhancements
 *
 * Consolidates 3 files into one:
 * - gfmr-ui-components.js
 * - gfmr-copy-button.js
 * - gfmr-i18n.js
 *
 * @package WpGfmRenderer
 * @since 2.0.0
 * @requires gfmr-utils-v2.js
 */

(function(global) {
    'use strict';

    /**
     * WP GFM UI - Unified User Interface Engine
     */
    class WPGFMUI {
        constructor(utils = null) {
            // Dependencies
            this.utils = utils || global.wpGfmUtils;
            if (!this.utils) {
                throw new Error('WPGFMUI requires WPGFMUtils');
            }

            // Component management
            this.components = new Map();
            this.componentInstances = new WeakMap();
            this.eventListeners = new Map();

            // Copy button management
            this.copyButtons = [];
            this.copyButtonInstances = new WeakMap();
            this.copySuccessTimeout = 2000;

            // Internationalization
            this.i18n = null;
            this.currentLocale = 'en';
            this.translations = new Map();
            this.rtlLanguages = new Set(['ar', 'he', 'fa', 'ur']);

            // Accessibility
            this.a11yConfig = {
                enabled: true,
                announcements: true,
                keyboardNavigation: true,
                highContrast: false,
                reducedMotion: false
            };

            // UI configuration
            this.config = {
                copyButton: {
                    enabled: this.utils.getConfig('ui.copyButton', true),
                    position: this.utils.getConfig('ui.copyButton.position', 'top-right'),
                    style: this.utils.getConfig('ui.copyButton.style', 'modern'),
                    showAnimation: this.utils.getConfig('ui.copyButton.animation', true)
                },
                accessibility: {
                    enabled: this.utils.getConfig('ui.accessibility', true),
                    announcements: this.utils.getConfig('ui.accessibility.announcements', true),
                    reducedMotion: this.utils.getConfig('ui.accessibility.reducedMotion', false)
                },
                theme: {
                    mode: this.utils.getConfig('ui.theme.mode', 'auto'), // auto, light, dark
                    customColors: this.utils.getConfig('ui.theme.colors', {})
                },
                responsive: {
                    enabled: this.utils.getConfig('ui.responsive', true),
                    breakpoints: {
                        mobile: 768,
                        tablet: 1024
                    }
                }
            };

            // Initialization
            this.setupI18n();
            this.setupAccessibility();
            this.setupThemeSystem();
            this.registerDefaultComponents();
        }

        // =============================================
        // Initialization and Setup
        // =============================================

        /**
         * Set up internationalization system
         */
        setupI18n() {
            // Detect current locale
            this.currentLocale = this.detectLocale();

            // Prioritize WordPress translations (from wpGfmConfig)
            if (global.wpGfmConfig?.translations) {
                this.translations.set(this.currentLocale, global.wpGfmConfig.translations);
            }

            // Load default translations as fallback
            this.loadDefaultTranslations();

            // RTL language support
            if (this.rtlLanguages.has(this.currentLocale)) {
                document.documentElement.dir = 'rtl';
                document.documentElement.classList.add('gfmr-rtl');
            }

            this.utils.debug('I18n system initialized', {
                locale: this.currentLocale,
                isRTL: this.rtlLanguages.has(this.currentLocale),
                hasWPTranslations: !!global.wpGfmConfig?.translations
            });
        }

        /**
         * Set up accessibility
         */
        setupAccessibility() {
            if (!this.config.accessibility.enabled) return;

            // Detect user preferences
            this.detectAccessibilityPreferences();

            // Enable keyboard navigation
            if (this.a11yConfig.keyboardNavigation) {
                this.enableKeyboardNavigation();
            }

            // Create announcement region
            if (this.a11yConfig.announcements) {
                this.createAnnouncementRegion();
            }

            this.utils.debug('Accessibility features initialized', this.a11yConfig);
        }

        /**
         * Set up theme system
         */
        setupThemeSystem() {
            // Apply theme mode
            this.applyThemeMode(this.config.theme.mode);

            // Apply custom colors
            if (this.config.theme.customColors) {
                this.applyCustomColors(this.config.theme.customColors);
            }

            // Watch media queries
            if (this.config.theme.mode === 'auto') {
                this.watchSystemTheme();
            }
        }

        /**
         * Register default components
         */
        registerDefaultComponents() {
            // Copy button component
            this.registerComponent('copyButton', {
                create: (element, options) => this.createCopyButton(element, options),
                destroy: (instance) => this.destroyCopyButton(instance)
            });

            // Loading spinner
            this.registerComponent('loadingSpinner', {
                create: (element, options) => this.createLoadingSpinner(element, options),
                destroy: (instance) => this.destroyLoadingSpinner(instance)
            });

            // Error display
            this.registerComponent('errorDisplay', {
                create: (element, options) => this.createErrorDisplay(element, options),
                destroy: (instance) => this.destroyErrorDisplay(instance)
            });

            // Tooltip
            this.registerComponent('tooltip', {
                create: (element, options) => this.createTooltip(element, options),
                destroy: (instance) => this.destroyTooltip(instance)
            });
        }

        // =============================================
        // Internationalization System
        // =============================================

        /**
         * Detect locale
         */
        detectLocale() {
            // Get from WordPress settings
            const wpLocale = global.wpGfmConfig?.locale;
            if (wpLocale) return wpLocale;

            // Get from HTML lang attribute
            const htmlLang = document.documentElement.lang;
            if (htmlLang) return htmlLang.substring(0, 2);

            // Get from browser settings
            const browserLang = navigator.language || navigator.userLanguage;
            return browserLang ? browserLang.substring(0, 2) : 'en';
        }

        /**
         * Load default translations (English fallback only)
         * Other languages are provided via WordPress translation system (wpGfmConfig.translations)
         */
        loadDefaultTranslations() {
            // English fallback translations
            const fallbackTranslations = {
                'copy': 'Copy',
                'copied': 'Copied!',
                'copy_code': 'Copy code',
                'code_copied': 'Code copied to clipboard',
                'copy_failed': 'Failed to copy',
                'loading': 'Loading...',
                'error': 'Error',
                'retry': 'Retry',
                'close': 'Close',
                'expand': 'Expand',
                'collapse': 'Collapse'
            };

            // Set English as fallback (only if not already set by WordPress translations)
            if (!this.translations.has('en')) {
                this.translations.set('en', fallbackTranslations);
            }
        }

        /**
         * Get translation text
         */
        translate(key, params = {}) {
            const localeTranslations = this.translations.get(this.currentLocale) ||
                                     this.translations.get('en') || {};

            let text = localeTranslations[key] || key;

            // Parameter replacement
            Object.entries(params).forEach(([param, value]) => {
                text = text.replace(new RegExp(`{${param}}`, 'g'), value);
            });

            return text;
        }

        /**
         * Add translations
         */
        addTranslations(locale, translations) {
            const existing = this.translations.get(locale) || {};
            this.translations.set(locale, { ...existing, ...translations });

            this.utils.debug(`Translations added for locale: ${locale}`, Object.keys(translations));
        }

        // =============================================
        // Copy Button Functionality
        // =============================================

        /**
         * Create copy button
         */
        createCopyButton(codeBlock, options = {}) {
            if (!this.config.copyButton.enabled) return null;

            // Check if Mermaid container
            const isMermaidContainer = this.isMermaidContainer(codeBlock);
            const buttonStyle = isMermaidContainer ? 'mermaid' : (options.style || this.config.copyButton.style);

            const button = document.createElement('button');
            button.type = 'button';
            button.className = `gfmr-copy-button gfmr-copy-button--${buttonStyle}`;
            button.setAttribute('aria-label', this.translate('copy_code'));
            button.title = this.translate('copy_code');

            // Icon and text
            button.innerHTML = `
                <svg class="gfmr-copy-icon" width="16" height="16" viewBox="0 0 16 16" fill="currentColor">
                    <path d="M4 2a2 2 0 0 1 2-2h8a2 2 0 0 1 2 2v8a2 2 0 0 1-2 2H6a2 2 0 0 1-2-2V2Zm2-1a1 1 0 0 0-1 1v8a1 1 0 0 0 1 1h8a1 1 0 0 0 1-1V2a1 1 0 0 0-1-1H6ZM2 5a1 1 0 0 0-1 1v8a1 1 0 0 0 1 1h8a1 1 0 0 0 1-1v-1h1v1a2 2 0 0 1-2 2H2a2 2 0 0 1-2-2V6a2 2 0 0 1 2-2h1v1H2Z"/>
                </svg>
                <span class="gfmr-copy-text">${this.translate('copy')}</span>
            `;

            // Event listener
            const clickHandler = async (event) => {
                event.preventDefault();
                await this.handleCopyClick(button, codeBlock);
            };

            button.addEventListener('click', clickHandler);

            // Keyboard event
            const keyHandler = (event) => {
                if (event.key === 'Enter' || event.key === ' ') {
                    event.preventDefault();
                    clickHandler(event);
                }
            };

            button.addEventListener('keydown', keyHandler);

            // Position button
            this.positionCopyButton(button, codeBlock, options);

            // Save instance information
            const instance = {
                button: button,
                codeBlock: codeBlock,
                clickHandler: clickHandler,
                keyHandler: keyHandler
            };

            this.copyButtonInstances.set(button, instance);
            this.copyButtons.push(instance);

            this.utils.debug('Copy button created', { position: this.config.copyButton.position });

            return instance;
        }

        /**
         * Position copy button
         */
        positionCopyButton(button, codeBlock, options) {
            const position = options.position || this.config.copyButton.position;
            const container = codeBlock.closest('pre') || codeBlock.parentElement;

            // Adjust container styles
            if (container) {
                container.style.position = 'relative';
                container.classList.add('gfmr-code-container');

                // Position button
                switch (position) {
                    case 'top-right':
                        button.style.cssText = `
                            position: absolute;
                            top: 8px;
                            right: 8px;
                            z-index: 10;
                        `;
                        break;

                    case 'top-left':
                        button.style.cssText = `
                            position: absolute;
                            top: 8px;
                            left: 8px;
                            z-index: 10;
                        `;
                        break;

                    case 'bottom-right':
                        button.style.cssText = `
                            position: absolute;
                            bottom: 8px;
                            right: 8px;
                            z-index: 10;
                        `;
                        break;
                }

                container.appendChild(button);
            }
        }

        /**
         * Handle copy click
         */
        async handleCopyClick(button, codeBlock) {
            try {
                // Get code text
                const codeText = this.extractCodeText(codeBlock);

                if (!codeText) {
                    throw new Error('No code content to copy');
                }

                // Copy to clipboard
                await this.copyToClipboard(codeText);

                // Success feedback
                this.showCopySuccess(button);

                // Accessibility announcement
                this.announce(this.translate('code_copied'));

                this.utils.debug('Code copied successfully', { length: codeText.length });

            } catch (error) {
                this.utils.handleError(error, 'copy_operation');
                this.showCopyError(button);
                this.announce(this.translate('copy_failed'));
            }
        }

        /**
         * Extract code text
         */
        extractCodeText(codeBlock) {
            // For Shiki processed elements
            if (codeBlock.classList.contains('shiki') || codeBlock.querySelector('.shiki')) {
                return this.extractFromShiki(codeBlock);
            }

            // Regular code block
            return codeBlock.textContent || codeBlock.innerText || '';
        }

        /**
         * Extract text from Shiki processed elements
         */
        extractFromShiki(element) {
            const shikiElement = element.classList.contains('shiki') ?
                                element : element.querySelector('.shiki');

            if (!shikiElement) {
                return element.textContent || '';
            }

            // Extract text while preserving Shiki line structure
            const lines = Array.from(shikiElement.querySelectorAll('.line, [class*="line"]'));

            if (lines.length > 0) {
                return lines.map(line => line.textContent || '').join('\n');
            }

            return shikiElement.textContent || '';
        }

        /**
         * Copy to clipboard
         */
        async copyToClipboard(text) {
            if (navigator.clipboard && window.isSecureContext) {
                // Modern Clipboard API
                await navigator.clipboard.writeText(text);
            } else {
                // Fallback method
                const textArea = document.createElement('textarea');
                textArea.value = text;
                textArea.style.cssText = `
                    position: fixed;
                    top: -999px;
                    left: -999px;
                    opacity: 0;
                    pointer-events: none;
                `;

                document.body.appendChild(textArea);
                textArea.select();
                textArea.setSelectionRange(0, 99999);

                if (!document.execCommand('copy')) {
                    throw new Error('Copy command failed');
                }

                document.body.removeChild(textArea);
            }
        }

        /**
         * Copy success feedback
         */
        showCopySuccess(button) {
            const originalContent = button.innerHTML;
            const originalText = button.getAttribute('aria-label');

            // Success icon and text
            button.innerHTML = `
                <svg class="gfmr-copy-icon gfmr-copy-success" width="16" height="16" viewBox="0 0 16 16" fill="currentColor">
                    <path d="M13.854 3.646a.5.5 0 0 1 0 .708l-7 7a.5.5 0 0 1-.708 0l-3.5-3.5a.5.5 0 1 1 .708-.708L6.5 10.293l6.646-6.647a.5.5 0 0 1 .708 0z"/>
                </svg>
                <span class="gfmr-copy-text">${this.translate('copied')}</span>
            `;

            button.setAttribute('aria-label', this.translate('copied'));
            button.classList.add('gfmr-copy-success');

            // Animation effect
            if (this.config.copyButton.showAnimation && !this.a11yConfig.reducedMotion) {
                button.style.transform = 'scale(1.1)';
                setTimeout(() => {
                    button.style.transform = '';
                }, 150);
            }

            // Restore original state
            setTimeout(() => {
                button.innerHTML = originalContent;
                button.setAttribute('aria-label', originalText);
                button.classList.remove('gfmr-copy-success');
            }, this.copySuccessTimeout);
        }

        /**
         * Copy error feedback
         */
        showCopyError(button) {
            const originalClasses = button.className;

            button.classList.add('gfmr-copy-error');

            if (this.config.copyButton.showAnimation && !this.a11yConfig.reducedMotion) {
                button.style.animation = 'shake 0.5s ease-in-out';
            }

            setTimeout(() => {
                button.className = originalClasses;
                button.style.animation = '';
            }, 2000);
        }

        // =============================================
        // Other UI Components
        // =============================================

        /**
         * Create loading spinner
         */
        createLoadingSpinner(element, _options = {}) { // eslint-disable-line no-unused-vars
            const spinner = document.createElement('div');
            spinner.className = 'gfmr-loading-spinner';
            spinner.setAttribute('role', 'status');
            spinner.setAttribute('aria-label', this.translate('loading'));

            spinner.innerHTML = `
                <div class="gfmr-spinner-circle"></div>
                <span class="gfmr-spinner-text">${this.translate('loading')}</span>
            `;

            if (element) {
                element.appendChild(spinner);
            }

            return { element: spinner, destroy: () => spinner.remove() };
        }

        /**
         * Create error display
         */
        createErrorDisplay(element, options = {}) {
            const errorDiv = document.createElement('div');
            errorDiv.className = 'gfmr-error-display';
            errorDiv.setAttribute('role', 'alert');

            const message = options.message || this.translate('error');
            const showRetry = options.retry !== false;

            errorDiv.innerHTML = `
                <div class="gfmr-error-icon">⚠️</div>
                <div class="gfmr-error-message">${this.utils.sanitizeString(message)}</div>
                ${showRetry ? `<button type="button" class="gfmr-error-retry">${this.translate('retry')}</button>` : ''}
            `;

            // Retry button event
            if (showRetry && options.onRetry) {
                const retryButton = errorDiv.querySelector('.gfmr-error-retry');
                retryButton.addEventListener('click', options.onRetry);
            }

            if (element) {
                element.appendChild(errorDiv);
            }

            return { element: errorDiv, destroy: () => errorDiv.remove() };
        }

        /**
         * Create tooltip
         */
        createTooltip(element, options = {}) {
            if (!element) return null;

            const tooltip = document.createElement('div');
            tooltip.className = 'gfmr-tooltip';
            tooltip.setAttribute('role', 'tooltip');
            tooltip.textContent = options.text || element.title || '';

            // Initially hidden
            tooltip.style.cssText = `
                position: absolute;
                visibility: hidden;
                opacity: 0;
                z-index: 9999;
                transition: opacity 0.2s, visibility 0.2s;
            `;

            document.body.appendChild(tooltip);

            // Event listeners
            const showTooltip = (event) => {
                this.positionTooltip(tooltip, element, event);
                tooltip.style.visibility = 'visible';
                tooltip.style.opacity = '1';
            };

            const hideTooltip = () => {
                tooltip.style.visibility = 'hidden';
                tooltip.style.opacity = '0';
            };

            element.addEventListener('mouseenter', showTooltip);
            element.addEventListener('mouseleave', hideTooltip);
            element.addEventListener('focus', showTooltip);
            element.addEventListener('blur', hideTooltip);

            return {
                element: tooltip,
                destroy: () => {
                    element.removeEventListener('mouseenter', showTooltip);
                    element.removeEventListener('mouseleave', hideTooltip);
                    element.removeEventListener('focus', showTooltip);
                    element.removeEventListener('blur', hideTooltip);
                    tooltip.remove();
                }
            };
        }

        // =============================================
        // Accessibility Features
        // =============================================

        /**
         * Detect accessibility preferences
         */
        detectAccessibilityPreferences() {
            // Detect reduced motion
            const prefersReducedMotion = window.matchMedia('(prefers-reduced-motion: reduce)');
            this.a11yConfig.reducedMotion = prefersReducedMotion.matches;

            prefersReducedMotion.addEventListener('change', (e) => {
                this.a11yConfig.reducedMotion = e.matches;
                this.updateAnimationSettings();
            });

            // Detect high contrast
            const prefersHighContrast = window.matchMedia('(prefers-contrast: high)');
            this.a11yConfig.highContrast = prefersHighContrast.matches;

            if (prefersHighContrast) {
                prefersHighContrast.addEventListener('change', (e) => {
                    this.a11yConfig.highContrast = e.matches;
                    this.updateContrastSettings();
                });
            }
        }

        /**
         * Enable keyboard navigation
         */
        enableKeyboardNavigation() {
            // Improve focus management
            document.addEventListener('keydown', (event) => {
                if (event.key === 'Tab') {
                    document.body.classList.add('gfmr-keyboard-navigation');
                }
            });

            document.addEventListener('mousedown', () => {
                document.body.classList.remove('gfmr-keyboard-navigation');
            });
        }

        /**
         * Create announcement region
         */
        createAnnouncementRegion() {
            const announcer = document.createElement('div');
            announcer.id = 'gfmr-announcer';
            announcer.setAttribute('aria-live', 'polite');
            announcer.setAttribute('aria-atomic', 'true');
            announcer.style.cssText = `
                position: absolute;
                top: -999px;
                left: -999px;
                width: 1px;
                height: 1px;
                overflow: hidden;
            `;

            document.body.appendChild(announcer);
            this.announcementRegion = announcer;
        }

        /**
         * Announce to screen readers
         */
        announce(message, priority = 'polite') {
            if (!this.a11yConfig.announcements || !this.announcementRegion) return;

            this.announcementRegion.setAttribute('aria-live', priority);
            this.announcementRegion.textContent = message;

            // Clear message (to allow re-announcement)
            setTimeout(() => {
                this.announcementRegion.textContent = '';
            }, 1000);
        }

        // =============================================
        // Theme System
        // =============================================

        /**
         * Apply theme mode
         */
        applyThemeMode(mode) {
            const root = document.documentElement;

            switch (mode) {
                case 'light':
                    root.classList.remove('gfmr-dark');
                    root.classList.add('gfmr-light');
                    break;
                case 'dark':
                    root.classList.remove('gfmr-light');
                    root.classList.add('gfmr-dark');
                    break;
                case 'auto':
                default:
                    this.applySystemTheme();
                    break;
            }
        }

        /**
         * Apply system theme
         */
        applySystemTheme() {
            const prefersDark = window.matchMedia('(prefers-color-scheme: dark)');
            const root = document.documentElement;

            if (prefersDark.matches) {
                root.classList.remove('gfmr-light');
                root.classList.add('gfmr-dark');
            } else {
                root.classList.remove('gfmr-dark');
                root.classList.add('gfmr-light');
            }
        }

        /**
         * Watch system theme
         */
        watchSystemTheme() {
            const prefersDark = window.matchMedia('(prefers-color-scheme: dark)');
            prefersDark.addEventListener('change', () => {
                if (this.config.theme.mode === 'auto') {
                    this.applySystemTheme();
                }
            });
        }

        /**
         * Apply custom colors
         */
        applyCustomColors(colors) {
            const root = document.documentElement;

            Object.entries(colors).forEach(([property, value]) => {
                root.style.setProperty(`--gfmr-${property}`, value);
            });
        }

        // =============================================
        // Component Management
        // =============================================

        /**
         * Register component
         */
        registerComponent(name, config) {
            if (!name || !config || typeof config.create !== 'function') {
                throw new Error('Invalid component configuration');
            }

            this.components.set(name, config);
            this.utils.debug(`Component registered: ${name}`);
        }

        /**
         * Create component
         */
        createComponent(name, element, options = {}) {
            const config = this.components.get(name);
            if (!config) {
                throw new Error(`Unknown component: ${name}`);
            }

            const instance = config.create(element, options);
            if (instance && element) {
                this.componentInstances.set(element, instance);
            }

            return instance;
        }

        /**
         * Destroy component
         */
        destroyComponent(element) {
            const instance = this.componentInstances.get(element);
            if (!instance) return false;

            if (instance.destroy && typeof instance.destroy === 'function') {
                instance.destroy();
            }

            this.componentInstances.delete(element);
            return true;
        }

        // =============================================
        // Batch Processing
        // =============================================

        /**
         * Add copy buttons to all code blocks
         */
        addCopyButtonsToAllCodeBlocks() {
            if (!this.config.copyButton.enabled) return;

            const selectors = [
                'pre code',
                '.wp-block-code code',
                '[class*="language-"]'
            ];

            const codeBlocks = document.querySelectorAll(selectors.join(', '));
            let addedCount = 0;

            codeBlocks.forEach(codeBlock => {
                // Skip if button already exists
                if (codeBlock.parentElement.querySelector('.gfmr-copy-button')) return;

                try {
                    this.createCopyButton(codeBlock);
                    addedCount++;
                } catch (error) {
                    this.utils.debug('Failed to add copy button:', error);
                }
            });

            this.utils.debug(`Copy buttons added to ${addedCount} code blocks`);
            return addedCount;
        }

        /**
         * Initialize all UI components
         */
        initializeAllComponents() {
            // Add copy buttons
            this.addCopyButtonsToAllCodeBlocks();

            // Other auto-initialization
            // Add as needed
        }

        // =============================================
        // Utilities
        // =============================================

        /**
         * Position tooltip
         */
        positionTooltip(tooltip, element, _event) { // eslint-disable-line no-unused-vars
            const rect = element.getBoundingClientRect();
            const tooltipRect = tooltip.getBoundingClientRect();

            let top = rect.bottom + 8;
            let left = rect.left + (rect.width - tooltipRect.width) / 2;

            // Adjust for screen edges
            if (left < 8) left = 8;
            if (left + tooltipRect.width > window.innerWidth - 8) {
                left = window.innerWidth - tooltipRect.width - 8;
            }

            if (top + tooltipRect.height > window.innerHeight - 8) {
                top = rect.top - tooltipRect.height - 8;
            }

            tooltip.style.top = `${top}px`;
            tooltip.style.left = `${left}px`;
        }

        /**
         * Update animation settings
         */
        updateAnimationSettings() {
            const root = document.documentElement;

            if (this.a11yConfig.reducedMotion) {
                root.style.setProperty('--gfmr-animation-duration', '0s');
                root.classList.add('gfmr-reduced-motion');
            } else {
                root.style.removeProperty('--gfmr-animation-duration');
                root.classList.remove('gfmr-reduced-motion');
            }
        }

        /**
         * Update contrast settings
         */
        updateContrastSettings() {
            const root = document.documentElement;

            if (this.a11yConfig.highContrast) {
                root.classList.add('gfmr-high-contrast');
            } else {
                root.classList.remove('gfmr-high-contrast');
            }
        }

        /**
         * Health check
         */
        healthCheck() {
            return {
                status: 'ok',
                copyButtonsEnabled: this.config.copyButton.enabled,
                copyButtonsCount: this.copyButtons.length,
                accessibilityEnabled: this.config.accessibility.enabled,
                currentLocale: this.currentLocale,
                themeMode: this.config.theme.mode,
                registeredComponents: Array.from(this.components.keys())
            };
        }

        /**
         * Get statistics
         */
        getStats() {
            return {
                copyButtons: this.copyButtons.length,
                components: this.components.size,
                translations: this.translations.size,
                accessibility: this.a11yConfig,
                theme: this.config.theme,
                healthCheck: this.healthCheck()
            };
        }

        /**
         * Check if element is Mermaid container
         */
        isMermaidContainer(element) {
            if (!element) return false;

            // Check if element itself is Mermaid container
            const mermaidClasses = ['gfmr-mermaid-container', 'gfmr-mermaid-inner-container', 'mermaid'];
            if (mermaidClasses.some(cls => element.classList?.contains(cls))) {
                return true;
            }

            // Search for Mermaid container in parent elements
            let parent = element.parentElement;
            while (parent && parent !== document.body) {
                if (mermaidClasses.some(cls => parent.classList?.contains(cls))) {
                    return true;
                }
                // SVG elements may also be Mermaid diagrams
                if (parent.tagName === 'SVG' && parent.closest('.gfmr-mermaid-container')) {
                    return true;
                }
                parent = parent.parentElement;
            }

            // Check data-mermaid and data-diagram attributes
            if (element.hasAttribute?.('data-mermaid') || element.hasAttribute?.('data-diagram')) {
                return true;
            }

            return false;
        }
    }

    // Create and expose global instance
    let wpGfmUI = null;

    // Initialization function
    function initializeUI() {
        if (wpGfmUI) return wpGfmUI;

        wpGfmUI = new WPGFMUI();

        // Expose as global variable
        global.wpGfmUI = wpGfmUI;
        global.WPGFMUI = WPGFMUI;

        // Legacy compatibility API
        global.wpGfmCreateCopyButton = (element, options) => {
            return wpGfmUI.createCopyButton(element, options);
        };
        global.wpGfmTranslate = (key, params) => {
            return wpGfmUI.translate(key, params);
        };

        // Initialization complete event
        if (global.document) {
            global.document.dispatchEvent(new CustomEvent('wpGfmUIReady', {
                detail: { ui: wpGfmUI }
            }));
        }

        return wpGfmUI;
    }

    // Control initialization timing
    if (global.document && global.document.readyState === 'loading') {
        global.document.addEventListener('DOMContentLoaded', () => {
            initializeUI();
            // Auto-initialize UI components
            setTimeout(() => {
                wpGfmUI.initializeAllComponents();
            }, 100);
        });
    } else {
        setTimeout(() => {
            initializeUI();
            wpGfmUI.initializeAllComponents();
        }, 0);
    }

    // Watch for utilities ready event
    if (global.document) {
        global.document.addEventListener('wpGfmUtilsReady', initializeUI);
    }

    // CommonJS / AMD support
    if (typeof module !== 'undefined' && module.exports) {
        module.exports = WPGFMUI;
    }

})(typeof window !== 'undefined' ? window : global);
