/**
 * GFMR Table of Contents (TOC) Module
 *
 * Automatically generates and manages table of contents
 * with scroll tracking and smooth navigation
 *
 * @package MarkdownRendererForGitHub
 * @since 1.12.0
 */

(function(global) {
	'use strict';

	// Prevent duplicate initialization
	if (global.wpGfmTocInitialized) {
		return;
	}
	global.wpGfmTocInitialized = true;

	const wpGfmToc = {
		config: {
			enabled: false,
			headingMin: 2,
			headingMax: 4,
			position: 'right',
			scrollOffset: 0,
			sticky: true,
			mobileHidden: true
		},
		headings: [],
		tocElement: null,
		wrapperElement: null,
		observer: null,
		mutationObserver: null,

		/**
		 * Initialize TOC module
		 */
		init: function() {
			try {
				// Merge config from localized settings
				if (global.wpGfmSettings && global.wpGfmSettings.toc) {
					this.config = Object.assign({}, this.config, global.wpGfmSettings.toc);
				}

				// Exit if TOC is disabled
				if (!this.config.enabled) {
					return;
				}

				// Use requestAnimationFrame to ensure content is painted
				global.requestAnimationFrame(() => this.build());
			} catch (error) {
				console.error('GFMR TOC: Initialization failed', error);
			}
		},

		/**
		 * Build TOC
		 */
		build: function() {
			try {
				this.collectHeadings();

				if (this.headings.length === 0) {
					console.log('GFMR TOC: No headings found');
					return;
				}

				this.ensureHeadingIds();
				const tocHTML = this.buildTocHTML();
				this.insertToc(tocHTML);
				this.setupScrollTracking();
				this.setupMutationObserver();

				console.log(`GFMR TOC: Initialized with ${this.headings.length} headings`);
			} catch (error) {
				console.error('GFMR TOC: Build failed', error);
			}
		},

		/**
		 * Collect heading elements from document
		 */
		collectHeadings: function() {
			const selectors = [];
			for (let i = this.config.headingMin; i <= this.config.headingMax; i++) {
				selectors.push(`h${i}`);
			}
			const selector = selectors.join(',');

			// Look for headings in .gfmr-markdown-rendered container
			const container = document.querySelector('.gfmr-markdown-rendered');
			if (container) {
				this.headings = Array.from(container.querySelectorAll(selector));
			} else {
				// Fallback to document-wide search
				this.headings = Array.from(document.querySelectorAll(selector));
			}
		},

		/**
		 * Ensure all headings have IDs
		 */
		ensureHeadingIds: function() {
			this.headings.forEach((heading, index) => {
				if (!heading.id) {
					// Generate slug from heading text
					const slug = this.generateSlug(heading.textContent);
					heading.id = slug || `gfmr-toc-heading-${index}`;
				}
			});
		},

		/**
		 * Generate URL-safe slug from text
		 */
		generateSlug: function(text) {
			return text
				.toLowerCase()
				.trim()
				.replace(/[^\w\s-]/g, '')
				.replace(/[\s_-]+/g, '-')
				.replace(/^-+|-+$/g, '');
		},

		/**
		 * Build TOC HTML
		 */
		buildTocHTML: function() {
			const title = '<h2 class="gfmr-toc-title">Table of Contents</h2>';

			const list = this.headings.map(heading => {
				const level = parseInt(heading.tagName.substring(1), 10);
				const text = this.escapeHtml(heading.textContent);
				const id = heading.id;

				return `<li class="gfmr-toc-item">
					<a href="#${id}" class="gfmr-toc-link" data-level="${level}" aria-label="Jump to ${text}">
						${text}
					</a>
				</li>`;
			}).join('');

			return `${title}<ul class="gfmr-toc-list" role="navigation" aria-label="Table of contents">${list}</ul>`;
		},

		/**
		 * Escape HTML entities
		 */
		escapeHtml: function(text) {
			const div = document.createElement('div');
			div.textContent = text;
			return div.innerHTML;
		},

		/**
		 * Insert TOC into DOM
		 */
		insertToc: function(html) {
			const tocContainer = document.createElement('aside');
			tocContainer.className = `gfmr-toc-sidebar gfmr-toc-sidebar--${this.config.position}`;
			tocContainer.setAttribute('role', 'navigation');
			tocContainer.setAttribute('aria-label', 'Table of contents');

			if (this.config.mobileHidden) {
				tocContainer.classList.add('gfmr-toc-mobile-hidden');
			}

			tocContainer.innerHTML = html;

			// Find content area
			const contentArea = document.querySelector('.gfmr-markdown-rendered, .gfmr-render');
			if (!contentArea) {
				// Fallback: append to body
				document.body.appendChild(tocContainer);
				this.tocElement = tocContainer;
				this.tocElement.addEventListener('click', this.handleTocClick.bind(this));
				return;
			}

			// Create wrapper for flex layout
			const wrapper = document.createElement('div');
			wrapper.className = 'gfmr-toc-layout';

			// Wrap content area with the new layout
			contentArea.parentNode.insertBefore(wrapper, contentArea);
			wrapper.appendChild(contentArea);

			// Insert TOC based on position
			if (this.config.position === 'left') {
				wrapper.insertBefore(tocContainer, contentArea);
			} else {
				wrapper.appendChild(tocContainer);
			}

			this.tocElement = tocContainer;
			this.wrapperElement = wrapper; // Store for cleanup
			this.tocElement.addEventListener('click', this.handleTocClick.bind(this));
		},

		/**
		 * Setup scroll tracking with IntersectionObserver
		 */
		setupScrollTracking: function() {
			if (!('IntersectionObserver' in global)) {
				console.warn('GFMR TOC: IntersectionObserver not supported');
				return;
			}

			const rootMargin = `-${this.config.scrollOffset}px 0px -66% 0px`;

			this.observer = new IntersectionObserver((entries) => {
				let activeEntry = null;

				// Find the first intersecting heading
				entries.forEach(entry => {
					if (entry.isIntersecting && (!activeEntry || entry.target.compareDocumentPosition(activeEntry.target) & Node.DOCUMENT_POSITION_FOLLOWING)) {
						activeEntry = entry;
					}
				});

				if (activeEntry) {
					this.highlightActive(activeEntry.target.getAttribute('id'));
				}
			}, {
				rootMargin: rootMargin,
				threshold: 0
			});

			this.headings.forEach(heading => this.observer.observe(heading));
		},

		/**
		 * Setup MutationObserver for dynamic content
		 */
		setupMutationObserver: function() {
			if (!('MutationObserver' in global)) {
				return;
			}

			const container = document.querySelector('.gfmr-markdown-rendered, .gfmr-render');
			if (!container) {
				return;
			}

			this.mutationObserver = new MutationObserver((mutations) => {
				let shouldRebuild = false;

				mutations.forEach(mutation => {
					mutation.addedNodes.forEach(node => {
						if (node.nodeType === Node.ELEMENT_NODE && node.matches('h1, h2, h3, h4, h5, h6')) {
							shouldRebuild = true;
						}
					});

					mutation.removedNodes.forEach(node => {
						if (node.nodeType === Node.ELEMENT_NODE && node.matches('h1, h2, h3, h4, h5, h6')) {
							shouldRebuild = true;
						}
					});
				});

				if (shouldRebuild) {
					this.rebuild();
				}
			});

			this.mutationObserver.observe(container, {
				childList: true,
				subtree: true
			});
		},

		/**
		 * Rebuild TOC
		 */
		rebuild: function() {
			this.destroy();
			global.requestAnimationFrame(() => this.build());
		},

		/**
		 * Highlight active TOC item
		 */
		highlightActive: function(id) {
			if (!this.tocElement) {
				return;
			}

			// Remove all active classes
			this.tocElement.querySelectorAll('.gfmr-toc-item--active').forEach(item => {
				item.classList.remove('gfmr-toc-item--active');
			});

			// Add active class to current item
			const link = this.tocElement.querySelector(`a[href="#${id}"]`);
			if (link && link.parentElement) {
				link.parentElement.classList.add('gfmr-toc-item--active');

				// Scroll TOC item into view if needed
				link.scrollIntoView({
					behavior: 'smooth',
					block: 'nearest'
				});
			}
		},

		/**
		 * Handle TOC link clicks
		 */
		handleTocClick: function(event) {
			if (event.target.tagName === 'A') {
				event.preventDefault();
				const id = event.target.getAttribute('href').substring(1);
				this.scrollToHeading(id);

				// Update URL without page reload
				if (history.pushState) {
					history.pushState(null, null, `#${id}`);
				}
			}
		},

		/**
		 * Scroll to heading with smooth behavior
		 */
		scrollToHeading: function(id) {
			const heading = document.getElementById(id);
			if (!heading) {
				return;
			}

			const y = heading.getBoundingClientRect().top + global.pageYOffset - this.config.scrollOffset;

			global.scrollTo({
				top: y,
				behavior: 'smooth'
			});
		},

		/**
		 * Clean up observers and remove TOC
		 */
		destroy: function() {
			if (this.observer) {
				this.observer.disconnect();
				this.observer = null;
			}

			if (this.mutationObserver) {
				this.mutationObserver.disconnect();
				this.mutationObserver = null;
			}

			// Unwrap content if wrapper exists
			if (this.wrapperElement) {
				const contentArea = this.wrapperElement.querySelector('.gfmr-markdown-rendered, .gfmr-render');
				if (contentArea) {
					this.wrapperElement.parentNode.insertBefore(contentArea, this.wrapperElement);
				}
				this.wrapperElement.remove();
				this.wrapperElement = null;
			}

			if (this.tocElement) {
				this.tocElement.removeEventListener('click', this.handleTocClick);
				this.tocElement.remove();
				this.tocElement = null;
			}

			this.headings = [];
		}
	};

	// Expose to global scope
	global.wpGfmToc = wpGfmToc;

	// Initialize when DOM is ready
	if (document.readyState === 'loading') {
		document.addEventListener('DOMContentLoaded', () => wpGfmToc.init());
	} else {
		wpGfmToc.init();
	}

})(typeof window !== 'undefined' ? window : global);
