/**
 * GFMR Multilingual Frontend Handler
 *
 * Handles language switching for multilingual Markdown blocks on the frontend.
 * Supports URL parameters, localStorage persistence, and browser language detection.
 *
 * @package WpGfmRenderer
 */

( function() {
	'use strict';

	// Language code validation regex (ISO 639-1 format)
	const VALID_LANG_REGEX = /^[a-z]{2}(-[A-Z]{2})?$/;

	/**
	 * GFMRMultilingual class for handling language switching
	 */
	class GFMRMultilingual {
		/**
		 * Constructor
		 * @param {HTMLElement} container - The multilingual block container
		 */
		constructor( container ) {
			this.container = container;
			this.multilingualId = container.dataset.multilingualId;
			this.languages = this.loadLanguages();
			this.defaultLang = this.sanitizeLangCode( container.dataset.defaultLanguage ) || 'en';
			this.availableLangs = this.parseAvailableLanguages( container.dataset.availableLanguages );
			this.showSwitcher = container.dataset.showSwitcher === 'true';
			this.switcher = null;

			// Get URL mode settings
			this.urlMode = window.wpGfmConfig?.multilingualUrlMode || 'query';
			this.configDefaultLang = window.wpGfmConfig?.multilingualDefaultLang || 'en';
			this.hideDefaultLang = window.wpGfmConfig?.multilingualHideDefaultLang !== false;
			this.configSupportedLangs = window.wpGfmConfig?.multilingualSupportedLangs || [ 'en', 'ja' ];

			// Detect and set initial language
			this.currentLang = this.detectLanguage();

			// Initialize UI
			if ( this.showSwitcher && this.availableLangs.length > 1 ) {
				this.renderSwitcher();
			}

			// Switch to detected language
			this.switchLanguage( this.currentLang );

			// Handle History API events
			window.addEventListener( 'popstate', () => this.handlePopState() );
		}

		/**
		 * Validate language code format
		 * @param {string} code - Language code to validate
		 * @returns {boolean}
		 */
		isValidLanguageCode( code ) {
			return typeof code === 'string' &&
				code.length <= 5 &&
				VALID_LANG_REGEX.test( code );
		}

		/**
		 * Sanitize language code
		 * @param {string} code - Language code to sanitize
		 * @returns {string|null}
		 */
		sanitizeLangCode( code ) {
			if ( ! code ) return null;
			const sanitized = code.toLowerCase().trim().slice( 0, 5 );
			return this.isValidLanguageCode( sanitized ) ? sanitized : null;
		}

		/**
		 * Load languages data from script tag
		 * @returns {Object}
		 */
		loadLanguages() {
			if ( ! this.multilingualId ) {
				return {};
			}

			const scriptElement = document.getElementById( this.multilingualId );
			if ( ! scriptElement ) {
				console.error( '[GFMR] Languages data script not found:', this.multilingualId );
				return {};
			}

			try {
				const data = JSON.parse( scriptElement.textContent );
				// Validate object keys
				const validated = {};
				for ( const [ key, value ] of Object.entries( data ) ) {
					if ( this.isValidLanguageCode( key ) && value?.html !== undefined ) {
						validated[ key ] = value;
					}
				}
				return validated;
			} catch ( e ) {
				console.error( '[GFMR] Failed to parse languages data:', e );
				return {};
			}
		}

		/**
		 * Parse available languages array safely
		 * @param {string} jsonStr - JSON string of available languages
		 * @returns {string[]}
		 */
		parseAvailableLanguages( jsonStr ) {
			try {
				const arr = JSON.parse( jsonStr || '[]' );
				return arr.filter( lang => this.isValidLanguageCode( lang ) );
			} catch ( e ) {
				return [];
			}
		}

		/**
		 * Detect the best language to display
		 * Priority: URL path (path mode) > URL param > localStorage > browser language > default
		 * @returns {string}
		 */
		detectLanguage() {
			// 1. URL path (path mode)
			if ( this.urlMode === 'path' ) {
				const pathLang = this.detectLanguageFromPath();
				if ( pathLang && this.availableLangs.includes( pathLang ) ) {
					return pathLang;
				}
			}

			// 2. URL parameter (highest priority)
			const params = new URLSearchParams( window.location.search );
			const urlLang = this.sanitizeLangCode( params.get( 'lang' ) );
			if ( urlLang && this.availableLangs.includes( urlLang ) ) {
				return urlLang;
			}

			// 3. localStorage (user preference)
			try {
				const savedLang = this.sanitizeLangCode( localStorage.getItem( 'gfmr_preferred_lang' ) );
				if ( savedLang && this.availableLangs.includes( savedLang ) ) {
					return savedLang;
				}
			} catch ( e ) {
				// localStorage may be disabled
			}

			// 4. Browser language
			const browserLang = navigator.language.split( '-' )[ 0 ].toLowerCase();
			if ( this.isValidLanguageCode( browserLang ) && this.availableLangs.includes( browserLang ) ) {
				return browserLang;
			}

			// 5. Default language
			return this.defaultLang;
		}

		/**
		 * Detect language from URL path (for path mode)
		 * @returns {string|null}
		 */
		detectLanguageFromPath() {
			const path = window.location.pathname;
			const segments = path.split( '/' ).filter( s => s.length > 0 );

			if ( segments.length > 0 && this.isValidLanguageCode( segments[ 0 ] ) ) {
				return segments[ 0 ].toLowerCase();
			}

			// Default language is hidden in URL, return default
			return this.configDefaultLang;
		}

		/**
		 * Render the language switcher UI
		 */
		renderSwitcher() {
			const switcher = document.createElement( 'div' );
			switcher.className = 'gfmr-language-switcher';

			this.availableLangs.forEach( lang => {
				const btn = document.createElement( 'button' );
				btn.type = 'button';
				btn.className = 'gfmr-lang-btn';
				btn.textContent = lang.toUpperCase();
				btn.dataset.lang = lang;
				btn.addEventListener( 'click', () => this.switchLanguage( lang ) );
				switcher.appendChild( btn );
			} );

			// Insert switcher before the content container
			this.container.parentElement.insertBefore( switcher, this.container );
			this.switcher = switcher;
		}

		/**
		 * Switch to a different language
		 * @param {string} lang - Language code to switch to
		 */
		switchLanguage( lang ) {
			// Validate language code
			if ( ! this.isValidLanguageCode( lang ) || ! this.languages[ lang ] ) {
				console.warn( '[GFMR] Invalid language code:', lang );
				return;
			}

			this.currentLang = lang;

			// Clean up dynamic content before switching (prevent ID conflicts)
			this.cleanupDynamicContent();

			// Update HTML content (server-side rendered with wp_kses_post)
			this.container.innerHTML = this.languages[ lang ].html || '';

			// Update button active states
			if ( this.switcher ) {
				this.switcher.querySelectorAll( '.gfmr-lang-btn' ).forEach( btn => {
					btn.classList.toggle( 'active', btn.dataset.lang === lang );
				} );
			}

			// Save user preference to localStorage
			try {
				localStorage.setItem( 'gfmr_preferred_lang', lang );
			} catch ( e ) {
				// localStorage may be disabled
			}

			// Update URL (only in path mode)
			this.updateLanguageUrl( lang );

			// Re-process dynamic content (Mermaid, Chart.js)
			this.reprocessDynamicContent();
		}

		/**
		 * Update URL when language changes (path mode only)
		 * @param {string} lang - New language code
		 */
		updateLanguageUrl( lang ) {
			if ( this.urlMode !== 'path' ) {
				return;
			}

			// If default language and hiding is enabled, don't change URL
			if ( this.hideDefaultLang && lang === this.configDefaultLang ) {
				return;
			}

			const newPath = this.buildLanguagePath( lang );
			if ( newPath && newPath !== window.location.pathname ) {
				history.pushState( { lang: lang }, '', newPath );
			}
		}

		/**
		 * Build language path for current URL
		 * @param {string} lang - Language code
		 * @returns {string}
		 */
		buildLanguagePath( lang ) {
			const currentPath = window.location.pathname;
			const basePath = this.stripLanguagePrefix( currentPath );

			// Default language is hidden in URL
			if ( this.hideDefaultLang && lang === this.configDefaultLang ) {
				return basePath;
			}

			// Ensure path starts with /
			const normalizedBase = basePath.startsWith( '/' ) ? basePath : '/' + basePath;

			// Add language prefix
			return '/' + lang + normalizedBase;
		}

		/**
		 * Strip language prefix from path
		 * @param {string} path - Current path
		 * @returns {string}
		 */
		stripLanguagePrefix( path ) {
			const langPattern = new RegExp( '^/(' + this.configSupportedLangs.join( '|' ) + ')(/|$)' );
			return path.replace( langPattern, '/' );
		}

		/**
		 * Handle History API popstate event (browser back/forward)
		 */
		handlePopState() {
			const lang = this.detectLanguageFromPath();
			if ( lang && lang !== this.currentLang ) {
				this.switchLanguage( lang );
			}
		}

		/**
		 * Clean up dynamic content before language switch
		 * Prevents ID conflicts for Mermaid SVGs and Chart.js canvases
		 */
		cleanupDynamicContent() {
			// Remove Mermaid SVGs
			this.container.querySelectorAll( '.mermaid svg, .gfmr-mermaid svg' ).forEach( svg => {
				svg.remove();
			} );

			// Destroy Chart.js instances
			this.container.querySelectorAll( 'canvas[data-chart-id]' ).forEach( canvas => {
				const chartId = canvas.dataset.chartId;
				if ( window.Chart && window.Chart.getChart ) {
					const chart = window.Chart.getChart( canvas );
					if ( chart ) {
						chart.destroy();
					}
				}
			} );
		}

		/**
		 * Re-process dynamic content after language switch
		 * Triggers Mermaid and Chart.js re-rendering
		 */
		reprocessDynamicContent() {
			// Re-render Mermaid diagrams
			if ( typeof window.wpGfmProcessMermaidBlocksUnified === 'function' ) {
				try {
					window.wpGfmProcessMermaidBlocksUnified( this.container, 'frontend' );
				} catch ( e ) {
					console.error( '[GFMR] Mermaid reprocessing failed:', e );
				}
			}

			// Re-render Chart.js charts
			if ( typeof window.wpGfmProcessChartBlocks === 'function' ) {
				try {
					window.wpGfmProcessChartBlocks( this.container );
				} catch ( e ) {
					console.error( '[GFMR] Chart.js reprocessing failed:', e );
				}
			}
		}
	}

	/**
	 * Initialize all multilingual blocks on the page
	 */
	function initMultilingualBlocks() {
		const containers = document.querySelectorAll( '.gfmr-markdown-rendered[data-multilingual-id]' );

		containers.forEach( container => {
			// Skip if already initialized
			if ( container.dataset.gfmrMultilingualInit ) {
				return;
			}

			// Mark as initialized
			container.dataset.gfmrMultilingualInit = 'true';

			// Create instance
			new GFMRMultilingual( container );
		} );
	}

	// Initialize on DOMContentLoaded
	if ( document.readyState === 'loading' ) {
		document.addEventListener( 'DOMContentLoaded', initMultilingualBlocks );
	} else {
		initMultilingualBlocks();
	}

	// Expose for external use
	window.GFMRMultilingual = GFMRMultilingual;
	window.gfmrInitMultilingualBlocks = initMultilingualBlocks;

} )();
