/**
 * GFMR Mermaid Lightbox Module
 * Click-to-expand functionality for Mermaid diagrams
 *
 * @package WpGfmRenderer
 * @since 1.11.0
 */

(function(global) {
	'use strict';

	// Prevent duplicate initialization
	if (global.gfmrMermaidLightboxInitialized) return;
	global.gfmrMermaidLightboxInitialized = true;

	class GFMRMermaidLightbox {
		constructor() {
			this.modal = null;
			this.currentDiagram = null;
			this.currentZoom = 1.0;
			this.zoomConfig = { min: 0.25, max: 4.0, step: 0.25 };
			this.panOffset = { x: 0, y: 0 };
			this.isDragging = false;
			this.dragStart = { x: 0, y: 0 };
			this.previousActiveElement = null;
			this.focusableElements = null;
			this.abortController = null;
			this.touchEndTime = 0;
		}

		init() {
			this.abortController = new AbortController();
			this.createModal();
			this.bindGlobalEvents();
			console.log('[GFMR Lightbox] Initialized');
		}

		destroy() {
			this.abortController?.abort();
			this.modal?.remove();
			global.gfmrMermaidLightboxInitialized = false;
		}

		createModal() {
			if (document.querySelector('.gfmr-mermaid-zoom-modal')) {
				this.modal = document.querySelector('.gfmr-mermaid-zoom-modal');
				return;
			}

			const modal = document.createElement('div');
			modal.className = 'gfmr-mermaid-zoom-modal';
			modal.setAttribute('role', 'dialog');
			modal.setAttribute('aria-modal', 'true');
			modal.setAttribute('aria-labelledby', 'gfmr-lightbox-title');
			modal.setAttribute('aria-describedby', 'gfmr-lightbox-instructions');
			modal.innerHTML = `
				<div class="modal-overlay" data-action="close"></div>
				<div class="modal-dialog">
					<div class="modal-header">
						<h3 class="modal-title" id="gfmr-lightbox-title">Diagram</h3>
						<div class="modal-controls">
							<button type="button" class="zoom-btn" data-action="zoom-out" aria-label="Zoom out">−</button>
							<button type="button" class="zoom-btn" data-action="zoom-reset" aria-label="Reset zoom">⌂</button>
							<button type="button" class="zoom-btn" data-action="zoom-in" aria-label="Zoom in">+</button>
							<button type="button" class="close-btn" data-action="close" aria-label="Close">×</button>
						</div>
					</div>
					<div class="modal-content">
						<div class="diagram-container">
							<div class="zoom-wrapper"></div>
						</div>
					</div>
					<div class="modal-footer">
						<div class="zoom-info">
							<span>Zoom:</span>
							<span class="zoom-level">100%</span>
						</div>
						<span class="instructions" id="gfmr-lightbox-instructions">
							Scroll to zoom, drag to pan
						</span>
					</div>
				</div>
			`;

			document.body.appendChild(modal);
			this.modal = modal;
			this.bindModalEvents();
		}

		bindGlobalEvents() {
			const signal = this.abortController.signal;

			document.addEventListener('keydown', (e) => {
				if (!this.modal?.classList.contains('active')) return;
				this.handleKeydown(e);
			}, { signal });
		}

		bindModalEvents() {
			// Modal control buttons
			this.modal.addEventListener('click', (e) => {
				const action = e.target.dataset.action;
				if (action === 'close') this.close();
				if (action === 'zoom-in') this.zoomIn();
				if (action === 'zoom-out') this.zoomOut();
				if (action === 'zoom-reset') this.zoomReset();
			});

			const container = this.modal.querySelector('.diagram-container');

			// Wheel zoom with throttling
			this.throttledZoom = this.throttle((delta) => {
				this.setZoom(this.currentZoom + delta);
			}, 50);

			container.addEventListener('wheel', (e) => {
				e.preventDefault();
				const delta = e.deltaY > 0 ? -this.zoomConfig.step : this.zoomConfig.step;
				this.throttledZoom(delta);
			});

			// Drag pan (mouse)
			container.addEventListener('mousedown', (e) => this.startDrag(e));
			container.addEventListener('mousemove', (e) => this.drag(e));
			container.addEventListener('mouseup', () => this.endDrag());
			container.addEventListener('mouseleave', () => this.endDrag());

			// Drag pan (touch)
			container.addEventListener('touchstart', (e) => this.startDrag(e), { passive: true });
			container.addEventListener('touchmove', (e) => {
				e.preventDefault();
				this.drag(e);
			}, { passive: false });
			container.addEventListener('touchend', () => this.endDrag());
		}

		handleKeydown(e) {
			switch (e.key) {
				case 'Escape':
					this.close();
					break;
				case 'Tab':
					this.handleTabKey(e);
					break;
				case '+':
				case '=':
					e.preventDefault();
					this.zoomIn();
					break;
				case '-':
					e.preventDefault();
					this.zoomOut();
					break;
				case '0':
					e.preventDefault();
					this.zoomReset();
					break;
			}
		}

		open(diagramElement) {
			const svg = diagramElement.querySelector('svg');
			if (!svg) return;

			this.previousActiveElement = document.activeElement;

			// Clone SVG with deep copy
		const clonedSvg = svg.cloneNode(true);

		// Preserve SVG ID to maintain CSS selector matching
		// Change ID to avoid conflicts with original SVG
		if (clonedSvg.id) {
			const originalId = clonedSvg.id;
			const newId = originalId + '-lightbox-' + Date.now();
			clonedSvg.id = newId;

			// Update ID selectors in <style> tags
			const styleElements = clonedSvg.querySelectorAll('style');
			styleElements.forEach(style => {
				// Replace #originalId with #newId in CSS selectors
				style.textContent = style.textContent.replace(
					new RegExp(`#${originalId}\\b`, 'g'),
					`#${newId}`
				);
			});
		}

			const wrapper = this.modal.querySelector('.zoom-wrapper');
			wrapper.innerHTML = '';
			wrapper.appendChild(clonedSvg);

			// Reset zoom/pan
			this.currentZoom = 1.0;
			this.panOffset = { x: 0, y: 0 };
			this.updateTransform();
			this.updateZoomLevel();

			// Set diagram type
			const diagramType = this.detectDiagramType(diagramElement);
			this.modal.querySelector('.modal-title').textContent = diagramType;

			// Show modal
			this.modal.classList.add('active');
			document.body.classList.add('gfmr-modal-open');

			// Update trigger element aria-expanded
			diagramElement.closest('[aria-haspopup="dialog"]')?.setAttribute('aria-expanded', 'true');

			// Focus management
			this.setupFocusTrap();
			this.modal.querySelector('.close-btn').focus();

			this.currentDiagram = diagramElement;
		}

		close() {
			this.modal.classList.remove('active');
			document.body.classList.remove('gfmr-modal-open');

			// Clear SVG clone (memory cleanup)
			const wrapper = this.modal.querySelector('.zoom-wrapper');
			wrapper.innerHTML = '';

			// Update trigger element aria-expanded
			this.currentDiagram?.closest('[aria-haspopup="dialog"]')?.setAttribute('aria-expanded', 'false');

			// Restore focus (check element still exists in DOM)
			if (this.previousActiveElement?.isConnected) {
				this.previousActiveElement.focus();
			}
			this.previousActiveElement = null;
			this.currentDiagram = null;
		}

		// Zoom methods
		zoomIn() {
			this.setZoom(this.currentZoom + this.zoomConfig.step);
		}

		zoomOut() {
			this.setZoom(this.currentZoom - this.zoomConfig.step);
		}

		zoomReset() {
			this.currentZoom = 1.0;
			this.panOffset = { x: 0, y: 0 };
			this.updateTransform();
			this.updateZoomLevel();
		}

		setZoom(newZoom) {
			this.currentZoom = Math.max(
				this.zoomConfig.min,
				Math.min(this.zoomConfig.max, newZoom)
			);
			this.updateTransform();
			this.updateZoomLevel();
		}

		updateTransform() {
			const wrapper = this.modal.querySelector('.zoom-wrapper');
			wrapper.style.transform = `translate(${this.panOffset.x}px, ${this.panOffset.y}px) scale(${this.currentZoom})`;
		}

		updateZoomLevel() {
			const percent = Math.round(this.currentZoom * 100);
			this.modal.querySelector('.zoom-level').textContent = `${percent}%`;
		}

		// Drag methods
		startDrag(e) {
			this.isDragging = true;
			const clientX = e.type.includes('touch') ? e.touches[0].clientX : e.clientX;
			const clientY = e.type.includes('touch') ? e.touches[0].clientY : e.clientY;
			this.dragStart = {
				x: clientX - this.panOffset.x,
				y: clientY - this.panOffset.y
			};
		}

		drag(e) {
			if (!this.isDragging) return;
			const clientX = e.type.includes('touch') ? e.touches[0].clientX : e.clientX;
			const clientY = e.type.includes('touch') ? e.touches[0].clientY : e.clientY;
			this.panOffset.x = clientX - this.dragStart.x;
			this.panOffset.y = clientY - this.dragStart.y;
			this.updateTransform();
		}

		endDrag() {
			this.isDragging = false;
		}

		// Focus management
		setupFocusTrap() {
			this.focusableElements = this.modal.querySelectorAll(
				'button:not([disabled]), [href], input:not([disabled]), select:not([disabled]), textarea:not([disabled]), [tabindex]:not([tabindex="-1"])'
			);
		}

		handleTabKey(e) {
			if (!this.focusableElements || this.focusableElements.length === 0) return;

			const firstElement = this.focusableElements[0];
			const lastElement = this.focusableElements[this.focusableElements.length - 1];

			if (e.shiftKey && document.activeElement === firstElement) {
				e.preventDefault();
				lastElement.focus();
			} else if (!e.shiftKey && document.activeElement === lastElement) {
				e.preventDefault();
				firstElement.focus();
			}
		}

		// Utilities
		detectDiagramType(element) {
			const source = element.querySelector('.mermaid-source')?.textContent || '';
			if (source.includes('flowchart') || source.includes('graph')) return 'Flowchart';
			if (source.includes('sequenceDiagram')) return 'Sequence Diagram';
			if (source.includes('classDiagram')) return 'Class Diagram';
			if (source.includes('stateDiagram')) return 'State Diagram';
			if (source.includes('erDiagram')) return 'ER Diagram';
			if (source.includes('gantt')) return 'Gantt Chart';
			if (source.includes('pie')) return 'Pie Chart';
			if (source.includes('gitGraph')) return 'Git Graph';
			if (source.includes('journey')) return 'User Journey';
			return 'Diagram';
		}

		throttle(func, limit) {
			let inThrottle;
			return (...args) => {
				if (!inThrottle) {
					func.apply(this, args);
					inThrottle = true;
					setTimeout(() => inThrottle = false, limit);
				}
			};
		}
	}

	// Create instance and expose in wpGfm namespace
	const lightbox = new GFMRMermaidLightbox();

	if (document.readyState === 'loading') {
		document.addEventListener('DOMContentLoaded', () => lightbox.init());
	} else {
		lightbox.init();
	}

	// Integrate with wpGfm namespace to reduce collision risk
	global.wpGfm = global.wpGfm || {};
	global.wpGfm.lightbox = lightbox;

})(typeof window !== 'undefined' ? window : this);
