/**
 * GFMR Main Module
 *
 * GitHub Flavored Markdown Renderer main functionality
 * Provides syntax highlighting and Mermaid diagrams
 *
 * @package WpGfmRenderer
 * @since 1.0.0
 */

(function (global) {
  "use strict";

  // Prevent duplicate execution
  if (global.gfmrMainInitialized) return;
  global.gfmrMainInitialized = true;

  console.log("[GFMR Main] Initialization started");

  /**
   * URL builder helper function
   * Local asset URL construction compliant with WordPress.org distribution
   *
   * @param {string} path - Relative path (e.g., 'assets/libs/shiki/shiki.min.js')
   * @returns {string} - Complete URL
   */
  function buildAssetUrl(path) {
    // Try to get plugin URL from wpGfmConfig (localized from PHP)
    let baseUrl = window.wpGfmConfig?.pluginUrl;

    // Fallback: detect from current script location
    if (!baseUrl) {
      const currentScript = document.currentScript;
      if (currentScript && currentScript.src) {
        // Extract base path from current script URL
        const scriptUrl = currentScript.src;
        const match = scriptUrl.match(/(.+\/wp-content\/plugins\/[^/]+)/);
        baseUrl = match
          ? match[1] + "/"
          : "/wp-content/plugins/markdown-renderer-for-github/";
      } else {
        // Last resort fallback
        baseUrl = "/wp-content/plugins/markdown-renderer-for-github/";
      }
      console.log("[WP GFM] Using fallback base URL:", baseUrl);
    }

    // Normalize trailing / in base URL
    const normalizedBase = baseUrl.replace(/\/+$/, "");
    // Normalize leading / in path
    const normalizedPath = path.replace(/^\/+/, "");
    return `${normalizedBase}/${normalizedPath}`;
  }

  // Expose globally (available in other files too)
  if (!global.wpGfmBuildAssetUrl) {
    global.wpGfmBuildAssetUrl = buildAssetUrl;
  }

  const hotfix = {
    shikiLoaded: false,
    mermaidLoaded: false,
    katexLoaded: false,
    processedElements: new WeakSet(),
    processedMathElements: new WeakSet(),
    isProcessing: false, // Processing flag to prevent recursion
    initialized: false, // Initialization completion flag

    // Shiki CDN loading (optimized version)
    async loadShiki() {
      if (global.shiki && global.wpGfmShikiHighlighter) {
        this.shikiLoaded = true;
        console.log("[WP GFM v2 Hotfix] Shiki already loaded");
        return;
      }

      console.log("[WP GFM v2 Hotfix] Loading Shiki...");
      return new Promise((resolve, reject) => {
        const script = document.createElement("script");
        script.src = buildAssetUrl("assets/libs/shiki/shiki.min.js");

        script.onload = async () => {
          try {
            // Get theme from wpGfmConfig/wpGfmSettings or auto-detect from system
            let shikiTheme =
              window.wpGfmConfig?.theme?.shiki_theme ??
              window.wpGfmSettings?.theme?.shiki_theme ??
              (window.matchMedia?.('(prefers-color-scheme: dark)').matches ? 'github-dark' : 'github-light');

            // Handle 'auto' theme - detect system preference
            if (shikiTheme === "auto") {
              const prefersDark =
                window.matchMedia &&
                window.matchMedia("(prefers-color-scheme: dark)").matches;
              shikiTheme = prefersDark ? "github-dark" : "github-light";
              console.log(
                "[WP GFM v2 Hotfix] Auto theme detected, using:",
                shikiTheme,
              );
            }

            console.log("[WP GFM v2 Hotfix] Using Shiki theme:", shikiTheme);

            // Apply CSS class to body for theme-aware styling
            if (shikiTheme === "github-dark") {
              document.body.classList.add("gfmr-dark");
              document.body.classList.remove("gfmr-light");
            } else {
              document.body.classList.add("gfmr-light");
              document.body.classList.remove("gfmr-dark");
            }

            // Initialize Shiki highlighter with extended language list
            global.wpGfmShikiHighlighter = await global.shiki.getHighlighter({
              theme: shikiTheme,
              langs: [
                "javascript",
                "js",
                "typescript",
                "ts",
                "python",
                "py",
                "java",
                "cpp",
                "c",
                "csharp",
                "cs",
                "html",
                "css",
                "scss",
                "sass",
                "json",
                "xml",
                "yaml",
                "yml",
                "bash",
                "shell",
                "sh",
                "zsh",
                "php",
                "ruby",
                "go",
                "rust",
                "sql",
                "markdown",
                "md",
                "diff",
              ],
            });

            this.shikiLoaded = true;
            console.log("[WP GFM v2 Hotfix] Shiki loading completed");
            resolve();
          } catch (error) {
            console.error(
              "[WP GFM v2 Hotfix] Shiki initialization error:",
              error,
            );
            reject(error);
          }
        };

        script.onerror = (error) => {
          console.error("[WP GFM v2 Hotfix] Shiki loading failed:", error);
          console.error("Script URL:", script.src);
          // Still resolve to allow degraded functionality
          resolve();
        };

        document.head.appendChild(script);
      });
    },

    // Mermaid CDN loading (optimized version)
    async loadMermaid() {
      if (global.mermaid && global.mermaid.render) {
        this.mermaidLoaded = true;
        console.log("[WP GFM v2 Hotfix] Mermaid already loaded");
        return;
      }

      console.log("[WP GFM v2 Hotfix] Loading Mermaid...");
      return new Promise((resolve, reject) => {
        const script = document.createElement("script");
        script.src = buildAssetUrl("assets/libs/mermaid/mermaid.min.js");

        script.onload = () => {
          try {
            global.mermaid.initialize({
              startOnLoad: false,
              theme: "default",
              // Use 'loose' instead of 'sandbox' to render inline SVG
              // This fixes Japanese text encoding issues (sandbox uses data URLs)
              securityLevel: "loose",
              fontFamily: "monospace",
              // Centering configuration for diagrams
              flowchart: {
                useMaxWidth: false,
                htmlLabels: true,
                curve: "basis",
              },
              sequence: {
                useMaxWidth: true,
                height: 65,
              },
              class: {
                useMaxWidth: true,
              },
              state: {
                useMaxWidth: false,
              },
              er: {
                useMaxWidth: false,
              },
              gantt: {
                useMaxWidth: true,
              },
              gitGraph: {
                useMaxWidth: false,
                mainLineWidth: 2,
                mainBranchOrder: 0,
              },
            });

            this.mermaidLoaded = true;
            console.log("[WP GFM v2 Hotfix] Mermaid loading completed");
            resolve();
          } catch (error) {
            console.error(
              "[WP GFM v2 Hotfix] Mermaid initialization error:",
              error,
            );
            reject(error);
          }
        };

        script.onerror = (error) => {
          console.error("[WP GFM v2 Hotfix] Mermaid loading failed:", error);
          console.error("Script URL:", script.src);
          // Still resolve to allow degraded functionality
          resolve();
        };

        document.head.appendChild(script);
      });
    },

    // KaTeX CSS loading
    async loadKaTeXCSS() {
      return new Promise((resolve) => {
        // Check if already loaded
        if (document.querySelector('link[href*="katex"]')) {
          resolve();
          return;
        }

        const link = document.createElement("link");
        link.rel = "stylesheet";
        link.href = buildAssetUrl("assets/libs/katex/katex.min.css");
        link.onload = resolve;
        link.onerror = () => {
          console.warn("[WP GFM v2 Hotfix] KaTeX CSS loading failed");
          resolve();
        };
        document.head.appendChild(link);
      });
    },

    // KaTeX library loading (conditional)
    async loadKaTeX() {
      if (global.katex) {
        this.katexLoaded = true;
        console.log("[WP GFM v2 Hotfix] KaTeX already loaded");
        return;
      }

      console.log("[WP GFM v2 Hotfix] Loading KaTeX...");

      // Load CSS first
      await this.loadKaTeXCSS();

      return new Promise((resolve) => {
        const script = document.createElement("script");
        script.src = buildAssetUrl("assets/libs/katex/katex.min.js");

        script.onload = () => {
          this.katexLoaded = true;
          console.log("[WP GFM v2 Hotfix] KaTeX loading completed");
          resolve();
        };

        script.onerror = (error) => {
          console.error("[WP GFM v2 Hotfix] KaTeX loading failed:", error);
          // Still resolve to allow degraded functionality
          resolve();
        };

        document.head.appendChild(script);
      });
    },

    // Check if page has math content
    hasMathContent() {
      // Check for existing math elements
      const mathElements = document.querySelectorAll(
        "[data-latex], .latex-formula, .katex, .math-tex, .gfmr-math",
      );
      if (mathElements.length > 0) {
        return true;
      }

      // Check for math patterns in text content
      const content = document.body.textContent || "";
      const patterns = [
        /\$\$[\s\S]+?\$\$/, // Block math: $$...$$
        /\$[^$\n]+\$/, // Inline math: $...$
        /\\\([^)]+\\\)/, // \(...\)
        /\\\[[\s\S]+?\\\]/, // \[...\]
        /\[latex\][\s\S]*?\[\/latex\]/i, // WordPress shortcode
      ];

      return patterns.some((p) => p.test(content));
    },

    // Check if page has chart content
    hasChartContent() {
      // Check for chart elements
      const chartElements = document.querySelectorAll(
        'code[class*="language-chart"], code.language-chart, code.language-chart-pro',
      );
      return chartElements.length > 0;
    },

    // Load gfmr-charts.js (lazy loading)
    async loadGfmrCharts() {
      if (global.wpGfmCharts) {
        console.log("[WP GFM v2 Hotfix] Charts module already loaded");
        return;
      }

      console.log("[WP GFM v2 Hotfix] Loading Charts module...");

      return new Promise((resolve, reject) => {
        const script = document.createElement("script");
        script.src = buildAssetUrl("assets/js/gfmr-charts.js");

        script.onload = () => {
          if (typeof global.wpGfmInitializeCharts === "function") {
            global.wpGfmInitializeCharts();
            console.log("[WP GFM v2 Hotfix] Charts module loaded successfully");
            resolve();
          } else {
            console.error("[WP GFM v2 Hotfix] Charts module initialization function not found");
            reject(new Error("Charts initialization failed"));
          }
        };

        script.onerror = (error) => {
          console.error("[WP GFM v2 Hotfix] Charts module loading failed:", error);
          reject(error);
        };

        document.head.appendChild(script);
      });
    },

    // Render block math ($$...$$)
    renderBlockMath() {
      if (!global.katex) return;

      const containers = document.querySelectorAll(
        ".gfmr-markdown-rendered, .entry-content, .post-content, article",
      );

      containers.forEach((container) => {
        const walker = document.createTreeWalker(
          container,
          NodeFilter.SHOW_TEXT,
          {
            acceptNode: (node) => {
              // Skip if already processed
              if (node.parentElement?.closest(".katex")) {
                return NodeFilter.FILTER_REJECT;
              }
              // Skip if inside code/pre elements
              if (node.parentElement?.closest("code, pre")) {
                return NodeFilter.FILTER_REJECT;
              }
              return NodeFilter.FILTER_ACCEPT;
            },
          },
        );

        const nodesToProcess = [];
        let node;
        while ((node = walker.nextNode()) !== null) {
          if (/\$\$[\s\S]+?\$\$/.test(node.textContent)) {
            nodesToProcess.push(node);
          }
        }

        nodesToProcess.forEach((textNode) => {
          this.processBlockMathNode(textNode);
        });
      });
    },

    // Process a single text node for block math
    processBlockMathNode(textNode) {
      if (!global.katex) return;
      if (this.processedMathElements.has(textNode)) return;

      const text = textNode.textContent;
      const blockMathRegex = /\$\$([\s\S]+?)\$\$/g;

      let match;
      let lastIndex = 0;
      const fragments = [];

      while ((match = blockMathRegex.exec(text)) !== null) {
        // Add text before the match
        if (match.index > lastIndex) {
          fragments.push(
            document.createTextNode(text.slice(lastIndex, match.index)),
          );
        }

        // Create math element
        const mathContainer = document.createElement("div");
        mathContainer.className = "gfmr-math gfmr-math-block";

        try {
          global.katex.render(match[1].trim(), mathContainer, {
            displayMode: true,
            throwOnError: false,
            output: "html",
          });
        } catch (error) {
          console.warn("[WP GFM v2 Hotfix] KaTeX render error:", error);
          mathContainer.textContent = match[0];
        }

        fragments.push(mathContainer);
        lastIndex = blockMathRegex.lastIndex;
      }

      // Add remaining text
      if (lastIndex < text.length) {
        fragments.push(document.createTextNode(text.slice(lastIndex)));
      }

      // Replace the text node if we found matches
      if (fragments.length > 0 && lastIndex > 0) {
        const parent = textNode.parentNode;
        fragments.forEach((fragment) => {
          parent.insertBefore(fragment, textNode);
        });
        parent.removeChild(textNode);
        this.processedMathElements.add(textNode);
      }
    },

    // Render inline math ($...$)
    renderInlineMath() {
      if (!global.katex) return;

      const containers = document.querySelectorAll(
        ".gfmr-markdown-rendered, .entry-content, .post-content, article",
      );

      containers.forEach((container) => {
        const walker = document.createTreeWalker(
          container,
          NodeFilter.SHOW_TEXT,
          {
            acceptNode: (node) => {
              // Skip if already processed
              if (node.parentElement?.closest(".katex, .gfmr-math")) {
                return NodeFilter.FILTER_REJECT;
              }
              // Skip if inside code/pre elements
              if (node.parentElement?.closest("code, pre")) {
                return NodeFilter.FILTER_REJECT;
              }
              return NodeFilter.FILTER_ACCEPT;
            },
          },
        );

        const nodesToProcess = [];
        let node;
        while ((node = walker.nextNode()) !== null) {
          // Match $...$ but not $$...$$
          if (/(?<!\$)\$(?!\$)[^$\n]+\$(?!\$)/.test(node.textContent)) {
            nodesToProcess.push(node);
          }
        }

        nodesToProcess.forEach((textNode) => {
          this.processInlineMathNode(textNode);
        });
      });
    },

    // Process a single text node for inline math
    processInlineMathNode(textNode) {
      if (!global.katex) return;
      if (this.processedMathElements.has(textNode)) return;

      const text = textNode.textContent;
      // Match $...$ but not $$...$$
      const inlineMathRegex = /(?<!\$)\$(?!\$)([^$\n]+)\$(?!\$)/g;

      let match;
      let lastIndex = 0;
      const fragments = [];

      while ((match = inlineMathRegex.exec(text)) !== null) {
        // Add text before the match
        if (match.index > lastIndex) {
          fragments.push(
            document.createTextNode(text.slice(lastIndex, match.index)),
          );
        }

        // Create math element
        const mathContainer = document.createElement("span");
        mathContainer.className = "gfmr-math gfmr-math-inline";

        try {
          global.katex.render(match[1].trim(), mathContainer, {
            displayMode: false,
            throwOnError: false,
            output: "html",
          });
        } catch (error) {
          console.warn("[WP GFM v2 Hotfix] KaTeX inline render error:", error);
          mathContainer.textContent = match[0];
        }

        fragments.push(mathContainer);
        lastIndex = inlineMathRegex.lastIndex;
      }

      // Add remaining text
      if (lastIndex < text.length) {
        fragments.push(document.createTextNode(text.slice(lastIndex)));
      }

      // Replace the text node if we found matches
      if (fragments.length > 0 && lastIndex > 0) {
        const parent = textNode.parentNode;
        fragments.forEach((fragment) => {
          parent.insertBefore(fragment, textNode);
        });
        parent.removeChild(textNode);
        this.processedMathElements.add(textNode);
      }
    },

    // Render LaTeX shortcodes [latex]...[/latex]
    renderLatexShortcodes() {
      if (!global.katex) return;

      const containers = document.querySelectorAll(
        ".gfmr-markdown-rendered, .entry-content, .post-content, article",
      );

      containers.forEach((container) => {
        const walker = document.createTreeWalker(
          container,
          NodeFilter.SHOW_TEXT,
          {
            acceptNode: (node) => {
              // Skip if already processed
              if (node.parentElement?.closest(".katex, .gfmr-math")) {
                return NodeFilter.FILTER_REJECT;
              }
              // Skip if inside code/pre elements
              if (node.parentElement?.closest("code, pre")) {
                return NodeFilter.FILTER_REJECT;
              }
              return NodeFilter.FILTER_ACCEPT;
            },
          },
        );

        const nodesToProcess = [];
        let node;
        while ((node = walker.nextNode()) !== null) {
          if (/\[latex\][\s\S]*?\[\/latex\]/i.test(node.textContent)) {
            nodesToProcess.push(node);
          }
        }

        nodesToProcess.forEach((textNode) => {
          this.processLatexShortcodeNode(textNode);
        });
      });
    },

    // Process a single text node for [latex] shortcodes
    processLatexShortcodeNode(textNode) {
      if (!global.katex) return;
      if (this.processedMathElements.has(textNode)) return;

      const text = textNode.textContent;
      const shortcodeRegex = /\[latex\]([\s\S]*?)\[\/latex\]/gi;

      let match;
      let lastIndex = 0;
      const fragments = [];

      while ((match = shortcodeRegex.exec(text)) !== null) {
        // Add text before the match
        if (match.index > lastIndex) {
          fragments.push(
            document.createTextNode(text.slice(lastIndex, match.index)),
          );
        }

        // Create math element
        const mathContainer = document.createElement("span");
        mathContainer.className = "gfmr-math gfmr-math-shortcode";

        try {
          global.katex.render(match[1].trim(), mathContainer, {
            displayMode: false,
            throwOnError: false,
            output: "html",
          });
        } catch (error) {
          console.warn(
            "[WP GFM v2 Hotfix] KaTeX shortcode render error:",
            error,
          );
          mathContainer.textContent = match[0];
        }

        fragments.push(mathContainer);
        lastIndex = shortcodeRegex.lastIndex;
      }

      // Add remaining text
      if (lastIndex < text.length) {
        fragments.push(document.createTextNode(text.slice(lastIndex)));
      }

      // Replace the text node if we found matches
      if (fragments.length > 0 && lastIndex > 0) {
        const parent = textNode.parentNode;
        fragments.forEach((fragment) => {
          parent.insertBefore(fragment, textNode);
        });
        parent.removeChild(textNode);
        this.processedMathElements.add(textNode);
      }
    },

    // Main math rendering function
    async renderMathFormulas() {
      if (!global.katex) {
        console.warn(
          "[WP GFM v2 Hotfix] KaTeX not loaded, skipping math rendering",
        );
        return;
      }

      console.log("[WP GFM v2 Hotfix] Starting math formula rendering");

      // Render in order: block first, then inline, then shortcodes
      this.renderBlockMath();
      this.renderInlineMath();
      this.renderLatexShortcodes();

      console.log("[WP GFM v2 Hotfix] Math formula rendering completed");
    },

    // Add copy button
    addCopyButton(container, content, type = "code") {
      // Check for existing copy button
      if (container.querySelector(".gfmr-copy-button")) {
        return;
      }

      // Load constants
      const constants = window.wpGfmConstants || {};
      const CLASSES = constants.CLASSES || {
        copyButton: "gfmr-copy-button",
        copyButtonCode: "gfmr-copy-button--code",
        copyButtonMermaid: "gfmr-copy-button--mermaid",
        copyButtonContainer: "gfmr-copy-button-container",
        copyButtonSuccess: "gfmr-copy-button--success",
        copyButtonError: "gfmr-copy-button--error",
      };
      const ICONS = constants.ICONS || {};
      const LABELS = constants.LABELS || {
        copy: "Copy",
        copied: "Copied!",
        failed: "Failed",
      };

      // Create copy button container
      const buttonContainer = document.createElement("div");
      buttonContainer.className = CLASSES.copyButtonContainer;
      buttonContainer.style.cssText = `
                position: absolute;
                top: 8px;
                right: 8px;
                z-index: 10;
            `;

      // Create copy button
      const button = document.createElement("button");
      const typeClass =
        type === "mermaid" ? CLASSES.copyButtonMermaid : CLASSES.copyButtonCode;

      button.className = `${CLASSES.copyButton} ${typeClass}`;
      button.setAttribute("aria-label", "Copy code");

      // Add icon and text
      const copyIcon =
        ICONS.copy ||
        `<svg width="16" height="16" viewBox="0 0 16 16" fill="currentColor">
                <path d="M0 6.75C0 5.784.784 5 1.75 5h1.5a.75.75 0 0 1 0 1.5h-1.5a.25.25 0 0 0-.25.25v7.5c0 .138.112.25.25.25h7.5a.25.25 0 0 0 .25-.25v-1.5a.75.75 0 0 1 1.5 0v1.5A1.75 1.75 0 0 1 9.25 16h-7.5A1.75 1.75 0 0 1 0 14.25Z"></path>
                <path d="M5 1.75C5 .784 5.784 0 6.75 0h7.5C15.216 0 16 .784 16 1.75v7.5A1.75 1.75 0 0 1 14.25 11h-7.5A1.75 1.75 0 0 1 5 9.25Zm1.75-.25a.25.25 0 0 0-.25.25v7.5c0 .138.112.25.25.25h7.5a.25.25 0 0 0 .25-.25v-7.5a.25.25 0 0 0-.25-.25Z"></path>
            </svg>`;

      button.innerHTML = `
                ${copyIcon}
                <span>${LABELS.copy}</span>
            `;

      // Hover effects are now handled by CSS

      // Click event handler
      button.addEventListener("click", async () => {
        try {
          // Get content to copy
          let textToCopy = "";
          if (type === "mermaid") {
            // For Mermaid, use original code
            const mermaidCode = container.querySelector(".mermaid-source");
            textToCopy = mermaidCode ? mermaidCode.textContent : content;
          } else {
            // For code blocks
            textToCopy = content || container.textContent;
          }

          // Copy to clipboard
          if (navigator.clipboard && window.isSecureContext) {
            await navigator.clipboard.writeText(textToCopy);
          } else {
            // Fallback for older browsers
            const textarea = document.createElement("textarea");
            textarea.value = textToCopy;
            textarea.style.position = "fixed";
            textarea.style.left = "-999999px";
            document.body.appendChild(textarea);
            textarea.select();
            document.execCommand("copy");
            document.body.removeChild(textarea);
          }

          // Success feedback
          const originalHTML = button.innerHTML;
          const successIcon =
            ICONS.success ||
            `<svg width="16" height="16" viewBox="0 0 16 16" fill="currentColor">
                        <path d="M13.78 4.22a.75.75 0 0 1 0 1.06l-7.25 7.25a.75.75 0 0 1-1.06 0L2.22 9.28a.751.751 0 0 1 .018-1.042.751.751 0 0 1 1.042-.018L6 10.94l6.72-6.72a.75.75 0 0 1 1.06 0Z"></path>
                    </svg>`;

          button.innerHTML = `
                        ${successIcon}
                        <span>${LABELS.copied}</span>
                    `;

          button.classList.add(CLASSES.copyButtonSuccess);

          // Reset to original state after 2 seconds
          setTimeout(() => {
            button.innerHTML = originalHTML;
            button.classList.remove(CLASSES.copyButtonSuccess);
          }, constants.ANIMATIONS?.feedbackDuration || 2000);
        } catch (err) {
          console.error("Copy failed:", err);
          const originalHTML = button.innerHTML;
          const errorIcon =
            ICONS.error ||
            `<svg width="16" height="16" viewBox="0 0 16 16" fill="currentColor">
                        <path d="M3.72 3.72a.75.75 0 0 1 1.06 0L8 6.94l3.22-3.22a.749.749 0 0 1 1.275.326.749.749 0 0 1-.215.734L9.06 8l3.22 3.22a.749.749 0 0 1-.326 1.275.749.749 0 0 1-.734-.215L8 9.06l-3.22 3.22a.751.751 0 0 1-1.042-.018.751.751 0 0 1-.018-1.042L6.94 8 3.72 4.78a.75.75 0 0 1 0-1.06Z"></path>
                    </svg>`;

          button.innerHTML = `
                        ${errorIcon}
                        <span>${LABELS.failed}</span>
                    `;

          button.classList.add(CLASSES.copyButtonError);

          setTimeout(() => {
            button.innerHTML = originalHTML;
            button.classList.remove(CLASSES.copyButtonError);
          }, constants.ANIMATIONS?.feedbackDuration || 2000);
        }
      });

      buttonContainer.appendChild(button);

      // Set container position to relative (only if not already set)
      if (!container.style.position || container.style.position === "static") {
        container.style.position = "relative";
      }

      // Add button to container
      container.appendChild(buttonContainer);
    },

    /**
     * Add lightbox trigger to Mermaid diagram container
     * @param {HTMLElement} container - Outer container
     * @param {HTMLElement} innerContainer - Inner container with SVG
     */
    addLightboxTrigger(container, innerContainer) {
      // Set ARIA attributes (cursor style is defined in CSS)
      container.setAttribute('role', 'button');
      container.setAttribute('tabindex', '0');
      container.setAttribute('aria-label', 'Click to expand diagram');
      container.setAttribute('aria-haspopup', 'dialog');
      container.setAttribute('aria-expanded', 'false');

      let touchEndTime = 0;

      const openLightbox = () => {
        if (window.wpGfm?.lightbox) {
          window.wpGfm.lightbox.open(innerContainer);
        }
      };

      // Record touch end time
      container.addEventListener('touchend', () => {
        touchEndTime = Date.now();
      });

      // Click handler (ignore clicks immediately after touch)
      container.addEventListener('click', (e) => {
        if (Date.now() - touchEndTime < 300) return;
        if (e.target.closest('.gfmr-copy-button')) return;
        openLightbox();
      });

      // Keyboard accessibility
      container.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' || e.key === ' ') {
          e.preventDefault();
          openLightbox();
        }
      });
    },

    // Language detection - delegate to external module
    detectLanguage(element) {
      if (window.wpGfmLanguageDetector) {
        return window.wpGfmLanguageDetector.detectLanguage(element);
      }
      // Fallback if module not loaded
      const className = element.className || "";
      const match = className.match(/language-(\w+)/);
      return match ? match[1] : "plaintext";
    },

    // Check if element is already syntax highlighted - delegate to external module
    isAlreadyHighlighted(element) {
      if (window.wpGfmLanguageDetector) {
        return window.wpGfmLanguageDetector.isAlreadyHighlighted(element);
      }
      // Fallback if module not loaded
      const hasShikiSpans = element.querySelector(
        'span.line, span[style*="color:"]'
      );
      const hasHighlightClass =
        element.classList.contains("hljs") ||
        element.classList.contains("shiki");
      return !!(hasShikiSpans || hasHighlightClass);
    },

    /**
     * Apply diff highlighting classes to code block.
     * Adds CSS classes to lines based on their diff prefix (+, -, @@).
     *
     * @param {HTMLElement} container - The pre element containing highlighted code
     */
    applyDiffHighlighting(container) {
      // Shiki output structure: <pre class="shiki"> > <code> > <span class="line">
      const codeElement = container.querySelector('code');
      if (!codeElement) {
        return; // Silent return - not a valid Shiki output
      }

      const lines = codeElement.querySelectorAll('.line');
      if (lines.length === 0) {
        return; // Silent return - no line elements found
      }

      lines.forEach(line => {
        const text = (line.textContent || '').trim();

        // Remove empty lines (including whitespace-only lines)
        if (!text) {
          line.remove();
          return;
        }

        // Skip file header lines (+++, ---, diff --git, index)
        if (text.startsWith('+') && !text.startsWith('+++')) {
          line.classList.add('diff', 'add');
        } else if (text.startsWith('-') && !text.startsWith('---')) {
          line.classList.add('diff', 'remove');
        } else if (text.startsWith('@@')) {
          line.classList.add('diff', 'hunk');
        }
        // Other lines (context, headers) remain unstyled
      });

      // Remove all text nodes (newlines) between .line elements to prevent visual empty lines
      // This is necessary because Shiki outputs newline characters between span.line elements
      // and white-space: pre causes them to be displayed as visual line breaks
      const childNodes = Array.from(codeElement.childNodes);
      childNodes.forEach(node => {
        if (node.nodeType === Node.TEXT_NODE) {
          node.remove();
        }
      });

      container.classList.add('has-diff');
    },

    // Code block highlighting process (processed code support)
    async highlightCodeBlock(codeElement) {
      if (this.processedElements.has(codeElement)) return;

      try {
        const code = codeElement.textContent || "";
        let language = window.wpGfmLanguageDetector.detectLanguage(codeElement);

        // Normalize diff-related aliases
        if (language === 'patch') {
          language = 'diff';
        }

        if (!code.trim()) return;

        // Get saved theme from parent container's data attribute
        const markdownRendered = codeElement.closest(".gfmr-markdown-rendered");
        const savedTheme =
          markdownRendered?.getAttribute("data-shiki-theme") || "";

        // Get current theme from settings (resolve 'auto' to actual theme)
        let currentTheme =
          window.wpGfmConfig?.theme?.shiki_theme ??
          window.wpGfmSettings?.theme?.shiki_theme ??
          (window.matchMedia?.('(prefers-color-scheme: dark)').matches ? 'github-dark' : 'github-light');
        if (currentTheme === "auto") {
          const prefersDark =
            window.matchMedia &&
            window.matchMedia("(prefers-color-scheme: dark)").matches;
          currentTheme = prefersDark ? "github-dark" : "github-light";
        }

        // Handle already highlighted code
        if (
          window.wpGfmLanguageDetector.isAlreadyHighlighted(codeElement) &&
          language !== "plaintext"
        ) {
          // Get saved language from element attributes
          const savedLanguage = codeElement.getAttribute('data-gfmr-language') ||
            codeElement.className.match(/language-(\w+)/)?.[1] || '';

          // Check if theme AND language match - skip re-rendering only if both match
          if (savedTheme && savedTheme === currentTheme && savedLanguage === language) {
            console.log(
              `[WP GFM v2 Hotfix] Theme and language match (${savedTheme}, ${language}), skipping re-render`,
            );
            this.processedElements.add(codeElement);
            codeElement.setAttribute("data-gfmr-processed", "true");
            codeElement.setAttribute("data-gfmr-language", language);

            // Ensure visibility
            const container =
              codeElement.closest("pre") || codeElement.parentElement;
            if (container) {
              container.style.visibility = "visible";
              container.style.opacity = "1";
            }
            return;
          }

          console.log(
            `[WP GFM v2 Hotfix] Already highlighted, reprocessing (saved: ${savedTheme}, current: ${currentTheme}): ${language}`,
          );

          // Mermaid requires separate processing
          if (language === "mermaid") {
            console.log(
              `[WP GFM v2 Hotfix] Adding Mermaid element to processing queue`,
            );
            await this.renderMermaidDiagram(codeElement);
            return;
          }

          // Re-highlight for other languages (theme mismatch)
          console.log(
            `[WP GFM v2 Hotfix] Executing re-highlighting: ${language}`,
          );
          const plainText =
            window.wpGfmLanguageDetector.extractPlainTextFromHighlighted(
              codeElement,
            );

          if (global.wpGfmShikiHighlighter && plainText.trim()) {
            try {
              console.log(
                `[DEBUG] Text sent to Shiki (${language}):`,
                plainText.substring(0, 300) + "...",
              );
              const highlighted = global.wpGfmShikiHighlighter.codeToHtml(
                plainText,
                { lang: language },
              );
              console.log(`[DEBUG] Re-highlighting successful: ${language}`);
              console.log(
                `[DEBUG] Generated HTML:`,
                highlighted.substring(0, 200) + "...",
              );

              // DOM update (enhanced debug version)
              const tempDiv = document.createElement("div");
              tempDiv.innerHTML = highlighted;

              const newPre = tempDiv.querySelector("pre");
              if (newPre && codeElement.parentElement) {
                const container =
                  codeElement.closest("pre") || codeElement.parentElement;
                console.log(
                  `[DEBUG] Container before update:`,
                  container.outerHTML.substring(0, 200) + "...",
                );

                if (container.parentNode) {
                  // Apply appropriate theme based on settings
                  const currentTheme =
                    window.wpGfmConfig?.theme?.shiki_theme ??
                    window.wpGfmSettings?.theme?.shiki_theme ??
                    (window.matchMedia?.('(prefers-color-scheme: dark)').matches ? 'github-dark' : 'github-light');
                  const existingClasses = container.className
                    .replace(/github-light/g, "")
                    .replace(/github-dark/g, "")
                    .replace(/shiki/g, "")
                    .trim();
                  newPre.className =
                    `shiki ${currentTheme} ${existingClasses}`.trim();

                  // Force style application and DOM update
                  container.innerHTML = newPre.innerHTML;
                  container.className = newPre.className;
                  
                  // Apply diff highlighting if language is diff
                  if (language === 'diff') {
                    this.applyDiffHighlighting(container);
                  }

                  // Apply theme-appropriate styles
                  // Light theme uses slightly darker background (#eaeef2) for visual separation
                  const isLightTheme = currentTheme.includes("light");
                  const bgColor = isLightTheme ? "#eaeef2" : "#24292e";
                  const textColor = isLightTheme ? "#24292f" : "#e1e4e8";

                  container.style.backgroundColor = bgColor;
                  container.style.color = textColor;
                  container.style.cssText = newPre.style.cssText || "";
                  container.style.setProperty(
                    "background-color",
                    bgColor,
                    "important",
                  );
                  container.style.setProperty("color", textColor, "important");
                  container.style.setProperty("font-size", "14px", "important");
                  container.style.setProperty(
                    "line-height",
                    "1.45",
                    "important",
                  );
                  container.style.setProperty(
                    "font-family",
                    "SFMono-Regular, Consolas, Liberation Mono, Menlo, Monaco, Courier New, monospace",
                    "important",
                  );
                  container.style.setProperty(
                    "border-radius",
                    "6px",
                    "important",
                  );
                  container.style.setProperty("padding", "16px", "important");

                  // Force apply color and font styles (theme-aware adjustment)
                  const codeElements = container.querySelectorAll("code, span");
                  codeElements.forEach((el) => {
                    if (el.style.color) {
                      // Apply theme colors as-is from Shiki
                      el.style.setProperty(
                        "color",
                        el.style.color,
                        "important",
                      );
                    }
                    // Make background transparent for proper theme display
                    if (el.style.backgroundColor) {
                      el.style.setProperty(
                        "background-color",
                        "transparent",
                        "important",
                      );
                    }
                  });

                  console.log(
                    `[DEBUG] Container after update:`,
                    container.outerHTML.substring(0, 200) + "...",
                  );

                  const processedElement =
                    container.querySelector("code") || container;
                  this.processedElements.add(processedElement);
                  // Mark as processed and show immediately
                  processedElement.setAttribute("data-gfmr-processed", "true");
                  processedElement.setAttribute("data-gfmr-language", language);

                  // Instant visibility for ultra-fast rendering
                  const visibleContainer =
                    processedElement.closest("pre") ||
                    processedElement.parentElement;
                  if (visibleContainer) {
                    visibleContainer.style.visibility = "visible";
                    visibleContainer.style.opacity = "1";
                  }

                  console.log(
                    `[WP GFM v2 Hotfix] Re-highlighting completed: ${language}`,
                  );

                  // Add copy button
                  this.addCopyButton(container, plainText);

                  // Immediate style application for theme override protection
                  const allSpans = container.querySelectorAll(
                    'span[style*="color"]',
                  );
                  allSpans.forEach((span) => {
                    const originalColor = span.style.color;
                    if (originalColor) {
                      span.style.setProperty(
                        "color",
                        originalColor,
                        "important",
                      );
                      span.style.setProperty(
                        "font-weight",
                        "inherit",
                        "important",
                      );
                    }
                  });

                  return;
                }
              }
            } catch (shikiError) {
              console.warn(
                `[DEBUG] Re-highlighting failed (${language}):`,
                shikiError,
              );
            }
          }

          // Mark existing process to prevent duplication if failed
          this.processedElements.add(codeElement);
          codeElement.setAttribute("data-gfmr-processed", "true");
          codeElement.setAttribute("data-gfmr-language", language);
          return;
        }

        console.log(`[WP GFM v2 Hotfix] Highlighting process: ${language}`);

        if (global.wpGfmShikiHighlighter) {
          // Shiki highlighting with language support check
          try {
            const highlighted = global.wpGfmShikiHighlighter.codeToHtml(code, {
              lang: language,
            });
            console.log(`[DEBUG] Shiki highlighting successful: ${language}`);

            // DOM update
            const tempDiv = document.createElement("div");
            tempDiv.innerHTML = highlighted;

            const newPre = tempDiv.querySelector("pre");
            if (newPre && codeElement.parentElement) {
              const container =
                codeElement.closest("pre") || codeElement.parentElement;
              if (container.parentNode) {
                // Apply appropriate theme based on settings
                const currentTheme =
                  window.wpGfmConfig?.theme?.shiki_theme ??
                  window.wpGfmSettings?.theme?.shiki_theme ??
                  (window.matchMedia?.('(prefers-color-scheme: dark)').matches ? 'github-dark' : 'github-light');
                const existingClasses = container.className
                  .replace(/github-light/g, "")
                  .replace(/github-dark/g, "")
                  .replace(/shiki/g, "")
                  .trim();
                newPre.className =
                  `shiki ${currentTheme} ${existingClasses}`.trim();

                // Apply theme-appropriate styles
                // Light theme uses slightly darker background (#eaeef2) for visual separation
                const isLightTheme = currentTheme.includes("light");
                const bgColor = isLightTheme ? "#eaeef2" : "#24292e";
                const textColor = isLightTheme ? "#24292f" : "#e1e4e8";

                newPre.style.setProperty(
                  "background-color",
                  bgColor,
                  "important",
                );
                newPre.style.setProperty("color", textColor, "important");
                newPre.style.setProperty("font-size", "14px", "important");
                newPre.style.setProperty("line-height", "1.45", "important");
                newPre.style.setProperty(
                  "font-family",
                  "SFMono-Regular, Consolas, Liberation Mono, Menlo, Monaco, Courier New, monospace",
                  "important",
                );
                newPre.style.setProperty("border-radius", "6px", "important");
                newPre.style.setProperty("padding", "16px", "important");

                container.parentNode.replaceChild(newPre, container);
                
                // Apply diff highlighting if language is diff
                if (language === 'diff') {
                  this.applyDiffHighlighting(newPre);
                }
                
                const processedElement = newPre.querySelector("code") || newPre;
                this.processedElements.add(processedElement);
                // Mark as processed and show immediately
                processedElement.setAttribute("data-gfmr-processed", "true");
                processedElement.setAttribute("data-gfmr-language", language);

                // Instant visibility for ultra-fast rendering
                const containerEl =
                  processedElement.closest("pre") ||
                  processedElement.parentElement;
                if (containerEl) {
                  containerEl.style.visibility = "visible";
                  containerEl.style.opacity = "1";
                }

                console.log(
                  `[WP GFM v2 Hotfix] Highlighting completed: ${language}`,
                );

                // Add copy button
                this.addCopyButton(newPre, code);
              }
            }
          } catch (shikiError) {
            console.debug(
              `[GFMR] Shiki: Language not supported: ${language}`,
            );
            // Process as plain text and show immediately
            this.processedElements.add(codeElement);
            codeElement.setAttribute("data-gfmr-processed", "true");

            // Instant visibility even without processing
            const container =
              codeElement.closest("pre") || codeElement.parentElement;
            if (container) {
              container.style.visibility = "visible";
              container.style.opacity = "1";
            }
          }
        } else {
          // Process as plain text if Shiki is not available and show immediately
          this.processedElements.add(codeElement);
          codeElement.setAttribute("data-gfmr-processed", "true");

          // Instant visibility even without Shiki
          const container =
            codeElement.closest("pre") || codeElement.parentElement;
          if (container) {
            container.style.visibility = "visible";
            container.style.opacity = "1";
          }
        }
      } catch (error) {
        console.error("[WP GFM v2 Hotfix] Highlighting error:", error);
        this.processedElements.add(codeElement);
        codeElement.setAttribute("data-gfmr-processed", "true");

        // Instant visibility even on error
        const container =
          codeElement.closest("pre") || codeElement.parentElement;
        if (container) {
          container.style.visibility = "visible";
          container.style.opacity = "1";
        }
      }
    },

    // Mermaid diagram processing (lazy loading support)
    async renderMermaidDiagram(element) {
      if (this.processedElements.has(element)) return;

      // Check if already processed by gfmr-diagrams-v2.js
      if (
        element.closest(".gfmr-mermaid-wrapper") ||
        element.parentElement?.classList?.contains("gfmr-mermaid-wrapper")
      ) {
        console.log(
          "[WP GFM v2 Hotfix] Already wrapped by gfmr-diagrams-v2.js, skipping",
        );
        this.processedElements.add(element);
        return;
      }

      try {
        let content = element.textContent || element.innerText || "";
        if (!content.trim()) return;

        // Remove markdown code block markers if present
        content = content.trim();
        content = content.replace(/^\s*```\s*mermaid\s*$/m, '');
        content = content.replace(/^\s*```\s*$/m, '');
        content = content.trim();

        console.log("[WP GFM v2 Hotfix] Starting Mermaid processing");

        // Load Mermaid library if not yet loaded
        if (!global.mermaid) {
          console.log("[WP GFM v2 Hotfix] Lazy loading Mermaid library");
          await this.loadMermaid();
        }

        if (global.mermaid) {
          // Generate unique ID
          const id =
            "mermaid-" +
            Date.now() +
            "-" +
            Math.random().toString(36).substr(2, 9);

          // Create outer container (Flexbox for centering)
          const outerContainer = document.createElement("div");
          outerContainer.id = id;
          outerContainer.className =
            "gfmr-mermaid-container gfmr-mermaid-container-unified mermaid-diagram";

          // Outer container: Flexbox centering only
          outerContainer.style.cssText = `
                        display: flex !important;
                        justify-content: center !important;
                        align-items: center !important;
                        width: 100% !important;
                        margin: 16px 0 !important;
                        padding: 0 !important;
                    `;

          // Create inner container (for copy button positioning)
          const innerContainer = document.createElement("div");
          innerContainer.className = "gfmr-mermaid-inner-container";
          innerContainer.style.cssText = `
                        position: relative !important;
                        display: block !important;
                        width: 100% !important;
                    `;

          // Get background color (from data-mermaid-bg-color attribute)
          const markdownContainer = element.closest(".gfmr-markdown-rendered");
          let bgColor = null;
          if (markdownContainer) {
            bgColor = markdownContainer.getAttribute("data-mermaid-bg-color");
          }

          // Mermaid rendering
          const { svg } = await global.mermaid.render(id + "-svg", content);
          innerContainer.innerHTML = svg;

          // Apply background color
          if (bgColor && bgColor !== "" && bgColor !== "transparent") {
            innerContainer.style.setProperty(
              "background",
              bgColor,
              "important",
            );
            innerContainer.style.setProperty("padding", "15px", "important");
            innerContainer.style.setProperty(
              "border-radius",
              "6px",
              "important",
            );
          }

          // Optimize SVG for proper display
          const svgElement = innerContainer.querySelector("svg");
          if (svgElement) {
            // Keep SVG at natural size (removing max-width prevents text clipping)
            svgElement.style.height = "auto";
          }

          // Assemble the structure
          outerContainer.appendChild(innerContainer);

          // Prepare for replacement
          const targetElement = element.closest("pre") || element.parentElement;

          // Save Mermaid source code as hidden element
          const sourceElement = document.createElement("div");
          sourceElement.className = "mermaid-source";
          sourceElement.textContent = content;
          sourceElement.style.display = "none";
          innerContainer.appendChild(sourceElement);

          // Replace original element
          if (targetElement && targetElement.parentNode) {
            targetElement.parentNode.replaceChild(
              outerContainer,
              targetElement,
            );

            this.processedElements.add(outerContainer);
            outerContainer.setAttribute("data-gfmr-processed", "true");

            // Add copy button (to inner container for proper positioning)
            this.addCopyButton(innerContainer, content, "mermaid");

            // Add click-to-expand functionality
            this.addLightboxTrigger(outerContainer, innerContainer);

            console.log(
              "[WP GFM v2 Hotfix] Mermaid processing completed with dual-container centering",
            );
          }
        } else {
          this.processedElements.add(element);
          element.setAttribute("data-gfmr-processed", "true");

          // Instant visibility even without Mermaid
          const container = element.closest("pre") || element.parentElement;
          if (container) {
            container.style.visibility = "visible";
            container.style.opacity = "1";
          }
        }
      } catch (error) {
        console.error("[WP GFM v2 Hotfix] Mermaid error:", error);
        this.processedElements.add(element);
        element.setAttribute("data-gfmr-processed", "true");

        // Instant visibility even on error
        const container = element.closest("pre") || element.parentElement;
        if (container) {
          container.style.visibility = "visible";
          container.style.opacity = "1";
        }
      }
    },

    // Process all code blocks
    async processAllCodeBlocks() {
      const selectors = [
        'pre code:not([data-ssr="true"])',
        '.wp-block-code code:not([data-ssr="true"])',
        '[class*="language-"]:not([data-ssr="true"])',
      ];

      const codeBlocks = document.querySelectorAll(selectors.join(", "));
      console.log(`[WP GFM v2 Hotfix] Found ${codeBlocks.length} code blocks (SSR-rendered excluded)`);

      for (const block of codeBlocks) {
        // Skip already processed elements
        if (block.hasAttribute("data-gfmr-processed")) continue;

        // Skip SSR-rendered elements (double check)
        if (block.getAttribute("data-ssr") === "true") continue;

        // Explicitly process SSR fallback elements (data-ssr="false")
        // These are code blocks that failed server-side rendering
        const isSSRFallback = block.closest('pre[data-ssr="false"]');
        if (isSSRFallback) {
          console.log('[WP GFM v2 Hotfix] Processing SSR fallback code block');
        }

        // Skip code blocks within Markdown source elements to prevent duplication
        if (block.closest(".gfmr-markdown-source")) {
          console.log(
            "[WP GFM v2 Hotfix] Skipping code blocks within Markdown source",
          );
          continue;
        }

        // Basic check within Markdown container (individual elements only)
        const markdownContainer = block.closest(".gfmr-markdown-container");
        if (markdownContainer) {
          const renderedSection = block.closest(".gfmr-markdown-rendered");
          if (!renderedSection) {
            console.log(
              "[WP GFM v2 Hotfix] Within Markdown container but outside rendering element, skipping processing",
            );
            continue;
          }
        }

        // Mermaid is handled separately
        if (window.wpGfmLanguageDetector.isMermaidBlock(block)) continue;

        await this.highlightCodeBlock(block);
      }
    },

    // Process all Mermaid blocks (improved version)
    async processAllMermaidDiagrams() {
      const mermaidSelectors = [
        '[class*="language-mermaid"]:not([data-ssr="true"])',
        '.language-mermaid:not([data-ssr="true"])',
        'code.language-mermaid:not([data-ssr="true"])',
        'pre.language-mermaid code:not([data-ssr="true"])',
        'pre[class*="language-mermaid"] code:not([data-ssr="true"])',
      ];

      // CSS class-based selectors
      const mermaidBlocks = document.querySelectorAll(
        mermaidSelectors.join(", "),
      );
      const detectedBlocks = new Set(Array.from(mermaidBlocks));
      console.log(
        `[WP GFM v2 Hotfix] Found ${mermaidBlocks.length} Mermaid blocks via CSS selectors (SSR-rendered excluded)`,
      );

      // Content-based detection for State Diagram, ER Diagram, GitGraph, User Journey, Class Diagram
      // More specific selectors for better performance on large pages
      const allCodeLikeElements = document.querySelectorAll(
        'pre, .wp-block-code code, .wp-block-code div[class*="language-"]',
      );
      let contentBasedCount = 0;

      for (const element of allCodeLikeElements) {
        // Skip if already detected via CSS selectors
        if (detectedBlocks.has(element)) continue;

        // Skip if already processed
        if (element.hasAttribute("data-gfmr-processed")) continue;

        // Check with isMermaidBlock method
        if (window.wpGfmLanguageDetector.isMermaidBlock(element)) {
          detectedBlocks.add(element);
          contentBasedCount++;
        }
      }

      console.log(
        `[WP GFM v2 Hotfix] Found ${contentBasedCount} additional Mermaid blocks via content detection`,
      );
      console.log(
        `[WP GFM v2 Hotfix] Total Mermaid blocks to process: ${detectedBlocks.size}`,
      );

      // Debug: Check all language class elements
      const allLanguageElements = document.querySelectorAll(
        '[class*="language-"]',
      );
      console.log(
        "[DEBUG] All language class elements:",
        Array.from(allLanguageElements).map((el) => ({
          tag: el.tagName,
          class: el.className,
          text: el.textContent.substring(0, 50),
        })),
      );

      for (const block of detectedBlocks) {
        // Skip already processed elements
        if (block.hasAttribute("data-gfmr-processed")) continue;

        // Skip SSR-rendered elements (double check)
        if (block.getAttribute("data-ssr") === "true") continue;

        // Skip Mermaid blocks within Markdown source elements to prevent duplication
        if (block.closest(".gfmr-markdown-source")) {
          console.log(
            "[WP GFM v2 Hotfix] Skipping Mermaid blocks within Markdown source",
          );
          continue;
        }

        // Check if already processed within Markdown rendered container
        const markdownContainer = block.closest(".gfmr-markdown-container");
        if (markdownContainer) {
          const renderedSection = block.closest(".gfmr-markdown-rendered");
          if (!renderedSection) {
            console.log(
              "[WP GFM v2 Hotfix] Within Markdown container but outside rendering element, skipping Mermaid",
            );
            continue;
          }
        }

        await this.renderMermaidDiagram(block);
      }
    },

    // Mermaid block detection - delegate to external module
    isMermaidBlock(element) {
      if (window.wpGfmLanguageDetector) {
        return window.wpGfmLanguageDetector.isMermaidBlock(element);
      }
      // Fallback if module not loaded
      const className = element.className || "";
      return (
        className.includes("language-mermaid") ||
        className.includes("mermaid")
      );
    },

    // Ultra-fast immediate processing (no idle callback delays)
    // Lazy loading threshold: blocks beyond this count use IntersectionObserver
    LAZY_LOAD_THRESHOLD: 5,
    // Number of blocks to process immediately even with lazy loading
    IMMEDIATE_PROCESS_COUNT: 3,
    // IntersectionObserver instance for lazy loading
    lazyLoadObserver: null,

    async processAllElementsImmediately() {
      const codeBlocks = document.querySelectorAll(
        'pre code:not([data-ssr="true"]), .wp-block-code code:not([data-ssr="true"]), [class*="language-"]:not([data-ssr="true"])',
      );

      // CSS-based Mermaid detection
      const mermaidBlocksCSS = document.querySelectorAll(
        '[class*="language-mermaid"]:not([data-ssr="true"]), .language-mermaid:not([data-ssr="true"]), code.language-mermaid:not([data-ssr="true"]), pre.language-mermaid code:not([data-ssr="true"]), pre[class*="language-mermaid"] code:not([data-ssr="true"])',
      );
      const detectedMermaidBlocks = new Set(Array.from(mermaidBlocksCSS));

      // Content-based Mermaid detection for additional diagram types (optimized selectors)
      const allCodeLikeElements = document.querySelectorAll(
        'pre:not([data-ssr="true"]), .wp-block-code code:not([data-ssr="true"]), .wp-block-code div[class*="language-"]:not([data-ssr="true"])',
      );
      for (const element of allCodeLikeElements) {
        if (detectedMermaidBlocks.has(element)) continue;
        if (element.hasAttribute("data-gfmr-processed")) continue;
        if (element.getAttribute("data-ssr") === "true") continue;

        if (window.wpGfmLanguageDetector.isMermaidBlock(element)) {
          detectedMermaidBlocks.add(element);
        }
      }

      // Filter out already processed, SSR-rendered, and Mermaid blocks from code blocks
      const unprocessedCodeBlocks = Array.from(codeBlocks).filter((block) => {
        if (block.hasAttribute("data-gfmr-processed")) return false;
        if (block.getAttribute("data-ssr") === "true") return false;
        if (block.closest(".gfmr-markdown-source")) return false;
        if (window.wpGfmLanguageDetector?.isMermaidBlock(block)) return false;
        return true;
      });

      const unprocessedMermaidBlocks = Array.from(detectedMermaidBlocks).filter(
        (block) => {
          if (block.hasAttribute("data-gfmr-processed")) return false;
          if (block.getAttribute("data-ssr") === "true") return false;
          if (block.closest(".gfmr-markdown-source")) return false;
          return true;
        },
      );

      const totalBlocks =
        unprocessedCodeBlocks.length + unprocessedMermaidBlocks.length;
      console.log(
        `[WP GFM v2 Hotfix] Total blocks to process: ${totalBlocks} (code: ${unprocessedCodeBlocks.length}, mermaid: ${unprocessedMermaidBlocks.length})`,
      );

      // Use lazy loading if there are many blocks
      if (totalBlocks > this.LAZY_LOAD_THRESHOLD) {
        console.log(
          `[WP GFM v2 Hotfix] Using lazy loading (threshold: ${this.LAZY_LOAD_THRESHOLD})`,
        );
        await this.processWithLazyLoading(
          unprocessedCodeBlocks,
          unprocessedMermaidBlocks,
        );
      } else {
        // Process all elements immediately with larger batches
        const processAllElements = async (elements, processor) => {
          const promises = elements.map((element) => {
            return processor.call(this, element);
          });
          return Promise.all(promises);
        };

        // Process code blocks and Mermaid blocks concurrently - no delays
        await Promise.all([
          processAllElements(unprocessedCodeBlocks, this.highlightCodeBlock),
          processAllElements(
            unprocessedMermaidBlocks,
            this.renderMermaidDiagram,
          ),
        ]);
      }

      console.log(
        "[WP GFM v2 Hotfix] Ultra-fast immediate processing completed",
      );
    },

    /**
     * Process blocks with lazy loading using IntersectionObserver.
     * First N blocks are processed immediately, rest are deferred until visible.
     */
    async processWithLazyLoading(codeBlocks, mermaidBlocks) {
      // Combine all blocks for priority processing
      const allBlocks = [
        ...codeBlocks.map((b) => ({ element: b, type: "code" })),
        ...mermaidBlocks.map((b) => ({ element: b, type: "mermaid" })),
      ];

      // Process first N blocks immediately
      const immediateBlocks = allBlocks.slice(0, this.IMMEDIATE_PROCESS_COUNT);
      const deferredBlocks = allBlocks.slice(this.IMMEDIATE_PROCESS_COUNT);

      console.log(
        `[WP GFM v2 Hotfix] Processing ${immediateBlocks.length} blocks immediately, ${deferredBlocks.length} deferred`,
      );

      // Process immediate blocks
      const immediatePromises = immediateBlocks.map(({ element, type }) => {
        if (type === "mermaid") {
          return this.renderMermaidDiagram(element);
        }
        return this.highlightCodeBlock(element);
      });
      await Promise.all(immediatePromises);

      // Setup IntersectionObserver for deferred blocks
      if (deferredBlocks.length > 0) {
        this.setupLazyLoadObserver(deferredBlocks);
      }
    },

    /**
     * Setup IntersectionObserver for lazy loading code blocks.
     */
    setupLazyLoadObserver(deferredBlocks) {
      // Clean up existing observer if any
      if (this.lazyLoadObserver) {
        this.lazyLoadObserver.disconnect();
      }

      // Create observer with generous root margin for pre-loading
      this.lazyLoadObserver = new IntersectionObserver(
        (entries) => {
          entries.forEach((entry) => {
            if (entry.isIntersecting) {
              const element = entry.target;
              const blockType = element.getAttribute("data-gfmr-lazy-type");

              // Stop observing this element
              this.lazyLoadObserver.unobserve(element);

              // Remove lazy load marker
              element.removeAttribute("data-gfmr-lazy-type");
              element.removeAttribute("data-gfmr-lazy");

              // Process the block
              if (blockType === "mermaid") {
                this.renderMermaidDiagram(element);
              } else {
                this.highlightCodeBlock(element);
              }

              console.log(
                `[WP GFM v2 Hotfix] Lazy loaded ${blockType} block on viewport entry`,
              );
            }
          });
        },
        {
          rootMargin: "200px 0px", // Start loading 200px before entering viewport
          threshold: 0,
        },
      );

      // Observe all deferred blocks
      deferredBlocks.forEach(({ element, type }) => {
        // Mark as lazy loaded for identification
        element.setAttribute("data-gfmr-lazy", "true");
        element.setAttribute("data-gfmr-lazy-type", type);

        // Add placeholder styling for unprocessed blocks
        const container = element.closest("pre") || element.parentElement;
        if (container && !container.hasAttribute("data-gfmr-lazy-styled")) {
          container.setAttribute("data-gfmr-lazy-styled", "true");
          container.style.minHeight = "3em"; // Prevent layout shift
        }

        this.lazyLoadObserver.observe(element);
      });

      console.log(
        `[WP GFM v2 Hotfix] IntersectionObserver setup for ${deferredBlocks.length} blocks`,
      );
    },

    // Monitor new elements (infinite loop prevention)
    startMutationObserver() {
      const observer = new MutationObserver((mutations) => {
        // Ignore during processing to prevent infinite loop
        if (this.isProcessing) return;

        let hasNewContent = false;

        mutations.forEach((mutation) => {
          if (mutation.type === "childList") {
            mutation.addedNodes.forEach((node) => {
              if (node.nodeType === Node.ELEMENT_NODE) {
                // Exclude elements created by ourselves
                const isOurElement =
                  node.classList?.contains("gfmr-mermaid-container") ||
                  node.hasAttribute?.("data-gfmr-processed") ||
                  node.closest?.(".gfmr-mermaid-container");

                if (!isOurElement) {
                  const hasCode =
                    node.querySelector('code, pre, [class*="language-"]') ||
                    node.matches('code, pre, [class*="language-"]');
                  if (hasCode) {
                    hasNewContent = true;
                  }
                }
              }
            });
          }
        });

        if (hasNewContent) {
          // Prevent continuous execution with debounce processing (extended to 500ms)
          clearTimeout(this.processTimeout);
          this.processTimeout = setTimeout(() => this.processNewContent(), 500);
        }
      });

      observer.observe(document.body, {
        childList: true,
        subtree: true,
      });
    },

    // Process new content (infinite loop prevention)
    async processNewContent() {
      if (this.isProcessing) return;

      this.isProcessing = true;
      try {
        await this.processAllCodeBlocks();
        await this.processAllMermaidDiagrams();

        // Process math formulas in new content
        if (this.hasMathContent()) {
          if (!this.katexLoaded) {
            await this.loadKaTeX();
          }
          if (this.katexLoaded) {
            await this.renderMathFormulas();
          }
        }

        // Process chart content in new content
        if (this.hasChartContent()) {
          try {
            await this.loadGfmrCharts();
            if (global.wpGfmCharts) {
              await global.wpGfmCharts.processAllCharts();
            }
          } catch (error) {
            console.error(
              "[WP GFM v2 Hotfix] Chart processing in new content failed:",
              error,
            );
          }
        }
      } finally {
        this.isProcessing = false;
      }
    },

    // Main initialization (optimized version)
    async initialize() {
      console.log("[WP GFM v2 Hotfix] Starting optimized initialization");

      try {
        // Simplified element detection - check for any code elements (excluding SSR-rendered)
        const codeElements = document.querySelectorAll(
          'pre code:not([data-ssr="true"]), code[class*="language-"]:not([data-ssr="true"]), [class*="language-"]:not([data-ssr="true"])',
        );

        // Check for math content
        const hasMath = this.hasMathContent();

        if (codeElements.length === 0 && !hasMath) {
          console.log(
            "[WP GFM v2 Hotfix] No code elements or math content found (SSR-rendered excluded) - skipping initialization",
          );
          return;
        }

        console.log(
          `[WP GFM v2 Hotfix] Found ${codeElements.length} code elements (SSR-rendered excluded), math content: ${hasMath} - Starting library loading`,
        );

        // Build list of libraries to load
        const librariesToLoad = [];

        if (codeElements.length > 0) {
          librariesToLoad.push(this.loadShiki());
          librariesToLoad.push(this.loadMermaid());
        }

        // Conditionally load KaTeX only if math content is detected
        if (hasMath) {
          console.log(
            "[WP GFM v2 Hotfix] Math content detected, adding KaTeX to load queue",
          );
          librariesToLoad.push(this.loadKaTeX());
        }

        // Check for chart content
        const hasCharts = this.hasChartContent();

        // Load libraries in parallel for maximum performance
        try {
          await Promise.all(librariesToLoad);
          console.log("[WP GFM v2 Hotfix] Library loading completed");
        } catch (error) {
          console.error(
            "[WP GFM v2 Hotfix] Some libraries failed to load, continuing with available functionality:",
            error,
          );
        }

        // Process code elements
        if (codeElements.length > 0) {
          // Ultra-fast immediate processing (no idle callback delays)
          await this.processAllElementsImmediately();
        }

        // Render math formulas if KaTeX was loaded
        if (hasMath && this.katexLoaded) {
          await this.renderMathFormulas();
        }

        // Initialize and process charts if chart content exists
        if (hasCharts) {
          try {
            await this.loadGfmrCharts();
            if (global.wpGfmCharts) {
              await global.wpGfmCharts.processAllCharts();
            }
          } catch (error) {
            console.error(
              "[WP GFM v2 Hotfix] Chart initialization failed:",
              error,
            );
          }
        }

        // Start monitoring new content
        this.startMutationObserver();

        // Setup OS theme change detection for 'auto' mode
        this.setupThemeChangeListener();

        console.log("[WP GFM v2 Hotfix] Initialization completed");
        this.initialized = true;

        // Fire completion event
        document.dispatchEvent(new CustomEvent("wpGfmV2HotfixReady"));
      } catch (error) {
        console.error("[WP GFM v2 Hotfix] Initialization error:", error);
      }
    },

    /**
     * Setup listener for OS theme changes (for 'auto' mode).
     * When the OS theme changes, re-render code blocks with the new theme.
     */
    setupThemeChangeListener() {
      const configTheme = window.wpGfmConfig?.theme?.shiki_theme;

      // Only listen for changes if theme is set to 'auto'
      if (configTheme !== "auto") {
        return;
      }

      console.log(
        "[WP GFM v2 Hotfix] Setting up OS theme change listener for auto mode",
      );

      const mediaQuery = window.matchMedia("(prefers-color-scheme: dark)");

      // Use addEventListener for modern browsers
      const handleThemeChange = async (e) => {
        const newTheme = e.matches ? "github-dark" : "github-light";
        console.log(
          `[WP GFM v2 Hotfix] OS theme changed to ${e.matches ? "dark" : "light"}, re-rendering with theme: ${newTheme}`,
        );

        // Clear processed elements to allow re-rendering
        this.processedElements = new WeakSet();

        // Re-process all code blocks with the new theme
        await this.processAllElementsImmediately();
      };

      // Add event listener (works in modern browsers)
      if (mediaQuery.addEventListener) {
        mediaQuery.addEventListener("change", handleThemeChange);
      } else if (mediaQuery.addListener) {
        // Fallback for older browsers
        mediaQuery.addListener(handleThemeChange);
      }
    },
  };

  // Ultra-fast initialization - execute as soon as body exists
  if (document.body) {
    // Body exists - initialize immediately for fastest rendering
    console.log(
      "[WP GFM v2 Hotfix] Body ready, starting ultra-fast initialization",
    );
    hotfix.initialize();
  } else if (document.readyState === "loading") {
    // Body doesn't exist yet - wait for DOMContentLoaded
    document.addEventListener("DOMContentLoaded", () => {
      console.log("[WP GFM v2 Hotfix] DOM loaded, starting initialization");
      hotfix.initialize();
    });
  } else {
    // DOM already loaded - execute immediately
    console.log(
      "[WP GFM v2 Hotfix] DOM ready, starting immediate initialization",
    );
    hotfix.initialize();
  }

  // Export globally
  global.wpGfmV2Hotfix = hotfix;
})(typeof window !== "undefined" ? window : global);
