/**
 * GFMR Highlighter - Unified Syntax Highlighting Engine
 *
 * Integrated module dedicated to syntax highlighting
 * Eliminates complex dependencies and provides reliable, high-performance highlighting
 *
 * Consolidates 4 files before integration into one file:
 * - gfmr-unified-highlighter.js
 * - gfmr-lazy-lang-loader.js
 * - gfmr-lang-aliases.js
 * - gfmr-indent-preprocessor.js
 *
 * @package WpGfmRenderer
 * @since 2.0.0
 * @requires gfmr-utils-v2.js
 */

(function(global) {
    'use strict';

    /**
     * WP GFM Highlighter - Integrated Syntax Highlighter
     */
    class WPGFMHighlighter {
        constructor(utils = null) {
            // Dependencies
            this.utils = utils || global.wpGfmUtils;
            if (!this.utils) {
                throw new Error('WPGFMHighlighter requires WPGFMUtils');
            }
            
            // Shiki related
            this.shikiInstance = null;
            this.shikiInitialized = false;
            this.shikiLoadPromise = null;

            // Language management
            this.loadedLanguages = new Set();
            this.languageAliases = new Map();
            this.languageLoadPromises = new Map();

            // Processing state
            this.processedBlocks = new WeakSet();
            this.processingQueue = [];
            this.isProcessing = false;

            // Configuration
            this.config = {
                theme: this.getThemeFromSettings(),
                coreLanguages: this.utils.getConfig('highlight.languages', ['javascript', 'python', 'php', 'html', 'css', 'json']),
                lazyLoading: this.utils.getConfig('highlight.lazyLoading', true),
                maxConcurrent: this.utils.getConfig('highlight.maxConcurrent', 3),
                timeout: this.utils.getConfig('highlight.timeout', 10000),
                retryAttempts: this.utils.getConfig('highlight.retryAttempts', 2)
            };
            
            // Initialization
            this.initializeLanguageAliases();
            this.setupPerformanceTracking();
            this.setupSystemThemeWatcher();
        }


        // =============================================
        // Initialization and Setup
        // =============================================

        /**
         * Initialize language aliases
         */
        initializeLanguageAliases() {
            const aliases = {
                // JavaScript related
                'js': 'javascript',
                'jsx': 'javascript',
                'node': 'javascript',

                // TypeScript related
                'ts': 'typescript',
                'tsx': 'typescript',

                // Python related
                'py': 'python',
                'python3': 'python',

                // Web related
                'htm': 'html',
                'xml': 'html',
                'svg': 'html',

                // Shell related
                'sh': 'bash',
                'shell': 'bash',
                'zsh': 'bash',

                // CSS related
                'scss': 'css',
                'sass': 'css',
                'less': 'css',

                // Data formats
                'yml': 'yaml',
                'md': 'markdown',
                'config': 'ini',

                // Others
                'c++': 'cpp',
                'c#': 'csharp',
                'cs': 'csharp',

                // Diff related
                'patch': 'diff'
            };
            
            Object.entries(aliases).forEach(([alias, canonical]) => {
                this.languageAliases.set(alias, canonical);
            });
            
            this.utils.debug('Language aliases initialized', { count: this.languageAliases.size });
        }

        /**
         * Setup performance tracking
         */
        setupPerformanceTracking() {
            // Initialize metrics
            this.utils.recordMetric('highlighter.blocks.processed', 0, 'count');
            this.utils.recordMetric('highlighter.languages.loaded', 0, 'count');
            this.utils.recordMetric('highlighter.errors', 0, 'count');
        }

        // =============================================
        // Shiki Initialization and Management
        // =============================================

        /**
         * Ensure Shiki is loaded
         */
        async ensureShikiLoaded() {
            if (this.shikiInitialized && this.shikiInstance) {
                return this.shikiInstance;
            }
            
            if (this.shikiLoadPromise) {
                return this.shikiLoadPromise;
            }
            
            this.shikiLoadPromise = this.loadShiki();
            return this.shikiLoadPromise;
        }

        /**
         * Shiki loading process
         */
        async loadShiki() {
            this.utils.startTimer('shiki.load');
            this.utils.debug('Loading Shiki highlighter...');

            try {
                // 1. Check for existing instance
                if (global.wpGfmShikiHighlighter) {
                    this.shikiInstance = global.wpGfmShikiHighlighter;
                    this.shikiInitialized = true;
                    this.utils.debug('Using existing Shiki instance');
                    return this.shikiInstance;
                }

                // 2. Load via asset loader
                if (global.wpGfmAssetLoader) {
                    try {
                        await global.wpGfmAssetLoader.loadLibrary('shiki');

                        // Wait for initialization to complete
                        let attempts = 0;
                        while (!global.wpGfmShikiHighlighter && attempts < 50) {
                            await this.delay(100);
                            attempts++;
                        }

                        if (global.wpGfmShikiHighlighter) {
                            this.shikiInstance = global.wpGfmShikiHighlighter;
                            this.shikiInitialized = true;
                            this.utils.debug('Shiki loaded via asset loader');
                            return this.shikiInstance;
                        }
                    } catch (error) {
                        this.utils.debug('Asset loader failed, trying direct load', error);
                    }
                }

                // 3. Direct CDN loading (fallback)
                this.shikiInstance = await this.loadShikiDirect();
                this.shikiInitialized = true;

                // Preload core languages
                await this.preloadCoreLanguages();

                return this.shikiInstance;
                
            } catch (error) {
                this.utils.handleError(error, 'shiki_load');
                throw error;
            } finally {
                const elapsed = this.utils.endTimer('shiki.load');
                this.utils.debug(`Shiki load completed in ${elapsed}ms`);
            }
        }

        /**
         * Direct Shiki loading
         */
        async loadShikiDirect() {
            return new Promise((resolve, reject) => {
                // Check for existing script tag
                if (document.querySelector('script[src*="shiki"]')) {
                    this.waitForShiki().then(resolve).catch(reject);
                    return;
                }
                
                const script = document.createElement('script');
                script.src = this.utils.buildAssetUrl('assets/libs/shiki/shiki.min.js');
                script.async = true;
                
                script.onload = async () => {
                    try {
                        const highlighter = await global.shiki.getHighlighter({
                            theme: this.config.theme,
                            langs: this.config.coreLanguages
                        });
                        
                        global.wpGfmShikiHighlighter = highlighter;
                        resolve(highlighter);
                    } catch (error) {
                        reject(error);
                    }
                };
                
                script.onerror = () => reject(new Error('Failed to load Shiki from CDN'));
                document.head.appendChild(script);
            });
        }

        /**
         * Wait for Shiki loading to complete
         */
        async waitForShiki(timeout = 10000) {
            const startTime = Date.now();
            
            while (!global.shiki && (Date.now() - startTime) < timeout) {
                await this.delay(100);
            }
            
            if (!global.shiki) {
                throw new Error('Shiki loading timeout');
            }
            
            return global.shiki.getHighlighter({
                theme: this.config.theme,
                langs: this.config.coreLanguages
            });
        }

        /**
         * Preload core languages
         */
        async preloadCoreLanguages() {
            const coreLanguages = this.config.coreLanguages;
            this.utils.debug('Preloading core languages', coreLanguages);
            
            for (const lang of coreLanguages) {
                this.loadedLanguages.add(lang);
            }
            
            this.utils.recordMetric('highlighter.languages.loaded', coreLanguages.length, 'count');
        }

        // =============================================
        // Theme Management
        // =============================================

        /**
         * Get theme from settings
         */
        getThemeFromSettings() {
            // Get current theme from WordPress settings
            if (global.wpGfmConfig && global.wpGfmConfig.theme) {
                const themeSettings = global.wpGfmConfig.theme;

                if (themeSettings.system_auto) {
                    // System settings linked
                    return this.getSystemPreferredTheme();
                }

                // Fixed theme
                return themeSettings.shiki_theme || 'github-light';
            }

            // Fallback: legacy settings or default
            return this.utils.getConfig('highlight.theme', 'github-light');
        }

        /**
         * Theme selection based on system settings
         */
        getSystemPreferredTheme() {
            if (global.matchMedia && global.matchMedia('(prefers-color-scheme: dark)').matches) {
                return 'github-dark';
            }
            return 'github-light';
        }

        /**
         * Dynamic theme switching
         * @param {string} newTheme - New theme name
         */
        async switchTheme(newTheme) {
            if (this.config.theme === newTheme) {
                return; // Do nothing if already using the same theme
            }

            this.utils.debug(`Switching theme from ${this.config.theme} to ${newTheme}`);

            const oldTheme = this.config.theme;
            this.config.theme = newTheme;

            try {
                // Update theme if Shiki instance exists
                if (this.shikiInstance) {
                    // Reinitialize Shiki with new theme
                    await this.reloadShikiWithTheme(newTheme);
                }

                // Reprocess already processed blocks
                await this.reprocessAllBlocks();

                this.utils.debug(`Theme switched successfully to ${newTheme}`);

                // Dispatch theme switch event
                if (global.document) {
                    global.document.dispatchEvent(new CustomEvent('wpGfmThemeChanged', {
                        detail: { oldTheme, newTheme }
                    }));
                }

            } catch (error) {
                // Revert to original theme on error
                this.config.theme = oldTheme;
                this.utils.handleError(error, 'theme_switch');
                throw error;
            }
        }

        /**
         * Reinitialize Shiki with new theme
         */
        async reloadShikiWithTheme(theme) {
            try {
                if (global.shiki) {
                    this.shikiInstance = await global.shiki.getHighlighter({
                        theme: theme,
                        langs: Array.from(this.loadedLanguages)
                    });
                    global.wpGfmShikiHighlighter = this.shikiInstance;
                }
            } catch (error) {
                this.utils.handleError(error, 'shiki_theme_reload');
                throw error;
            }
        }

        /**
         * Reprocess processed blocks
         */
        async reprocessAllBlocks() {
            // Clear WeakSet to allow reprocessing
            this.processedBlocks = new WeakSet();

            // Redetect and reprocess all blocks
            const codeBlocks = this.findAllCodeBlocks();
            if (codeBlocks.length > 0) {
                await this.highlightAll(codeBlocks);
            }
        }

        /**
         * Setup system settings change watcher
         */
        setupSystemThemeWatcher() {
            if (!global.matchMedia || !global.document) return;

            const mediaQuery = global.matchMedia('(prefers-color-scheme: dark)');

            const handleSystemThemeChange = (e) => {
                if (global.wpGfmConfig && global.wpGfmConfig.theme && global.wpGfmConfig.theme.system_auto) {
                    const newTheme = e.matches ? 'github-dark' : 'github-light';
                    this.switchTheme(newTheme).catch(error => {
                        this.utils.handleError(error, 'system_theme_change');
                    });
                }
            };

            // Use addEventListener for modern browsers
            if (mediaQuery.addEventListener) {
                mediaQuery.addEventListener('change', handleSystemThemeChange);
            } else {
                // Legacy browser support
                mediaQuery.addListener(handleSystemThemeChange);
            }

            // Setup custom event listener (integration with core functionality)
            global.document.addEventListener('wpGfmSystemThemeChanged', (event) => {
                const { newTheme } = event.detail;
                this.switchTheme(newTheme).catch(error => {
                    this.utils.handleError(error, 'custom_system_theme_change');
                });
            });

            // Admin settings change event listener (immediate reflection on settings page)
            global.document.addEventListener('wpGfmThemeChanged', (event) => {
                const { newTheme } = event.detail;
                this.switchTheme(newTheme).catch(error => {
                    this.utils.handleError(error, 'admin_theme_change');
                });
            });

            this.utils.debug('System theme watcher and event listeners setup completed');
        }

        // =============================================
        // Language Management
        // =============================================

        /**
         * Normalize language name
         */
        normalizeLanguage(language) {
            if (!language) return 'plaintext';
            
            const normalized = language.toLowerCase().trim();
            return this.languageAliases.get(normalized) || normalized;
        }

        /**
         * Dynamic language loading
         */
        async loadLanguage(language) {
            const normalizedLang = this.normalizeLanguage(language);
            
            if (this.isLanguageLoaded(normalizedLang)) {
                return true;
            }
            
            if (this.languageLoadPromises.has(normalizedLang)) {
                return this.languageLoadPromises.get(normalizedLang);
            }
            
            const loadPromise = this.doLoadLanguage(normalizedLang);
            this.languageLoadPromises.set(normalizedLang, loadPromise);
            
            try {
                const success = await loadPromise;
                if (success) {
                    this.loadedLanguages.add(normalizedLang);
                }
                return success;
            } finally {
                this.languageLoadPromises.delete(normalizedLang);
            }
        }

        /**
         * Actual language loading process
         */
        async doLoadLanguage(language) {
            if (!this.config.lazyLoading) {
                return false;
            }
            
            this.utils.startTimer(`language.${language}`);
            this.utils.debug(`Loading language: ${language}`);
            
            try {
                const shiki = await this.ensureShikiLoaded();
                
                // Add language to Shiki
                await shiki.loadLanguage(language);
                
                const elapsed = this.utils.endTimer(`language.${language}`);
                this.utils.debug(`Language ${language} loaded in ${elapsed}ms`);

                return true;
            } catch (error) {
                this.utils.handleError(error, `language_load_${language}`);
                return false;
            }
        }

        /**
         * Check language loading status
         */
        isLanguageLoaded(language) {
            const normalized = this.normalizeLanguage(language);
            return this.loadedLanguages.has(normalized);
        }

        // =============================================
        // Code Highlight Processing
        // =============================================

        /**
         * Main highlight function
         */
        async highlight(code, language = 'plaintext', options = {}) {
            this.utils.startTimer('highlight.single');

            try {
                // Input validation
                if (!code || typeof code !== 'string') {
                    return '';
                }

                // Normalize language
                const normalizedLang = this.normalizeLanguage(language);

                // Detect indent code block
                if (this.isIndentCodeBlock(code, normalizedLang, options.context)) {
                    const result = this.processPlainText(code);
                    this.utils.endTimer('highlight.single');
                    return result;
                }

                // Process plaintext
                if (normalizedLang === 'plaintext' || normalizedLang === 'text') {
                    const result = this.processPlainText(code);
                    this.utils.endTimer('highlight.single');
                    return result;
                }

                // Highlight with Shiki
                const result = await this.processWithShiki(code, normalizedLang, options);
                this.utils.endTimer('highlight.single');

                // Update metrics
                this.utils.recordMetric('highlighter.blocks.processed', 1, 'count');

                return result;

            } catch (error) {
                this.utils.handleError(error, 'highlight');
                this.utils.recordMetric('highlighter.errors', 1, 'count');

                // Process as plaintext on error
                const result = this.processPlainText(code);
                this.utils.endTimer('highlight.single');
                return result;
            }
        }

        /**
         * Actual highlight processing with Shiki
         */
        async processWithShiki(code, language, options = {}) {
            const shiki = await this.ensureShikiLoaded();

            // Load language if needed
            if (!this.isLanguageLoaded(language)) {
                const loaded = await this.loadLanguage(language);
                if (!loaded) {
                    this.utils.debug(`Language ${language} not available, using plaintext`);
                    return this.processPlainText(code);
                }
            }

            // Highlight with Shiki
            try {
                const html = shiki.codeToHtml(code, {
                    lang: language,
                    theme: options.theme || this.config.theme
                });

                return html;
            } catch (error) {
                this.utils.debug(`Shiki highlighting failed for ${language}:`, error);
                return this.processPlainText(code);
            }
        }

        /**
         * Process as plaintext
         */
        processPlainText(code) {
            // HTML escape
            const escaped = code
                .replace(/&/g, '&amp;')
                .replace(/</g, '&lt;')
                .replace(/>/g, '&gt;')
                .replace(/"/g, '&quot;')
                .replace(/'/g, '&#39;');

            return `<pre><code>${escaped}</code></pre>`;
        }

        /**
         * Detect indent code block
         */
        isIndentCodeBlock(code, language, context = null) {
            // Context-based detection
            if (context === 'indent' || context === 'indented') {
                return true;
            }

            // If language is explicitly specified, it's a language block
            if (language && language !== 'plaintext' && language !== 'text') {
                return false;
            }

            // Detection based on code content (simplified)
            const lines = code.split('\n');
            const nonEmptyLines = lines.filter(line => line.trim().length > 0);

            if (nonEmptyLines.length === 0) {
                return true;
            }

            // Check if all lines are indented at the same level
            const indentLevels = nonEmptyLines.map(line => {
                const match = line.match(/^(\s*)/);
                return match ? match[1].length : 0;
            });

            const minIndent = Math.min(...indentLevels);
            if (minIndent >= 4) {
                // Has 4+ spaces of indentation
                return true;
            }

            return false;
        }

        // =============================================
        // Batch Processing
        // =============================================

        /**
         * Batch process multiple code blocks
         */
        async highlightAll(codeBlocks = null) {
            this.utils.startTimer('highlight.batch');
            this.utils.debug('Starting batch highlighting');

            // Auto-detect code blocks
            if (!codeBlocks) {
                codeBlocks = this.findAllCodeBlocks();
            }

            if (codeBlocks.length === 0) {
                this.utils.debug('No code blocks found');
                this.utils.endTimer('highlight.batch');
                return [];
            }

            this.utils.debug(`Processing ${codeBlocks.length} code blocks`);

            const results = [];
            const semaphore = Array(this.config.maxConcurrent).fill(Promise.resolve());

            const processingTasks = codeBlocks.map(async (block, index) => {
                // Concurrency limit via semaphore
                await semaphore[index % this.config.maxConcurrent];

                try {
                    const result = await this.processCodeBlock(block, index);
                    results[index] = result;
                } catch (error) {
                    this.utils.handleError(error, `batch_process_${index}`);
                    results[index] = { success: false, error: error.message };
                }

                // Release semaphore
                semaphore[index % this.config.maxConcurrent] = Promise.resolve();
            });

            await Promise.all(processingTasks);

            const elapsed = this.utils.endTimer('highlight.batch');
            this.utils.debug(`Batch highlighting completed in ${elapsed}ms`, {
                total: codeBlocks.length,
                successful: results.filter(r => r.success).length
            });

            return results;
        }

        /**
         * Detect code blocks
         */
        findAllCodeBlocks() {
            const selectors = [
                'pre code',
                '.wp-block-code code', 
                'code[class*="language-"]',
                '.highlight code'
            ];
            
            const codeBlocks = [];
            
            selectors.forEach(selector => {
                const elements = document.querySelectorAll(selector);
                elements.forEach(el => {
                    if (!this.processedBlocks.has(el)) {
                        codeBlocks.push(el);
                    }
                });
            });
            
            return codeBlocks;
        }

        /**
         * Process individual code block
         */
        async processCodeBlock(block, index) {
            if (this.processedBlocks.has(block)) {
                return { success: true, message: 'Already processed', index };
            }

            try {
                const code = block.textContent || '';
                const language = this.detectLanguageFromBlock(block);

                // Skip chart and chart-pro blocks - handled by gfmr-charts.js
                if (language === 'chart' || language === 'chart-pro') {
                    this.utils.debug(`Skipping ${language} block - handled by Chart.js renderer`);
                    return { success: true, message: 'Skipped (chart)', language, index };
                }

                this.utils.debug(`Processing block ${index}: ${language}`);

                const highlighted = await this.highlight(code, language, {
                    context: this.detectContext(block)
                });

                // DOM update
                this.updateBlockContent(block, highlighted);
                this.processedBlocks.add(block);

                return { success: true, language, index };

            } catch (error) {
                this.utils.handleError(error, `process_block_${index}`);
                return { success: false, error: error.message, index };
            }
        }

        /**
         * Detect language from block
         */
        detectLanguageFromBlock(block) {
            // Detect from class name
            const className = block.className || '';
            const classMatch = className.match(/language-(\w+)/);
            if (classMatch) {
                return classMatch[1];
            }

            // Detect from data attribute
            if (block.dataset && block.dataset.language) {
                return block.dataset.language;
            }

            // Detect from parent element class
            const parent = block.parentElement;
            if (parent && parent.className) {
                const parentMatch = parent.className.match(/language-(\w+)/);
                if (parentMatch) {
                    return parentMatch[1];
                }
            }

            return 'plaintext';
        }

        /**
         * Detect context
         */
        detectContext(block) {
            const parent = block.parentElement;
            if (parent) {
                if (parent.classList.contains('wp-block-code')) {
                    return 'wordpress-block';
                }
                if (parent.tagName.toLowerCase() === 'pre' && parent.style.marginLeft) {
                    return 'indent';
                }
            }
            return 'standard';
        }

        /**
         * Update block content
         */
        updateBlockContent(block, highlightedHtml) {
            const tempDiv = document.createElement('div');
            tempDiv.innerHTML = highlightedHtml;

            const newPre = tempDiv.querySelector('pre');
            if (newPre && block.parentElement.tagName.toLowerCase() === 'pre') {
                // Preserve existing classes
                const existingClasses = block.parentElement.className;
                newPre.className = `${newPre.className} ${existingClasses}`.trim();

                // Replace
                block.parentElement.replaceWith(newPre);
            }
        }

        // =============================================
        // Utility
        // =============================================

        /**
         * Delay processing
         */
        delay(ms) {
            return new Promise(resolve => setTimeout(resolve, ms));
        }

        /**
         * Health check
         */
        healthCheck() {
            return {
                status: 'ok',
                shikiLoaded: this.shikiInitialized,
                loadedLanguages: Array.from(this.loadedLanguages),
                processedBlocks: this.processedBlocks ? 'available' : 'unavailable',
                config: this.config
            };
        }

        /**
         * Get statistics
         */
        getStats() {
            return {
                loadedLanguages: this.loadedLanguages.size,
                processedBlocksCount: this.processedBlocks ? 'WeakSet' : 0,
                languageAliases: this.languageAliases.size,
                shikiInitialized: this.shikiInitialized,
                performance: {
                    blocks: this.utils.getMetricStats('highlighter.blocks.processed'),
                    languages: this.utils.getMetricStats('highlighter.languages.loaded'),
                    errors: this.utils.getMetricStats('highlighter.errors')
                }
            };
        }
    }

    // Create and expose global instance
    let wpGfmHighlighter = null;

    // Initialization function
    function initializeHighlighter() {
        if (wpGfmHighlighter) return wpGfmHighlighter;

        wpGfmHighlighter = new WPGFMHighlighter();

        // Expose as global variable
        global.wpGfmHighlighter = wpGfmHighlighter;
        global.WPGFMHighlighter = WPGFMHighlighter;

        // Legacy compatible API
        global.wpGfmUnifiedHighlight = (code, lang, options) => {
            return wpGfmHighlighter.highlight(code, lang, options);
        };

        // Theme switch API
        global.wpGfmSwitchTheme = (newTheme) => {
            return wpGfmHighlighter.switchTheme(newTheme);
        };

        // Theme get API
        global.wpGfmGetCurrentTheme = () => {
            return wpGfmHighlighter.config.theme;
        };

        // Initialization complete event
        if (global.document) {
            global.document.dispatchEvent(new CustomEvent('wpGfmHighlighterReady', {
                detail: { highlighter: wpGfmHighlighter }
            }));
        }

        return wpGfmHighlighter;
    }

    // Control initialization timing
    if (global.document && global.document.readyState === 'loading') {
        global.document.addEventListener('DOMContentLoaded', initializeHighlighter);
    } else {
        setTimeout(initializeHighlighter, 0);
    }

    // Watch for utility ready event
    if (global.document) {
        global.document.addEventListener('wpGfmUtilsReady', initializeHighlighter);
    }

    // CommonJS / AMD support
    if (typeof module !== 'undefined' && module.exports) {
        module.exports = WPGFMHighlighter;
    }

})(typeof window !== 'undefined' ? window : global);