/**
 * GFMR Diagrams - Unified Diagram Rendering Engine
 *
 * Integrated module for Mermaid diagram rendering
 * Combines error handling, zoom, and debug features for stable rendering
 *
 * Consolidates 5 files into one:
 * - gfmr-mermaid-handler.js
 * - gfmr-mermaid-unified.js
 * - gfmr-mermaid-error-handler.js
 * - gfmr-mermaid-zoom.js
 * - gfmr-mermaid-debug.js
 *
 * @package WpGfmRenderer
 * @since 2.0.0
 * @requires gfmr-utils-v2.js
 */

(function(global) {
    'use strict';

    /**
     * WP GFM Diagrams - Unified Diagram Rendering Engine
     */
    class WPGFMDiagrams {
        constructor(utils = null) {
            // Dependencies
            this.utils = utils || global.wpGfmUtils;
            if (!this.utils) {
                throw new Error('WPGFMDiagrams requires WPGFMUtils');
            }

            // Mermaid-related properties
            this.mermaidInstance = null;
            this.mermaidInitialized = false;
            this.mermaidLoadPromise = null;

            // Rendering state
            this.processedDiagrams = new WeakSet();
            this.renderingQueue = [];
            this.isProcessing = false;
            this.renderingTasks = new Map();

            // Zoom functionality
            this.zoomConfig = {
                enabled: false,
                zoomLevels: [0.5, 0.75, 1.0, 1.25, 1.5, 2.0],
                currentZoom: 1.0,
                minZoom: 0.25,
                maxZoom: 4.0
            };

            // Error handling
            this.errorConfig = {
                showErrors: true,
                fallbackToText: true,
                retryAttempts: 2,
                retryDelay: 1000
            };

            // Configuration
            this.config = {
                theme: this.utils.getConfig('mermaid.theme', 'default'),
                maxWidth: this.utils.getConfig('mermaid.maxWidth', 1200),
                maxHeight: this.utils.getConfig('mermaid.maxHeight', 800),
                fontSize: this.utils.getConfig('mermaid.fontSize', 14),
                securityLevel: this.utils.getConfig('mermaid.securityLevel', 'strict'),
                startOnLoad: false,
                flowchart: {
                    useMaxWidth: false,  // Prevent text clipping (consistent with gfmr-main.js)
                    htmlLabels: true
                },
                sequence: {
                    useMaxWidth: true,
                    diagramMarginX: 50,
                    diagramMarginY: 10,
                    actorMargin: 50,
                    width: 150,
                    height: 65
                },
                class: {
                    useMaxWidth: true
                },
                state: {
                    useMaxWidth: false
                },
                er: {
                    useMaxWidth: false
                },
                gantt: {
                    useMaxWidth: true,
                    titleTopMargin: 25,
                    barHeight: 20,
                    fontSize: 11,
                    sidePadding: 75,
                    gridLineStartPadding: 35
                }
            };
            
            // Initialization
            this.setupDiagramTypes();
            this.setupErrorHandling();
            this.setupPerformanceTracking();
        }


        // =============================================
        // Initialization and Setup
        // =============================================

        /**
         * Setup diagram types
         */
        setupDiagramTypes() {
            this.supportedTypes = new Set([
                'flowchart', 'graph', 'sequence', 'sequenceDiagram',
                'gantt', 'classDiagram', 'stateDiagram', 'journey',
                'pie', 'gitgraph', 'er', 'erDiagram'
            ]);
            
            this.typeAliases = new Map([
                ['flow', 'flowchart'],
                ['seq', 'sequence'],
                ['class', 'classDiagram'],
                ['state', 'stateDiagram'],
                ['entity', 'erDiagram']
            ]);
            
            this.utils.debug('Diagram types configured', { 
                supported: Array.from(this.supportedTypes),
                aliases: Object.fromEntries(this.typeAliases)
            });
        }

        /**
         * Setup error handling
         */
        setupErrorHandling() {
            // Capture Mermaid errors
            this.utils.addErrorHandler((errorEntry) => {
                const error = errorEntry.error;
                if (error && (error.message || '').toLowerCase().includes('mermaid')) {
                    this.handleMermaidError(error, errorEntry.context);
                }
            });
        }

        /**
         * Setup performance tracking
         */
        setupPerformanceTracking() {
            this.utils.recordMetric('diagrams.rendered', 0, 'count');
            this.utils.recordMetric('diagrams.errors', 0, 'count');
            this.utils.recordMetric('diagrams.cache.hits', 0, 'count');
        }

        // =============================================
        // Mermaid Initialization and Management
        // =============================================

        /**
         * Ensure Mermaid is loaded
         */
        async ensureMermaidLoaded() {
            if (this.mermaidInitialized && this.mermaidInstance) {
                return this.mermaidInstance;
            }
            
            if (this.mermaidLoadPromise) {
                return this.mermaidLoadPromise;
            }
            
            this.mermaidLoadPromise = this.loadMermaid();
            return this.mermaidLoadPromise;
        }

        /**
         * Load Mermaid library
         */
        async loadMermaid() {
            this.utils.startTimer('mermaid.load');
            this.utils.debug('Loading Mermaid diagram library...');

            try {
                // 1. Check for existing instance
                if (global.mermaid) {
                    this.mermaidInstance = global.mermaid;
                    this.mermaidInitialized = true;
                    this.configureMermaid();
                    this.utils.debug('Using existing Mermaid instance');
                    return this.mermaidInstance;
                }

                // 2. Load via asset loader
                if (global.wpGfmAssetLoader) {
                    try {
                        await global.wpGfmAssetLoader.loadLibrary('mermaid');

                        // Wait for initialization to complete
                        let attempts = 0;
                        while (!global.mermaid && attempts < 50) {
                            await this.delay(100);
                            attempts++;
                        }

                        if (global.mermaid) {
                            this.mermaidInstance = global.mermaid;
                            this.configureMermaid();
                            this.mermaidInitialized = true;
                            this.utils.debug('Mermaid loaded via asset loader');
                            return this.mermaidInstance;
                        }
                    } catch (error) {
                        this.utils.debug('Asset loader failed, trying direct load', error);
                    }
                }

                // 3. Direct CDN load (fallback)
                this.mermaidInstance = await this.loadMermaidDirect();
                this.configureMermaid();
                this.mermaidInitialized = true;

                return this.mermaidInstance;
                
            } catch (error) {
                this.utils.handleError(error, 'mermaid_load');
                throw error;
            } finally {
                const elapsed = this.utils.endTimer('mermaid.load');
                this.utils.debug(`Mermaid load completed in ${elapsed}ms`);
            }
        }

        /**
         * Load Mermaid directly
         */
        async loadMermaidDirect() {
            return new Promise((resolve, reject) => {
                // Check for existing script tag
                if (document.querySelector('script[src*="mermaid"]')) {
                    this.waitForMermaid().then(resolve).catch(reject);
                    return;
                }

                const script = document.createElement('script');
                script.src = this.utils.buildAssetUrl('assets/libs/mermaid/mermaid.min.js');
                script.async = true;

                script.onload = async () => {
                    try {
                        const mermaid = await this.waitForMermaid();
                        resolve(mermaid);
                    } catch (error) {
                        reject(error);
                    }
                };

                script.onerror = () => reject(new Error('Failed to load Mermaid from CDN'));
                document.head.appendChild(script);
            });
        }

        /**
         * Wait for Mermaid to load
         */
        async waitForMermaid(timeout = 10000) {
            const startTime = Date.now();

            while (!global.mermaid && (Date.now() - startTime) < timeout) {
                await this.delay(100);
            }

            if (!global.mermaid) {
                throw new Error('Mermaid loading timeout');
            }

            return global.mermaid;
        }

        /**
         * Configure Mermaid
         */
        configureMermaid() {
            if (!this.mermaidInstance) return;
            
            const config = {
                startOnLoad: false,
                theme: this.config.theme,
                securityLevel: this.config.securityLevel,
                maxWidth: this.config.maxWidth,
                fontFamily: 'Arial, sans-serif',
                fontSize: this.config.fontSize,
                flowchart: this.config.flowchart,
                sequence: this.config.sequence,
                class: this.config.class,
                state: this.config.state,
                er: this.config.er,
                gantt: this.config.gantt,
                gitGraph: {
                    useMaxWidth: true,
                    mainLineWidth: 2,
                    mainBranchOrder: 0,
                },
                logLevel: this.utils.getConfig('debug.enabled', false) ? 'debug' : 'error'
            };
            
            this.mermaidInstance.initialize(config);
            this.utils.debug('Mermaid configured', config);
        }

        // =============================================
        // Diagram Rendering
        // =============================================

        /**
         * Main diagram rendering function
         */
        async renderDiagram(content, element, options = {}) {
            this.utils.startTimer('diagram.render');

            try {
                // Input validation
                if (!content || typeof content !== 'string') {
                    throw new Error('Invalid diagram content');
                }

                if (!element || !element.nodeType) {
                    throw new Error('Invalid target element');
                }

                // Detect diagram type
                const diagramType = this.detectDiagramType(content);
                if (!this.isTypeSupported(diagramType)) {
                    throw new Error(`Unsupported diagram type: ${diagramType}`);
                }

                this.utils.debug(`Rendering ${diagramType} diagram`, {
                    contentLength: content.length,
                    elementTag: element.tagName
                });

                // Check if already processed
                if (this.processedDiagrams.has(element)) {
                    this.utils.debug('Diagram already processed, skipping');
                    return;
                }

                // Render with Mermaid
                const result = await this.renderWithMermaid(content, element, diagramType, options);

                // Mark as processed
                this.processedDiagrams.add(element);

                // Update metrics
                this.utils.recordMetric('diagrams.rendered', 1, 'count');
                this.utils.endTimer('diagram.render');

                return result;

            } catch (error) {
                this.utils.recordMetric('diagrams.errors', 1, 'count');
                this.utils.endTimer('diagram.render');

                // Error handling
                return this.handleRenderError(error, content, element, options);
            }
        }

        /**
         * Actual rendering with Mermaid
         */
        async renderWithMermaid(content, element, diagramType, options = {}) {
            const mermaid = await this.ensureMermaidLoaded();

            // Generate unique ID
            const diagramId = this.generateDiagramId(element);

            // Preprocess content
            const processedContent = this.preprocessDiagramContent(content, diagramType);

            try {
                // Render with Mermaid
                const renderResult = await mermaid.render(diagramId, processedContent);

                if (renderResult && renderResult.svg) {
                    // Apply SVG result
                    this.applySvgResult(element, renderResult.svg, diagramId, options);

                    // Add zoom functionality
                    if (options.enableZoom || this.utils.getConfig('mermaid.zoom', false)) {
                        this.enableZoomForDiagram(element, diagramId);
                    }

                    return renderResult;
                } else {
                    throw new Error('Mermaid render returned no SVG content');
                }
            } catch (renderError) {
                this.utils.debug('Mermaid render failed:', renderError);
                throw renderError;
            }
        }

        /**
         * Detect diagram type
         */
        detectDiagramType(content) {
            const lines = content.trim().split('\n');
            const firstLine = lines[0].trim().toLowerCase();

            // Direct diagram type declaration
            for (const type of this.supportedTypes) {
                if (firstLine.includes(type)) {
                    return type;
                }
            }

            // Check aliases
            for (const [alias, canonical] of this.typeAliases) {
                if (firstLine.includes(alias)) {
                    return canonical;
                }
            }

            // Pattern-based detection
            if (firstLine.includes('graph') || firstLine.includes('flowchart')) {
                return 'flowchart';
            }
            if (firstLine.includes('sequenceDiagram') || firstLine.includes('sequence')) {
                return 'sequence';
            }
            if (firstLine.includes('classDiagram')) {
                return 'classDiagram';
            }
            if (firstLine.includes('gantt')) {
                return 'gantt';
            }
            if (firstLine.includes('pie')) {
                return 'pie';
            }

            // Default is flowchart
            return 'flowchart';
        }

        /**
         * Check if diagram type is supported
         */
        isTypeSupported(diagramType) {
            return this.supportedTypes.has(diagramType);
        }

        /**
         * Preprocess diagram content
         */
        preprocessDiagramContent(content, diagramType) {
            let processed = content.trim();

            // Common preprocessing
            processed = processed.replace(/^\s*```\s*mermaid\s*$/m, '');
            processed = processed.replace(/^\s*```\s*$/m, '');
            processed = processed.trim();

            // Normalize gitgraph to gitGraph (case-insensitive)
            const firstLine = processed.split('\n')[0].toLowerCase();
            if (firstLine === 'gitgraph' || firstLine.startsWith('gitgraph ')) {
              processed = processed.replace(/^gitgraph/i, 'gitGraph');
            }

            // Normalize to Mermaid v10+ syntax: colon to equals
            // id: "msg" → id="msg", state: "msg" → state="msg"
            processed = processed.replace(/(\w+):\s*"([^"]*)"/g, '$1="$2"');

            // Type-specific preprocessing
            switch (diagramType) {
                case 'flowchart':
                case 'graph':
                    // Ensure flowchart direction
                    if (!processed.match(/graph\s+(TD|LR|TB|RL|BT)/)) {
                        if (processed.startsWith('graph')) {
                            processed = processed.replace(/^graph/, 'graph TD');
                        } else {
                            processed = `graph TD\n${processed}`;
                        }
                    }
                    break;

                case 'sequence':
                    // Ensure sequence diagram title
                    if (!processed.startsWith('sequenceDiagram')) {
                        processed = `sequenceDiagram\n${processed}`;
                    }
                    break;
            }

            return processed;
        }

        /**
         * Apply SVG result
         */
        applySvgResult(element, svg, diagramId, options = {}) {
            // Preprocess SVG
            const processedSvg = this.processSvg(svg, diagramId, options);

            // Create wrapper container
            const wrapper = document.createElement('div');
            wrapper.className = 'gfmr-mermaid-wrapper';
            wrapper.setAttribute('data-diagram-id', diagramId);
            wrapper.innerHTML = processedSvg;

            // Apply background color (from data-mermaid-bg-color attribute)
            const markdownContainer = element.closest('.gfmr-markdown-rendered');

            if (markdownContainer) {
                const bgColor = markdownContainer.getAttribute('data-mermaid-bg-color');

                if (bgColor && bgColor !== '' && bgColor !== 'transparent') {
                    // Set data attribute (used by CSS)
                    wrapper.setAttribute('data-bg-color', bgColor);
                    // Also apply background color via inline style for certainty
                    wrapper.style.setProperty('background', bgColor, 'important');
                    wrapper.style.setProperty('padding', '15px', 'important');
                    wrapper.style.setProperty('border-radius', '6px', 'important');
                }
            }

            // Replace existing content
            if (element.parentNode) {
                element.parentNode.replaceChild(wrapper, element);
            } else {
                element.innerHTML = '';
                element.appendChild(wrapper);
            }

            this.utils.debug(`Diagram ${diagramId} rendered successfully`);
        }

        /**
         * Process and optimize SVG
         */
        processSvg(svg, diagramId, options = {}) {
            let processed = svg;

            // Process all attributes in a single replace (merge existing class/style)
            processed = processed.replace(
                /<svg([^>]*)>/,
                (match, attrs) => {
                    let newAttrs = attrs;

                    // Merge class attribute
                    if (/class\s*=/.test(attrs)) {
                        newAttrs = newAttrs.replace(
                            /class\s*=\s*["']([^"']*)["']/,
                            'class="$1 gfmr-mermaid-diagram"'
                        );
                    } else {
                        newAttrs += ' class="gfmr-mermaid-diagram"';
                    }

                    // Merge style attribute
                    if (/style\s*=/.test(attrs)) {
                        newAttrs = newAttrs.replace(
                            /style\s*=\s*["']([^"']*)["']/,
                            'style="$1; height: auto;"'
                        );
                    } else {
                        newAttrs += ' style="height: auto;"';
                    }

                    // Add accessibility attributes if not present
                    if (!/role\s*=/.test(attrs)) {
                        newAttrs += ` role="img" aria-label="Mermaid diagram: ${diagramId}"`;
                    }

                    return `<svg${newAttrs}>`;
                }
            );

            // Apply custom styles
            if (options.customCss) {
                const styleTag = `<style>${options.customCss}</style>`;
                processed = processed.replace('</svg>', `${styleTag}</svg>`);
            }

            return processed;
        }

        // =============================================
        // Zoom Functionality
        // =============================================

        /**
         * Add zoom functionality to diagram
         */
        enableZoomForDiagram(wrapper, diagramId) {
            if (!wrapper || this.zoomConfig.enabled === false) {
                return;
            }

            const svg = wrapper.querySelector('svg');
            if (!svg) return;

            // Create zoom controls
            const zoomControls = this.createZoomControls(diagramId);
            wrapper.appendChild(zoomControls);

            // Setup zoom events
            this.setupZoomEvents(wrapper, svg, diagramId);

            this.utils.debug(`Zoom enabled for diagram ${diagramId}`);
        }

        /**
         * Create zoom controls
         */
        createZoomControls(_diagramId) { // eslint-disable-line no-unused-vars
            const controls = document.createElement('div');
            controls.className = 'gfmr-mermaid-zoom-controls';
            controls.innerHTML = `
                <button type="button" class="zoom-in" title="Zoom In">+</button>
                <button type="button" class="zoom-out" title="Zoom Out">−</button>
                <button type="button" class="zoom-reset" title="Reset Zoom">⌂</button>
                <span class="zoom-level">100%</span>
            `;

            return controls;
        }

        /**
         * Setup zoom events
         */
        setupZoomEvents(wrapper, svg, _diagramId) { // eslint-disable-line no-unused-vars
            const controls = wrapper.querySelector('.gfmr-mermaid-zoom-controls');
            if (!controls) return;

            let currentZoom = 1.0;

            // Zoom in
            controls.querySelector('.zoom-in').addEventListener('click', () => {
                currentZoom = Math.min(currentZoom * 1.2, this.zoomConfig.maxZoom);
                this.applyZoom(svg, controls, currentZoom);
            });

            // Zoom out
            controls.querySelector('.zoom-out').addEventListener('click', () => {
                currentZoom = Math.max(currentZoom / 1.2, this.zoomConfig.minZoom);
                this.applyZoom(svg, controls, currentZoom);
            });

            // Reset zoom
            controls.querySelector('.zoom-reset').addEventListener('click', () => {
                currentZoom = 1.0;
                this.applyZoom(svg, controls, currentZoom);
            });

            // Mouse wheel zoom
            svg.addEventListener('wheel', (event) => {
                event.preventDefault();
                const delta = event.deltaY > 0 ? 0.9 : 1.1;
                currentZoom = Math.max(this.zoomConfig.minZoom,
                               Math.min(this.zoomConfig.maxZoom, currentZoom * delta));
                this.applyZoom(svg, controls, currentZoom);
            });
        }

        /**
         * Apply zoom
         */
        applyZoom(svg, controls, zoom) {
            svg.style.transform = `scale(${zoom})`;
            svg.style.transformOrigin = 'center center';

            const zoomPercent = Math.round(zoom * 100);
            controls.querySelector('.zoom-level').textContent = `${zoomPercent}%`;
        }

        // =============================================
        // Error Handling
        // =============================================

        /**
         * Handle rendering errors
         */
        handleRenderError(error, content, element, options = {}) {
            this.utils.handleError(error, 'diagram_render');

            const errorInfo = {
                message: error.message,
                type: 'render_error',
                content: content.substring(0, 200) + (content.length > 200 ? '...' : ''),
                timestamp: new Date().toISOString()
            };

            if (this.errorConfig.showErrors) {
                this.displayError(element, errorInfo, options);
            }

            // Retry processing
            if (options.retryCount < this.errorConfig.retryAttempts) {
                const retryOptions = { ...options, retryCount: (options.retryCount || 0) + 1 };

                this.utils.debug(`Retrying diagram render (attempt ${retryOptions.retryCount})`);

                return new Promise((resolve) => {
                    setTimeout(() => {
                        resolve(this.renderDiagram(content, element, retryOptions));
                    }, this.errorConfig.retryDelay);
                });
            }

            return null;
        }

        /**
         * Handle Mermaid-specific errors
         */
        handleMermaidError(error, context) {
            this.utils.debug('Mermaid-specific error handling', { error: error.message, context });

            // Handle common Mermaid error patterns
            if (error.message.includes('Parse error') || error.message.includes('Syntax error')) {
                // Don't attempt auto-repair for syntax errors
                return false;
            }

            if (error.message.includes('Cannot read') || error.message.includes('undefined')) {
                // Attempt re-initialization for initialization-related errors
                this.mermaidInitialized = false;
                this.mermaidLoadPromise = null;
                return true;
            }

            return false;
        }

        /**
         * Display error
         */
        displayError(element, errorInfo, _options = {}) { // eslint-disable-line no-unused-vars
            const errorDiv = document.createElement('div');
            errorDiv.className = 'gfmr-mermaid-error';
            errorDiv.innerHTML = `
                <div class="error-header">
                    <strong>Diagram Render Error</strong>
                    <small>${errorInfo.timestamp}</small>
                </div>
                <div class="error-message">${this.utils.sanitizeString(errorInfo.message)}</div>
                ${this.errorConfig.fallbackToText ? `
                    <details class="error-details">
                        <summary>Original Content</summary>
                        <pre><code>${this.utils.sanitizeString(errorInfo.content)}</code></pre>
                    </details>
                ` : ''}
            `;

            // Apply error styling
            errorDiv.style.cssText = `
                border: 2px solid #dc3545;
                background: #f8d7da;
                color: #721c24;
                padding: 15px;
                border-radius: 6px;
                margin: 10px 0;
                font-family: monospace;
            `;

            if (element.parentNode) {
                element.parentNode.replaceChild(errorDiv, element);
            } else {
                element.innerHTML = '';
                element.appendChild(errorDiv);
            }
        }

        // =============================================
        // Batch Processing
        // =============================================

        /**
         * Batch render multiple diagrams
         */
        async renderAllDiagrams(diagrams = null) {
            this.utils.startTimer('diagrams.batch');
            this.utils.debug('Starting batch diagram rendering');

            // Auto-detect diagrams
            if (!diagrams) {
                diagrams = this.findAllDiagrams();
            }

            if (diagrams.length === 0) {
                this.utils.debug('No diagrams found');
                this.utils.endTimer('diagrams.batch');
                return [];
            }

            this.utils.debug(`Processing ${diagrams.length} diagrams`);

            const results = [];
            const maxConcurrent = this.utils.getConfig('mermaid.maxConcurrent', 2);

            // Limit concurrent execution with semaphore
            const semaphore = Array(maxConcurrent).fill(Promise.resolve());

            const processingTasks = diagrams.map(async (diagram, index) => {
                await semaphore[index % maxConcurrent];

                try {
                    const result = await this.processDiagramBlock(diagram, index);
                    results[index] = result;
                } catch (error) {
                    this.utils.handleError(error, `batch_diagram_${index}`);
                    results[index] = { success: false, error: error.message };
                }

                semaphore[index % maxConcurrent] = Promise.resolve();
            });

            await Promise.all(processingTasks);

            const elapsed = this.utils.endTimer('diagrams.batch');
            this.utils.debug(`Batch diagram rendering completed in ${elapsed}ms`, {
                total: diagrams.length,
                successful: results.filter(r => r && r.success).length
            });

            return results;
        }

        /**
         * Find all diagram blocks
         */
        findAllDiagrams() {
            const selectors = [
                'code[class*="mermaid"]',
                'pre code[class*="mermaid"]',
                '.language-mermaid',
                '[data-diagram-type="mermaid"]'
            ];

            const diagrams = [];

            selectors.forEach(selector => {
                const elements = document.querySelectorAll(selector);
                elements.forEach(el => {
                    if (!this.processedDiagrams.has(el)) {
                        diagrams.push(el);
                    }
                });
            });

            return diagrams;
        }

        /**
         * Process individual diagram block
         */
        async processDiagramBlock(element, index) {
            if (this.processedDiagrams.has(element)) {
                return { success: true, message: 'Already processed', index };
            }

            try {
                const content = element.textContent || element.innerText || '';
                this.utils.debug(`Processing diagram ${index}:`, content.substring(0, 50) + '...');

                const result = await this.renderDiagram(content, element, {
                    blockIndex: index
                });

                return { success: true, result, index };

            } catch (error) {
                this.utils.handleError(error, `process_diagram_${index}`);
                return { success: false, error: error.message, index };
            }
        }

        // =============================================
        // Utilities
        // =============================================

        /**
         * Generate diagram ID
         */
        generateDiagramId(element) {
            const timestamp = Date.now();
            const random = Math.random().toString(36).substr(2, 5);
            const elementHash = element.outerHTML ?
                element.outerHTML.substring(0, 50).replace(/\W/g, '') :
                'unknown';

            return `wpgfm-mermaid-${timestamp}-${random}-${elementHash}`.substring(0, 50);
        }

        /**
         * Delay processing
         */
        delay(ms) {
            return new Promise(resolve => setTimeout(resolve, ms));
        }

        /**
         * Health check
         */
        healthCheck() {
            return {
                status: 'ok',
                mermaidLoaded: this.mermaidInitialized,
                supportedTypes: Array.from(this.supportedTypes),
                processedDiagrams: this.processedDiagrams ? 'available' : 'unavailable',
                config: this.config,
                zoomEnabled: this.zoomConfig.enabled
            };
        }

        /**
         * Get statistics
         */
        getStats() {
            return {
                processedDiagramsCount: this.processedDiagrams ? 'WeakSet' : 0,
                supportedTypes: this.supportedTypes.size,
                typeAliases: this.typeAliases.size,
                mermaidInitialized: this.mermaidInitialized,
                renderingQueue: this.renderingQueue.length,
                performance: {
                    rendered: this.utils.getMetricStats('diagrams.rendered'),
                    errors: this.utils.getMetricStats('diagrams.errors'),
                    cacheHits: this.utils.getMetricStats('diagrams.cache.hits')
                }
            };
        }
    }

    // Global instance creation and exposure
    let wpGfmDiagrams = null;

    // Initialization function
    function initializeDiagrams() {
        if (wpGfmDiagrams) return wpGfmDiagrams;

        wpGfmDiagrams = new WPGFMDiagrams();

        // Expose as global variables
        global.wpGfmDiagrams = wpGfmDiagrams;
        global.WPGFMDiagrams = WPGFMDiagrams;

        // Legacy compatible API
        global.wpGfmRenderMermaid = (content, element, options) => {
            return wpGfmDiagrams.renderDiagram(content, element, options);
        };

        // Initialization complete event
        if (global.document) {
            global.document.dispatchEvent(new CustomEvent('wpGfmDiagramsReady', {
                detail: { diagrams: wpGfmDiagrams }
            }));
        }

        return wpGfmDiagrams;
    }

    // Initialization timing control
    if (global.document && global.document.readyState === 'loading') {
        global.document.addEventListener('DOMContentLoaded', initializeDiagrams);
    } else {
        setTimeout(initializeDiagrams, 0);
    }

    // Monitor utilities ready event
    if (global.document) {
        global.document.addEventListener('wpGfmUtilsReady', initializeDiagrams);
    }

    // CommonJS / AMD support
    if (typeof module !== 'undefined' && module.exports) {
        module.exports = WPGFMDiagrams;
    }

})(typeof window !== 'undefined' ? window : global);