/**
 * WP GFM Core v2.0 - Main Orchestration Engine
 * 
 * Main engine handling initialization, coordination, and external API provision
 * Integrates all v2 architecture modules and provides external API
 * 
 * Consolidates 4 files into one:
 * - gfmr-renderer.js (main processing)
 * - gfmr-unified-renderer.js (integration)
 * - gfmr-metadata-manager.js (metadata management)
 * - gfmr-editor-integration.js (editor integration)
 * 
 * @package WpGfmRenderer
 * @since 2.0.0
 * @requires gfmr-utils-v2.js
 */

(function(global) {
    'use strict';

    /**
     * WP GFM Core - Main Orchestration Engine
     */
    class WPGFMCore {
        constructor() {
            // Core modules
            this.utils = null;
            this.highlighter = null;
            this.diagrams = null;
            this.assets = null;
            this.ui = null;
            
            // Initialization state
            this.initialized = false;
            this.initializationPromise = null;
            this.modules = new Map();
            this.moduleLoadOrder = ['utils', 'assets', 'highlighter', 'diagrams', 'ui'];
            
            // Processing state
            this.processedContent = new WeakSet();
            this.processingQueue = [];
            this.isProcessing = false;
            
            // Metadata management
            this.contentMetadata = new Map();
            this.blockMetadata = new Map();
            this.renderingHistory = new Map();
            
            // Editor integration
            this.editorIntegration = {
                enabled: false,
                hooks: new Map(),
                observers: new Map()
            };
            
            // Performance statistics
            this.performanceStats = {
                initTime: 0,
                processedBlocks: 0,
                errors: 0,
                cacheHits: 0
            };
            
            // Configuration
            this.config = {
                autoInitialize: true,
                enabledFeatures: {
                    highlighting: true,
                    diagrams: true,
                    copyButtons: true,
                    errorRecovery: true,
                    performance: true
                },
                processing: {
                    batchSize: 5,
                    maxConcurrency: 3,
                    timeout: 30000
                },
                compatibility: {
                    legacyApi: true,
                    wordpressBlocks: true,
                    classicEditor: true
                }
            };
            
            // Auto initialization
            if (this.config.autoInitialize) {
                this.scheduleInitialization();
            }
        }

        // =============================================
        // Initialization System
        // =============================================

        /**
         * Schedule initialization
         */
        scheduleInitialization() {
            if (global.document) {
                if (global.document.readyState === 'loading') {
                    global.document.addEventListener('DOMContentLoaded', () => this.initialize());
                } else {
                    setTimeout(() => this.initialize(), 0);
                }
            }
        }

        /**
         * Main initialization process
         */
        async initialize() {
            if (this.initialized || this.initializationPromise) {
                return this.initializationPromise || Promise.resolve();
            }
            
            const startTime = performance.now();
            
            this.initializationPromise = this.doInitialize();
            
            try {
                await this.initializationPromise;
                this.performanceStats.initTime = performance.now() - startTime;
                this.initialized = true;
                
                this.debug(`WP GFM Core initialized in ${this.performanceStats.initTime.toFixed(2)}ms`);
                
                // Initialization complete event
                if (global.document) {
                    global.document.dispatchEvent(new CustomEvent('wpGfmCoreReady', {
                        detail: { 
                            core: this,
                            initTime: this.performanceStats.initTime
                        }
                    }));
                }
                
                // Start auto processing
                this.startAutoProcessing();
                
                return this;
            } catch (error) {
                console.error('[WP GFM Core] Initialization failed:', error);
                throw error;
            }
        }

        /**
         * Actual initialization process
         */
        async doInitialize() {
            // 1. Load core modules
            await this.loadCoreModules();

            // 2. Merge settings
            this.integrateConfigurations();

            // 3. Initialize editor integration
            if (this.config.compatibility.wordpressBlocks || this.config.compatibility.classicEditor) {
                this.initializeEditorIntegration();
            }

            // 4. Provide legacy API
            if (this.config.compatibility.legacyApi) {
                this.provideLegacyApi();
            }

            // 5. Start performance monitoring
            if (this.config.enabledFeatures.performance) {
                this.startPerformanceMonitoring();
            }

            // 6. Initialize theme settings
            this.initializeThemeSettings();

            // 7. Initialize Shiki for admin screen
            if (this.isAdminPreview()) {
                await this.initializeAdminShiki();
            }

            this.debug('Core initialization completed');
        }

        /**
         * Load core modules
         */
        async loadCoreModules() {
            const moduleChecks = {
                utils: () => global.wpGfmUtils,
                assets: () => global.wpGfmAssets,
                highlighter: () => global.wpGfmHighlighter,
                diagrams: () => global.wpGfmDiagrams,
                ui: () => global.wpGfmUI
            };
            
            // Wait for and load modules
            for (const moduleName of this.moduleLoadOrder) {
                await this.waitForModule(moduleName, moduleChecks[moduleName]);
            }

            // Set module references
            this.utils = global.wpGfmUtils;
            this.assets = global.wpGfmAssets;
            this.highlighter = global.wpGfmHighlighter;
            this.diagrams = global.wpGfmDiagrams;
            this.ui = global.wpGfmUI;
            
            this.debug('All core modules loaded');
        }

        /**
         * Wait for module
         */
        async waitForModule(name, checkFunction, timeout = 15000) {
            const startTime = Date.now();

            // Special handling for admin screen context
            if (this.isAdminPreview()) {
                timeout = 20000; // Longer timeout for admin screen
            }

            // Assets module may take longer to load
            if (name === 'assets' || name === 'ui') {
                timeout = Math.max(timeout, 20000);
                this.debug(`🔧 Critical module ${name}: Extended timeout (${timeout}ms)`);
            }
            
            let lastLogTime = startTime;
            while (!checkFunction() && (Date.now() - startTime) < timeout) {
                await this.delay(100);

                // Progress log (every 5 seconds)
                const elapsed = Date.now() - startTime;
                if (elapsed - (lastLogTime - startTime) >= 5000) {
                    this.debug(`⏳ Still waiting for module ${name}... (${elapsed}ms elapsed)`);
                    lastLogTime = Date.now();
                }
            }
            
            if (!checkFunction()) {
                // Provide detailed error information
                const availableModules = Object.keys(global).filter(key => key.startsWith('wpGfm'));
                const errorDetails = {
                    module: name,
                    timeout: timeout,
                    availableModules: availableModules,
                    isAdmin: this.isAdminPreview(),
                    globalCheck: name === 'utils' ? typeof global.wpGfmUtils : 'N/A',
                    documentReady: document.readyState,
                    scriptsLoaded: this.getLoadedScripts()
                };
                
                console.warn(`[WP GFM Core] Module ${name} loading timeout - attempting graceful degradation`);
                this.debug('❌ Module loading failed - details:', errorDetails);

                // For non-critical modules, just warn and continue
                if (this.isNonCriticalModule(name)) {
                    console.warn(`[WP GFM Core] Non-critical module ${name} failed to load - continuing with degraded functionality`);
                    this.modules.set(name, null); // Mark with null value
                    return;
                }
                
                throw new Error(`Critical module ${name} failed to load within timeout (${timeout}ms)`);
            }
            
            this.modules.set(name, checkFunction());
            this.debug(`✅ Module ${name} loaded successfully`);
        }

        /**
         * Merge settings
         */
        integrateConfigurations() {
            if (!this.utils) return;

            // Load WordPress settings
            const wpConfig = global.wpGfmConfig || {};

            // Feature enable/disable control
            this.config.enabledFeatures = {
                ...this.config.enabledFeatures,
                ...wpConfig.features
            };

            // Merge processing settings
            this.config.processing = {
                ...this.config.processing,
                ...wpConfig.processing
            };
            
            this.utils.debug('Configurations integrated', this.config);
        }

        /**
         * Initialize theme settings
         */
        initializeThemeSettings() {
            if (!global.document || !global.wpGfmConfig) return;
            
            const themeSettings = global.wpGfmConfig.theme;
            if (!themeSettings) return;
            
            const body = global.document.body;
            if (!body) return;
            
            try {
                // Initialize system theme detection
                this.setupSystemThemeDetection();

                // Set data-gfmr-theme attribute based on current theme settings
                let themeAttribute = 'light'; // Default

                if (themeSettings.system_auto) {
                    themeAttribute = 'auto';
                    // Set actual color scheme based on system theme with additional class
                    this.applySystemTheme();
                } else if (themeSettings.current === 'dark') {
                    themeAttribute = 'dark';
                } else if (themeSettings.current === 'light') {
                    themeAttribute = 'light';
                }
                
                body.setAttribute('data-gfmr-theme', themeAttribute);

                // Set listener for theme change events
                global.document.addEventListener('wpGfmThemeChanged', (event) => {
                    this.handleThemeChange(event.detail);
                });
                
                this.debug(`Theme initialized: ${themeAttribute}`);
                
            } catch (error) {
                this.debug('Failed to initialize theme settings', error);
            }
        }

        /**
         * Setup system theme detection
         */
        setupSystemThemeDetection() {
            if (!global.window || !global.window.matchMedia) {
                this.debug('matchMedia not supported - system theme detection disabled');
                return;
            }

            try {
                // Setup prefers-color-scheme media query
                const darkThemeQuery = global.window.matchMedia('(prefers-color-scheme: dark)');

                // Detect initial system theme
                const systemPrefersDark = darkThemeQuery.matches;
                this.debug(`System theme detected: ${systemPrefersDark ? 'dark' : 'light'}`);

                // Set listener for system theme changes
                const handleSystemThemeChange = (e) => {
                    this.debug(`System theme changed: ${e.matches ? 'dark' : 'light'}`);

                    // Only update if current setting is system_auto
                    const themeSettings = global.wpGfmConfig && global.wpGfmConfig.theme;
                    if (themeSettings && themeSettings.system_auto) {
                        this.applySystemTheme();

                        // Fire custom event to notify modules like Shiki
                        const event = new CustomEvent('wpGfmSystemThemeChanged', {
                            detail: {
                                prefersDark: e.matches,
                                newTheme: e.matches ? 'github-dark' : 'github-light'
                            }
                        });
                        global.document.dispatchEvent(event);
                    }
                };

                // Modern browser support
                if (darkThemeQuery.addEventListener) {
                    darkThemeQuery.addEventListener('change', handleSystemThemeChange);
                }
                // Legacy browser support
                else if (darkThemeQuery.addListener) {
                    darkThemeQuery.addListener(handleSystemThemeChange);
                }

                this.debug('System theme detection initialized');
                
            } catch (error) {
                this.debug('Failed to setup system theme detection', error);
            }
        }

        /**
         * Apply system theme
         */
        applySystemTheme() {
            if (!global.window || !global.window.matchMedia || !global.document) return;

            try {
                const darkThemeQuery = global.window.matchMedia('(prefers-color-scheme: dark)');
                const systemPrefersDark = darkThemeQuery.matches;
                const body = global.document.body;
                
                if (body) {
                    // Add/remove class corresponding to system theme
                    body.classList.remove('gfmr-system-light', 'gfmr-system-dark');
                    body.classList.add(systemPrefersDark ? 'gfmr-system-dark' : 'gfmr-system-light');

                    // Smooth transition effect
                    body.classList.add('gfmr-theme-transitioning');
                    setTimeout(() => {
                        body.classList.remove('gfmr-theme-transitioning');
                    }, 300);

                    this.debug(`Applied system theme: ${systemPrefersDark ? 'dark' : 'light'}`);
                }
                
            } catch (error) {
                this.debug('Failed to apply system theme', error);
            }
        }

        /**
         * Theme change handler
         */
        handleThemeChange(detail) {
            if (!global.document) return;
            
            const body = global.document.body;
            if (!body) return;
            
            try {
                const { newTheme } = detail;
                let themeAttribute = 'light';
                
                if (newTheme === 'github-dark') {
                    themeAttribute = 'dark';
                } else if (newTheme === 'github-light') {
                    themeAttribute = 'light';
                } else if (newTheme === 'auto') {
                    themeAttribute = 'auto';
                }
                
                body.setAttribute('data-gfmr-theme', themeAttribute);
                body.classList.add('gfmr-theme-switching');

                // Remove class after animation completes
                setTimeout(() => {
                    body.classList.remove('gfmr-theme-switching');
                }, 400);
                
                this.debug(`Theme changed to: ${themeAttribute}`);
                
            } catch (error) {
                this.debug('Failed to handle theme change', error);
            }
        }

        // =============================================
        // Content Processing Engine
        // =============================================

        /**
         * Main content processing function
         */
        async processContent(content, options = {}) {
            if (!this.initialized) {
                await this.initialize();
            }
            
            try {
                const startTime = performance.now();

                // Preprocess content
                const preprocessed = this.preprocessContent(content, options);

                // Extract metadata
                const metadata = this.extractContentMetadata(preprocessed, options);

                // Processing result
                const result = {
                    original: content,
                    processed: preprocessed,
                    metadata: metadata,
                    blocks: [],
                    timing: {}
                };

                // Process code blocks
                if (this.config.enabledFeatures.highlighting) {
                    const codeBlocks = this.findCodeBlocks(preprocessed);
                    result.blocks.push(...await this.processCodeBlocks(codeBlocks, options));
                }

                // Process diagrams
                if (this.config.enabledFeatures.diagrams) {
                    const diagramBlocks = this.findDiagramBlocks(preprocessed);
                    result.blocks.push(...await this.processDiagramBlocks(diagramBlocks, options));
                }

                // Add UI components
                if (this.config.enabledFeatures.copyButtons && this.ui) {
                    this.ui.addCopyButtonsToAllCodeBlocks();
                }
                
                result.timing.total = performance.now() - startTime;
                this.performanceStats.processedBlocks += result.blocks.length;
                
                return result;
                
            } catch (error) {
                this.performanceStats.errors++;
                this.handleError(error, 'content_processing', { content, options });
                throw error;
            }
        }

        /**
         * Preprocess content
         */
        preprocessContent(content, options = {}) {
            if (!content || typeof content !== 'string') {
                return '';
            }

            let processed = content;

            // WordPress-specific preprocessing
            if (options.isWordPressContent) {
                // Process block comments
                processed = this.processWordPressBlocks(processed);

                // Preprocess shortcodes
                processed = this.preprocessShortcodes(processed);
            }

            // Protect metadata markers
            processed = this.protectMetadataMarkers(processed);
            
            return processed;
        }

        /**
         * Process code blocks
         */
        async processCodeBlocks(codeBlocks, options = {}) {
            if (!this.highlighter || codeBlocks.length === 0) {
                return [];
            }

            const results = [];
            const batchSize = this.config.processing.batchSize;

            // Batch processing
            for (let i = 0; i < codeBlocks.length; i += batchSize) {
                const batch = codeBlocks.slice(i, i + batchSize);
                const batchResults = await Promise.all(
                    batch.map(async (block, index) => {
                        try {
                            return await this.processSingleCodeBlock(block, i + index, options);
                        } catch (error) {
                            this.handleError(error, 'code_block_processing', { block, index: i + index });
                            return { success: false, error: error.message, index: i + index };
                        }
                    })
                );
                
                results.push(...batchResults);

                // Brief pause between batches
                if (i + batchSize < codeBlocks.length) {
                    await this.delay(10);
                }
            }
            
            return results;
        }

        /**
         * Process single code block
         */
        async processSingleCodeBlock(block, index, options = {}) {
            const codeText = block.textContent || '';
            const language = this.detectLanguage(block, options);

            // Attempt to restore metadata
            const metadata = this.restoreBlockMetadata(codeText, block);
            const finalLanguage = metadata?.language || language;

            // Highlight processing
            const highlighted = await this.highlighter.highlight(codeText, finalLanguage, {
                context: options.context || 'core-processing',
                metadata: metadata,
                blockIndex: index
            });
            
            if (highlighted && highlighted !== codeText) {
                // DOM update
                this.updateBlockInDOM(block, highlighted, index);
                return { success: true, language: finalLanguage, index, highlighted: true };
            }
            
            return { success: true, language: finalLanguage, index, highlighted: false };
        }

        /**
         * Process diagram blocks
         */
        async processDiagramBlocks(diagramBlocks, options = {}) {
            if (!this.diagrams || diagramBlocks.length === 0) {
                return [];
            }
            
            const results = [];
            
            for (const [index, block] of diagramBlocks.entries()) {
                try {
                    const content = block.textContent || '';
                    await this.diagrams.renderDiagram(content, block, {
                        context: options.context || 'core-processing',
                        blockIndex: index
                    });
                    
                    results.push({ success: true, type: 'diagram', index });
                } catch (error) {
                    this.handleError(error, 'diagram_processing', { block, index });
                    results.push({ success: false, error: error.message, index });
                }
            }
            
            return results;
        }

        // =============================================
        // Metadata Management System
        // =============================================

        /**
         * Extract content metadata
         */
        extractContentMetadata(content, options = {}) {
            const metadata = {
                timestamp: Date.now(),
                contentLength: content.length,
                contentHash: this.generateContentHash(content),
                context: options.context || 'unknown',
                blocks: []
            };

            // WordPress-specific metadata
            if (options.isWordPressContent) {
                metadata.wordpress = this.extractWordPressMetadata(content);
            }

            // Block-level metadata
            metadata.blocks = this.extractBlockMetadata(content);
            
            this.contentMetadata.set(metadata.contentHash, metadata);
            return metadata;
        }

        /**
         * Restore block metadata
         */
        restoreBlockMetadata(codeText, element) {
            // Search by content hash
            const contentHash = this.generateContentHash(codeText);
            const cached = this.blockMetadata.get(contentHash);

            if (cached) {
                this.performanceStats.cacheHits++;
                return cached;
            }

            // Restore from DOM attributes
            let current = element;
            while (current && current !== document.body) {
                if (current.getAttribute && current.getAttribute('data-gfmr-meta')) {
                    try {
                        const metadata = JSON.parse(current.getAttribute('data-gfmr-meta'));
                        this.blockMetadata.set(contentHash, metadata);
                        return metadata;
                    } catch (error) {
                        this.debug('Failed to parse metadata attribute:', error);
                    }
                }
                current = current.parentElement;
            }
            
            return null;
        }

        /**
         * Generate content hash
         */
        generateContentHash(content) {
            // Simple hash function
            let hash = 0;
            for (let i = 0; i < content.length; i++) {
                const char = content.charCodeAt(i);
                hash = ((hash << 5) - hash) + char;
                hash = hash & hash; // Convert to 32-bit integer
            }
            return hash.toString(36);
        }

        // =============================================
        // Editor Integration System
        // =============================================

        /**
         * Initialize editor integration
         */
        initializeEditorIntegration() {
            // Gutenberg block editor support
            if (global.wp && global.wp.blocks) {
                this.initializeGutenbergIntegration();
            }

            // Classic editor support
            if (global.tinyMCE) {
                this.initializeClassicEditorIntegration();
            }

            // Other editor support
            this.initializeGenericEditorIntegration();
            
            this.editorIntegration.enabled = true;
            this.debug('Editor integration initialized');
        }

        /**
         * Gutenberg editor integration
         */
        initializeGutenbergIntegration() {
            // Hook for block insertion
            if (global.wp.data && global.wp.data.subscribe) {
                let previousBlockCount = 0;

                global.wp.data.subscribe(() => {
                    const blocks = global.wp.data.select('core/block-editor')?.getBlocks() || [];

                    if (blocks.length > previousBlockCount) {
                        // New block has been added
                        setTimeout(() => this.processNewContent(), 100);
                    }

                    previousBlockCount = blocks.length;
                });
            }
        }

        /**
         * Classic editor integration
         */
        initializeClassicEditorIntegration() {
            // Monitor TinyMCE changes
            if (global.tinyMCE) {
                global.tinyMCE.on('AddEditor', (e) => {
                    const editor = e.editor;
                    editor.on('NodeChange SetContent', () => {
                        setTimeout(() => this.processNewContent(), 200);
                    });
                });
            }
        }

        /**
         * Generic editor integration
         */
        initializeGenericEditorIntegration() {
            // Generic monitoring using MutationObserver
            const observer = new MutationObserver((mutations) => {
                let shouldProcess = false;
                
                mutations.forEach(mutation => {
                    if (mutation.type === 'childList') {
                        mutation.addedNodes.forEach(node => {
                            if (node.nodeType === Node.ELEMENT_NODE) {
                                const hasCode = node.querySelector('code, pre') || 
                                              node.matches('code, pre');
                                if (hasCode) {
                                    shouldProcess = true;
                                }
                            }
                        });
                    }
                });
                
                if (shouldProcess) {
                    setTimeout(() => this.processNewContent(), 100);
                }
            });
            
            observer.observe(document.body, {
                childList: true,
                subtree: true
            });
            
            this.editorIntegration.observers.set('generic', observer);
        }

        // =============================================
        // Auto Processing System
        // =============================================

        /**
         * Start auto processing
         */
        startAutoProcessing() {
            // Process initial content
            this.processExistingContent();

            // Periodic detection of new content
            setInterval(() => {
                this.processNewContent();
            }, 2000);
            
            this.debug('Auto processing started');
        }

        /**
         * Process existing content
         */
        async processExistingContent() {
            const codeBlocks = this.findAllCodeBlocks();
            const diagramBlocks = this.findAllDiagramBlocks();

            this.debug(`Processing ${codeBlocks.length} code blocks and ${diagramBlocks.length} diagrams`);

            // Batch processing
            const codeResults = await this.processCodeBlocks(codeBlocks, { context: 'initial-load' });
            const diagramResults = await this.processDiagramBlocks(diagramBlocks, { context: 'initial-load' });
            
            this.debug(`Processed ${codeResults.length + diagramResults.length} blocks total`);
        }

        /**
         * Process new content
         */
        async processNewContent() {
            if (this.isProcessing) return;
            
            this.isProcessing = true;
            
            try {
                const unprocessedCode = this.findUnprocessedCodeBlocks();
                const unprocessedDiagrams = this.findUnprocessedDiagramBlocks();
                
                if (unprocessedCode.length > 0 || unprocessedDiagrams.length > 0) {
                    this.debug(`Processing ${unprocessedCode.length} new code blocks and ${unprocessedDiagrams.length} new diagrams`);
                    
                    await Promise.all([
                        this.processCodeBlocks(unprocessedCode, { context: 'dynamic-load' }),
                        this.processDiagramBlocks(unprocessedDiagrams, { context: 'dynamic-load' })
                    ]);
                }
            } catch (error) {
                this.handleError(error, 'auto_processing');
            } finally {
                this.isProcessing = false;
            }
        }

        // =============================================
        // Block Detection System
        // =============================================

        /**
         * Detect all code blocks
         */
        findAllCodeBlocks() {
            const selectors = [
                'pre code',
                '.wp-block-code code',
                'code[class*="language-"]',
                '.highlight code'
            ];
            
            return Array.from(document.querySelectorAll(selectors.join(', ')));
        }

        /**
         * Detect unprocessed code blocks
         */
        findUnprocessedCodeBlocks() {
            const allBlocks = this.findAllCodeBlocks();
            return allBlocks.filter(block => !this.processedContent.has(block));
        }

        /**
         * Detect all diagram blocks
         */
        findAllDiagramBlocks() {
            const selectors = [
                'code[class*="mermaid"]',
                'pre code[class*="mermaid"]',
                '.language-mermaid'
            ];
            
            return Array.from(document.querySelectorAll(selectors.join(', ')));
        }

        /**
         * Detect unprocessed diagram blocks
         */
        findUnprocessedDiagramBlocks() {
            const allBlocks = this.findAllDiagramBlocks();
            return allBlocks.filter(block => !this.processedContent.has(block));
        }

        /**
         * Detect code blocks from content
         */
        findCodeBlocks(_content) { // eslint-disable-line no-unused-vars
            // This implementation is string parsing based and returns actual DOM elements
            // For simplicity, uses existing DOM detection method
            return this.findAllCodeBlocks();
        }

        /**
         * Detect diagram blocks from content
         */
        findDiagramBlocks(_content) { // eslint-disable-line no-unused-vars
            return this.findAllDiagramBlocks();
        }

        // =============================================
        // Legacy API Provision
        // =============================================

        /**
         * Provide legacy API
         */
        provideLegacyApi() {
            // Main rendering function
            global.wpGfmRenderer = {
                processCodeBlocks: (blocks) => this.processCodeBlocks(blocks || this.findAllCodeBlocks()),
                renderContent: (content, options) => this.processContent(content, options),
                initialize: () => this.initialize()
            };

            // Individual feature API
            global.wpGfmHighlight = (code, language, options) => {
                return this.highlighter ? this.highlighter.highlight(code, language, options) : null;
            };

            global.wpGfmRenderDiagram = (content, element, options) => {
                return this.diagrams ? this.diagrams.renderDiagram(content, element, options) : null;
            };

            // Unified processing API
            global.wpGfmUnifiedHighlight = (code, language, options) => {
                return this.processUnifiedHighlight(code, language, options);
            };
            
            this.debug('Legacy API provided');
        }

        /**
         * Unified highlight processing (legacy compatible)
         */
        async processUnifiedHighlight(code, language, options = {}) {
            if (!this.initialized) {
                await this.initialize();
            }

            if (!this.highlighter) {
                return code; // Fallback
            }
            
            return await this.highlighter.highlight(code, language, {
                ...options,
                context: 'legacy-api'
            });
        }

        // =============================================
        // Utilities
        // =============================================

        /**
         * Language detection
         */
        detectLanguage(element, _options = {}) { // eslint-disable-line no-unused-vars
            // Detect from class name
            const className = element.className || '';
            const match = className.match(/language-(\w+)/);
            if (match) return match[1];

            // Detect from data attribute
            if (element.dataset && element.dataset.language) {
                return element.dataset.language;
            }

            // Detect from parent element
            const parent = element.closest('pre, div');
            if (parent && parent.className) {
                const parentMatch = parent.className.match(/language-(\w+)/);
                if (parentMatch) return parentMatch[1];
            }
            
            return 'plaintext';
        }

        /**
         * Update DOM block
         */
        updateBlockInDOM(block, highlightedContent, index) {
            try {
                const tempDiv = document.createElement('div');
                tempDiv.innerHTML = highlightedContent;
                
                const newPre = tempDiv.querySelector('pre');
                if (newPre && block.parentElement) {
                    const container = block.closest('pre') || block.parentElement;
                    if (container.parentNode) {
                        container.parentNode.replaceChild(newPre, container);
                        this.processedContent.add(newPre.querySelector('code') || newPre);
                    }
                }
            } catch (error) {
                this.handleError(error, 'dom_update', { block, index });
            }
        }

        /**
         * Process WordPress blocks
         */
        processWordPressBlocks(content) {
            // Process WordPress block comments
            return content.replace(/<!-- wp:[\s\S]*? -->/g, '');
        }

        /**
         * Preprocess shortcodes
         */
        preprocessShortcodes(content) {
            // Basic shortcode protection
            return content.replace(/\[([^\]]+)\]/g, '{{SHORTCODE:$1}}');
        }

        /**
         * Protect metadata markers
         */
        protectMetadataMarkers(content) {
            return content.replace(/<!--\s*gfmr-meta:[\s\S]*?-->/g, (match) => {
                return `{{META:${btoa(match)}}}`;
            });
        }

        /**
         * Extract WordPress metadata
         */
        extractWordPressMetadata(content) {
            const metadata = {};

            // Extract block information
            const blockMatches = content.match(/<!-- wp:(\w+\/\w+).*?-->/g);
            if (blockMatches) {
                metadata.blocks = blockMatches.map(match => {
                    const blockType = match.match(/wp:(\w+\/\w+)/);
                    return blockType ? blockType[1] : 'unknown';
                });
            }
            
            return metadata;
        }

        /**
         * Extract block metadata
         */
        extractBlockMetadata(content) {
            const blocks = [];

            // Code block pattern
            const codeBlockRegex = /```(\w+)?\s*\n([\s\S]*?)```/g;
            let match;
            
            while ((match = codeBlockRegex.exec(content)) !== null) {
                blocks.push({
                    type: 'code',
                    language: match[1] || 'plaintext',
                    content: match[2],
                    position: match.index
                });
            }

            // Mermaid block pattern
            const mermaidRegex = /```mermaid\s*\n([\s\S]*?)```/g;
            while ((match = mermaidRegex.exec(content)) !== null) {
                blocks.push({
                    type: 'diagram',
                    diagramType: 'mermaid',
                    content: match[1],
                    position: match.index
                });
            }
            
            return blocks;
        }

        /**
         * Error handling
         */
        handleError(error, context, details = {}) {
            if (this.utils) {
                this.utils.handleError(error, context, details);
            } else {
                console.error(`[WP GFM Core] Error in ${context}:`, error, details);
            }
        }

        /**
         * Debug output
         */
        debug(message, ...args) {
            if (this.utils) {
                this.utils.debug(message, ...args);
            } else if (global.wpGfmDebug) {
                console.log(`[WP GFM Core] ${message}`, ...args);
            }
        }

        /**
         * Delay processing
         */
        delay(ms) {
            return new Promise(resolve => setTimeout(resolve, ms));
        }

        /**
         * Start performance monitoring
         */
        startPerformanceMonitoring() {
            if (!this.utils) return;

            // Periodic statistics collection
            setInterval(() => {
                this.collectPerformanceStats();
            }, 30000); // 30 second interval
        }

        /**
         * Collect performance statistics
         */
        collectPerformanceStats() {
            const stats = {
                core: this.getStats(),
                modules: {}
            };
            
            if (this.utils) stats.modules.utils = this.utils.getStats();
            if (this.highlighter) stats.modules.highlighter = this.highlighter.getStats();
            if (this.diagrams) stats.modules.diagrams = this.diagrams.getStats();
            if (this.assets) stats.modules.assets = this.assets.getStats();
            if (this.ui) stats.modules.ui = this.ui.getStats();
            
            this.debug('Performance stats collected', stats);
            return stats;
        }

        // =============================================
        // External API
        // =============================================

        /**
         * Manually process content
         */
        async render(content, options = {}) {
            return await this.processContent(content, options);
        }

        /**
         * Highlight specific code block
         */
        async highlight(code, language = 'plaintext', options = {}) {
            if (!this.initialized) await this.initialize();
            return this.highlighter ? await this.highlighter.highlight(code, language, options) : code;
        }

        /**
         * Render diagram
         */
        async renderDiagram(content, element, options = {}) {
            if (!this.initialized) await this.initialize();
            return this.diagrams ? await this.diagrams.renderDiagram(content, element, options) : null;
        }

        /**
         * Reprocess all content
         */
        async reprocessAll() {
            this.processedContent = new WeakSet();
            this.contentMetadata.clear();
            this.blockMetadata.clear();
            
            await this.processExistingContent();
        }

        /**
         * Health check
         */
        healthCheck() {
            return {
                status: this.initialized ? 'ok' : 'initializing',
                initialized: this.initialized,
                modules: {
                    utils: !!this.utils,
                    highlighter: !!this.highlighter,
                    diagrams: !!this.diagrams,
                    assets: !!this.assets,
                    ui: !!this.ui
                },
                editorIntegration: this.editorIntegration.enabled,
                performance: this.performanceStats,
                config: this.config
            };
        }

        /**
         * Get statistics
         */
        getStats() {
            return {
                initialized: this.initialized,
                initTime: this.performanceStats.initTime,
                processedBlocks: this.performanceStats.processedBlocks,
                errors: this.performanceStats.errors,
                cacheHits: this.performanceStats.cacheHits,
                contentMetadata: this.contentMetadata.size,
                blockMetadata: this.blockMetadata.size,
                modulesLoaded: this.modules.size,
                healthCheck: this.healthCheck()
            };
        }

        // =============================================
        // Admin Screen Support
        // =============================================

        /**
         * Determine if this is an admin page
         */
        isAdminPage() {
            return global.wpGfmConfig && global.wpGfmConfig.isAdmin;
        }

        /**
         * Determine if this is an admin preview
         */
        isAdminPreview() {
            return global.wpGfmConfig && global.wpGfmConfig.adminPreview;
        }

        /**
         * Initialize Shiki for admin screen
         */
        async initializeAdminShiki() {
            if (!this.isAdminPreview()) return;

            try {
                this.debug('🔧 GFMR Admin: Initializing Shiki for admin preview...');
                
                // Dynamically import Shiki
                const { createHighlighter } = await import('https://esm.sh/shiki@1.0.0');
                
                // Create highlighter with both themes
                global.gfmrAdminHighlighter = await createHighlighter({
                    themes: ['github-light', 'github-dark'],
                    langs: ['javascript', 'typescript', 'python', 'php', 'html', 'css', 'rust']
                });
                
                this.debug('✅ GFMR Admin: Shiki highlighter initialized');

                // Processing when jQuery is ready
                if (global.jQuery) {
                    global.jQuery(global.document).ready(() => {
                        this.setupAdminJQueryIntegration();
                    });
                }
                
            } catch (error) {
                this.debug('❌ GFMR Admin: Error initializing Shiki', error);
            }
        }

        /**
         * Setup admin jQuery integration
         */
        setupAdminJQueryIntegration() {
            if (!global.jQuery || !this.isAdminPreview()) return;

            const $ = global.jQuery;

            // Monitor preview code element
            const $previewCode = $('#gfmr-preview-code');
            if ($previewCode.length) {
                this.debug('✅ GFMR Admin: Found preview code element, ready for highlighting');

                // Apply initial highlighting
                this.applyAdminPreviewHighlighting($previewCode);
            }
        }

        /**
         * Apply admin preview highlighting
         */
        async applyAdminPreviewHighlighting($element) {
            if (!global.gfmrAdminHighlighter || !$element.length) return;
            
            try {
                const code = $element.text();
                const themeSettings = global.wpGfmConfig && global.wpGfmConfig.theme;
                const currentTheme = this.getEffectiveAdminTheme(themeSettings);
                
                const html = global.gfmrAdminHighlighter.codeToHtml(code, {
                    lang: 'javascript',
                    theme: currentTheme
                });
                
                $element.parent().html(html);
                this.debug(`✅ GFMR Admin: Applied highlighting with theme: ${currentTheme}`);
                
            } catch (error) {
                this.debug('❌ GFMR Admin: Error applying preview highlighting', error);
            }
        }

        /**
         * Get effective admin theme
         */
        getEffectiveAdminTheme(themeSettings) {
            if (!themeSettings) return 'github-light';
            
            if (themeSettings.current === 'system') {
                const prefersDark = global.window.matchMedia && 
                                   global.window.matchMedia('(prefers-color-scheme: dark)').matches;
                return prefersDark ? 'github-dark' : 'github-light';
            }
            
            return themeSettings.shiki_theme || 'github-light';
        }
        
        /**
         * Determine if module is non-critical
         */
        isNonCriticalModule(moduleName) {
            const nonCritical = ['diagrams', 'highlighter'];
            return nonCritical.includes(moduleName);
        }
        
        /**
         * Get loaded scripts
         */
        getLoadedScripts() {
            const scripts = Array.from(document.querySelectorAll('script[src]'));
            return scripts
                .filter(script => script.src.includes('gfmr-') || script.src.includes('wp-gfm'))
                .map(script => ({
                    src: script.src.split('/').pop(),
                    loaded: script.readyState === 'complete' || script.readyState === 'loaded'
                }));
        }
    }

    // Create and expose global instance
    const wpGfmCore = new WPGFMCore();

    // Expose as global variables
    global.wpGfmCore = wpGfmCore;
    global.WPGFMCore = WPGFMCore;

    // Main entry point
    global.wpGfm = wpGfmCore;
    
    // CommonJS / AMD support
    if (typeof module !== 'undefined' && module.exports) {
        module.exports = WPGFMCore;
    }

})(typeof window !== 'undefined' ? window : global);
