/**
 * GFMR Constants Module
 * 
 * Centralized configuration and constants for the GFM Renderer
 * 
 * @package WpGfmRenderer
 * @since 1.0.0
 */

(function(global) {
    'use strict';

    // Copy Button Icons
    const ICONS = {
        copy: `<svg width="16" height="16" viewBox="0 0 16 16" fill="currentColor">
            <path d="M0 6.75C0 5.784.784 5 1.75 5h1.5a.75.75 0 0 1 0 1.5h-1.5a.25.25 0 0 0-.25.25v7.5c0 .138.112.25.25.25h7.5a.25.25 0 0 0 .25-.25v-1.5a.75.75 0 0 1 1.5 0v1.5A1.75 1.75 0 0 1 9.25 16h-7.5A1.75 1.75 0 0 1 0 14.25Z"></path>
            <path d="M5 1.75C5 .784 5.784 0 6.75 0h7.5C15.216 0 16 .784 16 1.75v7.5A1.75 1.75 0 0 1 14.25 11h-7.5A1.75 1.75 0 0 1 5 9.25Zm1.75-.25a.25.25 0 0 0-.25.25v7.5c0 .138.112.25.25.25h7.5a.25.25 0 0 0 .25-.25v-7.5a.25.25 0 0 0-.25-.25Z"></path>
        </svg>`,
        success: `<svg width="16" height="16" viewBox="0 0 16 16" fill="currentColor">
            <path d="M13.78 4.22a.75.75 0 0 1 0 1.06l-7.25 7.25a.75.75 0 0 1-1.06 0L2.22 9.28a.751.751 0 0 1 .018-1.042.751.751 0 0 1 1.042-.018L6 10.94l6.72-6.72a.75.75 0 0 1 1.06 0Z"></path>
        </svg>`,
        error: `<svg width="16" height="16" viewBox="0 0 16 16" fill="currentColor">
            <path d="M3.72 3.72a.75.75 0 0 1 1.06 0L8 6.94l3.22-3.22a.749.749 0 0 1 1.275.326.749.749 0 0 1-.215.734L9.06 8l3.22 3.22a.749.749 0 0 1-.326 1.275.749.749 0 0 1-.734-.215L8 9.06l-3.22 3.22a.751.751 0 0 1-1.042-.018.751.751 0 0 1-.018-1.042L6.94 8 3.72 4.78a.75.75 0 0 1 0-1.06Z"></path>
        </svg>`
    };

    // Button Labels
    const LABELS = {
        copy: 'Copy',
        copied: 'Copied!',
        failed: 'Failed',
        copyCode: 'Copy code'
    };

    // Animation Durations
    const ANIMATIONS = {
        feedbackDuration: 2000,  // How long to show success/error feedback
        transitionDuration: 200, // CSS transition duration
        debounceDelay: 500      // Debounce delay for mutation observer
    };

    // Supported Languages for Syntax Highlighting
    const LANGUAGES = {
        shiki: [
            'javascript', 'js', 'typescript', 'ts',
            'python', 'py', 'java', 'cpp', 'c', 'csharp', 'cs',
            'html', 'css', 'scss', 'sass',
            'json', 'xml', 'yaml', 'yml',
            'bash', 'shell', 'sh', 'zsh',
            'php', 'ruby', 'go', 'rust',
            'sql', 'markdown', 'md'
        ],
        aliases: {
            'js': 'javascript',
            'ts': 'typescript',
            'py': 'python',
            'cs': 'csharp',
            'yml': 'yaml',
            'sh': 'bash',
            'md': 'markdown'
        }
    };

    // Theme Configuration
    const THEMES = {
        shiki: {
            default: 'auto', // Follow system preference
            light: 'github-light',
            dark: 'github-dark'
        },
        mermaid: {
            default: 'default',
            dark: 'dark',
            forest: 'forest',
            neutral: 'neutral'
        }
    };

    // Style Classes
    const CLASSES = {
        copyButton: 'gfmr-copy-button',
        copyButtonCode: 'gfmr-copy-button--code',
        copyButtonMermaid: 'gfmr-copy-button--mermaid',
        copyButtonSuccess: 'gfmr-copy-button--success',
        copyButtonError: 'gfmr-copy-button--error',
        copyButtonContainer: 'gfmr-copy-button-container',
        
        mermaidContainer: 'gfmr-mermaid-container',
        mermaidInnerContainer: 'gfmr-mermaid-inner-container',
        mermaidSource: 'mermaid-source',
        
        codeContainer: 'gfmr-code-container',
        shikiProcessed: 'gfmr-shiki-processed',
        
        processed: 'data-gfmr-processed',
        language: 'data-gfmr-language'
    };

    // Selectors for Finding Elements
    const SELECTORS = {
        codeBlocks: [
            'pre code',
            '.wp-block-code code',
            '[class*="language-"]'
        ],
        mermaidBlocks: [
            '[class*="language-mermaid"]',
            '.language-mermaid',
            'code.language-mermaid',
            'pre.language-mermaid code',
            'pre[class*="language-mermaid"] code'
        ],
        markdownContainers: [
            '.gfmr-markdown-container',
            '.gfmr-markdown-rendered',
            '.gfmr-markdown-source'
        ]
    };

    // Mermaid Configuration
    const MERMAID_CONFIG = {
        startOnLoad: false,
        theme: 'default',
        securityLevel: 'sandbox',
        fontFamily: 'monospace',
        flowchart: {
            useMaxWidth: false,  // Consistent with other files
            htmlLabels: true,
            curve: 'basis'
        },
        sequence: {
            useMaxWidth: true,
            width: 150,
            height: 65
        }
    };

    // Console Log Prefixes
    const LOG_PREFIXES = {
        main: '[GFMR Main]',
        hotfix: '[WP GFM v2 Hotfix]',
        debug: '[DEBUG]',
        error: '[ERROR]',
        warning: '[WARNING]',
        info: '[INFO]'
    };

    // Export the constants
    global.wpGfmConstants = {
        ICONS,
        LABELS,
        ANIMATIONS,
        LANGUAGES,
        THEMES,
        CLASSES,
        SELECTORS,
        MERMAID_CONFIG,
        LOG_PREFIXES
    };

    // Helper function to get a constant value with fallback
    global.wpGfmGetConstant = function(path, defaultValue) {
        const keys = path.split('.');
        let value = global.wpGfmConstants;
        
        for (const key of keys) {
            if (value && typeof value === 'object' && key in value) {
                value = value[key];
            } else {
                return defaultValue;
            }
        }
        
        return value;
    };

})(typeof window !== 'undefined' ? window : global);