/**
 * GFMR Charts Module
 * Chart.js integration for WordPress (UMD version)
 *
 * @package WpGfmRenderer
 * @since 2.1.0
 */
(function(global) {
    'use strict';

    class WPGFMCharts {
        constructor() {
            this.chartjsLoaded = false;
            this.chartjsLoadPromise = null;
            this.processedCharts = new WeakSet();

            // All supported chart types (all free)
            this.supportedTypes = new Set([
                'line', 'bar', 'pie', 'doughnut',
                'scatter', 'bubble', 'radar', 'polarArea'
            ]);

            // Chart configuration defaults
            this.defaultConfig = {
                responsive: true,
                maintainAspectRatio: true,
                animation: { duration: 750 }
            };
        }

        /**
         * Get current theme mode
         * Charts always use light mode for consistency
         */
        isDarkTheme() {
            // Charts always use light mode
            return false;
        }

        /**
         * Get theme colors for charts
         * Charts always use light theme colors for consistency
         */
        getThemeColors() {
            // Charts always use light theme colors
            return {
                text: '#24292f',
                grid: '#d0d7de',
                border: '#d0d7de'
            };
        }

        /**
         * Merge theme colors into chart config options
         * Preserves user-specified colors, applies theme colors only where unspecified
         * @param {Object} config - Chart.js configuration object
         * @returns {Object} Object containing isDark and colors for container styling
         */
        mergeThemeColors(config) {
            const colors = this.getThemeColors();

            // Deep merge helper - preserves user-specified values
            const deepMerge = (target, source) => {
                const result = { ...target };
                for (const key in source) {
                    if (source[key] && typeof source[key] === 'object' && !Array.isArray(source[key])) {
                        result[key] = deepMerge(target[key] || {}, source[key]);
                    } else if (target[key] === undefined) {
                        // Only apply source value if target doesn't have it
                        result[key] = source[key];
                    }
                }
                return result;
            };

            // Theme options (applied only if not user-specified)
            const themeOptions = {
                color: colors.text,
                borderColor: colors.border,
                scales: {
                    x: {
                        grid: { color: colors.grid },
                        ticks: { color: colors.text }
                    },
                    y: {
                        grid: { color: colors.grid },
                        ticks: { color: colors.text }
                    }
                },
                plugins: {
                    legend: { labels: { color: colors.text } },
                    title: { color: colors.text },
                    tooltip: {
                        backgroundColor: 'rgba(255, 255, 255, 0.95)',
                        titleColor: colors.text,
                        bodyColor: colors.text,
                        borderColor: colors.border,
                        borderWidth: 1
                    }
                }
            };

            // Merge: user options override theme options
            config.options = deepMerge(themeOptions, config.options || {});

            // Charts always use light mode
            return { isDark: false, colors };
        }

        /**
         * Apply theme colors to Chart.js global defaults
         * Charts always use light theme colors
         */
        applyThemeToDefaults() {
            if (!global.Chart) return;

            const colors = this.getThemeColors();

            // Global text color
            global.Chart.defaults.color = colors.text;

            // Global border color
            global.Chart.defaults.borderColor = colors.border;

            // Scale defaults (affects all axes)
            if (global.Chart.defaults.scale) {
                global.Chart.defaults.scale.grid = global.Chart.defaults.scale.grid || {};
                global.Chart.defaults.scale.grid.color = colors.grid;
                global.Chart.defaults.scale.ticks = global.Chart.defaults.scale.ticks || {};
                global.Chart.defaults.scale.ticks.color = colors.text;
            }

            // Plugins
            if (global.Chart.defaults.plugins) {
                if (global.Chart.defaults.plugins.legend?.labels) {
                    global.Chart.defaults.plugins.legend.labels.color = colors.text;
                }
                if (global.Chart.defaults.plugins.title) {
                    global.Chart.defaults.plugins.title.color = colors.text;
                }
                if (global.Chart.defaults.plugins.tooltip) {
                    global.Chart.defaults.plugins.tooltip.backgroundColor = 'rgba(255, 255, 255, 0.95)';
                    global.Chart.defaults.plugins.tooltip.titleColor = colors.text;
                    global.Chart.defaults.plugins.tooltip.bodyColor = colors.text;
                    global.Chart.defaults.plugins.tooltip.borderColor = colors.border;
                    global.Chart.defaults.plugins.tooltip.borderWidth = 1;
                }
            }
        }

        /**
         * Load Chart.js library (UMD version)
         */
        async ensureChartJSLoaded() {
            if (this.chartjsLoaded && global.Chart) {
                return global.Chart;
            }

            if (this.chartjsLoadPromise) {
                return this.chartjsLoadPromise;
            }

            this.chartjsLoadPromise = new Promise((resolve, reject) => {
                if (global.Chart) {
                    this.chartjsLoaded = true;
                    resolve(global.Chart);
                    return;
                }

                const script = document.createElement('script');
                // UMD version
                script.src = global.wpGfmBuildAssetUrl('assets/libs/chartjs/chart.umd.min.js');

                script.onload = () => {
                    this.chartjsLoaded = true;
                    console.log('[GFMR Charts] Chart.js UMD loaded');
                    // Apply theme defaults after Chart.js is loaded
                    this.applyThemeToDefaults();
                    resolve(global.Chart);
                };

                script.onerror = (error) => {
                    console.error('[GFMR Charts] Failed to load Chart.js:', error);
                    reject(error);
                };

                document.head.appendChild(script);
            });

            return this.chartjsLoadPromise;
        }

        /**
         * Check if chart type is supported
         */
        isChartAllowed(element, config) {
            // All chart types are allowed (no Pro/Free distinction)
            const chartType = config.type;

            if (!this.supportedTypes.has(chartType)) {
                return {
                    allowed: false,
                    reason: 'unsupported_type',
                    message: `Unsupported chart type: ${chartType}`
                };
            }

            return { allowed: true };
        }

        /**
         * Parse chart configuration from JSON
         */
        parseChartConfig(content) {
            try {
                const config = JSON.parse(content);

                if (!config.type) {
                    throw new Error('Chart type is required');
                }
                if (!config.data) {
                    throw new Error('Chart data is required');
                }

                config.options = {
                    ...this.defaultConfig,
                    ...(config.options || {})
                };

                return config;
            } catch (error) {
                throw new Error(`Invalid chart config: ${error.message}`);
            }
        }

        /**
         * Render a single chart
         */
        async renderChart(element, config) {
            if (this.processedCharts.has(element)) {
                return;
            }

            try {
                // Fence-priority check
                const checkResult = this.isChartAllowed(element, config);
                if (global.wpGfmConfig?.debug) {
                    console.log('[GFMR Charts] Permission check:', config.type, checkResult.allowed ? 'allowed' : checkResult.reason);
                }

                if (!checkResult.allowed) {
                    this.renderBlockedMessage(element, checkResult);
                    return;
                }

                const Chart = await this.ensureChartJSLoaded();

                // Merge theme colors into config (preserves user-specified colors)
                // Charts always use light mode
                this.mergeThemeColors(config);

                // Create container (light theme fixed)
                const container = document.createElement('div');
                container.className = 'gfmr-chart-container';

                const canvas = document.createElement('canvas');
                canvas.id = 'gfmr-chart-' + Date.now() + '-' +
                    Math.random().toString(36).substr(2, 9);
                container.appendChild(canvas);

                // Replace original element
                const targetElement = element.closest('pre') || element.parentElement;
                if (targetElement?.parentNode) {
                    targetElement.parentNode.replaceChild(container, targetElement);
                }

                new Chart(canvas.getContext('2d'), config);

                if (global.wpGfmConfig?.debug) {
                    console.log('[GFMR Charts] Chart rendered:', config.type);
                }

                this.processedCharts.add(container);
                container.setAttribute('data-gfmr-processed', 'true');

            } catch (error) {
                console.error('[GFMR Charts] Render error:', error);
                this.renderErrorMessage(element, error.message);
            }
        }

        /**
         * Render blocked message (unsupported chart type)
         */
        renderBlockedMessage(element, checkResult) {
            if (global.wpGfmConfig?.debug) {
                console.log('[GFMR Charts] Rendering blocked message:', checkResult.reason);
            }

            const container = document.createElement('div');
            container.className = 'gfmr-chart-blocked';

            const message = checkResult.message || 'Chart not available';

            container.innerHTML = `
                <div class="gfmr-chart-error-content">
                    <div class="gfmr-chart-error-icon">📊</div>
                    <strong>${this.escapeHtml(message)}</strong>
                </div>
            `;

            const targetElement = element.closest('pre') || element.parentElement;
            if (targetElement?.parentNode) {
                targetElement.parentNode.replaceChild(container, targetElement);
            }

            this.processedCharts.add(container);
        }

        /**
         * Render error message
         */
        renderErrorMessage(element, message) {
            const container = document.createElement('div');
            container.className = 'gfmr-chart-error';
            container.innerHTML = `
                <div class="gfmr-chart-error-content">
                    <strong>Chart Error:</strong> ${this.escapeHtml(message)}
                </div>
            `;

            const targetElement = element.closest('pre') || element.parentElement;
            if (targetElement?.parentNode) {
                targetElement.parentNode.replaceChild(container, targetElement);
            }
        }

        /**
         * HTML escape utility
         */
        escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        /**
         * Find and process all chart blocks
         */
        async processAllCharts() {
            // Search for both chart and chart-pro (exclude processed)
            const chartBlocks = document.querySelectorAll(
                'code[class*="language-chart"]:not([data-gfmr-chart-processed]), ' +
                'code.language-chart:not([data-gfmr-chart-processed]), ' +
                'code.language-chart-pro:not([data-gfmr-chart-processed])'
            );

            console.log(`[GFMR Charts] Found ${chartBlocks.length} chart blocks`);

            for (const block of chartBlocks) {
                block.setAttribute('data-gfmr-chart-processed', 'true');

                const content = block.textContent || '';

                try {
                    const config = this.parseChartConfig(content);
                    await this.renderChart(block, config);
                } catch (error) {
                    this.renderErrorMessage(block, error.message);
                }
            }
        }
    }

    // Initialize and export
    function initializeCharts() {
        const charts = new WPGFMCharts();
        global.wpGfmCharts = charts;

        // If Chart.js is already loaded, apply theme defaults
        if (global.Chart) {
            charts.applyThemeToDefaults();
        }

        return charts;
    }

    global.WPGFMCharts = WPGFMCharts;
    global.wpGfmInitializeCharts = initializeCharts;

})(typeof window !== 'undefined' ? window : global);
