/**
 * WP GFM Assets v2.0 - Unified Asset Management Engine
 *
 * Unified module for dynamic library loading and asset management.
 * Reliable asset management with integrated performance measurement, caching, and error handling.
 *
 * Consolidation of 2 files into 1:
 * - gfmr-asset-loader.js
 * - gfmr-performance-metrics.js
 *
 * @package WpGfmRenderer
 * @since 2.0.0
 * @requires gfmr-utils-v2.js
 */

(function(global) {
    'use strict';

    /**
     * WP GFM Assets - Unified Asset Management Engine
     */
    class WPGFMAssets {
        constructor(utils = null) {
            // Dependencies
            this.utils = utils || global.wpGfmUtils;
            if (!this.utils) {
                throw new Error('WPGFMAssets requires WPGFMUtils');
            }
            
            // Library management
            this.loadedLibraries = new Set();
            this.loadPromises = new Map();
            this.libraryInstances = new Map();
            this.libraryCallbacks = new Map();
            
            // Performance measurement
            this.loadTimes = new Map();
            this.metrics = new Map();
            this.resourceSizes = new Map();
            
            // Cache management
            this.cache = new Map();
            this.cacheConfig = {
                enabled: true,
                maxAge: 1000 * 60 * 30, // 30 minutes
                maxSize: 50
            };
            
            // Configuration
            this.config = {
                timeout: this.utils.getConfig('assets.timeout', 30000),
                retryAttempts: this.utils.getConfig('assets.retryAttempts', 3),
                retryDelay: this.utils.getConfig('assets.retryDelay', 1000),
                maxConcurrent: this.utils.getConfig('assets.maxConcurrent', 3),
                useCache: this.utils.getConfig('assets.cache', true),
                preload: this.utils.getConfig('assets.preload', [])
            };
            
            // Library definitions (WordPress.org compliant - using local assets)
            this.libraries = new Map([
                ['shiki', {
                    name: 'Shiki',
                    urls: {
                        production: this.utils.buildAssetUrl('assets/libs/shiki/shiki.min.js'),
                        development: this.utils.buildAssetUrl('assets/libs/shiki/shiki.min.js')
                    },
                    globalName: 'shiki',
                    dependencies: [],
                    priority: 10,
                    size: '~2MB'
                }],
                ['mermaid', {
                    name: 'Mermaid',
                    urls: {
                        production: this.utils.buildAssetUrl('assets/libs/mermaid/mermaid.min.js'),
                        development: this.utils.buildAssetUrl('assets/libs/mermaid/mermaid.min.js')
                    },
                    globalName: 'mermaid',
                    dependencies: [],
                    priority: 8,
                    size: '~1.5MB'
                }],
                ['katex', {
                    name: 'KaTeX',
                    urls: {
                        production: this.utils.buildAssetUrl('assets/libs/katex/katex.min.js'),
                        development: this.utils.buildAssetUrl('assets/libs/katex/katex.min.js')
                    },
                    css: this.utils.buildAssetUrl('assets/libs/katex/katex.min.css'),
                    globalName: 'katex',
                    dependencies: [],
                    priority: 6,
                    size: '~500KB'
                }],
                ['highlight.js', {
                    name: 'Highlight.js',
                    urls: {
                        production: this.utils.buildAssetUrl('assets/libs/highlight.js/highlight.min.js'),
                        development: this.utils.buildAssetUrl('assets/libs/highlight.js/highlight.min.js')
                    },
                    css: this.utils.buildAssetUrl('assets/libs/highlight.js/styles/github.min.css'),
                    globalName: 'hljs',
                    dependencies: [],
                    priority: 5,
                    size: '~200KB'
                }]
            ]);
            
            // Initialization
            this.setupPerformanceTracking();
            this.setupCacheManagement();
            this.preloadAssets();
        }


        // =============================================
        // Initialization and Setup
        // =============================================

        /**
         * Set up performance tracking
         */
        setupPerformanceTracking() {
            this.utils.recordMetric('assets.loaded', 0, 'count');
            this.utils.recordMetric('assets.errors', 0, 'count');
            this.utils.recordMetric('assets.cache.hits', 0, 'count');
            this.utils.recordMetric('assets.cache.misses', 0, 'count');
            this.utils.recordMetric('assets.total.size', 0, 'bytes');
        }

        /**
         * Set up cache management
         */
        setupCacheManagement() {
            if (!this.cacheConfig.enabled) return;
            
            // Periodic cache cleanup
            setInterval(() => {
                this.cleanupCache();
            }, 1000 * 60 * 5); // 5 minute interval
            
            this.utils.debug('Cache management initialized', {
                maxAge: this.cacheConfig.maxAge,
                maxSize: this.cacheConfig.maxSize
            });
        }

        /**
         * Preload processing
         */
        async preloadAssets() {
            const preloadList = this.config.preload;
            if (!preloadList || preloadList.length === 0) return;
            
            this.utils.debug('Starting asset preload', preloadList);
            
            for (const libraryName of preloadList) {
                try {
                    await this.loadLibrary(libraryName, { preload: true });
                } catch (error) {
                    this.utils.debug(`Preload failed for ${libraryName}:`, error);
                }
            }
        }

        // =============================================
        // Library Loading
        // =============================================

        /**
         * Main library loading function
         */
        async loadLibrary(libraryName, options = {}) {
            const startTime = performance.now();
            this.utils.startTimer(`library.${libraryName}`);
            
            // Input validation
            if (!libraryName || typeof libraryName !== 'string') {
                throw new Error('Invalid library name');
            }
            
            const normalizedName = libraryName.toLowerCase();
            
            try {
                
                // Check if already loaded
                if (this.isLibraryLoaded(normalizedName)) {
                    this.utils.debug(`Library ${normalizedName} already loaded`);
                    this.utils.recordMetric('assets.cache.hits', 1, 'count');
                    return this.libraryInstances.get(normalizedName);
                }
                
                // Check if loading is in progress
                if (this.loadPromises.has(normalizedName)) {
                    this.utils.debug(`Library ${normalizedName} loading in progress, waiting...`);
                    return this.loadPromises.get(normalizedName);
                }
                
                // Get library definition
                const libraryConfig = this.libraries.get(normalizedName);
                if (!libraryConfig) {
                    throw new Error(`Unknown library: ${normalizedName}`);
                }
                
                // Start loading process
                const loadPromise = this.doLoadLibrary(normalizedName, libraryConfig, options);
                this.loadPromises.set(normalizedName, loadPromise);
                
                const result = await loadPromise;
                
                // Post-processing on success
                this.loadedLibraries.add(normalizedName);
                this.libraryInstances.set(normalizedName, result);
                this.utils.recordMetric('assets.loaded', 1, 'count');
                
                const loadTime = performance.now() - startTime;
                this.loadTimes.set(normalizedName, loadTime);
                this.utils.endTimer(`library.${libraryName}`);
                
                this.utils.debug(`Library ${normalizedName} loaded successfully in ${loadTime.toFixed(2)}ms`);
                
                return result;
                
            } catch (error) {
                this.utils.recordMetric('assets.errors', 1, 'count');
                this.utils.handleError(error, `library_load_${libraryName}`);
                throw error;
            } finally {
                this.loadPromises.delete(normalizedName);
            }
        }

        /**
         * Actual library loading process
         */
        async doLoadLibrary(libraryName, libraryConfig, options = {}) {
            // Load dependencies first
            if (libraryConfig.dependencies && libraryConfig.dependencies.length > 0) {
                this.utils.debug(`Loading dependencies for ${libraryName}:`, libraryConfig.dependencies);
                
                for (const dependency of libraryConfig.dependencies) {
                    await this.loadLibrary(dependency, { ...options, isDependency: true });
                }
            }
            
            // Cache check
            if (this.config.useCache && !options.ignoreCache) {
                const cached = this.getCachedLibrary(libraryName);
                if (cached) {
                    this.utils.recordMetric('assets.cache.hits', 1, 'count');
                    return cached;
                }
            }
            
            this.utils.recordMetric('assets.cache.misses', 1, 'count');
            
            // Select URL based on environment
            const environment = this.utils.getConfig('debug.enabled', false) ? 'development' : 'production';
            const scriptUrl = libraryConfig.urls[environment] || libraryConfig.urls.production;
            
            // Load CSS (if needed)
            if (libraryConfig.css && !this.isCssLoaded(libraryConfig.css)) {
                await this.loadCss(libraryConfig.css, libraryName);
            }
            
            // Load JavaScript
            const instance = await this.loadScript(scriptUrl, libraryConfig, options);
            
            // Save to cache
            if (this.config.useCache && instance) {
                this.setCachedLibrary(libraryName, instance);
            }
            
            return instance;
        }

        /**
         * Load script
         */
        async loadScript(url, libraryConfig, _options = {}) { // eslint-disable-line no-unused-vars
            return new Promise((resolve, reject) => {
                // Check for existing script tag
                const existingScript = document.querySelector(`script[src="${url}"]`);
                if (existingScript) {
                    this.waitForGlobalVariable(libraryConfig.globalName)
                        .then(resolve)
                        .catch(reject);
                    return;
                }
                
                const script = document.createElement('script');
                script.src = url;
                script.async = true;
                script.defer = false;
                
                // Timeout setting
                const timeoutId = setTimeout(() => {
                    script.remove();
                    reject(new Error(`Timeout loading ${libraryConfig.name} from ${url}`));
                }, this.config.timeout);
                
                script.onload = async () => {
                    clearTimeout(timeoutId);
                    
                    try {
                        // Wait for global variable
                        const instance = await this.waitForGlobalVariable(libraryConfig.globalName);
                        
                        // Record estimated resource size
                        this.recordResourceSize(libraryConfig.name, libraryConfig.size);
                        
                        resolve(instance);
                    } catch (error) {
                        reject(error);
                    }
                };
                
                script.onerror = (error) => {
                    clearTimeout(timeoutId);
                    script.remove();
                    reject(new Error(`Failed to load ${libraryConfig.name} from ${url}: ${error.message || 'Unknown error'}`));
                };
                
                // Set Integrity attribute (if needed)
                if (libraryConfig.integrity) {
                    script.integrity = libraryConfig.integrity;
                    script.crossOrigin = 'anonymous';
                }
                
                document.head.appendChild(script);
                this.utils.debug(`Loading script: ${url}`);
            });
        }

        /**
         * Load CSS
         */
        async loadCss(url, libraryName) {
            return new Promise((resolve, _reject) => { // eslint-disable-line no-unused-vars
                const existingLink = document.querySelector(`link[href="${url}"]`);
                if (existingLink) {
                    resolve();
                    return;
                }
                
                const link = document.createElement('link');
                link.rel = 'stylesheet';
                link.href = url;
                
                link.onload = () => {
                    this.utils.debug(`CSS loaded for ${libraryName}: ${url}`);
                    resolve();
                };
                
                link.onerror = (error) => {
                    this.utils.debug(`CSS load failed for ${libraryName}: ${url}`, error);
                    // CSS load failure is not a fatal error
                    resolve();
                };
                
                document.head.appendChild(link);
            });
        }

        /**
         * Wait for global variable
         */
        async waitForGlobalVariable(globalName, timeout = 10000) {
            const startTime = Date.now();
            
            while (!global[globalName] && (Date.now() - startTime) < timeout) {
                await this.delay(50);
            }
            
            if (!global[globalName]) {
                throw new Error(`Global variable '${globalName}' not available after timeout`);
            }
            
            return global[globalName];
        }

        // =============================================
        // Cache Management
        // =============================================

        /**
         * Get library from cache
         */
        getCachedLibrary(libraryName) {
            if (!this.cacheConfig.enabled) return null;
            
            const cached = this.cache.get(libraryName);
            if (!cached) return null;
            
            // Expiration check
            const now = Date.now();
            if (now - cached.timestamp > this.cacheConfig.maxAge) {
                this.cache.delete(libraryName);
                return null;
            }
            
            return cached.instance;
        }

        /**
         * Save library to cache
         */
        setCachedLibrary(libraryName, instance) {
            if (!this.cacheConfig.enabled) return;
            
            this.cache.set(libraryName, {
                instance: instance,
                timestamp: Date.now()
            });
            
            // Cache size limit
            if (this.cache.size > this.cacheConfig.maxSize) {
                const oldestKey = this.cache.keys().next().value;
                this.cache.delete(oldestKey);
            }
        }

        /**
         * Cache cleanup
         */
        cleanupCache() {
            const now = Date.now();
            const expiredKeys = [];
            
            for (const [key, cached] of this.cache) {
                if (now - cached.timestamp > this.cacheConfig.maxAge) {
                    expiredKeys.push(key);
                }
            }
            
            expiredKeys.forEach(key => this.cache.delete(key));
            
            if (expiredKeys.length > 0) {
                this.utils.debug(`Cleaned up ${expiredKeys.length} expired cache entries`);
            }
        }

        // =============================================
        // Performance Measurement
        // =============================================

        /**
         * Record resource size
         */
        recordResourceSize(libraryName, sizeInfo) {
            // Extract numeric value from size info (e.g., "~2MB" -> 2000000)
            let sizeBytes = 0;
            if (typeof sizeInfo === 'string') {
                const match = sizeInfo.match(/(\d+(?:\.\d+)?)\s*(KB|MB|GB)/i);
                if (match) {
                    const value = parseFloat(match[1]);
                    const unit = match[2].toUpperCase();
                    
                    switch (unit) {
                        case 'KB': sizeBytes = value * 1024; break;
                        case 'MB': sizeBytes = value * 1024 * 1024; break;
                        case 'GB': sizeBytes = value * 1024 * 1024 * 1024; break;
                    }
                }
            } else if (typeof sizeInfo === 'number') {
                sizeBytes = sizeInfo;
            }
            
            this.resourceSizes.set(libraryName, sizeBytes);
            this.utils.recordMetric('assets.total.size', sizeBytes, 'bytes');
        }

        /**
         * Generate performance report
         */
        getPerformanceReport() {
            const report = {
                summary: {
                    totalLibraries: this.loadedLibraries.size,
                    totalLoadTime: 0,
                    totalSize: 0,
                    averageLoadTime: 0,
                    cacheHitRate: 0
                },
                libraries: [],
                cache: {
                    size: this.cache.size,
                    maxSize: this.cacheConfig.maxSize,
                    enabled: this.cacheConfig.enabled
                },
                metrics: {}
            };
            
            // Library details
            for (const [name, loadTime] of this.loadTimes) {
                const size = this.resourceSizes.get(name) || 0;
                report.libraries.push({
                    name: name,
                    loadTime: Math.round(loadTime * 100) / 100,
                    size: size,
                    sizeFormatted: this.formatBytes(size)
                });
                
                report.summary.totalLoadTime += loadTime;
                report.summary.totalSize += size;
            }
            
            // Calculate aggregate values
            if (this.loadedLibraries.size > 0) {
                report.summary.averageLoadTime = report.summary.totalLoadTime / this.loadedLibraries.size;
            }
            
            // Metrics statistics
            report.metrics = {
                loaded: this.utils.getMetricStats('assets.loaded'),
                errors: this.utils.getMetricStats('assets.errors'),
                cacheHits: this.utils.getMetricStats('assets.cache.hits'),
                cacheMisses: this.utils.getMetricStats('assets.cache.misses')
            };
            
            // Calculate cache hit rate
            const hits = report.metrics.cacheHits?.count || 0;
            const misses = report.metrics.cacheMisses?.count || 0;
            if (hits + misses > 0) {
                report.summary.cacheHitRate = (hits / (hits + misses)) * 100;
            }
            
            return report;
        }

        /**
         * Format byte size
         */
        formatBytes(bytes) {
            if (bytes === 0) return '0 B';
            
            const k = 1024;
            const sizes = ['B', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        }

        // =============================================
        // Utilities and Management
        // =============================================

        /**
         * Check library load status
         */
        isLibraryLoaded(libraryName) {
            const normalizedName = libraryName.toLowerCase();
            return this.loadedLibraries.has(normalizedName);
        }

        /**
         * Check CSS load status
         */
        isCssLoaded(url) {
            return !!document.querySelector(`link[href="${url}"]`);
        }

        /**
         * Unload all libraries
         */
        unloadAllLibraries() {
            // Remove script tags
            for (const [libraryName, config] of this.libraries) {
                if (this.loadedLibraries.has(libraryName)) {
                    const environment = this.utils.getConfig('debug.enabled', false) ? 'development' : 'production';
                    const url = config.urls[environment] || config.urls.production;
                    
                    const script = document.querySelector(`script[src="${url}"]`);
                    if (script) {
                        script.remove();
                    }
                    
                    // Clean up global variables
                    if (config.globalName && global[config.globalName]) {
                        delete global[config.globalName];
                    }
                }
            }
            
            // Reset internal state
            this.loadedLibraries.clear();
            this.libraryInstances.clear();
            this.loadPromises.clear();
            this.cache.clear();
            this.loadTimes.clear();
            this.resourceSizes.clear();
            
            this.utils.debug('All libraries unloaded');
        }

        /**
         * Manually add library
         */
        addLibrary(name, config) {
            if (!name || !config) {
                throw new Error('Invalid library definition');
            }
            
            const requiredFields = ['name', 'urls', 'globalName'];
            for (const field of requiredFields) {
                if (!config[field]) {
                    throw new Error(`Missing required field '${field}' in library config`);
                }
            }
            
            this.libraries.set(name.toLowerCase(), {
                priority: 1,
                dependencies: [],
                size: 'unknown',
                ...config
            });
            
            this.utils.debug(`Library ${name} added to registry`);
        }

        /**
         * Delay processing
         */
        delay(ms) {
            return new Promise(resolve => setTimeout(resolve, ms));
        }

        /**
         * Health check
         */
        healthCheck() {
            return {
                status: 'ok',
                loadedLibraries: Array.from(this.loadedLibraries),
                availableLibraries: Array.from(this.libraries.keys()),
                cacheEnabled: this.cacheConfig.enabled,
                cacheSize: this.cache.size,
                pendingLoads: this.loadPromises.size,
                config: this.config
            };
        }

        /**
         * Get statistics
         */
        getStats() {
            return {
                loadedCount: this.loadedLibraries.size,
                availableCount: this.libraries.size,
                cacheSize: this.cache.size,
                pendingLoads: this.loadPromises.size,
                performanceReport: this.getPerformanceReport(),
                healthCheck: this.healthCheck()
            };
        }
    }

    // Create and expose global instance
    let wpGfmAssets = null;

    // Initialization function
    function initializeAssets() {
        if (wpGfmAssets) return wpGfmAssets;
        
        wpGfmAssets = new WPGFMAssets();
        
        // Expose as global variable
        global.wpGfmAssets = wpGfmAssets;
        global.WPGFMAssets = WPGFMAssets;
        
        // Legacy compatible API
        global.wpGfmLoadLibrary = (name, options) => {
            return wpGfmAssets.loadLibrary(name, options);
        };
        
        // Maintain asset loader compatibility
        global.wpGfmAssetLoader = {
            loadLibrary: (name, options) => wpGfmAssets.loadLibrary(name, options),
            isLoaded: (name) => wpGfmAssets.isLibraryLoaded(name),
            getStats: () => wpGfmAssets.getStats()
        };
        
        // Initialization complete event
        if (global.document) {
            global.document.dispatchEvent(new CustomEvent('wpGfmAssetsReady', {
                detail: { assets: wpGfmAssets }
            }));
        }
        
        return wpGfmAssets;
    }
    
    // Control initialization timing
    if (global.document && global.document.readyState === 'loading') {
        global.document.addEventListener('DOMContentLoaded', initializeAssets);
    } else {
        setTimeout(initializeAssets, 0);
    }
    
    // Monitor utility ready event
    if (global.document) {
        global.document.addEventListener('wpGfmUtilsReady', initializeAssets);
    }
    
    // CommonJS / AMD support
    if (typeof module !== 'undefined' && module.exports) {
        module.exports = WPGFMAssets;
    }

})(typeof window !== 'undefined' ? window : global);