/**
 * GFMR Admin Settings Page JavaScript
 *
 * Handles theme preview and settings page interactions.
 *
 * @package MarkdownRendererForGitHub
 * @since 1.8.0
 */

(function($) {
	'use strict';

	$(document).ready(function() {
		const previewContainer = $('#gfmr-theme-preview');
		const previewCode = $('#gfmr-preview-code');
		const themeDisplay = $('#gfmr-theme-display');

		// Exit if not on settings page
		if (previewContainer.length === 0) {
			return;
		}

		let isHighlightingInitialized = false;
		const originalCodeContent = previewCode.text();

		// Get localized data from PHP
		const gfmrAdminSettings = window.gfmrAdminSettings || {};
		const optionName = gfmrAdminSettings.optionName || 'gfmr_theme_settings';
		const i18n = gfmrAdminSettings.i18n || {
			light: 'Light',
			dark: 'Dark',
			system: 'System',
			applyingTheme: 'Applying theme...'
		};

		// Shiki theme mapping for admin preview
		const shikiThemeMapping = {
			light: 'github-light',
			dark: 'github-dark',
			system: 'auto'
		};

		/**
		 * Check system theme preference
		 *
		 * @return {string} Shiki theme name based on system preference
		 */
		function getSystemTheme() {
			const prefersDark = window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches;
			console.log('GFMR Admin Preview: System prefers', prefersDark ? 'dark' : 'light', 'theme');
			return prefersDark ? 'github-dark' : 'github-light';
		}

		/**
		 * Get effective Shiki theme based on selection
		 *
		 * @param {string} selectedTheme - The user-selected theme option
		 * @return {string} Effective Shiki theme name
		 */
		function getEffectiveShikiTheme(selectedTheme) {
			if (selectedTheme === 'system') {
				return getSystemTheme();
			}
			return shikiThemeMapping[selectedTheme] || 'github-light';
		}

		/**
		 * Initialize Shiki highlighting for admin preview
		 */
		function initializeHighlighting() {
			if (isHighlightingInitialized) {
				return;
			}

			try {
				console.log('GFMR Admin Preview: Initializing with basic syntax highlighting...');

				// Mark as initialized to avoid retry
				isHighlightingInitialized = true;

				// Apply basic syntax highlighting immediately
				applyBasicSyntaxHighlighting();

				// Apply initial theme styling
				const currentTheme = previewContainer.data('current-theme');
				updateThemePreview(currentTheme, false);

				console.log('GFMR Admin Preview: Basic highlighting initialized');

			} catch (error) {
				console.warn('GFMR Admin Preview: Error initializing highlighting', error);
				isHighlightingInitialized = true;
			}
		}

		/**
		 * Apply basic syntax highlighting with proper HTML handling
		 */
		function applyBasicSyntaxHighlighting() {
			try {
				const code = originalCodeContent;

				// Apply syntax highlighting first (before HTML escaping)
				let highlightedCode = code
					.replace(/\/\/(.*)/g, '##COMMENT_START##$1##COMMENT_END##')
					.replace(/\b(const|let|var|function|return|if|else|for|while|class|async|await)\b/g, '##KEYWORD_START##$1##KEYWORD_END##')
					.replace(/\b(console|window|document|Array|Object)\b/g, '##PROPERTY_START##$1##PROPERTY_END##')
					.replace(/(['"`])((?:\\.|(?!\1)[^\\])*?)\1/g, '##STRING_START##$1$2$1##STRING_END##')
					.replace(/\b(\d+)\b/g, '##NUMBER_START##$1##NUMBER_END##');

				// Now safely escape HTML entities
				highlightedCode = highlightedCode
					.replace(/&/g, '&amp;')
					.replace(/</g, '&lt;')
					.replace(/>/g, '&gt;')
					.replace(/"/g, '&quot;')
					.replace(/'/g, '&#x27;');

				// Convert markers to proper HTML spans
				highlightedCode = highlightedCode
					.replace(/##COMMENT_START##/g, '<span class="gfmr-comment">')
					.replace(/##COMMENT_END##/g, '</span>')
					.replace(/##KEYWORD_START##/g, '<span class="gfmr-keyword">')
					.replace(/##KEYWORD_END##/g, '</span>')
					.replace(/##PROPERTY_START##/g, '<span class="gfmr-property">')
					.replace(/##PROPERTY_END##/g, '</span>')
					.replace(/##STRING_START##/g, '<span class="gfmr-string">')
					.replace(/##STRING_END##/g, '</span>')
					.replace(/##NUMBER_START##/g, '<span class="gfmr-number">')
					.replace(/##NUMBER_END##/g, '</span>');

				// Update preview with properly structured HTML
				$('#gfmr-preview-wrapper').html(
					'<pre class="gfmr-admin-preview"><code>' + highlightedCode + '</code></pre>'
				);
				console.log('GFMR Admin Preview: Applied basic syntax highlighting');

			} catch (error) {
				console.warn('GFMR Admin Preview: Basic highlighting failed', error);
				// Fallback to plain text with proper escaping
				const escapedFallback = originalCodeContent
					.replace(/&/g, '&amp;')
					.replace(/</g, '&lt;')
					.replace(/>/g, '&gt;')
					.replace(/"/g, '&quot;')
					.replace(/'/g, '&#x27;');
				$('#gfmr-preview-wrapper').html(
					'<pre class="gfmr-admin-preview"><code>' + escapedFallback + '</code></pre>'
				);
			}
		}

		/**
		 * Update theme preview display
		 *
		 * @param {string} theme - Theme to preview
		 * @param {boolean} applyHighlighting - Whether to reapply highlighting
		 */
		function updateThemePreview(theme, applyHighlighting) {
			// Update preview container data attribute
			previewContainer.attr('data-preview-theme', theme);

			console.log('GFMR Admin Preview: Updating theme preview to:', theme);

			// Always reapply basic syntax highlighting when theme changes
			if (applyHighlighting && isHighlightingInitialized) {
				applyBasicSyntaxHighlighting();
			}

			// Add visual feedback during theme switching
			previewContainer.addClass('gfmr-theme-switching');

			setTimeout(function() {
				previewContainer.removeClass('gfmr-theme-switching');
			}, 600);
		}

		/**
		 * Update theme display text
		 *
		 * @param {string} theme - Selected theme
		 */
		function updateThemeDisplay(theme) {
			const themeNames = {
				'light': i18n.light,
				'dark': i18n.dark,
				'system': i18n.system
			};
			themeDisplay.text(themeNames[theme] || theme);
		}

		// Expose getEffectiveShikiTheme for potential external use
		window.gfmrGetEffectiveShikiTheme = getEffectiveShikiTheme;

		// Start initialization
		setTimeout(initializeHighlighting, 500);

		// Handle theme selection changes
		$('input[name="' + optionName + '[theme]"]').on('change', function() {
			const selectedTheme = $(this).val();
			updateThemePreview(selectedTheme, true);
			updateThemeDisplay(selectedTheme);
		});

		// System theme detection
		if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
			console.log('GFMR Admin Preview: System prefers dark theme');
		} else {
			console.log('GFMR Admin Preview: System prefers light theme');
		}

		// Handle system theme changes for 'system' option
		if (window.matchMedia) {
			const darkModeQuery = window.matchMedia('(prefers-color-scheme: dark)');
			darkModeQuery.addEventListener('change', function() {
				const currentSelection = $('input[name="' + optionName + '[theme]"]:checked').val();
				if (currentSelection === 'system') {
					updateThemePreview('system', true);
				}
			});
		}

		/**
		 * Tab switching functionality
		 */
		const form = $('#gfmr-settings-form');
		let hiddenInput = form.find('input[name="gfmr_active_tab"]');
		if (hiddenInput.length === 0) {
			hiddenInput = $('<input>').attr({
				type: 'hidden',
				name: 'gfmr_active_tab'
			});
			form.append(hiddenInput);
		}

		function switchTab(tabId) {
			// Hide all tab contents
			$('.gfmr-tab-content').hide();

			// Remove active class from all tabs
			$('.nav-tab').removeClass('nav-tab-active');

			// Show selected tab content
			$('#tab-' + tabId).show();

			// Add active class to clicked tab
			$('.nav-tab[data-tab="' + tabId + '"]').addClass('nav-tab-active');

			// Update hidden input
			hiddenInput.val(tabId);

			// Update URL hash without scrolling
			history.replaceState(null, null, '#tab-' + tabId);
		}

		// Tab click event handler
		$('.nav-tab').on('click', function(e) {
			e.preventDefault();
			const tabId = $(this).data('tab');
			switchTab(tabId);
		});

		// Restore active tab from URL hash on page load
		const urlHash = window.location.hash;
		if (urlHash && urlHash.startsWith('#tab-')) {
			const tabId = urlHash.substring(5); // Remove '#tab-' prefix
			switchTab(tabId);
		}
	});
})(jQuery);
