<?php
/**
 * Plugin Name: MameTech Chat Button
 * Plugin URI: https://www.mametech.com/mametech-chat-button
 * Description: Plugin completo con widget, shortcode, orari, numeri multipli, statistiche e temi per chat flottante
 * Version: 2.1.3
 * Author: Mario Merola
 * Author URI: https://www.mametech.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: mametech-chat-button
 * Domain Path: /languages
 */

// Impedisce l'accesso diretto al file
if (!defined('ABSPATH')) {
    exit;
}

// Definizione delle costanti del plugin
define('MAMETECH_CB_VERSION', '2.1.3');
define('MAMETECH_CB_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('MAMETECH_CB_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MAMETECH_CB_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Classe principale del plugin
 */
class MameTech_Chat_Button {
    
    /**
     * Costruttore
     */
    public function __construct() {
        // Hook per l'attivazione del plugin
        register_activation_hook(__FILE__, array($this, 'activate'));
        
        // Hook per la disattivazione del plugin
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        
        // Aggiungi menu admin
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // Registra le impostazioni
        add_action('admin_init', array($this, 'register_settings'));
        
        // Aggiungi il pulsante al frontend
        add_action('wp_footer', array($this, 'add_whatsapp_button'));
        
        // Carica gli stili e script admin
        add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));
        
        // Carica gli stili frontend
        add_action('wp_enqueue_scripts', array($this, 'frontend_scripts'));
        
        // Registra shortcode
        add_shortcode('mametech_chat_button', array($this, 'shortcode_button'));
        
        // Registra widget
        add_action('widgets_init', array($this, 'register_widget'));
        
        // AJAX per statistiche
        add_action('wp_ajax_mametech_cb_track_click', array($this, 'track_click'));
        add_action('wp_ajax_nopriv_mametech_cb_track_click', array($this, 'track_click'));
        
        // AJAX per gestione numeri multipli
        add_action('wp_ajax_mametech_cb_save_numbers', array($this, 'save_multiple_numbers'));
        add_action('wp_ajax_mametech_cb_delete_number', array($this, 'delete_number'));
    }
    
    /**
     * Attivazione plugin
     */
    public function activate() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        
        // Crea tabella per statistiche
        $table_stats = $wpdb->prefix . 'mametech_cb_stats';
        $sql_stats = "CREATE TABLE IF NOT EXISTS $table_stats (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            button_id varchar(50) NOT NULL DEFAULT 'main',
            click_date datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            user_ip varchar(45),
            user_agent text,
            page_url text,
            PRIMARY KEY  (id),
            KEY button_id (button_id),
            KEY click_date (click_date)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql_stats);
        
        // Opzioni di base
        $defaults = array(
            'mametech_cb_phone_number' => '',
            'mametech_cb_enabled' => '0',
            'mametech_cb_message' => 'Ciao! Come posso aiutarti?',
            'mametech_cb_position' => 'bottom-right',
            'mametech_cb_button_size' => '60',
            'mametech_cb_theme' => 'default',
            'mametech_cb_schedule_enabled' => '0',
            'mametech_cb_schedule_hours' => array(),
            'mametech_cb_multiple_numbers' => array(),
            'mametech_cb_enable_stats' => '1',
            'mametech_cb_tooltip_text' => 'Chatta con noi!',
            'mametech_cb_show_tooltip' => '1'
        );
        
        foreach ($defaults as $key => $value) {
            if (get_option($key) === false) {
                add_option($key, $value);
            }
        }
    }
    
    /**
     * Disattivazione plugin
     */
    public function deactivate() {
        // Cleanup se necessario
    }
    
    /**
     * Assicura che la tabella statistiche esista
     */
    private function ensure_stats_table_exists() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'mametech_cb_stats';
        
        // Controlla se la tabella esiste
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $table_exists = $wpdb->get_var(
            $wpdb->prepare(
                "SHOW TABLES LIKE %s",
                $table_name
            )
        );
        
        // Se non esiste, creala
        if ($table_exists !== $table_name) {
            $charset_collate = $wpdb->get_charset_collate();
            
            $sql = "CREATE TABLE $table_name (
                id bigint(20) NOT NULL AUTO_INCREMENT,
                button_id varchar(50) NOT NULL DEFAULT 'main',
                click_date datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
                user_ip varchar(45),
                user_agent text,
                page_url text,
                PRIMARY KEY  (id),
                KEY button_id (button_id),
                KEY click_date (click_date)
            ) $charset_collate;";
            
            require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
            dbDelta($sql);
        }
    }
    
    /**
     * Aggiungi menu admin
     */
    public function add_admin_menu() {
        add_menu_page(
            'MameTech Chat Button',
            'Chat Button PRO',
            'manage_options',
            'mametech-chat-button',
            array($this, 'admin_page'),
            'dashicons-whatsapp',
            80
        );
        
        add_submenu_page(
            'mametech-chat-button',
            'Impostazioni',
            'Impostazioni',
            'manage_options',
            'mametech-chat-button',
            array($this, 'admin_page')
        );
        
        add_submenu_page(
            'mametech-chat-button',
            'Statistiche',
            'Statistiche',
            'manage_options',
            'mametech-chat-button-stats',
            array($this, 'stats_page')
        );
        
        add_submenu_page(
            'mametech-chat-button',
            'Numeri Multipli',
            'Numeri Multipli',
            'manage_options',
            'mametech-chat-button-numbers',
            array($this, 'numbers_page')
        );
    }
    
    /**
     * Registra le impostazioni
     */
    public function register_settings() {
        register_setting('mametech_cb_settings_group', 'mametech_cb_phone_number', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_phone'),
            'default' => ''
        ));
        
        register_setting('mametech_cb_settings_group', 'mametech_cb_enabled', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => '0'
        ));
        
        register_setting('mametech_cb_settings_group', 'mametech_cb_message', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_textarea_field',
            'default' => 'Ciao! Come posso aiutarti?'
        ));
        
        register_setting('mametech_cb_settings_group', 'mametech_cb_position', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_position'),
            'default' => 'bottom-right'
        ));
        
        register_setting('mametech_cb_settings_group', 'mametech_cb_button_size', array(
            'type' => 'integer',
            'sanitize_callback' => 'absint',
            'default' => 60
        ));
        
        register_setting('mametech_cb_settings_group', 'mametech_cb_theme', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_theme'),
            'default' => 'default'
        ));
        
        register_setting('mametech_cb_settings_group', 'mametech_cb_schedule_enabled', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => '0'
        ));
        
        register_setting('mametech_cb_settings_group', 'mametech_cb_schedule_hours', array(
            'type' => 'array',
            'sanitize_callback' => array($this, 'sanitize_schedule_hours'),
            'default' => array()
        ));
        
        register_setting('mametech_cb_settings_group', 'mametech_cb_enable_stats', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => '1'
        ));
        
        register_setting('mametech_cb_settings_group', 'mametech_cb_tooltip_text', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => 'Chatta con noi!'
        ));
        
        register_setting('mametech_cb_settings_group', 'mametech_cb_show_tooltip', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => '1'
        ));
        
        register_setting('mametech_cb_settings_group', 'mametech_cb_multiple_menu_text', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => 'Scegli con chi parlare'
        ));
    }
    
    /**
     * Sanitizza checkbox (0 o 1)
     */
    public function sanitize_checkbox($value) {
        return ($value === '1' || $value === 1 || $value === true) ? '1' : '0';
    }
    
    /**
     * Sanitizza la posizione
     */
    public function sanitize_position($value) {
        $valid_positions = array('bottom-right', 'bottom-left', 'top-right', 'top-left');
        return in_array($value, $valid_positions, true) ? $value : 'bottom-right';
    }
    
    /**
     * Sanitizza il tema
     */
    public function sanitize_theme($value) {
        $valid_themes = array('default', 'minimal', 'gradient', 'pulse', 'dark');
        return in_array($value, $valid_themes, true) ? $value : 'default';
    }
    
    /**
     * Sanitizza gli orari programmati
     */
    public function sanitize_schedule_hours($value) {
        if (!is_array($value)) {
            return array();
        }
        
        $sanitized = array();
        $valid_days = array('monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday');
        
        foreach ($value as $day => $schedule) {
            if (!in_array($day, $valid_days, true)) {
                continue;
            }
            
            $sanitized[$day] = array(
                'enabled' => isset($schedule['enabled']) ? (bool) $schedule['enabled'] : false,
                'start' => isset($schedule['start']) ? sanitize_text_field($schedule['start']) : '09:00',
                'end' => isset($schedule['end']) ? sanitize_text_field($schedule['end']) : '18:00'
            );
        }
        
        return $sanitized;
    }
    
    /**
     * Sanitizza il numero di telefono
     */
    public function sanitize_phone($phone) {
        // Rimuovi tutto tranne numeri e +
        $phone = preg_replace('/[^0-9+]/', '', $phone);
        
        // Se non inizia con +, aggiungi +39 (Italia)
        if (!empty($phone) && substr($phone, 0, 1) !== '+') {
            $phone = '+39' . $phone;
        }
        
        return $phone;
    }
    
    /**
     * Carica script admin
     */
    public function admin_scripts($hook) {
        if (strpos($hook, 'mametech-chat-button') === false) {
            return;
        }
        
        wp_enqueue_script('jquery');
        wp_enqueue_script('jquery-ui-sortable');
        
        wp_enqueue_style('mametech-cb-admin-style', MAMETECH_CB_PLUGIN_URL . 'assets/css/admin.css', array(), MAMETECH_CB_VERSION);
        
        // Enqueue admin scripts
        wp_enqueue_script('mametech-cb-admin-scripts', MAMETECH_CB_PLUGIN_URL . 'assets/js/admin-scripts.js', array('jquery'), MAMETECH_CB_VERSION, true);
        
        // Enqueue numbers management scripts if on numbers page
        // Use $hook parameter instead of $_GET to avoid nonce warnings
        if (strpos($hook, 'mametech-chat-button-numbers') !== false) {
            wp_enqueue_script('mametech-cb-numbers-scripts', MAMETECH_CB_PLUGIN_URL . 'assets/js/numbers-scripts.js', array('jquery', 'jquery-ui-sortable'), MAMETECH_CB_VERSION, true);
        }
        
        wp_localize_script('mametech-cb-admin-scripts', 'mametechCbAjax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('mametech_cb_nonce')
        ));
    }
    
    /**
     * Carica stili e script frontend
     */
    public function frontend_scripts() {
        // Carica sempre il CSS per supportare shortcode anche senza floating button
        wp_enqueue_style('mametech-cb-frontend-style', MAMETECH_CB_PLUGIN_URL . 'assets/css/frontend.css', array(), MAMETECH_CB_VERSION);
        
        // Script e logica solo se il floating button è abilitato
        if ($this->should_display_button()) {
            // Aggiungi stili dinamici per le dimensioni del pulsante
            $size = get_option('mametech_cb_button_size', '60');
            $dynamic_css = sprintf(
                '.wfb-floating-button a { width: %1$dpx; height: %1$dpx; padding: %2$dpx; }
                @media (max-width: 768px) {
                    .wfb-floating-button a { width: %3$dpx; height: %3$dpx; }
                    .wfb-tooltip { display: none; }
                }',
                intval($size),
                intval($size * 0.2),
                intval($size * 0.8)
            );
            wp_add_inline_style('mametech-cb-frontend-style', $dynamic_css);
            
            wp_enqueue_script('mametech-cb-frontend', MAMETECH_CB_PLUGIN_URL . 'assets/js/frontend.js', array('jquery'), MAMETECH_CB_VERSION, true);
            
            $multiple_numbers = get_option('mametech_cb_multiple_numbers', array());
            
            wp_localize_script('mametech-cb-frontend', 'mametechCbData', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('mametech_cb_track_nonce'),
                'enable_stats' => get_option('mametech_cb_enable_stats', '1'),
                'multiple_numbers' => $multiple_numbers,
                'button_position' => get_option('mametech_cb_position', 'bottom-right'),
                'default_message' => get_option('mametech_cb_message', 'Ciao! Come posso aiutarti?'),
                'multiple_menu_text' => get_option('mametech_cb_multiple_menu_text', 'Scegli con chi parlare')
            ));
        }
    }
    
    /**
     * Verifica se il pulsante deve essere mostrato
     */
    private function should_display_button() {
        if (get_option('mametech_cb_enabled') !== '1') {
            return false;
        }
        
        // Verifica orari se abilitati
        if (get_option('mametech_cb_schedule_enabled') === '1') {
            return $this->is_within_schedule();
        }
        
        return true;
    }
    
    /**
     * Verifica se siamo negli orari programmati
     */
    private function is_within_schedule() {
        $schedule = get_option('mametech_cb_schedule_hours', array());
        
        if (empty($schedule)) {
            return true;
        }
        
        $current_day = strtolower(current_time('l'));
        $current_time = current_time('H:i');
        
        if (!isset($schedule[$current_day])) {
            return false;
        }
        
        $day_schedule = $schedule[$current_day];
        
        if (!$day_schedule['enabled']) {
            return false;
        }
        
        return ($current_time >= $day_schedule['start'] && $current_time < $day_schedule['end']);
    }
    
    /**
     * Pagina impostazioni admin
     */
    public function admin_page() {
        ?>
        <div class="wrap wfb-admin-wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <?php settings_errors(); ?>
            
            <div class="wfb-tabs">
                <button class="wfb-tab-button active" data-tab="general">Generali</button>
                <button class="wfb-tab-button" data-tab="appearance">Aspetto</button>
                <button class="wfb-tab-button" data-tab="schedule">Orari</button>
                <button class="wfb-tab-button" data-tab="advanced">Avanzate</button>
            </div>
            
            <form method="post" action="options.php">
                <?php
                settings_fields('mametech_cb_settings_group');
                do_settings_sections('mametech_cb_settings_group');
                ?>
                
                <!-- Tab Generali -->
                <div class="wfb-tab-content active" id="tab-general">
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="mametech_cb_enabled">Attiva Pulsante WhatsApp</label>
                            </th>
                            <td>
                                <label class="wfb-switch">
                                    <input type="checkbox" name="mametech_cb_enabled" id="mametech_cb_enabled" value="1" 
                                           <?php checked(get_option('mametech_cb_enabled'), '1'); ?>>
                                    <span class="wfb-slider"></span>
                                </label>
                                <p class="description">Mostra il pulsante WhatsApp sul sito</p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="mametech_cb_phone_number">Numero di Telefono Principale</label>
                            </th>
                            <td>
                                <input type="text" name="mametech_cb_phone_number" id="mametech_cb_phone_number" 
                                       value="<?php echo esc_attr(get_option('mametech_cb_phone_number')); ?>" 
                                       class="regular-text" placeholder="+393331234567">
                                <p class="description">
                                    Inserisci il numero completo con prefisso internazionale (es: +393331234567)
                                    <br>Per configurare numeri multipli, vai alla sezione <a href="<?php echo esc_url(admin_url('admin.php?page=mametech-chat-button-numbers')); ?>">Numeri Multipli</a>
                                </p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="mametech_cb_message">Messaggio Predefinito</label>
                            </th>
                            <td>
                                <textarea name="mametech_cb_message" id="mametech_cb_message" rows="3" class="large-text"
                                          placeholder="Ciao! Come posso aiutarti?"><?php echo esc_textarea(get_option('mametech_cb_message')); ?></textarea>
                                <p class="description">Messaggio che apparirà precompilato nella chat WhatsApp</p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="mametech_cb_enable_stats">Abilita Statistiche</label>
                            </th>
                            <td>
                                <label class="wfb-switch">
                                    <input type="checkbox" name="mametech_cb_enable_stats" id="mametech_cb_enable_stats" value="1" 
                                           <?php checked(get_option('mametech_cb_enable_stats', '1'), '1'); ?>>
                                    <span class="wfb-slider"></span>
                                </label>
                                <p class="description">Traccia i click sul pulsante WhatsApp</p>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <!-- Tab Aspetto -->
                <div class="wfb-tab-content" id="tab-appearance">
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="mametech_cb_theme">Tema</label>
                            </th>
                            <td>
                                <select name="mametech_cb_theme" id="mametech_cb_theme" class="regular-text">
                                    <option value="default" <?php selected(get_option('mametech_cb_theme', 'default'), 'default'); ?>>
                                        Default (Verde WhatsApp)
                                    </option>
                                    <option value="minimal" <?php selected(get_option('mametech_cb_theme'), 'minimal'); ?>>
                                        Minimale
                                    </option>
                                    <option value="gradient" <?php selected(get_option('mametech_cb_theme'), 'gradient'); ?>>
                                        Gradiente
                                    </option>
                                    <option value="pulse" <?php selected(get_option('mametech_cb_theme'), 'pulse'); ?>>
                                        Pulsante
                                    </option>
                                    <option value="dark" <?php selected(get_option('mametech_cb_theme'), 'dark'); ?>>
                                        Scuro
                                    </option>
                                </select>
                                <p class="description">Scegli lo stile del pulsante</p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="mametech_cb_position">Posizione del Pulsante</label>
                            </th>
                            <td>
                                <select name="mametech_cb_position" id="mametech_cb_position">
                                    <option value="bottom-right" <?php selected(get_option('mametech_cb_position'), 'bottom-right'); ?>>
                                        Basso a Destra
                                    </option>
                                    <option value="bottom-left" <?php selected(get_option('mametech_cb_position'), 'bottom-left'); ?>>
                                        Basso a Sinistra
                                    </option>
                                    <option value="top-right" <?php selected(get_option('mametech_cb_position'), 'top-right'); ?>>
                                        Alto a Destra
                                    </option>
                                    <option value="top-left" <?php selected(get_option('mametech_cb_position'), 'top-left'); ?>>
                                        Alto a Sinistra
                                    </option>
                                </select>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="mametech_cb_button_size">Dimensione Pulsante (px)</label>
                            </th>
                            <td>
                                <input type="range" name="mametech_cb_button_size" id="mametech_cb_button_size" 
                                       value="<?php echo esc_attr(get_option('mametech_cb_button_size', '60')); ?>" 
                                       min="40" max="100" step="5">
                                <span id="size-value"><?php echo esc_html(get_option('mametech_cb_button_size', '60')); ?>px</span>
                                <p class="description">Dimensione del pulsante in pixel (da 40 a 100)</p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="mametech_cb_show_tooltip">Mostra Tooltip</label>
                            </th>
                            <td>
                                <label class="wfb-switch">
                                    <input type="checkbox" name="mametech_cb_show_tooltip" id="mametech_cb_show_tooltip" value="1" 
                                           <?php checked(get_option('mametech_cb_show_tooltip', '1'), '1'); ?>>
                                    <span class="wfb-slider"></span>
                                </label>
                                <p class="description">Mostra il tooltip quando si passa sopra il pulsante</p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="mametech_cb_tooltip_text">Testo Tooltip</label>
                            </th>
                            <td>
                                <input type="text" name="mametech_cb_tooltip_text" id="mametech_cb_tooltip_text" 
                                       value="<?php echo esc_attr(get_option('mametech_cb_tooltip_text', 'Chatta con noi!')); ?>" 
                                       class="regular-text" placeholder="Chatta con noi!">
                                <p class="description">Testo che appare nel tooltip</p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="mametech_cb_multiple_menu_text">Testo Menu Numeri Multipli</label>
                            </th>
                            <td>
                                <input type="text" name="mametech_cb_multiple_menu_text" id="mametech_cb_multiple_menu_text" 
                                       value="<?php echo esc_attr(get_option('mametech_cb_multiple_menu_text', 'Scegli con chi parlare')); ?>" 
                                       class="regular-text" placeholder="Scegli con chi parlare">
                                <p class="description">Testo mostrato nel menu quando ci sono numeri multipli</p>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <!-- Tab Orari -->
                <div class="wfb-tab-content" id="tab-schedule">
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="mametech_cb_schedule_enabled">Abilita Programmazione Orari</label>
                            </th>
                            <td>
                                <label class="wfb-switch">
                                    <input type="checkbox" name="mametech_cb_schedule_enabled" id="mametech_cb_schedule_enabled" value="1" 
                                           <?php checked(get_option('mametech_cb_schedule_enabled'), '1'); ?>>
                                    <span class="wfb-slider"></span>
                                </label>
                                <p class="description">Mostra il pulsante solo in determinati orari</p>
                            </td>
                        </tr>
                    </table>
                    
                    <div id="schedule-settings" style="<?php echo get_option('mametech_cb_schedule_enabled') !== '1' ? 'display:none;' : ''; ?>">
                        <h3>Imposta gli orari per ogni giorno</h3>
                        <?php
                        $schedule = get_option('mametech_cb_schedule_hours', array());
                        $days = array(
                            'monday' => 'Lunedì',
                            'tuesday' => 'Martedì',
                            'wednesday' => 'Mercoledì',
                            'thursday' => 'Giovedì',
                            'friday' => 'Venerdì',
                            'saturday' => 'Sabato',
                            'sunday' => 'Domenica'
                        );
                        
                        foreach ($days as $day => $label) {
                            $enabled = isset($schedule[$day]['enabled']) ? $schedule[$day]['enabled'] : false;
                            $start = isset($schedule[$day]['start']) ? $schedule[$day]['start'] : '09:00';
                            $end = isset($schedule[$day]['end']) ? $schedule[$day]['end'] : '18:00';
                            ?>
                            <div class="wfb-schedule-day">
                                <label>
                                    <input type="checkbox" name="mametech_cb_schedule_hours[<?php echo esc_attr($day); ?>][enabled]" value="1" 
                                           <?php checked($enabled, true); ?>>
                                    <strong><?php echo esc_html($label); ?></strong>
                                </label>
                                <label>
                                    Dalle: <input type="time" name="mametech_cb_schedule_hours[<?php echo esc_attr($day); ?>][start]" 
                                                  value="<?php echo esc_attr($start); ?>">
                                </label>
                                <label>
                                    Alle: <input type="time" name="mametech_cb_schedule_hours[<?php echo esc_attr($day); ?>][end]" 
                                                 value="<?php echo esc_attr($end); ?>">
                                </label>
                            </div>
                            <?php
                        }
                        ?>
                    </div>
                </div>
                
                <!-- Tab Avanzate -->
                <div class="wfb-tab-content" id="tab-advanced">
                    <h3>Shortcode e Widget</h3>
                    <table class="form-table">
                        <tr>
                            <th scope="row">Shortcode</th>
                            <td>
                                <code>[mametech_chat_button]</code>
                                <p class="description">
                                    Usa questo shortcode per inserire il pulsante manualmente in qualsiasi pagina o post.<br>
                                    Parametri opzionali:<br>
                                    - <code>phone="+393331234567"</code> - Numero specifico<br>
                                    - <code>message="Ciao!"</code> - Messaggio personalizzato<br>
                                    - <code>text="Contattaci"</code> - Testo del pulsante<br>
                                    - <code>size="medium"</code> - Dimensione (small, medium, large)
                                </p>
                                <p><strong>Esempio:</strong> <code>[mametech_chat_button phone="+393331234567" message="Vorrei informazioni" text="Scrivici su WhatsApp" size="large"]</code></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Widget</th>
                            <td>
                                <p class="description">
                                    Il widget "WhatsApp Button" è disponibile in <a href="<?php echo esc_url(admin_url('widgets.php')); ?>">Aspetto > Widget</a>
                                    per essere inserito in qualsiasi area widget del tuo tema.
                                </p>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <?php submit_button('Salva Impostazioni'); ?>
            </form>
            
            <!-- Anteprima -->
            <div class="wfb-preview-section">
                <h2>Anteprima</h2>
                <p>Il pulsante apparirà così sul tuo sito:</p>
                <div class="wfb-preview-container">
                    <div id="wfb-preview-button">
                        <svg viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg">
                            <path fill="currentColor" d="M16 0C7.164 0 0 7.164 0 16c0 2.829.747 5.482 2.051 7.792L.051 31.95l8.367-2.174A15.936 15.936 0 0016 32c8.836 0 16-7.164 16-16S24.836 0 16 0zm0 29.333c-2.453 0-4.789-.667-6.832-1.899l-.491-.293-5.109 1.328 1.328-5.109-.293-.491A13.268 13.268 0 012.667 16C2.667 8.636 8.636 2.667 16 2.667S29.333 8.636 29.333 16 23.364 29.333 16 29.333zm7.448-9.781c-.405-.203-2.4-1.184-2.773-1.32-.373-.136-.645-.203-.917.203-.272.405-1.053 1.32-1.291 1.593-.237.272-.475.305-.88.102-.405-.203-1.709-.629-3.255-2.008-1.203-1.072-2.016-2.396-2.253-2.801-.237-.405-.025-.624.177-.826.183-.183.405-.475.608-.713.203-.237.271-.405.407-.677.136-.272.068-.509-.034-.713-.102-.203-.917-2.208-1.256-3.025-.331-.797-.667-.688-.917-.701-.237-.013-.509-.017-.781-.017s-.713.102-1.087.509c-.373.405-1.423 1.391-1.423 3.396s1.457 3.937 1.66 4.209c.203.272 2.859 4.365 6.927 6.119.968.417 1.724.667 2.312.853.972.309 1.856.265 2.555.161.78-.116 2.4-.981 2.737-1.928.337-.947.337-1.759.237-1.928-.102-.169-.373-.272-.78-.475z"/>
                        </svg>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Pagina statistiche
     */
    public function stats_page() {
        global $wpdb;
        
        // Assicurati che la tabella esista
        $this->ensure_stats_table_exists();
        
        // Ottieni periodo selezionato
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $period = isset($_GET['period']) ? sanitize_text_field(wp_unslash($_GET['period'])) : '7days';
        
        // Calcola date
        $date_from = current_time('Y-m-d 00:00:00');
        switch($period) {
            case 'today':
                $date_from = current_time('Y-m-d 00:00:00');
                break;
            case '7days':
                $date_from = gmdate('Y-m-d 00:00:00', current_time('timestamp') - (7 * DAY_IN_SECONDS));
                break;
            case '30days':
                $date_from = gmdate('Y-m-d 00:00:00', current_time('timestamp') - (30 * DAY_IN_SECONDS));
                break;
            case 'all':
                $date_from = '1970-01-01 00:00:00';
                break;
        }
        
        // Query statistiche
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $total_clicks = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}mametech_cb_stats WHERE click_date >= %s",
                $date_from
            )
        );
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $clicks_by_day = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT DATE(click_date) as day, COUNT(*) as clicks 
                 FROM {$wpdb->prefix}mametech_cb_stats 
                 WHERE click_date >= %s 
                 GROUP BY DATE(click_date) 
                 ORDER BY day ASC",
                $date_from
            ),
            ARRAY_A
        );
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $top_pages = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT page_url, COUNT(*) as clicks 
                 FROM {$wpdb->prefix}mametech_cb_stats 
                 WHERE click_date >= %s 
                 GROUP BY page_url 
                 ORDER BY clicks DESC 
                 LIMIT 10",
                $date_from
            ),
            ARRAY_A
        );
        
        ?>
        <div class="wrap">
            <h1>Statistiche WhatsApp Button</h1>
            
            <div class="wfb-stats-filters">
                <a href="?page=mametech-chat-button-stats&period=today" 
                   class="button <?php echo $period === 'today' ? 'button-primary' : ''; ?>">Oggi</a>
                <a href="?page=mametech-chat-button-stats&period=7days" 
                   class="button <?php echo $period === '7days' ? 'button-primary' : ''; ?>">Ultimi 7 giorni</a>
                <a href="?page=mametech-chat-button-stats&period=30days" 
                   class="button <?php echo $period === '30days' ? 'button-primary' : ''; ?>">Ultimi 30 giorni</a>
                <a href="?page=mametech-chat-button-stats&period=all" 
                   class="button <?php echo $period === 'all' ? 'button-primary' : ''; ?>">Tutti</a>
            </div>
            
            <div class="wfb-stats-boxes">
                <div class="wfb-stat-box">
                    <h3>Click Totali</h3>
                    <div class="wfb-stat-number"><?php echo number_format($total_clicks); ?></div>
                </div>
            </div>
            
            <div class="wfb-chart-container">
                <h2>Click per Giorno</h2>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th>Data</th>
                            <th>Click</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($clicks_by_day)): ?>
                            <tr>
                                <td colspan="2">Nessun dato disponibile</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($clicks_by_day as $day_data): ?>
                                <tr>
                                    <td><?php echo esc_html($day_data['day']); ?></td>
                                    <td><?php echo number_format($day_data['clicks']); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            
            <div class="wfb-table-container">
                <h2>Pagine più Cliccate</h2>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th>URL Pagina</th>
                            <th>Click</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($top_pages)): ?>
                            <tr>
                                <td colspan="2">Nessun dato disponibile</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($top_pages as $page): ?>
                                <tr>
                                    <td><?php echo esc_html($page['page_url']); ?></td>
                                    <td><?php echo number_format($page['clicks']); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php
    }
    
    /**
     * Pagina numeri multipli
     */
    public function numbers_page() {
        $numbers = get_option('mametech_cb_multiple_numbers', array());
        $editing_index = -1;
        $success_message = '';
        
        // Gestione modifica
        if (isset($_GET['edit'])) {
            $editing_index = intval($_GET['edit']);
            // Se l'index non esiste, resetta
            if (!isset($numbers[$editing_index])) {
                $editing_index = -1;
            }
        }
        
        // Gestione salvataggio/aggiornamento
        if (isset($_POST['mametech_cb_save_number']) && check_admin_referer('mametech_cb_save_number_action', 'mametech_cb_save_number_nonce')) {
            $number_data = array(
                'name' => isset($_POST['number_name']) ? sanitize_text_field(wp_unslash($_POST['number_name'])) : '',
                'phone' => isset($_POST['number_phone']) ? $this->sanitize_phone(sanitize_text_field(wp_unslash($_POST['number_phone']))) : '',
                'department' => isset($_POST['number_department']) ? sanitize_text_field(wp_unslash($_POST['number_department'])) : '',
                'message' => isset($_POST['number_message']) ? sanitize_text_field(wp_unslash($_POST['number_message'])) : ''
            );
            
            if (!empty($number_data['name']) && !empty($number_data['phone'])) {
                $edit_index = isset($_POST['edit_index']) ? intval($_POST['edit_index']) : -1;
                
                if ($edit_index >= 0 && isset($numbers[$edit_index])) {
                    // Modifica numero esistente
                    $numbers[$edit_index] = $number_data;
                    $success_message = 'Numero aggiornato con successo!';
                } else {
                    // Aggiungi nuovo numero
                    $numbers[] = $number_data;
                    $success_message = 'Numero aggiunto con successo!';
                }
                
                update_option('mametech_cb_multiple_numbers', $numbers);
                $editing_index = -1; // Reset modalità edit dopo salvataggio
                
                // Ricarica i numeri aggiornati
                $numbers = get_option('mametech_cb_multiple_numbers', array());
            }
        }
        
        // Gestione eliminazione
        if (isset($_GET['delete']) && check_admin_referer('mametech_cb_delete_number_' . intval($_GET['delete']))) {
            $index = intval($_GET['delete']);
            if (isset($numbers[$index])) {
                unset($numbers[$index]);
                $numbers = array_values($numbers);
                update_option('mametech_cb_multiple_numbers', $numbers);
                $success_message = 'Numero eliminato!';
            }
        }
        
        // Dati per il form di modifica
        $form_data = array(
            'name' => '',
            'phone' => '',
            'department' => '',
            'message' => ''
        );
        
        if ($editing_index >= 0 && isset($numbers[$editing_index])) {
            $form_data = $numbers[$editing_index];
        }
        
        ?>
        <div class="wrap">
            <h1>Gestione Numeri Multipli</h1>
            
            <?php if (!empty($success_message)): ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php echo esc_html($success_message); ?></p>
                </div>
            <?php endif; ?>
            
            <p>Aggiungi più numeri WhatsApp per diversi reparti o persone. Gli utenti potranno scegliere chi contattare.</p>
            
            <?php if (!empty($numbers)): ?>
            <table class="wp-list-table widefat fixed striped" id="sortable-numbers">
                <thead>
                    <tr>
                        <th style="width: 30px;"></th>
                        <th>Nome</th>
                        <th>Numero</th>
                        <th>Reparto</th>
                        <th>Messaggio</th>
                        <th style="width: 120px;">Azioni</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($numbers as $index => $number): ?>
                    <tr data-id="<?php echo esc_attr($index); ?>">
                        <td class="handle" style="cursor: move;">☰</td>
                        <td><?php echo esc_html($number['name']); ?></td>
                        <td><?php echo esc_html($number['phone']); ?></td>
                        <td><?php echo esc_html($number['department']); ?></td>
                        <td><?php echo esc_html($number['message']); ?></td>
                        <td>
                            <a href="<?php echo esc_url(admin_url('admin.php?page=mametech-chat-button-numbers&edit=' . $index)); ?>" 
                               class="button button-small">
                                Modifica
                            </a>
                            <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=mametech-chat-button-numbers&delete=' . $index), 'mametech_cb_delete_number_' . $index)); ?>" 
                               class="button button-small"
                               onclick="return confirm('Sei sicuro di voler eliminare questo numero?');">
                                Elimina
                            </a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; ?>
            
            <div class="wfb-add-number-form">
                <h2><?php echo $editing_index >= 0 ? 'Modifica Numero' : 'Aggiungi Nuovo Numero'; ?></h2>
                <?php if ($editing_index >= 0): ?>
                    <p><a href="<?php echo esc_url(admin_url('admin.php?page=mametech-chat-button-numbers')); ?>" class="button">← Annulla Modifica</a></p>
                <?php endif; ?>
                <form method="post" action="">
                    <?php wp_nonce_field('mametech_cb_save_number_action', 'mametech_cb_save_number_nonce'); ?>
                    <?php if ($editing_index >= 0): ?>
                        <input type="hidden" name="edit_index" value="<?php echo esc_attr($editing_index); ?>">
                    <?php endif; ?>
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="number_name">Nome *</label></th>
                            <td><input type="text" id="number_name" name="number_name" class="regular-text" value="<?php echo esc_attr($form_data['name']); ?>" required></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="number_phone">Numero WhatsApp *</label></th>
                            <td>
                                <input type="text" id="number_phone" name="number_phone" class="regular-text" placeholder="+393331234567" value="<?php echo esc_attr($form_data['phone']); ?>" required>
                                <p class="description">Formato internazionale: +[prefisso][numero]</p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="number_department">Reparto</label></th>
                            <td><input type="text" id="number_department" name="number_department" class="regular-text" placeholder="Es: Vendite, Supporto" value="<?php echo esc_attr($form_data['department']); ?>"></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="number_message">Messaggio predefinito</label></th>
                            <td><input type="text" id="number_message" name="number_message" class="regular-text" value="<?php echo esc_attr($form_data['message']); ?>"></td>
                        </tr>
                    </table>
                    <?php submit_button($editing_index >= 0 ? 'Aggiorna Numero' : 'Aggiungi Numero', 'primary', 'mametech_cb_save_number'); ?>
                </form>
            </div>
        </div>
        <?php
    }
    
    /**
     * Salva numeri multipli (AJAX)
     */
    public function save_multiple_numbers() {
        check_ajax_referer('mametech_cb_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permessi insufficienti'));
        }
        
        $number = isset($_POST['number']) ? array_map('sanitize_text_field', wp_unslash($_POST['number'])) : array();
        
        if (empty($number['name']) || empty($number['phone'])) {
            wp_send_json_error(array('message' => 'Nome e numero sono obbligatori'));
        }
        
        $numbers = get_option('mametech_cb_multiple_numbers', array());
        $numbers[] = array(
            'name' => sanitize_text_field($number['name']),
            'phone' => $this->sanitize_phone($number['phone']),
            'department' => sanitize_text_field($number['department']),
            'message' => sanitize_text_field($number['message'])
        );
        
        update_option('mametech_cb_multiple_numbers', $numbers);
        
        wp_send_json_success();
    }
    
    /**
     * Elimina numero (AJAX)
     */
    public function delete_number() {
        check_ajax_referer('mametech_cb_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error();
        }
        
        $index = isset($_POST['index']) ? intval($_POST['index']) : -1;
        
        $numbers = get_option('mametech_cb_multiple_numbers', array());
        
        if (isset($numbers[$index])) {
            unset($numbers[$index]);
            $numbers = array_values($numbers); // Reindex array
            update_option('mametech_cb_multiple_numbers', $numbers);
            wp_send_json_success();
        }
        
        wp_send_json_error();
    }
    
    /**
     * Traccia click (AJAX)
     */
    public function track_click() {
        check_ajax_referer('mametech_cb_track_nonce', 'nonce');
        
        if (get_option('mametech_cb_enable_stats') !== '1') {
            wp_send_json_success();
            return;
        }
        
        global $wpdb;
        
        // Assicurati che la tabella esista
        $this->ensure_stats_table_exists();
        
        $table_stats = $wpdb->prefix . 'mametech_cb_stats';
        
        $button_id = isset($_POST['button_id']) ? sanitize_text_field(wp_unslash($_POST['button_id'])) : 'main';
        $page_url = isset($_POST['page_url']) ? esc_url_raw(wp_unslash($_POST['page_url'])) : '';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $wpdb->insert(
            $table_stats,
            array(
                'button_id' => $button_id,
                'user_ip' => $this->get_user_ip(),
                'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '',
                'page_url' => $page_url,
                'click_date' => current_time('mysql')
            ),
            array('%s', '%s', '%s', '%s', '%s')
        );
        
        wp_send_json_success();
    }
    
    /**
     * Ottieni IP utente
     */
    private function get_user_ip() {
        $ip = '';
        
        if (isset($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = sanitize_text_field(wp_unslash($_SERVER['HTTP_CLIENT_IP']));
        } elseif (isset($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = sanitize_text_field(wp_unslash($_SERVER['HTTP_X_FORWARDED_FOR']));
        } elseif (isset($_SERVER['REMOTE_ADDR'])) {
            $ip = sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR']));
        }
        
        return sanitize_text_field($ip);
    }
    
    /**
     * Registra widget
     */
    public function register_widget() {
        register_widget('MameTech_CB_Widget');
    }
    
    /**
     * Shortcode pulsante
     */
    public function shortcode_button($atts) {
        $atts = shortcode_atts(array(
            'phone' => get_option('mametech_cb_phone_number'),
            'message' => get_option('mametech_cb_message'),
            'text' => 'Contattaci su WhatsApp',
            'size' => 'medium'
        ), $atts);
        
        if (empty($atts['phone'])) {
            return '';
        }
        
        $whatsapp_url = 'https://wa.me/' . $atts['phone'];
        if (!empty($atts['message'])) {
            $whatsapp_url .= '?text=' . urlencode($atts['message']);
        }
        
        $size_class = 'wfb-btn-' . $atts['size'];
        
        $output = '<a href="' . esc_url($whatsapp_url) . '" class="wfb-shortcode-button ' . esc_attr($size_class) . '" target="_blank" rel="noopener noreferrer">';
        $output .= '<svg viewBox="0 0 32 32" class="wfb-icon">';
        $output .= '<path fill="currentColor" d="M16 0C7.164 0 0 7.164 0 16c0 2.829.747 5.482 2.051 7.792L.051 31.95l8.367-2.174A15.936 15.936 0 0016 32c8.836 0 16-7.164 16-16S24.836 0 16 0zm0 29.333c-2.453 0-4.789-.667-6.832-1.899l-.491-.293-5.109 1.328 1.328-5.109-.293-.491A13.268 13.268 0 012.667 16C2.667 8.636 8.636 2.667 16 2.667S29.333 8.636 29.333 16 23.364 29.333 16 29.333zm7.448-9.781c-.405-.203-2.4-1.184-2.773-1.32-.373-.136-.645-.203-.917.203-.272.405-1.053 1.32-1.291 1.593-.237.272-.475.305-.88.102-.405-.203-1.709-.629-3.255-2.008-1.203-1.072-2.016-2.396-2.253-2.801-.237-.405-.025-.624.177-.826.183-.183.405-.475.608-.713.203-.237.271-.405.407-.677.136-.272.068-.509-.034-.713-.102-.203-.917-2.208-1.256-3.025-.331-.797-.667-.688-.917-.701-.237-.013-.509-.017-.781-.017s-.713.102-1.087.509c-.373.405-1.423 1.391-1.423 3.396s1.457 3.937 1.66 4.209c.203.272 2.859 4.365 6.927 6.119.968.417 1.724.667 2.312.853.972.309 1.856.265 2.555.161.78-.116 2.4-.981 2.737-1.928.337-.947.337-1.759.237-1.928-.102-.169-.373-.272-.78-.475z"/>';
        $output .= '</svg>';
        $output .= '<span>' . esc_html($atts['text']) . '</span>';
        $output .= '</a>';
        
        return $output;
    }
    
    /**
     * Aggiungi il pulsante WhatsApp al frontend
     */
    public function add_whatsapp_button() {
        if (!$this->should_display_button()) {
            return;
        }
        
        $phone = get_option('mametech_cb_phone_number');
        
        if (empty($phone)) {
            return;
        }
        
        $message = get_option('mametech_cb_message', 'Ciao! Come posso aiutarti?');
        $position = get_option('mametech_cb_position', 'bottom-right');
        $size = get_option('mametech_cb_button_size', '60');
        $theme = get_option('mametech_cb_theme', 'default');
        $show_tooltip = get_option('mametech_cb_show_tooltip', '1');
        $tooltip_text = get_option('mametech_cb_tooltip_text', 'Chatta con noi!');
        
        $whatsapp_url = 'https://wa.me/' . $phone;
        if (!empty($message)) {
            $whatsapp_url .= '?text=' . urlencode($message);
        }
        
        $position_class = 'wfb-position-' . $position;
        $theme_class = 'wfb-theme-' . $theme;
        
        ?>
        <div class="wfb-floating-button <?php echo esc_attr($position_class . ' ' . $theme_class); ?>" 
             data-button-id="main">
            <a href="<?php echo esc_url($whatsapp_url); ?>" 
               target="_blank" 
               rel="noopener noreferrer"
               aria-label="Contattaci su WhatsApp"
               class="wfb-button-link">
                <svg viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg">
                    <path fill="currentColor" d="M16 0C7.164 0 0 7.164 0 16c0 2.829.747 5.482 2.051 7.792L.051 31.95l8.367-2.174A15.936 15.936 0 0016 32c8.836 0 16-7.164 16-16S24.836 0 16 0zm0 29.333c-2.453 0-4.789-.667-6.832-1.899l-.491-.293-5.109 1.328 1.328-5.109-.293-.491A13.268 13.268 0 012.667 16C2.667 8.636 8.636 2.667 16 2.667S29.333 8.636 29.333 16 23.364 29.333 16 29.333zm7.448-9.781c-.405-.203-2.4-1.184-2.773-1.32-.373-.136-.645-.203-.917.203-.272.405-1.053 1.32-1.291 1.593-.237.272-.475.305-.88.102-.405-.203-1.709-.629-3.255-2.008-1.203-1.072-2.016-2.396-2.253-2.801-.237-.405-.025-.624.177-.826.183-.183.405-.475.608-.713.203-.237.271-.405.407-.677.136-.272.068-.509-.034-.713-.102-.203-.917-2.208-1.256-3.025-.331-.797-.667-.688-.917-.701-.237-.013-.509-.017-.781-.017s-.713.102-1.087.509c-.373.405-1.423 1.391-1.423 3.396s1.457 3.937 1.66 4.209c.203.272 2.859 4.365 6.927 6.119.968.417 1.724.667 2.312.853.972.309 1.856.265 2.555.161.78-.116 2.4-.981 2.737-1.928.337-.947.337-1.759.237-1.928-.102-.169-.373-.272-.78-.475z"/>
                </svg>
            </a>
            <?php if ($show_tooltip === '1'): ?>
                <div class="wfb-tooltip"><?php echo esc_html($tooltip_text); ?></div>
            <?php endif; ?>
        </div>
        <?php
    }
}

/**
 * Widget WhatsApp Button
 */
class MameTech_CB_Widget extends WP_Widget {
    
    public function __construct() {
        parent::__construct(
            'wfb_widget',
            'Floating Chat Button',
            array('description' => 'Aggiungi un pulsante chat nella sidebar')
        );
    }
    
    public function widget($args, $instance) {
        $title = !empty($instance['title']) ? $instance['title'] : '';
        $phone = !empty($instance['phone']) ? $instance['phone'] : get_option('mametech_cb_phone_number');
        $message = !empty($instance['message']) ? $instance['message'] : get_option('mametech_cb_message');
        $button_text = !empty($instance['button_text']) ? $instance['button_text'] : 'Scrivici su WhatsApp';
        
        if (empty($phone)) {
            return;
        }
        
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        echo $args['before_widget'];
        
        if (!empty($title)) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            echo $args['before_title'] . esc_html($title) . $args['after_title'];
        }
        
        $whatsapp_url = 'https://wa.me/' . $phone;
        if (!empty($message)) {
            $whatsapp_url .= '?text=' . urlencode($message);
        }
        
        ?>
        <div class="wfb-widget-content">
            <a href="<?php echo esc_url($whatsapp_url); ?>" 
               class="wfb-widget-button" 
               target="_blank" 
               rel="noopener noreferrer">
                <svg viewBox="0 0 32 32" class="wfb-widget-icon">
                    <path fill="currentColor" d="M16 0C7.164 0 0 7.164 0 16c0 2.829.747 5.482 2.051 7.792L.051 31.95l8.367-2.174A15.936 15.936 0 0016 32c8.836 0 16-7.164 16-16S24.836 0 16 0zm0 29.333c-2.453 0-4.789-.667-6.832-1.899l-.491-.293-5.109 1.328 1.328-5.109-.293-.491A13.268 13.268 0 012.667 16C2.667 8.636 8.636 2.667 16 2.667S29.333 8.636 29.333 16 23.364 29.333 16 29.333zm7.448-9.781c-.405-.203-2.4-1.184-2.773-1.32-.373-.136-.645-.203-.917.203-.272.405-1.053 1.32-1.291 1.593-.237.272-.475.305-.88.102-.405-.203-1.709-.629-3.255-2.008-1.203-1.072-2.016-2.396-2.253-2.801-.237-.405-.025-.624.177-.826.183-.183.405-.475.608-.713.203-.237.271-.405.407-.677.136-.272.068-.509-.034-.713-.102-.203-.917-2.208-1.256-3.025-.331-.797-.667-.688-.917-.701-.237-.013-.509-.017-.781-.017s-.713.102-1.087.509c-.373.405-1.423 1.391-1.423 3.396s1.457 3.937 1.66 4.209c.203.272 2.859 4.365 6.927 6.119.968.417 1.724.667 2.312.853.972.309 1.856.265 2.555.161.78-.116 2.4-.981 2.737-1.928.337-.947.337-1.759.237-1.928-.102-.169-.373-.272-.78-.475z"/>
                </svg>
                <span><?php echo esc_html($button_text); ?></span>
            </a>
        </div>
        <?php
        
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        echo $args['after_widget'];
    }
    
    public function form($instance) {
        $title = !empty($instance['title']) ? $instance['title'] : '';
        $phone = !empty($instance['phone']) ? $instance['phone'] : '';
        $message = !empty($instance['message']) ? $instance['message'] : '';
        $button_text = !empty($instance['button_text']) ? $instance['button_text'] : 'Scrivici su WhatsApp';
        ?>
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('title')); ?>">Titolo:</label>
            <input class="widefat" id="<?php echo esc_attr($this->get_field_id('title')); ?>" 
                   name="<?php echo esc_attr($this->get_field_name('title')); ?>" type="text" 
                   value="<?php echo esc_attr($title); ?>">
        </p>
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('phone')); ?>">Numero (lascia vuoto per usare quello principale):</label>
            <input class="widefat" id="<?php echo esc_attr($this->get_field_id('phone')); ?>" 
                   name="<?php echo esc_attr($this->get_field_name('phone')); ?>" type="text" 
                   value="<?php echo esc_attr($phone); ?>" placeholder="+393331234567">
        </p>
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('message')); ?>">Messaggio (lascia vuoto per usare quello principale):</label>
            <textarea class="widefat" id="<?php echo esc_attr($this->get_field_id('message')); ?>" 
                      name="<?php echo esc_attr($this->get_field_name('message')); ?>" rows="3"><?php echo esc_textarea($message); ?></textarea>
        </p>
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('button_text')); ?>">Testo Pulsante:</label>
            <input class="widefat" id="<?php echo esc_attr($this->get_field_id('button_text')); ?>" 
                   name="<?php echo esc_attr($this->get_field_name('button_text')); ?>" type="text" 
                   value="<?php echo esc_attr($button_text); ?>">
        </p>
        <?php
    }
    
    public function update($new_instance, $old_instance) {
        $instance = array();
        $instance['title'] = (!empty($new_instance['title'])) ? sanitize_text_field($new_instance['title']) : '';
        $instance['phone'] = (!empty($new_instance['phone'])) ? sanitize_text_field($new_instance['phone']) : '';
        $instance['message'] = (!empty($new_instance['message'])) ? sanitize_text_field($new_instance['message']) : '';
        $instance['button_text'] = (!empty($new_instance['button_text'])) ? sanitize_text_field($new_instance['button_text']) : '';
        
        return $instance;
    }
}

// Inizializza il plugin
if ( ! function_exists( 'mametech_chat_button_init' ) ) {
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedFunctionFound
    function mametech_chat_button_init() {
        new MameTech_Chat_Button();
    }
}
add_action( 'plugins_loaded', 'mametech_chat_button_init' );
