<?php

/**
 * Silence is golden.
 *
 * Prevents direct access and directory listing.
 */
defined('ABSPATH') || exit;

final class MAKARHQ_Plugin
{
    const OPTION_KEY = 'makarhq_options';
    const NONCE_ACTION = 'makarhq_validate';
    const NONCE_NAME = 'makarhq_nonce';

    private static $instance = null;

    public static function instance()
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        // Defaults on activate
        register_activation_hook(__FILE__, [__CLASS__, 'activate']);

        // Admin settings
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_menu', [$this, 'add_settings_page']);

        // Frontend hooks
        add_action('login_form', [$this, 'render_login_widget']);
        add_filter('authenticate', [$this, 'validate_on_login'], 30, 3);

        // Generic POST validator for shortcode-embedded forms
        add_action('init', [$this, 'maybe_validate_generic_post']);

        // Shortcode for custom forms
        add_shortcode('makarhq_human_check', [$this, 'shortcode']);

        // Styling (very small)
        add_action('login_enqueue_scripts', [$this, 'enqueue_styles']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_styles']);
    }

    public static function activate()
    {
        $defaults = [
            'enabled_login' => 1,
            'enabled_shortcode' => 1,
            'min_number' => 3,
            'max_number' => 12,
            'ops' => ['+', '-', '×'], // use × for display; map to * in code
            'time_limit' => 120, // seconds
            'division_tolerance' => 0, // integers only by default
            'error_message' => __('Please solve the human check correctly.', 'makuruwan-arithmetic-human-quiz'),
            'accessibility_hint' => __('Solve the arithmetic question below to continue.', 'makuruwan-arithmetic-human-quiz'),
            'bypass_roles' => [],
        ];
        $current = get_option(self::OPTION_KEY);
        if (!is_array($current)) {
            add_option(self::OPTION_KEY, $defaults);
        } else {
            update_option(self::OPTION_KEY, wp_parse_args($current, $defaults));
        }
    }

    // ===== Settings =====
    public function register_settings()
    {
        register_setting(self::OPTION_KEY, self::OPTION_KEY, [$this, 'sanitize_options']);

        add_settings_section('makarhq_main', __('Human Quiz Settings', 'makuruwan-arithmetic-human-quiz'), function () {
            echo '<p>' . esc_html__('Configure the arithmetic challenge used to protect login and custom forms.', 'makuruwan-arithmetic-human-quiz') . '</p>';
        }, self::OPTION_KEY);

        $fields = [
            ['enabled_login', __('Protect WP Login', 'makuruwan-arithmetic-human-quiz'), 'checkbox'],
            ['enabled_shortcode', __('Enable Shortcode [makarhq_human_check]', 'makuruwan-arithmetic-human-quiz'), 'checkbox'],
            ['min_number', __('Minimum number', 'makuruwan-arithmetic-human-quiz'), 'number'],
            ['max_number', __('Maximum number', 'makuruwan-arithmetic-human-quiz'), 'number'],
            ['ops', __('Operations', 'makuruwan-arithmetic-human-quiz'), 'ops'],
            ['time_limit', __('Time limit (seconds)', 'makuruwan-arithmetic-human-quiz'), 'number'],
            ['division_tolerance', __('Division tolerance (± allowed error)', 'makuruwan-arithmetic-human-quiz'), 'number'],
            ['error_message', __('Error message', 'makuruwan-arithmetic-human-quiz'), 'text'],
            ['accessibility_hint', __('Accessibility hint (ARIA description)', 'makuruwan-arithmetic-human-quiz'), 'text'],
        ];

        foreach ($fields as $f) {
            add_settings_field($f[0], $f[1], [$this, 'render_field'], self::OPTION_KEY, 'makarhq_main', ['key' => $f[0], 'type' => $f[2]]);
        }
    }

    public function add_settings_page()
    {
        add_options_page(
            __('Arithmetic Human Quiz', 'makuruwan-arithmetic-human-quiz'),
            __('Human Quiz', 'makuruwan-arithmetic-human-quiz'),
            'manage_options',
            self::OPTION_KEY,
            [$this, 'render_settings_page']
        );
    }

    public function render_settings_page()
    {
        echo '<div class="wrap">';
        echo '<h1>' . esc_html__('Arithmetic Human Quiz', 'makuruwan-arithmetic-human-quiz') . '</h1>';
        echo '<form method="post" action="options.php">';
        settings_fields(self::OPTION_KEY);
        do_settings_sections(self::OPTION_KEY);
        submit_button();
        echo '</form>';
        echo '<hr/><p><strong>[makarhq_human_check]</strong> ' . esc_html__('Use this shortcode inside any custom form. Ensure your form submits the quiz fields by including the shortcode inside the <form> tag.', 'makuruwan-arithmetic-human-quiz') . '</p>';
        echo '</div>';
    }

    public function render_field($args)
    {
        $opts = $this->get_options();
        $key = $args['key'];
        $type = $args['type'];
        $val = isset($opts[$key]) ? $opts[$key] : '';
        switch ($type) {
            case 'checkbox':
                printf(
                    '<label><input type="checkbox" name="%1$s[%2$s]" value="1" %3$s /> %4$s</label>',
                    esc_attr(self::OPTION_KEY),
                    esc_attr($key),
                    checked($val, 1, false),
                    esc_html__('Enabled', 'makuruwan-arithmetic-human-quiz')
                );
                break;
            case 'number':
                printf('<input type="number" class="small-text" name="%1$s[%2$s]" value="%3$s" />', esc_attr(self::OPTION_KEY), esc_attr($key), esc_attr($val));
                break;
            case 'text':
                printf('<input type="text" class="regular-text" name="%1$s[%2$s]" value="%3$s" />', esc_attr(self::OPTION_KEY), esc_attr($key), esc_attr($val));
                break;
            case 'ops':
                $ops = ['+' => '+', '-' => '−', '×' => '×', '÷' => '÷'];
                echo '<fieldset>';
                foreach ($ops as $code => $label) {
                    $is_checked = is_array($val) && in_array($code, $val, true);
                    printf(
                        '<label style="margin-right:12px"><input type="checkbox" name="%1$s[ops][]" value="%2$s" %3$s /> %4$s</label>',
                        esc_attr(self::OPTION_KEY),
                        esc_attr($code),
                        checked($is_checked, true, false),
                        esc_html($label)
                    );
                }
                echo '</fieldset>';
                break;
        }
    }

    public function sanitize_options($input)
    {
        $out = $this->get_options();
        $out['enabled_login'] = empty($input['enabled_login']) ? 0 : 1;
        $out['enabled_shortcode'] = empty($input['enabled_shortcode']) ? 0 : 1;
        $out['min_number'] = isset($input['min_number']) ? max(0, intval($input['min_number'])) : 0;
        $out['max_number'] = isset($input['max_number']) ? max($out['min_number'], intval($input['max_number'])) : $out['min_number'];
        $allowed_ops = ['+', '-', '×', '÷'];
        $out['ops'] = isset($input['ops']) && is_array($input['ops']) ? array_values(array_intersect($allowed_ops, array_map('sanitize_text_field', $input['ops']))) : ['+'];
        $out['time_limit'] = isset($input['time_limit']) ? max(5, intval($input['time_limit'])) : 120;
        $out['division_tolerance'] = isset($input['division_tolerance']) ? max(0, intval($input['division_tolerance'])) : 0;
        $out['error_message'] = isset($input['error_message']) ? sanitize_text_field($input['error_message']) : __('Please solve the human check correctly.', 'makuruwan-arithmetic-human-quiz');
        $out['accessibility_hint'] = isset($input['accessibility_hint']) ? sanitize_text_field($input['accessibility_hint']) : '';
        return $out;
    }

    private function get_options()
    {
        $opts = get_option(self::OPTION_KEY, []);
        if (!is_array($opts))
            $opts = [];
        return $opts;
    }

    // ===== Rendering & Validation =====
    public function enqueue_styles()
    {
        $css = '.makarhq-box{margin-top:12px;padding:12px;border:1px solid #ddd;border-radius:6px;background:#fff}.makarhq-row{display:flex;align-items:center;gap:8px}.makarhq-q{font-weight:600}.makarhq-input{width:90px}.makarhq-note{font-size:12px;color:#666;margin-top:6px}';
        wp_register_style('makarhq-inline', false, [], '1.0.1');
        wp_enqueue_style('makarhq-inline');
        wp_add_inline_style('makarhq-inline', $css);
    }

    /**
     * Display on WP login form
     */
    public function render_login_widget()
    {
        $opts = $this->get_options();
        if (empty($opts['enabled_login'])) {
            return;
        }
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- HTML is escaped within render_widget_html()
        echo $this->render_widget_html('login');
    }

    /**
     * Shortcode renderer for custom forms.
     * Ensure this lives inside the <form> markup so fields are submitted.
     */
    public function shortcode($atts = [])
    {
        $opts = $this->get_options();
        if (empty($opts['enabled_shortcode']))
            return '';
        return $this->render_widget_html('shortcode');
    }

    private function render_widget_html($context = 'generic')
    {
        $opts = $this->get_options();
        $challenge = $this->generate_challenge($opts);
        $state = $this->pack_state($challenge);
        $label_id = 'makarhq_answer_' . wp_generate_uuid4();
        ob_start();
        ?>
        <div class="makarhq-box" role="group"
            aria-label="<?php echo esc_attr__('Human verification', 'makuruwan-arithmetic-human-quiz'); ?>">
            <p class="screen-reader-text"><?php echo esc_html($opts['accessibility_hint']); ?></p>
            <div class="makarhq-row">
                <span class="makarhq-q" aria-hidden="true"><?php echo esc_html($challenge['display']); ?></span>
                <label class="screen-reader-text"
                    for="<?php echo esc_attr($label_id); ?>"><?php esc_html_e('Answer to the arithmetic question', 'makuruwan-arithmetic-human-quiz'); ?></label>
                <input id="<?php echo esc_attr($label_id); ?>" class="input makarhq-input" type="text" name="makarhq_answer"
                    inputmode="numeric" autocomplete="off" required />
            </div>
            <input type="hidden" name="makarhq_state" value="<?php echo esc_attr($state['state']); ?>" />
            <input type="hidden" name="makarhq_sig" value="<?php echo esc_attr($state['sig']); ?>" />
            <input type="hidden" name="makarhq_present" value="1" />
            <?php wp_nonce_field(self::NONCE_ACTION, self::NONCE_NAME); ?>
            <div class="makarhq-note">
                <?php echo esc_html__('This quick question helps us keep bots out.', 'makuruwan-arithmetic-human-quiz'); ?>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    private function generate_challenge($opts)
    {
        $min = max(0, (int) $opts['min_number']);
        $max = max($min, (int) $opts['max_number']);
        $ops = !empty($opts['ops']) ? $opts['ops'] : ['+'];
        $op = $ops[array_rand($ops)];

        $a = random_int($min, $max);
        $b = random_int($min, $max);

        // avoid negative results for subtraction (optional UX nicety)
        if ($op === '-' && $a < $b) {
            [$a, $b] = [$b, $a];
        }

        // avoid fractions for division when tolerance is 0
        if ($op === '÷') {
            if (intval($opts['division_tolerance']) === 0) {
                // ensure divisible
                $b = max(1, random_int($min ?: 1, max(1, $max)));
                $a = $b * random_int($min, $max);
            } else {
                $b = max(1, $b);
            }
        }

        $display = sprintf('%d %s %d = ?', $a, $op, $b);
        $ts = time();

        return [
            'a' => $a,
            'b' => $b,
            'op' => $op,
            'display' => $display,
            'ts' => $ts,
            'limit' => (int) $opts['time_limit'],
        ];
    }

    private function pack_state($challenge)
    {
        $json = wp_json_encode($challenge);
        $state = rtrim(strtr(base64_encode($json), '+/', '-_'), '=');
        $sig = $this->sign($state);
        return ['state' => $state, 'sig' => $sig];
    }

    private function unpack_state($state, $sig)
    {
        if (!$this->verify($state, $sig))
            return null;
        $json = base64_decode(strtr($state, '-_', '+/'));
        $data = json_decode($json, true);
        return is_array($data) ? $data : null;
    }

    private function sign($state)
    {
        $key = defined('AUTH_SALT') ? AUTH_SALT : wp_salt('auth');
        return hash_hmac('sha256', $state, $key);
    }

    private function verify($state, $sig)
    {
        if (!$state || !$sig)
            return false;
        $calc = $this->sign($state);
        return hash_equals($calc, $sig);
    }

    private function compute_answer($a, $b, $op)
    {
        switch ($op) {
            case '+':
                return $a + $b;
            case '-':
                return $a - $b;
            case '×':
                return $a * $b;
            case '÷':
                return $b == 0 ? INF : $a / $b;
        }
        return null;
    }

    private function validate_request(&$error_out = '')
    {
        if (empty($_POST['makarhq_present']))
            return true; // Widget not present; allow pass-through

        // Nonce
        if (!isset($_POST[self::NONCE_NAME]) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST[self::NONCE_NAME])), self::NONCE_ACTION)) {
            $error_out = __('Security check failed. Please try again.', 'makuruwan-arithmetic-human-quiz');
            return false;
        }

        $state = isset($_POST['makarhq_state']) ? sanitize_text_field(wp_unslash($_POST['makarhq_state'])) : '';
        $sig = isset($_POST['makarhq_sig']) ? sanitize_text_field(wp_unslash($_POST['makarhq_sig'])) : '';
        $answer_raw = isset($_POST['makarhq_answer'])
            ? sanitize_text_field(wp_unslash($_POST['makarhq_answer']))
            : '';
        $answer_raw = trim($answer_raw);


        if ($state === '' || $sig === '' || $answer_raw === '') {
            $error_out = __('Please answer the human verification question.', 'makuruwan-arithmetic-human-quiz');
            return false;
        }

        $data = $this->unpack_state($state, $sig);
        if (!$data) {
            $error_out = __('Session expired. Please try again.', 'makuruwan-arithmetic-human-quiz');
            return false;
        }

        // Time window
        $now = time();
        $limit = isset($data['limit']) ? intval($data['limit']) : 120;
        if (($now - intval($data['ts'])) > $limit) {
            $error_out = __('Time limit exceeded. Please try again.', 'makuruwan-arithmetic-human-quiz');
            return false;
        }

        // Compute expected answer
        $expected = $this->compute_answer(intval($data['a']), intval($data['b']), $data['op']);
        $opts = $this->get_options();
        $tol = ($data['op'] === '÷') ? intval($opts['division_tolerance']) : 0;

        // Normalize answer: allow comma/period, spaces
        $norm = str_replace([',', ' '], ['.', ''], $answer_raw);
        if (!is_numeric($norm)) {
            $error_out = $this->get_options()['error_message'];
            return false;
        }
        $given = (float) $norm;

        $ok = ($tol > 0) ? (abs($given - $expected) <= $tol) : ((string) $given === (string) (float) $expected && intval($given) == $expected);

        if (!$ok) {
            $error_out = $this->get_options()['error_message'];
            return false;
        }

        return true;
    }

    // ===== Login integration =====
    public function validate_on_login($user, $username, $password)
    {
        $opts = $this->get_options();
        if (empty($opts['enabled_login']))
            return $user;

        $error = '';
        if (!$this->validate_request($error)) {
            return new WP_Error('makarhq_failed', $error ?: $opts['error_message']);
        }
        return $user;
    }

    // ===== Generic POST integration for shortcode =====
    public function maybe_validate_generic_post()
    {
        if (empty($_POST) || empty($_POST['makarhq_present'])) {
            return;
        }
        // Verify nonce before any processing to satisfy scanners.
        if (
            !isset($_POST[self::NONCE_NAME]) ||
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST[self::NONCE_NAME])), self::NONCE_ACTION)
        ) {
            return; // or optionally set a transient error and bail.
        }

        $error = '';
        if (!$this->validate_request($error)) {
            set_transient('makarhq_last_error_' . $this->client_key(), $error, 60);
            do_action('makarhq_validation_failed', $error);
        } else {
            delete_transient('makarhq_last_error_' . $this->client_key());
            do_action('makarhq_validation_passed');
        }
    }

    private function client_key()
    {
        $ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '';
        $ua = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '';
        $ip = rest_is_ip_address($ip) ? $ip : '';
        return md5($ip . '|' . substr($ua, 0, 128));
    }
}