<?php
/*
 * Plugin Name: MakeMyBrand
 * Plugin URI: https://makemybrand.ai/
 * Description: Adds an infographic to new posts
 * Version: 1.03
 * Author: Make My Brand
 * Author URI: https://makemybrand.ai
 * License: GPLv2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 *
 * Copyright (C) 2024 MakeMyBrand.AI
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */
?>
<?php


if (!defined('ABSPATH')) exit; // Exit if accessed directly

function makemybrand_fetch_userinfo() {
    check_ajax_referer('makemybrand_fetch_userinfo_nonce', 'nonce');
    if (!current_user_can('edit_posts')) {
        wp_send_json_error('Unauthorized');
        return;
    }

    $user_token = get_option('makemybrand_user_authtoken');
    if (!$user_token) {
        wp_send_json_error('USER_TOKEN_NOT_FOUND');
        return;
    }

    $api_url = 'https://api.makemybrand.ai/user_info';
    $response = wp_remote_get($api_url, array(
        'headers' => array(
            'authorization' => $user_token,
        ),
        'timeout' => 30,
    ));
    if (is_wp_error($response)) {
        wp_send_json_error('API call failed');
        return;
    }
    $response_body = json_decode($response['body'], true);
    wp_send_json_success($response_body);
}
add_action('wp_ajax_makemybrand_fetch_userinfo', 'makemybrand_fetch_userinfo');

function makemybrand_poll_status($task_id, $user_token, $max_attempts = 20) {
    $api_url = 'https://api.makemybrand.ai/poll';
    $attempt = 0;
    set_time_limit(300); // Set to 300 seconds

    while ($attempt < $max_attempts) {
        $response = wp_remote_post($api_url, array(
            'method'    => 'POST',
            'body'      => wp_json_encode(array(
                'task_id' => $task_id
            )),
            'headers'   => array(
                'Content-Type'  => 'application/json',
                'authorization' => $user_token,
            ),
            'timeout'   => 30,
        ));

        if (is_wp_error($response)) {
            return array('success' => false, 'error' => 'API call failed');
        }

        $response_body = json_decode($response['body'], true);
        
        if (!isset($response_body['status'])) {
            return array('success' => false, 'error' => 'Invalid response format');
        }

        if ($response_body['status'] === 'success') {
            return array('success' => true, 'data' => $response_body);
        }

        if ($response_body['status'] === 'failed') {
            return array('success' => false, 'error' => 'Task failed');
        }

        $attempt++;
        if ($attempt < $max_attempts) {
            sleep(10); // Wait 10 seconds before next poll
        }
    }

    return array('success' => false, 'error' => 'Maximum polling attempts reached');
}

function makemybrand_add_infographic() {
    ini_set('log_errors', 1);

    check_ajax_referer('makemybrand_add_infographic_nonce', 'nonce');
    if (!current_user_can('edit_posts')) {
        wp_send_json_error('Unauthorized');
        return;
    }

    $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
    $template_id = isset($_POST['template_id']) ? sanitize_text_field(wp_unslash($_POST['template_id'])) : '';
    $user_token = get_option('makemybrand_user_authtoken');

    if (!$post_id || !$template_id || !$user_token) {
        wp_send_json_error('Missing required parameters');
        return;
    }

    $post = get_post($post_id);
    if (!$post) {
        wp_send_json_error('Invalid post ID');
        return;
    }
    
    $post_title = $post->post_title;
    $post_content = $post->post_content;

    $api_url = 'https://api.makemybrand.ai/generate';

    $response = wp_remote_post($api_url, array(
        'method'    => 'POST',
        'body'      => wp_json_encode(array(
            'post_id'    => $post_id,
            'template_id' => $template_id,
            'title'     => $post_title,
            'blogpost'   => $post_title . ' ' . wp_strip_all_tags($post_content),
            'from' => 'wordpress'
        )),
        'headers'   => array(
            'Content-Type'  => 'application/json',
            'authorization' => $user_token,
        ),
        'timeout'   => 120,
    ));

    if (is_wp_error($response)) {
        wp_send_json_error('API call failed');
        return;
    }

    $response_body = json_decode($response['body'], true);
    if (!isset($response_body['task_id'])) {
        wp_send_json_error('Invalid response format');
        return;
    }
    // Poll for the result
    $poll_result = makemybrand_poll_status($response_body['task_id'], $user_token);
    
    if (!$poll_result['success']) {
        wp_send_json_error($poll_result['error']);
        return;
    }

    $result_data = $poll_result['data'];
    if (!isset($result_data['result']) || !isset($result_data['result']['url'])) {
        wp_send_json_error('Missing URL in response');
        return;
    }

    $infographic = esc_url($result_data['result']['url']);

    if (isset($result_data['result']['design_id'])) {
        $design_id = $result_data['result']['design_id'];
    } else {
        $design_id = '';
    }
    

    $attachment_id = add_image_to_gallery_and_post($infographic, $post_id);

    if (is_wp_error($attachment_id)) {
        wp_send_json_error($attachment_id->get_error_message());
    } else {
        $image_src = wp_get_attachment_url($attachment_id);
        $response_data = [
            'imageUrl' => $image_src,
            'designId' => $design_id,
        ];
        wp_send_json_success($response_data);
    }
}
function add_image_to_gallery_and_post($image_url, $post_id) {
    // Download the image to a temporary location
    $temp_file = download_url($image_url);

    // Handle download errors
    if (is_wp_error($temp_file)) {
        return $temp_file;
    }
    if (!file_exists($temp_file)) {
        return new WP_Error('upload_error', 'Temporary file not found.');
    }
    // Get the file name and type
    $file_array = array(
        'name'     => basename(parse_url($image_url, PHP_URL_PATH)), // Extract file name without query params
        'tmp_name' => $temp_file, // Temporary file path
    );
    
    // Upload the image to the WordPress media library
    $attachment_id = media_handle_sideload($file_array, $post_id);

    // If there was an error, delete the temporary file and return the error
    if (is_wp_error($attachment_id)) {
        @unlink($temp_file);
        return $attachment_id;
    }

    // Generate metadata for the attachment and update the database
    $attachment_data = wp_generate_attachment_metadata($attachment_id, get_attached_file($attachment_id));
    wp_update_attachment_metadata($attachment_id, $attachment_data);

    // Get the attachment URL
    $image_src = wp_get_attachment_url($attachment_id);

    // Insert the image into the post content
    $post_content = get_post_field('post_content', $post_id);
    $post_content .= '<img src="' . esc_url($image_src) . '" alt="">';
    wp_update_post(array(
        'ID'           => $post_id,
        'post_content' => $post_content,
    ));

    return $attachment_id;
}
add_action('wp_ajax_makemybrand_add_infographic', 'makemybrand_add_infographic');