<?php
/*
 * Plugin Name: MakeMyBrand
 * Plugin URI: https://makemybrand.ai/
 * Description: Adds an infographic to new posts
 * Version: 1.03
 * Author: Make My Brand
 * Author URI: https://makemybrand.ai
 * License: GPLv2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 *
 * Copyright (C) 2024 MakeMyBrand.AI
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */
?>

<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

add_action('admin_menu', 'makemybrand_add_admin_menu');
add_action('admin_init', 'makemybrand_register_settings');

function makemybrand_register_settings() {
    if (isset($_POST['makemybrand_nonce'])) {
        $nonce = sanitize_text_field(wp_unslash($_POST['makemybrand_nonce']));
        if (!wp_verify_nonce($nonce, 'makemybrand_update_settings')) {
            return; // Nonce is invalid, do not proceed
        }
    }    
    register_setting(
        'makemybrand_options_group', 
        'makemybrand_user_authtoken', 
        array(
            'type'              => 'string',
            'sanitize_callback' => 'sanitize_text_field'
        )
    );
}

function makemybrand_add_admin_menu() {
    add_options_page('Make My Brand', 'Make My Brand', 'manage_options', 'makemybrand', 'makemybrand_render_settings_page');
}

function makemybrand_render_settings_page() {
    ?>
    <div class="wrap">
        <h1>Make My Brand Settings</h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('makemybrand_options_group');
            wp_nonce_field('makemybrand_update_settings', 'makemybrand_nonce');
            $token = get_option('makemybrand_user_authtoken');
            ?>
            <table class="form-table">
                <tr valign="top">
                    <th scope="row">Token</th>
                    <td>
                        <input style="width:100%" type="text" name="makemybrand_user_authtoken" value="<?php echo esc_attr($token); ?>" />
                        <p class="makemybrand-info-line"><a target="_blank" href='https://app.makemybrand.ai/login?redirect=token'>Click here</a> to find your token.</p>

                    </td>

                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}


// this is for updating token from the edit-post page (when user saves the token from the dialog)
add_action('wp_ajax_update_user_token', function () {   
    check_ajax_referer('makemybrand_update_token_nonce', 'nonce');

    // Check if the current user has permission to update the token
    if (!current_user_can('edit_posts')) {
        wp_send_json_error(array('message' => 'Permission denied'), 403);
    }

    // Get and sanitize the token
    if (isset($_POST['token'])) {
        $new_token = sanitize_text_field(wp_unslash($_POST['token']));

        $response = wp_remote_get('https://api.makemybrand.ai/user_info', array(
            'headers' => array(
                'authorization' => $new_token,
            ),
            'timeout' => 30,
        ));
        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => 'Invalid token'), 400);
            return;
        }
        // Update the token in the database
        update_option('makemybrand_user_authtoken', $new_token);

        // Return the updated token in the response
        wp_send_json_success(array('userinfo' => json_decode($response['body'], true)));
    } else {
        wp_send_json_error(array('message' => 'No token provided'), 400);
    }
});
